/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.profiler;

import org.apache.jetspeed.om.profile.*;
import org.apache.jetspeed.capability.CapabilityMap;
import org.apache.turbine.services.Service;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.DynamicURI;
import org.apache.jetspeed.util.MimeType;
import java.util.Iterator;

/**
 * <P>This interface is a facade for all profile related operations</P>
 *
 * @see org.apache.jetspeed.om.profile.Profile
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @author <a href="mailto:morciuch@apache.org">Mark Orciuch</a>
 * @version $Id: ProfilerService.java,v 1.17 2003/03/04 00:05:09 sgala Exp $
 */

public interface ProfilerService extends Service
{

    /** The name of this service */
    public String SERVICE_NAME = "Profiler";

    /**
     *  get the Profile object using the Rundata state and capability map
     * this is the mapping functionality of the profiler
     *
     * @param rundata The rundata object for the current request.
     * @param cm  The device capability map.
     * @return A Profile object if found by the manager or null.
     */
    public Profile getProfile(RunData rundata, CapabilityMap cm)
        throws ProfileException;

    /**
     * get the Profile object using the Rundata state and capability map
     * this is the mapping functionality of the profiler
     *
     * @param rundata The rundata object for the current request.
     * @return A new Profile object.
     */
    public Profile getProfile(RunData rundata)
                throws ProfileException;

    /**
     * get the Profile object using the Rundata state and capability map
     * this is the mapping functionality of the profiler
     *
     * @deprecated Do not use a profiler method based on MimeType
     * @param rundata the rundata object for the current request
     * @return a new Profile object
     */
    public Profile getProfile(RunData data, MimeType mt)
        throws ProfileException;

    /**
     *  get the Profile object using a profile locator
     *
     * @param locator The locator containing criteria describing the profile.
     * @return a new Profile object
     */
    public Profile getProfile(ProfileLocator locator)
        throws ProfileException;

    /**
     * Creates a dynamic URI
     *
     * @param rundata the rundata object for the current request
     * @param locator The description of the profile.
     * @return A new dynamic URI representing all profile parameters from the locator.
     */
    public DynamicURI makeDynamicURI( RunData data, ProfileLocator locator )
        throws ProfileException;

    /**
     * Creates a new Profile object that can be successfully managed by
     * the current Profiler implementation
     *
     * @return A new Profile object
     */
    public Profile createProfile();

    /**
     * Creates a new Profile object for a specific locator.
     *
     * @param locator The description of the profile.
     * @return A new Profile object
     */
    public Profile createProfile(ProfileLocator locator);

    /**
     * Creates a new ProfileLocator object that can be successfully managed by
     * the current Profiler implementation
     *
     * @return A new ProfileLocator object
     */
    public ProfileLocator createLocator();

    /**
     * Create a new profile. The profile parameter's document will be cloned.
     *
     * @param rundata The rundata object for the current request.
     * @param profile The description of the new profile to be created.
     * @return The newly created profile.
     */
    public Profile createProfile( RunData data, Profile profile )
        throws ProfileException;

    /** Create a new profile.
     *
     * @param rundata The rundata object for the current request.
     * @param profile The description of the new profile to be created.
     * @param mt The specific mime type, which is converted to a mediatype.
     * @return The newly created profile.
     */
    public Profile createProfile( RunData data, Profile profile, MimeType mt )
        throws ProfileException;

    /** Create a new profile.
     *
     * @param locator The description of the new profile to be created.
     * @param portlets The PSML tree
     */
    public Profile createProfile(ProfileLocator locator, Portlets portlets)
        throws ProfileException;

   /**
     *  Removes a profile.
     *
     * @param locator The profile locator criteria.
     */
    public void removeProfile( ProfileLocator locator )
        throws ProfileException;

    /** Query for a collection of profiles given a profile locator criteria.
     *
     * @param locator The profile locator criteria.
     * @return The list of profiles matching the locator criteria.
     */
    public Iterator query( QueryLocator locator );

    /**
     * Returns status of role profile merging feature
     *
     * @return True if role profile merging is active
     */
    public boolean useRoleProfileMerging();

}
