package org.apache.jetspeed.services.logging;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Turbine" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// Java classes
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.io.FileInputStream;

// Servlet API
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

// Turbine classes
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.resources.ResourceService;
import org.apache.turbine.services.resources.TurbineResources;
import org.apache.turbine.services.logging.LoggingService;
import org.apache.turbine.services.logging.LoggingConfig;
import org.apache.turbine.services.logging.Logger;
import org.apache.turbine.util.RunData;
import org.apache.turbine.Turbine;

// commons logging classes
import org.apache.commons.logging.LogFactory; 

// log4j stuff
import org.apache.log4j.PropertyConfigurator;

/**
 * The default implementation of the logging service in Jetspeed.
 *
 * This service functions as a logger provider.
 * It allows access to loggers: explicit by the getLogger method,
 * or by printing methods (info, error...).
 * Real work is done by classes that implement interface: Logger.
 * The configuration of the service is read from the TurbineResources.properties.
 * The rest of the configuration is done through a defined LoggingConfig class.
 *
 * Names of the loggers, classes, log levels, destinations are defined in that file.
 *
 * @see org.apache.turbine.services.logging.Logger
 * @author <a href="mailto:morciuch@apache.org">Mark Orciuch</a>
 * @version $Id: JetspeedLoggingService.java,v 1.1 2002/12/23 23:14:49 morciuch Exp $
 */
public class JetspeedLoggingService
extends TurbineBaseService
implements LoggingService
{

    // configuration keys
    private static final String CONFIG_LOG4J_PROPERTIES = "log4j.properties";
    private static final String CONFIG_LOG4J_PROPERTIES_DEFAULT = "/WEB-INF/conf/log4j.properties";

    /**
     * loggers repository
     */
    private Hashtable loggersTable;

    /**
     * logger for methods without target
     */
    private Logger defaultLogger;

    /**
     * bootstrap and shutdown logger using context.log
     */
    private Logger simpleLogger;

    /**
     * context for resolving paths and servlet logging
     */
    private ServletContext context = null;

    /**
     * Resources for this Service
     */
    private ResourceService resources = null;

    public JetspeedLoggingService()
    {
        loggersTable = new Hashtable();
        defaultLogger = null;
    }

    /**
     * Load all configured components and initialize them. This is
     * a zero parameter variant which queries the Turbine Servlet
     * for its config.
     *
     * @throws InitializationException Something went wrong in the init
     *         stage
     */ 
    public void init()
    throws InitializationException
    {
        ServletConfig conf = Turbine.getTurbineServletConfig();
        init(conf);
    }

    /**
     * Inits the service using servlet parameters to obtain path to the
     * configuration file. Change relatives paths.
     *
     * @param config The ServletConfiguration from Turbine
     *
     * @throws InitializationException Something went wrong when starting up.
     */
    public void init(ServletConfig config) 
    throws InitializationException
    {
        context = config.getServletContext();

        // Create bootstrap logger, for handling exceptions during service
        // initialization.
        defaultLogger = new BaseLogger(LogFactory.getLog("default"));

        simpleLogger = defaultLogger;

        internalInit();
        setInit(true);
    }

    /**
     * This gets the ResourceService associated to this Service
     */
    public ResourceService getResources()
    {
        if (resources == null)
        {
            // Get the properties for this Service
            resources = TurbineResources
                        .getResources(TurbineServices.SERVICE_PREFIX 
                                      + LoggingService.SERVICE_NAME);

            //add webappRoot manually - cos logging is a primary
            //service and so it is not yet defined
            String webappRoot = context.getRealPath("/");
            resources.setProperty(Turbine.WEBAPP_ROOT, webappRoot);
        }
        return (resources);
    }

    /**
     * This method initializes the service.
     */
    private void internalInit() throws InitializationException
    {
        ResourceService props = getResources();
        if (props == null)
        {
            throw new InitializationException("LoggingService failed to " 
                                              + "get access to the properties for this service.");
        }

        //looking for default logger name
        String defaultLoggerName = props.getString(LoggingConfig.DEFAULT);

        //checking whether default logger is properly configured
        if (defaultLoggerName == null)
        {
            throw new InitializationException("LoggingService can't find " 
                                              + "default logger name in the configuration file.");
        }

        // Configure log4j logging
        String log4jProperties = props.getString(this.CONFIG_LOG4J_PROPERTIES, this.CONFIG_LOG4J_PROPERTIES_DEFAULT);
        if (log4jProperties != null)
        {
            configureLog4J(log4jProperties);
        }

        // Create default logger
        defaultLogger = new BaseLogger(LogFactory.getLog(defaultLoggerName));
        loggersTable.put(defaultLoggerName, defaultLogger);

        //checking whether default logger is properly configured
        if (defaultLogger == null)
        {
            throw new InitializationException("LoggingService can't find " 
                                              + "default logger in working loggers.");
        }
    }

    private void configureLog4J(String log4jProperties) throws InitializationException
    {
        try
        {
            Properties p = new Properties();        
            p.load(new FileInputStream(Turbine.getRealPath(log4jProperties)));
            p.setProperty("webappRoot", this.context.getRealPath("/"));
            PropertyConfigurator.configure(p);
        }
        catch (Exception e)
        {
            throw new InitializationException("Failed to load " + log4jProperties + " - " + e.toString());
        }
        
    }

    /**
     * Shutdowns all loggers. After shutdown servlet logger is still available
     * using the servlet log method
     */
    public void shutdown()
    {
        if (!getInit())
        {
            return;
        }
        Enumeration iter = loggersTable.elements();

        while (iter.hasMoreElements())
        {
            ((Logger) iter.nextElement()).shutdown();
        }

        //HACK!!!!!
        //some services may log using our services after shutdown
        loggersTable.clear();
        defaultLogger = simpleLogger;

        //we don't set init as false, because we can still log.
    }

    /**
     * This method returns default logger for Turbine System
     */
    public final Logger getLogger()
    {
        return defaultLogger;
    }

    /**
     * This method returns logger with given name.
     */
    public Logger getLogger(String logName)
    {
        Logger logger = (Logger) loggersTable.get(logName);
        if (logger == null)
        {
            logger = new BaseLogger(LogFactory.getLog(logName));
            if (logger == null)
            {
                return defaultLogger;
            }
            loggersTable.put(logName, logger);
        }
        return logger;
    }

    /**
     * This method sets the log level of the default logger.
     */
    public void setLogLevel(int level)
    {
        defaultLogger.setLogLevel(level);
    }

    /**
     * This method sets the log level of the logger of given name.
     */
    public void setLogLevel(String logName, int level)
    {
        Logger logger = (Logger) loggersTable.get(logName);
        if (logger != null)
        {
            logger.setLogLevel(level);
        }
    }

    /**
     * This method sets format style of the default logger
     */
    public void setFormat(String format)
    {
        defaultLogger.setFormat(format);
    }

    /**
     * This method sets format style of the given logger.
     */
    public void setFormat(String logName, String format)
    {
        Logger logger = (Logger) loggersTable.get(logName);
        if (logger != null)
        {
            logger.setFormat(format);
        }
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the default logger
     */
    public void debug(String message)
    {
        defaultLogger.debug(message);
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the default logger
     */
    public void debug(String message, Throwable t)
    {
        defaultLogger.debug(message, t);
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the given logger
     */
    public void debug(String logName, String message, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.debug(message, t);
        }
        else
        {
            defaultLogger.debug("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the given logger
     */
    public void debug(String logName, String message)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.debug(message);
        }
        else
        {
            defaultLogger.debug("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the default logger
     */
    public void debug(String message, RunData data)
    {
        defaultLogger.debug(message);
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the default logger
     */
    public void debug(String message, RunData data, Throwable t)
    {
        defaultLogger.debug(message, t);
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the given logger
     */
    public void debug(String logName, String message, RunData data, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.debug(message, data, t);
        }
        else
        {
            defaultLogger.debug("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == DEBUG, printing is done by
     * the given logger
     */
    public void debug(String logName, String message, RunData data)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.debug(message, data);
        }
        else
        {
            defaultLogger.debug("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the default logger
     */
    public void info(String message)
    {
        defaultLogger.info(message);
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the default logger
     */
    public void info(String message, Throwable t)
    {
        defaultLogger.info(message, t);
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the given logger
     */
    public void info(String logName, String message)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.info(message);
        }
        else
        {
            defaultLogger.info("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the given logger
     */
    public void info(String logName, String message, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.info(message, t);
        }
        else
        {
            defaultLogger.info("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the default logger
     */
    public void info(String message, RunData data)
    {
        defaultLogger.info(message);
    }

    /**
     * This is a log method with logLevel == INFO,printing is done by
     * the default logger
     */
    public void info(String message, RunData data, Throwable t)
    {
        defaultLogger.info(message, t);
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the given logger
     */
    public void info(String logName, String message, RunData data)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.info(message, data);
        }
        else
        {
            defaultLogger.info("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == INFO, printing is done by
     * the given logger
     */
    public void info(String logName, String message, RunData data, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.info(message, data, t);
        }
        else
        {
            defaultLogger.info("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the default logger
     */
    public void warn(String message)
    {
        defaultLogger.warn(message);
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the default logger
     */
    public void warn(String message, Throwable t)
    {
        defaultLogger.warn(message, t);
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the given logger
     */
    public void warn(String logName, String message)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.warn(message);
        }
        else
        {
            defaultLogger.warn("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the given logger
     */
    public void warn(String logName, String message, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.warn(message, t);
        }
        else
        {
            defaultLogger.warn("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == WARN,printing is done by
     * the default logger
     */
    public void warn(String message, RunData data)
    {
        defaultLogger.warn(message);
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the default logger
     */
    public void warn(String message, RunData data, Throwable t)
    {
        defaultLogger.warn(message, t);
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the given logger
     */
    public void warn(String logName, String message, RunData data)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.warn(message, data);
        }
        else
        {
            defaultLogger.warn("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == WARN, printing is done by
     * the given logger
     */
    public void warn(String logName, String message, RunData data, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.warn(message, data, t);
        }
        else
        {
            defaultLogger.warn("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the default logger
     */
    public void error(String message)
    {
        defaultLogger.error(message);
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the default logger
     */
    public void error(String message, Throwable t)
    {
        defaultLogger.error(message, t);
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the given logger
     */
    public void error(String logName, String message)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.error(message);
        }
        else
        {
            defaultLogger.error("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the given logger
     */
    public void error(String logName, String message, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.error(message, t);
        }
        else
        {
            defaultLogger.error("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the default logger
     */
    public void error(String message, RunData data)
    {
        defaultLogger.error(message);
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the default logger
     */
    public void error(String message, RunData data, Throwable t)
    {
        defaultLogger.error(message, t);
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the given logger
     */
    public void error(String logName, String message, RunData data)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.error(message, data);
        }
        else
        {
            defaultLogger.error("FROM logger:" + logName + ": " + message);
        }
    }

    /**
     * This is a log method with logLevel == ERROR, printing is done by
     * the given logger
     */
    public void error(String logName, String message, RunData data, Throwable t)
    {
        Logger logger = getLogger(logName);
        if (logger != null)
        {
            logger.error(message, data, t);
        }
        else
        {
            defaultLogger.error("FROM logger:" + logName + ": " + message);
        }
    }
}
