/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.om.registry.base;

import org.apache.jetspeed.om.registry.PortletRegistry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.PortletEntry;
import org.apache.jetspeed.om.registry.InvalidEntryException;
import org.apache.jetspeed.om.registry.Category;
import org.apache.jetspeed.om.registry.RegistryException;
import org.apache.jetspeed.services.Registry;

import org.apache.turbine.util.Log;

import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Provides base functionality within a Portlet Registry.
 *
 * <p>To avoid loops, a RegistryService implementation using this class
 * nees to call the addLocalEntry/removeLocalEntry methods to modify
 * the in memory state of this Registry</p>
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: BasePortletRegistry.java,v 1.5 2003/03/04 00:05:00 sgala Exp $
 */
public class BasePortletRegistry extends BaseRegistry implements PortletRegistry
{

    private Map catMap = new TreeMap();

    /**
    @see Registry#setEntry
    */
    public void setEntry( RegistryEntry entry ) throws InvalidEntryException
    {
		// Delegate to the RegistryService to ensure correct handling of
		// persistence if using file fragments

		try
		{
			Registry.addEntry(Registry.PORTLET, entry);
		}
		catch (RegistryException e)
		{
			Log.error(e);
		}
    }

    /**
    @see Registry#addEntry
    */
    public void addEntry( RegistryEntry entry ) throws InvalidEntryException
    {
		// Delegate to the RegistryService to ensure correct handling of
		// persistence if using file fragments

		try
		{
			Registry.addEntry(Registry.PORTLET, entry);
		}
		catch (RegistryException e)
		{
			Log.error(e);
		}
    }

    /**
    @see Registry#removeEntry
    */
    public void removeEntry( String name )
    {
		// Delegate to the RegistryService to ensure correct handling of
		// persistence if using file fragments

		Registry.removeEntry(Registry.PORTLET, name);
    }

    /**
    @see Registry#removeEntry
    */
    public void removeEntry( RegistryEntry entry )
    {
		// Delegate to the RegistryService to ensure correct handling of
		// persistence if using file fragments

		if (entry != null)
		{
			Registry.removeEntry(Registry.PORTLET, entry.getName());
		}
    }

    protected void setPortletEntry(PortletEntry entry) throws InvalidEntryException
    {
        synchronized (catMap)
        {
            int count = 0;
            Iterator it = ((PortletEntry)entry).listCategories();
            while (it.hasNext())
            {
                Category category = (Category)it.next();
                String key = getCategoryKey(category);
                HashMap bucket = (HashMap)this.catMap.get(key);
                if (null == bucket)
                {
                    bucket = new HashMap();
                    bucket.put(entry.getName(), entry);
                    this.catMap.put(key, bucket);
                }
                else
                {
                    bucket.put(entry.getName(), entry);
                }
                count++;
            }
            /*
            TODO: this feature could be optional
            */
            if (0 == count)
            {
                StringBuffer key = new StringBuffer(128);
                key.append(PortletEntry.DEFAULT_GROUP);
                key.append(".");
                if (entry.getType().equals(PortletEntry.TYPE_ABSTRACT))
                    key.append(PortletEntry.DEFAULT_CATEGORY_ABSTRACT);
                else
                    key.append(PortletEntry.DEFAULT_CATEGORY_REF);

                HashMap bucket = (HashMap)this.catMap.get(key.toString());
                if (null == bucket)
                {
                    bucket = new HashMap();
                    bucket.put(entry.getName(), entry);
                    this.catMap.put(key.toString(), bucket);
                }
                else
                {
                    bucket.put(entry.getName(), entry);
                }
            }

        }
    }


    public String getCategoryKey(Category category)
    {
        if (category == null)
        {
            return PortletEntry.DEFAULT_GROUP;
		}

        String categoryName = category.getName();

        if ((categoryName == null) || categoryName.equals(""))
        {
            return category.getGroup();
		}

        return category.getGroup() + "." + categoryName;
    }


    /*
     * Find portlets in this registry, looking up by category in the default category group.
     *
     * @param category The category and optional subcategories.
     * @return Iterator The result as an iterator.
     */
    public Iterator findPortletsByCategory(String category)
    {
    	String key;

        if ((category == null) || category.equals(""))
        {
        	key = PortletEntry.DEFAULT_GROUP;
	    }
        else
        {
            key = PortletEntry.DEFAULT_GROUP + "." + category;
		}

        CategoryIterator iterator = new CategoryIterator((SortedMap)catMap, key);

        return iterator;
    }

    /*
     * Find portlets in this registry, looking up by category and category group.
     *
     * @param group The group to search for categories in.
     * @param category The category and optional subcategories.
     * @return Iterator The result as an iterator.
     */
    public Iterator findPortletsByGroupCategory(String group, String category)
    {
        if ((group == null) || group.equals(""))
        {
            group = PortletEntry.DEFAULT_GROUP;
		}

        String key = group + "." + category;

        CategoryIterator iterator = new CategoryIterator((SortedMap)catMap, key);

        return iterator;
    }

    /*
     * List all portlets in this registry, sorted by category
     *
     * @return Iterator The result as an iterator.
     */
    public Iterator listByCategory()
    {
        CategoryIterator iterator = new CategoryIterator((SortedMap)catMap, null);
        return iterator;
    }

    /**
     * Creates a new RegistryEntry instance compatible with the current
     * Registry instance implementation
     *
     * @return the newly created RegistryEntry
     */
    public RegistryEntry createEntry()
    {
		return new BasePortletEntry();
	}

    /**
    @see Registry#setEntry
    */
    public void setLocalEntry( RegistryEntry entry ) throws InvalidEntryException
    {
        super.setLocalEntry(entry);
        setPortletEntry((PortletEntry)entry);
    }

    /**
    @see Registry#addEntry
    */
    public void addLocalEntry( RegistryEntry entry ) throws InvalidEntryException
    {
        super.addLocalEntry(entry);
        setPortletEntry((PortletEntry)entry);
    }

    /**
    @see Registry#removeEntry
    */
    public void removeLocalEntry( String name )
    {
        if (name == null)
        {
            return;
		}

        RegistryEntry entry = (RegistryEntry)this.entries.get( name ) ;

        if (entry == null)
        {
            return;
		}

        removeLocalEntry(entry);
    }

    /**
    @see Registry#removeEntry
    */
    public void removeLocalEntry( RegistryEntry entry )
    {
        synchronized(catMap)
        {
            int count = 0;
            Iterator it = ((PortletEntry)entry).listCategories();
            while (it.hasNext())
            {
                Category category = (Category)it.next();
                HashMap map = (HashMap)catMap.get(getCategoryKey(category));
                if (map != null)
                {
                    map.remove(entry.getName());
                    if (0 == map.size())
                    {
                        catMap.remove(getCategoryKey(category));
                    }
                }
            }
        }
        super.removeLocalEntry(entry);
    }

}