/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.modules.parameters;

//ecs stuff
import org.apache.ecs.html.Select;
import org.apache.ecs.html.Option;

// java stuff
import java.util.Map;
import java.util.Arrays;
import java.util.StringTokenizer;

//turbine support
import org.apache.turbine.util.RunData;

/**
 * Returns simple list box control.
 * <p>Options:
 * <UL>
 * <LI><code>items</code> - comma-separated list of list box items</LI>
 * <LI><code>layout</code> [<strong>$combo</strong>|$list] - combo box vs list box</LI> 
 * <LI><code>size</code> - size of the list box for $list style</LI> 
 * <LI><code>sort</code> [<strong>false</strong>|true] - return sorted list of items</LI>  
 * <LI><code>multiplechoice</code> [<strong>false</strong>|true] - allow multiple selections</LI>  
 * <LI><code>null-if-empty</code> [<strong>false</strong>|true] - do not return a select control if item list is empty</LI>   
 * </UL> 
 * @author <a href="mark_orciuch@ngsltd.com">Mark Orciuch</a>
 * @author <a href="mailto:solutioncenter@infointegrators.com">Dave Trapp</a>
 * @version $Id: ListBox.java,v 1.2 2002/10/01 20:06:35 morciuch Exp $  
 */
public class ListBox extends ParameterPresentationStyle
{

    public static final String SORT = "sort";
    public static final String ITEMS = "items";
    public static final String LAYOUT = "layout";
    public static final String LAYOUT_COMBO = "$combo";
    public static final String LAYOUT_LIST = "$list";
    public static final String LIST_SIZE = "listsize";
    public static final String MULTIPLE_CHOICE = "multiplechoice";
    public static final String NULL_IF_EMPTY = "null-if-empty";

    protected String layout = null;
    protected String items[] = null; 
    protected String size = null;
    protected boolean multiple = false;

    /**
     * Returns presentation control
     * 
     * @param data
     * @param name
     * @param value
     * @param parms
     * @return string
     */
    public String getContent(RunData data, String name, String value, Map parms)
    {

        init(data);

        Select select = null;
        if ( layout.equalsIgnoreCase(this.LAYOUT_LIST) )
        {
            select = new Select(name, new Integer(size).intValue());
        } else
        {
            select = new Select(name);
        }

        if ( multiple )
        {
            select.setMultiple(multiple);
        }

        if ( items != null )
        {

            boolean sort = new Boolean((String)this.getParm(this.SORT, "false")).booleanValue();
            if ( sort )
            {
                Arrays.sort(items);
            }

            for ( int i=0; i < items.length; i++ )
            {
                Option option = new Option(items[i]).addElement(items[i]);
                option.setSelected(items[i].equalsIgnoreCase(value));
                select.addElement(option);
            }
        }

        // If no items to display, do not display empty control
        boolean nullIfEmpty = new Boolean((String)this.getParm(this.NULL_IF_EMPTY, "false")).booleanValue();
        if ( this.items == null || (nullIfEmpty && items.length == 0) )
        {
            return null;
        }

        return select.toString();

    }

    /**
     * Initialize options
     * 
     * @param data
     */
    protected void init(RunData data)
    {

        this.layout = (String)this.getParm(this.LAYOUT, this.LAYOUT_COMBO);
        this.items = this.getItems(data); 
        this.size = (String)this.getParm(this.LIST_SIZE, "1");
        this.multiple = new Boolean((String)this.getParm(this.MULTIPLE_CHOICE, "false")).booleanValue();

    }

    /**
     * Parse items into an arrary of strings
     * 
     * @param data
     * @return string array
     */
    protected String[] getItems(RunData data)
    {

        String[] result = null;
        String list = (String)this.getParm(this.ITEMS, ""); 

        StringTokenizer it = new StringTokenizer(list, ",");
        int size = it.countTokens();

        if ( size > 0 )
        {
            result = new String[size];

            int i = 0;
            while ( it.hasMoreTokens() )
            {
                String item = (String)it.nextToken();
                result[i] = item;
                i++;
            }
        }

        return result; 

    }

}
