/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.cache.disk;

import java.io.*;

/**
<p>
A way to store remote documents locally.  This can increase performance by 
eliminating TCP connection latency.
</p>

<p>
All implementations of a DiskCache should implement this interface.
</p>

@author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
@author <a href="mailto:sgala@hisitech.com">Santiago Gala</a>
@version $Id: DiskCache.java,v 1.14 2002/01/10 09:12:39 kimptoc Exp $
*/
public interface DiskCache {
    
    /**
    <p>
    Take the given remove URL and place it in the disk cache.  Additionaly 
    operations include building a DiskCacheEntry and then returning it.
    </p>
    
    <p>getEntry may attempt to pull down the URL if it is not in the disk cache.
    Most implementations may provide support for changing this behavior so that
    you have to explicitly call add.  This is done for performance reasons
    so that HTTP clients don't end up triggering a URL connection to fetch 
    the given URL.</p>
    
    */
    public DiskCacheEntry getEntry( String url ) throws IOException;

    /**
    <p>Get an Entry from the from the cache but force this URL to be fetched and
    then cached no matter what configuration options Jetspeed provides.
    
    @see #getEntry( String url )
    */
    public DiskCacheEntry getEntry( String url,
                                    boolean force ) throws IOException;
    
    /**
    Get a entry based on a URL but you should already have the content.  This is
    usually used to specify an alternative Reader (maybe StringReader).

    <p>getEntry may attempt to pull down the URL if it is not in the disk cache.
    Most implementations may provide support for changing this behavior so that
    you have to explicitly call add.  This is done for performance reasons
    so that HTTP clients don't end up triggering a URL connection to fetch 
    the given URL.</p>
    
    */
    public DiskCacheEntry getEntry( String url, Reader is ) throws IOException;
    
    /**
    Get a list of all the documents within the cache...
    
    */
    public DiskCacheEntry[] getEntries();

    /**
    Add this URL to the disk cache
    
    */
    public void add( String url ) throws IOException;
    
    /**
    Remove this URL from the disk cache.
    
    */
    public void remove( String url ) throws IOException;
    

    /**
    Get the URL from the Internet and then place it in the File dest.
    
    */
    public String fetch( String url, String destination ) throws IOException ;

    /**
    Return the root of this DiskCache.
    
    */
    public String getRoot();

    /**
    Tell the DiskCache that this URL should be refreshed.  This will do this in 
    a threaded and asynchronous manner.
    
    */
    public void refresh( String url );

    /**
    Ask if a url is in the DiskCache  
    
    */
    public boolean isCached( String url );

}
