/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.ui.views;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PlatformUI;

import benten.cat.glossary.core.BentenGlossaryEngine;
import benten.cat.glossary.core.BentenGlossarySearchResult;
import benten.cat.glossary.ui.CatGlossaryUiPlugin;
import benten.cat.glossary.ui.actions.CopyGlossarySourceAction;
import benten.cat.glossary.ui.actions.CopyGlossaryTargetAction;
import benten.cat.glossary.ui.views.messages.GlossaryViewMessages;
import benten.core.CorePlugin;
import benten.ui.views.AutoColumnWidthTableViewer;
import benten.ui.views.CopySupportTableView;
import benten.ui.views.TableColumnSortAdapter;
import benten.ui.views.TableLabelProvider;

/**
 * 用語集ビュー。
 *
 * <UL>
 * <LI>エディターなどで選択された文字列に対し、用語集を表示するためのビュー・クラスです。
 * <LI>用語集の検索は {@link BentenGlossaryEngine} へ委譲されます。
 * </UL>
 *
 * ★基本設計「翻訳支援機能: 用語対訳集検索機能」に対応します。
 *
 * @author YAMAMOTO Koji
 * @author KASHIHARA Shinji
 */
public class GlossaryView extends CopySupportTableView {

	/**
	 * 用語集ビューのためのメッセージ。
	 */
	protected static final GlossaryViewMessages fMsg = new GlossaryViewMessages();

	/** コピー・ソース・アクション */
	private Action copySourceAction;

	/** コピー・ターゲット・アクション */
	private Action copyTargetAction;

	/** リロード・アクション */
	private Action reloadAction;

	/**
	 * コンストラクター。
	 */
	public GlossaryView() {
	}

	@Override
	public void createTableViewer(final AutoColumnWidthTableViewer viewer) {
		viewer.addColumn(fMsg.getSource(), 10).addSelectionListener(new GlossarySourceSortListener(0));
		viewer.addColumn(fMsg.getTarget(), 10).addSelectionListener(new GlossaryTargetSortListener(1));
		viewer.addColumn(fMsg.getNote(), 10).addSelectionListener(new GlossaryNoteSortListener(2));

		viewer.setLabelProvider(new ViewLabelProvider());
		viewer.setInput(new ArrayList<BentenGlossarySearchResult>());

		makeActions();
		createContextMenu();
		contributeToActionBars();
	}

	/**
	 * コンテキスト・メニューの作成。
	 */
	protected void createContextMenu() {
		final MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(final IMenuManager manager) {
				GlossaryView.this.fillContextMenu(manager);
			}
		});
		final Menu menu = menuMgr.createContextMenu(getViewer().getControl());
		getViewer().getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, getViewer());
	}

	/**
	 * アクション・バーのコントリビュート。
	 */
	protected void contributeToActionBars() {
		final IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	/**
	 * ローカル・プルダウンの設定。
	 * @param manager メニュー・マネージャー
	 */
	private void fillLocalPullDown(final IMenuManager manager) {
		manager.add(reloadAction);
		manager.add(new Separator());
	}

	/**
	 * コンテキスト・メニューの設定。
	 * @param manager メニュー・マネージャー
	 */
	private void fillContextMenu(final IMenuManager manager) {
		final boolean isEmpty = getViewer().getSelection().isEmpty();
		copySourceAction.setEnabled(!isEmpty);
		manager.add(copySourceAction);
		copyTargetAction.setEnabled(!isEmpty);
		manager.add(copyTargetAction);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/**
	 * ローカル・ツールバーの設定。
	 * @param manager ツール･バー・マネージャー
	 */
	private void fillLocalToolBar(final IToolBarManager manager) {
		manager.add(reloadAction);
	}

	/**
	 * アクションの作成。
	 */
	private void makeActions() {
		reloadAction = new Action() {
			@Override
			public void run() {
				CatGlossaryUiPlugin.getDefault().newGlossaryEnginePluginInstance().load();
			}
		};
		reloadAction.setText(fMsg.getReloadGlossary());
		reloadAction.setToolTipText(fMsg.getReloadGlossaryToolTipText());
		reloadAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(
				ISharedImages.IMG_OBJS_INFO_TSK));

		copySourceAction = new CopyGlossarySourceAction(this, fMsg.getCopySource());
		copyTargetAction = new CopyGlossaryTargetAction(this, fMsg.getCopyTarget());
	}

	/**
	 * 用語集の表示。
	 * @param string 用語文字列
	 */
	public void showGlossary(final String string) {
		final StringBuffer targetStrBuf = new StringBuffer();
		if (string != null) {
			targetStrBuf.append(string);
		}
		final BentenGlossaryEngine glossaryEngine = CatGlossaryUiPlugin.getDefault().getGlossaryEngine();
		final List<BentenGlossarySearchResult> entries = glossaryEngine.search(targetStrBuf.toString());

		if (CorePlugin.IS_DEBUG) {
			final CatGlossaryUiPlugin plugin = CatGlossaryUiPlugin.getDefault();
			plugin.log(fMsg.getMsg001(BigDecimal.valueOf(entries.size())));
			for (int i = 0; i < entries.size(); i++) {
				final BentenGlossarySearchResult glossaryEntry = entries.get(i);
				plugin.log("------------------------------------"); //$NON-NLS-1$
				plugin.log(fMsg.getMsg002(BigDecimal.valueOf(i + 1)));
				plugin.log(glossaryEntry.getSource());
				plugin.log(glossaryEntry.getTarget());
				plugin.log(glossaryEntry.getNote());
			}
			plugin.log("------------------------------------"); //$NON-NLS-1$
			plugin.log(""); //$NON-NLS-1$
		}

		getViewer().setInput(entries);
	}

	/**
	 * ビューのラベル・プロバイダー。
	 */
	private static class ViewLabelProvider extends TableLabelProvider {
		public String getColumnText(final Object obj, final int index) {
			final BentenGlossarySearchResult entry = (BentenGlossarySearchResult) obj;
			switch (index) {
			case 0:
				return entry.getSource();
			case 1:
				return entry.getTarget();
			case 2:
				return entry.getNote();
			}
			return getText(obj);
		}
	}

	//-------------------------------------------------------------------------
	// ソート・リスナー・クラス

	/**
	 * 用語集のソース・ソート・リスナー。
	 */
	protected class GlossarySourceSortListener extends TableColumnSortAdapter<BentenGlossarySearchResult> {
		public GlossarySourceSortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenGlossarySearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenGlossarySearchResult>() {
				@Override
				public int compare(final BentenGlossarySearchResult tm1, final BentenGlossarySearchResult tm2) {
					return tm1.getSource().compareToIgnoreCase(tm2.getSource());
				}
			};
		}
	}

	/**
	 * 用語集のターゲット・ソート・リスナー。
	 */
	protected class GlossaryTargetSortListener extends TableColumnSortAdapter<BentenGlossarySearchResult> {
		public GlossaryTargetSortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenGlossarySearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenGlossarySearchResult>() {
				@Override
				public int compare(final BentenGlossarySearchResult tm1, final BentenGlossarySearchResult tm2) {
					return tm1.getTarget().compareToIgnoreCase(tm2.getTarget());
				}
			};
		}
	}

	/**
	 * 用語集のノート・ソート・リスナー。
	 */
	protected class GlossaryNoteSortListener extends TableColumnSortAdapter<BentenGlossarySearchResult> {
		public GlossaryNoteSortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenGlossarySearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenGlossarySearchResult>() {
				@Override
				public int compare(final BentenGlossarySearchResult tm1, final BentenGlossarySearchResult tm2) {
					return tm1.getNote().compareToIgnoreCase(tm2.getNote());
				}
			};
		}
	}
}
