/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.dialogs;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import benten.twa.BentenTwaConstants;
import benten.twa.filter.messages.BentenImportHtmlMessages;
import benten.twa.filter.status.BentenImportHtmlStatus;
import benten.twa.ui.dialogs.AbstractWorkflowDialog;
import benten.ui.UiStatus;
import benten.ui.UiStatusException;
import benten.ui.fields.FolderTextField;
import benten.ui.fields.ResourceFolderTextField;
import benten.ui.fields.TextField;
import benten.ui.preference.BentenProjectProperty;
import benten.ui.preference.BentenProjectProperty.ProjectProperty;

/**
 * 翻訳対象のインポート・ダイアログ。
 *
 * <pre>
 * 翻訳対象物をプロジェクトにインポートします。
 *   1.  翻訳対象物のファイルをプロジェクトにインポートします。
 *   2.  インポートされた翻訳対象物は、デフォルトではプロジェクト内の [.benten/source] ディレクトリ以下に配置されます。
 *   3.  インポートされた翻訳対象物は、エクスポートの際に利用されます。
 *   4.  インポートされた翻訳対象物は、直接編集しないでください。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTMLインポート機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class ImportHtmlDialog extends AbstractWorkflowDialog {

	/**
	 * HTMLインポート機能のためのメッセージ。
	 */
	protected static final BentenImportHtmlMessages fMsg = new BentenImportHtmlMessages();

	/** コントロール・コンテナー */
	private final ImportHtmlControls controls = new ImportHtmlControls();

	/** コントロール・コンテナー・クラス */
	public static class ImportHtmlControls {

		/** 入力ディレクトリー・テキスト */
		public FolderTextField fromDirText;

		/** 出力ディレクトリー・テキスト */
		public ResourceFolderTextField toDirText;

		/** trans-target id テキスト */
		public TextField transTargetIdText;

		/** HTML 正規化チェックボックス */
		public Button normalizeHtmlCheck;
	}

	/**
	 * コンストラクター。
	 * @param parentShell 親シェル
	 * @param selection 選択
	 */
	public ImportHtmlDialog(final Shell parentShell, final IStructuredSelection selection) {
		super(parentShell, selection, fMsg.getWindowTitle());
	}

	@Override
	public void createControls(final Composite composite) {
		composite.setLayout(new GridLayout(3, false));
		IPreferenceStore projectStore = null;

		controls.transTargetIdText = new TextField(composite, fMsg.getDialogCtrlLabelTransTargetId());
		controls.transTargetIdText.addModifyListener(this);
		if (getSelectionResource() != null) {
			projectStore = BentenProjectProperty.getStore(getSelectionResource().getProject());
			final String transTargetId = projectStore.getString(ProjectProperty.TRANS_TARGET_ID.name());
			controls.transTargetIdText.setText(transTargetId);
		}
		controls.transTargetIdText.setInfoDecoration(fMsg.getDialogCtrlLabelTransTargetIdInfoDescription());
		// グリッドが崩れないようにするためのダミー。
		new Label(composite, SWT.LEFT);

		controls.fromDirText = new FolderTextField(composite, fMsg.getDialogCtrlLabelFromDir());
		controls.fromDirText.addModifyListener(this);
		if (projectStore != null) {
			final String fromDir = projectStore.getString(ProjectProperty.TRANS_TARGET_IMPORT_FROM_DIR.name());
			controls.fromDirText.setText(fromDir);
		}

		controls.toDirText = new ResourceFolderTextField(composite, fMsg.getDialogCtrlLabelToDir());
		controls.toDirText.addModifyListener(this);
		if (getSelectionResource() != null) {
			// デフォルトは [/.benten/source] とします。
			// このディレクトリには、現在は html しか格納されませんが、
			// 将来は docx や txt が入ることを想定しています。
			controls.toDirText.setText(getSelectionResource().getProject().getName() + "/" //$NON-NLS-1$
					+ BentenTwaConstants.DEFAULT_SOURCE_DIR);
		}
		controls.toDirText.setInfoDecoration(fMsg
				.getDialogCtrlLabelToDirInfoDescription(BentenTwaConstants.DEFAULT_SOURCE_DIR));

		controls.normalizeHtmlCheck = new Button(composite, SWT.CHECK);
		controls.normalizeHtmlCheck.setText(fMsg.getDialogCtrlButtonHtmlCheck());
		controls.normalizeHtmlCheck.setSelection(true);
		final GridData checkGrid = new GridData();
		checkGrid.horizontalSpan = 3;
		controls.normalizeHtmlCheck.setLayoutData(checkGrid);
	}

	@Override
	public boolean okPressedPrompt() {
		final IProject project = controls.toDirText.getResource().getProject();
		final IPreferenceStore projectStore = BentenProjectProperty.getStore(project);

		// このプロジェクトで処理済みとマークされている場合は確認ダイアログを表示
		if (projectStore.getBoolean(ProjectProperty.TRANS_TARGET_IMPORTED.name())) {

			final String prompt = projectStore.getString(ProjectProperty.PROMPT_TRANS_TARGET_IMPORTED.name());
			if (!MessageDialogWithToggle.ALWAYS.equals(prompt)) {

				final MessageDialogWithToggle dialog = MessageDialogWithToggle.openOkCancelConfirm(getShell(), fMsg
						.getDialogMsg01(), fMsg.getDialogMsg02(), fMsg.getDialogMsg03(), false, projectStore,
						ProjectProperty.PROMPT_TRANS_TARGET_IMPORTED.name());
				return dialog.getReturnCode() == Window.OK;
			}
		}
		return true;
	}

	@Override
	public IStatus getResultStatus() {
		final BentenImportHtmlStatus status = new BentenImportHtmlStatus();
		status.setFromDir(controls.fromDirText.getFile());
		status.setToDir(controls.toDirText.getFile());
		status.setNormalizeHtml(controls.normalizeHtmlCheck.getSelection());
		status.setTransTargetId(controls.transTargetIdText.getText());
		status.setProject(controls.toDirText.getResource());

		final IPreferenceStore projectStore = BentenProjectProperty.getStore(status.getProject());
		projectStore.setValue(ProjectProperty.TRANS_TARGET_ID.name(), status.getTransTargetId());
		projectStore.setValue(ProjectProperty.TRANS_TARGET_IMPORT_FROM_DIR.name(), controls.fromDirText.getText());
		return status;
	}

	@Override
	public IStatus validate() throws UiStatusException {
		validatePath(controls.fromDirText);
		validatePath(controls.toDirText);
		validateRequire(controls.transTargetIdText);
		validatePattern(controls.transTargetIdText, "[a-z][a-z0-9_\\-\\.]*[a-z0-9]", ".*[_\\-\\.]{2,}.*"); //$NON-NLS-1$ //$NON-NLS-2$

		// このプロジェクトで処理済みとマークされている場合は警告を表示
		if (controls.toDirText != null) {
			final IProject project = controls.toDirText.getResource().getProject();
			final IPreferenceStore projectStore = BentenProjectProperty.getStore(project);
			if (projectStore.getBoolean(ProjectProperty.TRANS_TARGET_IMPORTED.name())) {
				throw new UiStatusException(IStatus.WARNING, fMsg.getDialogMsg04());
			}
		}
		return UiStatus.OK_STATUS;
	}
}
