/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import benten.core.BentenConstants;
import benten.core.io.Files;
import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.valueobject.BentenExportHtmlProcessInput;
import benten.twa.filter.messages.BentenExportHtmlMessages;
import benten.twa.filter.model.HtmlInlineTextBuilder;
import benten.twa.filter.model.HtmlInlineTextBuilder.InlineText;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 翻訳成果のエクスポート
 *
 * <pre>
 * 翻訳成果物をプロジェクトからエクスポートします。
 *   1.  XLIFF とインポートされた翻訳対象物をもちいて、翻訳成果物をエクスポートします。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTMLエクスポート機能」に対応します。
 *
 * @author KASHIHARA Shinji
 * @author IGA Tosiki
 */
public class BentenExportHtmlProcessImpl extends AbstractTraverseDir implements BentenExportHtmlProcess {
	/**
	 * HTMLエクスポート機能のためのメッセージ。
	 */
	protected static final BentenExportHtmlMessages fMsg = new BentenExportHtmlMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenExportHtmlProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenExportHtmlProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenExportHtmlProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenExportHtmlProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirSourcexliff = new File(fInput.getSourcexliffdir());
		final File dirSourcehtml = new File(fInput.getSourcehtmldir());
		if (dirSourcexliff.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getSourcexliffdir()));
		}
		if (dirSourcexliff.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getSourcexliffdir()));
		}
		if (dirSourcehtml.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getSourcehtmldir()));
		}
		if (dirSourcehtml.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getSourcehtmldir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenExportHtmlProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				// 処理進捗が 2 度呼ばれます。
				fCounter += 2;
			}

			@Override
			public void processFilterdFile(final File file, final String baseDir) throws IOException {
				fCounter++;
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(dirSourcehtml);

		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirSourcehtml);

		if (progress(fMsg.getCoreP004(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".html") || file.getName().toLowerCase().endsWith(".htm")) { //$NON-NLS-1$ //$NON-NLS-2$
			return BentenTwaFilterHtmlUtil.isHtmlTagExist(file);
		} else {
			return false;
		}
	}

	@Override
	public void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenExportHtmlProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		final String relativePath = Files.relativePath(new File(fInput.getSourcehtmldir()), file);
		final File toDir = new File(fInput.getTargetdir(), relativePath).getParentFile();
		final File xliffDir = new File(fInput.getSourcexliffdir(), relativePath).getParentFile();

		writeHtmlFromXliff(file, xliffDir, toDir);
	}

	@Override
	public void processFilterdFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenExportHtmlProcessImpl#processFilterdFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP013(file.getName()))) {
			return;
		}

		final String relativePath = Files.relativePath(new File(fInput.getSourcehtmldir()), file);
		final File outputFile = new File(fInput.getTargetdir(), relativePath);
		Files.copyFile(file, outputFile);

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}

	/**
	 * HTML と XLIFF を入力して、HTML を出力します。
	 * @param htmlFile HTML ファイル
	 * @param xliffDir XLIFF ディレクトリー
	 * @param toDir 出力ディレクトリー
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected void writeHtmlFromXliff(final File htmlFile, final File xliffDir, final File toDir) throws IOException {
		final String xliffFileName = htmlFile.getName() + BentenConstants.FILE_EXT_XLIFF;
		final File inXliffFile = new File(xliffDir, xliffFileName);
		final File outHtmlFile = (new File(toDir, htmlFile.getName()));

		if (progress(fMsg.getCoreP012(htmlFile.getName()))) {
			return;
		}

		if (inXliffFile.exists() == false || inXliffFile.isFile() == false) {
			// XLIFF ファイルがないので、しかたなく、HTML ファイルをそのままコピーします。
			Files.copyFile(htmlFile, outHtmlFile);
			getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			return;
		}

		final BentenXliff xliff = BentenXliff.loadInstance(inXliffFile);
		final Iterator<BlancoXliffTransUnit> units = xliff.getAllTransUnitList().iterator();
		final HtmlInlineTextBuilder htmlBuilder = new HtmlInlineTextBuilder(htmlFile);

		for (final InlineText inline : htmlBuilder) {
			final StringBuilder target = new StringBuilder();

			while (units.hasNext()) {
				fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

				final BlancoXliffTransUnit unit = units.next();
				final BlancoXliffTarget t = unit.getTarget();

				if (t != null && t.getTarget() != null && !t.getTarget().equals("")) { //$NON-NLS-1$
					target.append(t.getTarget());
				} else {
					target.append(unit.getSource());
				}

				if (!HelpTransUnitId.hasContinue(unit.getId())) {
					break;
				}
			}

			if (target.length() > 0) {
				inline.setText(target.toString());
			} else {
				inline.setText(inline.getText());
			}
		}

		// 該当ディレクトリーが無い場合には、ディレクトリー作成。
		// すでに存在する場合は、それはそれで良い。
		if (toDir.exists() == false) {
			if (toDir.mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE008(toDir.getAbsolutePath()));
			}
		} else {
			if (toDir.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE009(fInput.getTargetdir()));
			}
		}

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		htmlBuilder.writeTo(outStream, "UTF-8"); //$NON-NLS-1$
		outStream.flush();
		byte[] bytes = outStream.toByteArray();

		// Content-type 指定が存在しない場合には、これを追加します。
		bytes = resolveContentType(bytes);

		Files.writeByteArrayToFile(outHtmlFile, bytes);

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}

	/**
	 * コンテンツ・タイプの解決。
	 * @param bytes HTML バイト・データ
	 * @return HTML バイト・データ
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected byte[] resolveContentType(byte[] bytes) throws IOException {
		// エンコーディングは UTF-8 固定です。
		final String encoding = "UTF-8"; //$NON-NLS-1$
		final String html = new String(bytes, encoding);

		// HTML の meta charset 指定が存在しない場合 (HTML が正規化されている前提)
		if (!html
				.replaceAll("<!--.*?-->", "").matches("(?i)(?s).+?<meta[^>]*?\\s+content=\"text/html;\\s+charset=.+?\".+")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			// meta タグの作成
			final StringBuilder metaTag = new StringBuilder();
			metaTag.append("\n<meta http-equiv=\"Content-Type\" content=\"text/html; charset="); //$NON-NLS-1$
			metaTag.append(encoding);
			metaTag.append("\">\n"); //$NON-NLS-1$

			// head タグの後に meta タグを挿入
			String resultHtml = insertMetaTag(html, "<head>", metaTag.toString()); //$NON-NLS-1$

			// head タグがない場合は作成
			if (resultHtml == null) {
				resultHtml = insertMetaTag(html, "<html>", "\n<head>" + metaTag.toString() + "</head>\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
			bytes = resultHtml.getBytes(encoding);
		}
		return bytes;
	}

	/**
	 * meta タグの挿入。
	 * @param html html 文字列
	 * @param beforeTag 挿入する前のタグ
	 * @param metaTag メタタグ
	 * @return meta タグ挿入後の html 文字列。指定された beforeTag が見つからない場合は null。
	 */
	private String insertMetaTag(final String html, final String beforeTag, final String metaTag) {
		final StringBuilder sb = new StringBuilder();
		final int beforeTagIndex = html.toLowerCase().indexOf(beforeTag.toLowerCase());
		if (beforeTagIndex != -1) {
			sb.append(html.substring(0, beforeTagIndex + beforeTag.length()));
			sb.append(metaTag);
			sb.append(html.substring(beforeTagIndex + beforeTag.length()));
			return sb.toString();
		}
		return null;
	}
}
