/*****************************************************************************
 * Copyright (c) 2004, 2007 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and
 * is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: IBM Corporation - initial API and implementation
 ****************************************************************************/
/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui.fields;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import benten.ui.fields.messages.TextFieldMessages;

/**
 * ファイルやリソースを選択するための抽象テキスト・フィールド。
 *
 * <UL>
 * <LI>{@link TextField} にファイルなどの参照ボタンを追加したフィールド基底クラスです。
 * <LI>参照ボタンが押下されたときの挙動はサブクラスで実装する必要があります。
 * <LI>※ボタン横幅の算出には org.eclipse.jface.dialogs.DialogPage 由来のソースコードが含まれます。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public abstract class AbstractBrowseTextField extends TextField implements IBrowseTextField {

	/**
	 * (*)テキストフィールドのためのメッセージ。
	 */
	protected static final TextFieldMessages fMsg = new TextFieldMessages();

	/** 前回入力した値のマップ */
	protected static final Map<String, String> previousInputMap = new HashMap<String, String>();

	/** ボタン */
	protected final Button button;

	/** 拡張子の配列 */
	protected final String[] extensions;

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param labelText ラベル・テキスト
	 * @param extensions 拡張子
	 */
	public AbstractBrowseTextField(final Composite parent, final String labelText, final String... extensions) {
		super(parent, labelText);
		initializeDialogUnits(parent);

		final String prevDir = previousInputMap.get(labelText);
		text.setText(prevDir == null ? "" : prevDir); //$NON-NLS-1$

		button = new Button(parent, SWT.PUSH);
		button.setText(fMsg.getMsg001());
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				openDialog();
			}
		});
		setButtonLayoutData(button);

		this.extensions = new String[extensions.length];
		for (int i = 0; i < extensions.length; i++) {
			this.extensions[i] = "*" + extensions[i]; //$NON-NLS-1$
		}
	}

	@Override
	public void setText(String path) {
		path = resolvePath(path);
		if (new File(path).exists()) {
			super.setText(path);
		} else {
			final String prevDir = previousInputMap.get(labelText);
			super.setText(prevDir);
		}
	}

	@Override
	public String getText() {
		final String path = super.getText();
		if (path.length() > 0) {
			previousInputMap.put(labelText, path);
		}
		return path;
	}

	/**
	 * {@inheritDoc}
	 */
	public File getFile() {
		return new File(getText());
	}

	/**
	 * ダイアログのオープン。
	 */
	abstract protected void openDialog();

	/**
	 * パスの解決。
	 * @param path パス
	 * @return 解決後のパス
	 */
	protected String resolvePath(final String path) {
		return path;
	}

	/**
	 * パスが不正か判定。
	 * @return 不正な場合は true
	 */
	public boolean invalidExtension() {
		if (extensions.length == 0) {
			return false;
		}
		final String path = super.getText();
		if (path == null || path.equals("")) { //$NON-NLS-1$
			return false;
		}
		for (final String extension : extensions) {
			final String ext = extension.replace("*", ""); //$NON-NLS-1$ //$NON-NLS-2$
			if (path.endsWith(ext)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public void setEnabled(final boolean enabled) {
		text.setEnabled(enabled);
		button.setEnabled(enabled);
	}

	/**
	 * 前回入力した値のマップをクリア。
	 */
	public static void clearPreviousInput() {
		previousInputMap.clear();
	}

	// -------------------------------------------------------------------------
	// この場所以降は、ボタンの横幅を算出するためのコードです。
	// org.eclipse.jface.dialogs.DialogPage を由来とするものです。

	/** フォント・メトリクス */
	private FontMetrics fontMetrics;

	/**
	 * ボタン・レイアウト・データのセット。
	 * @param button ボタン
	 * @return グリッド・データ
	 */
	private GridData setButtonLayoutData(final Button button) {
		final GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		final int widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
		final Point minSize = button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		data.widthHint = Math.max(widthHint, minSize.x);
		button.setLayoutData(data);
		return data;
	}

	/**
	 * 水平ダイアログ単位の数に対応するピクセル数の取得。
	 * @param dlus 水平ダイアログ単位の数
	 * @return ピクセル数
	 */
	private int convertHorizontalDLUsToPixels(final int dlus) {
		if (fontMetrics == null) {
			return 0;
		}
		return Dialog.convertHorizontalDLUsToPixels(fontMetrics, dlus);
	}

	/**
	 * ダイアログ単位の初期化。
	 * @param testControl テストするコントロール
	 */
	private void initializeDialogUnits(final Control testControl) {
		final GC gc = new GC(testControl);
		gc.setFont(JFaceResources.getDialogFont());
		fontMetrics = gc.getFontMetrics();
		gc.dispose();
	}
}
