/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.xliff.core;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

import benten.core.BentenConstants;
import benten.core.model.BentenXliff;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.xliff.core.valueobject.BentenCleanXliffProcessInput;
import benten.twa.xliff.messages.BentenCleanXliffMessages;
import blanco.commons.util.BlancoStringUtil;
import blanco.xliff.valueobject.BlancoXliffAltTrans;
import blanco.xliff.valueobject.BlancoXliffNote;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * XLIFF フラグメントのクリーン
 *
 * <pre>
 * XLIFF フラグメントを、もとになった XLIFF にクリーンします。
 *   1.  所定のルールにもとづき、XLIFF をクリーンします。
 * </pre>
 * 
 * @author IGA Tosiki
 */
public class BentenCleanXliffProcessImpl extends AbstractTraverseDir implements BentenCleanXliffProcess {
	/**
	 * XLIFF クリーン機能のためのメッセージ。
	 */
	protected static final BentenCleanXliffMessages fMsg = new BentenCleanXliffMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenCleanXliffProcessInput fInput;

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenCleanXliffProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenCleanXliffProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenCleanXliffProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File fileTargetdir = new File(input.getTargetdir());
		if (fileTargetdir.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE021(fInput.getTargetdir()));
		}
		if (fileTargetdir.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE022(fInput.getTargetdir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenCleanXliffProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				fCounter += 1;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
				// こちらはカウントしません。
			}

			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(new File(input.getTargetdir()));
		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(new File(input.getTargetdir()));

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * 処理の内部処理でアイテムが処理されるたびに進捗報告としてコールバック。
	 *
	 * @param argProgressMessage 現在処理しているアイテムに関するメッセージ。
	 * @return 処理をそのまま継続する場合は false。処理中断をリクエストしたい場合は true。
	 */
	public boolean progressInner(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	public void processFile(final File fromXliffFile, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenCleanXliffProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(fromXliffFile.getName()))) {
			return;
		}

		final BentenXliff inXliff = BentenXliff.loadInstance(fromXliffFile);

		class MyTraverseDir extends AbstractTraverseDir {
			boolean fIsSaved = false;

			public void execute(final BentenXliff toXliff, final File dirTo) throws IOException {
				processDir(dirTo);
			}

			@Override
			protected boolean canProcess(final File file) {
				return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
			}

			/**
			 * 保管がおこなわれたかどうか。
			 * @return 保管が行われていれば true。
			 */
			public boolean isSaved() {
				return fIsSaved;
			}

			@Override
			protected void processFile(final File toXliffFile, final String baseDir) throws IOException {
				final BentenXliff toXliff = BentenXliff.loadInstance(toXliffFile);
				int transUnitCount = 0;
				final int transUnitTotalCount = toXliff.getAllTransUnitList().size();
				for (final BlancoXliffTransUnit transUnit : toXliff.getAllTransUnitList()) {
					if (progressInner(fMsg.getCoreP101(toXliffFile.getName(), BigDecimal.valueOf(++transUnitCount),
							BigDecimal.valueOf(transUnitTotalCount)))) {
						break;
					}

					cleanAltTrans(transUnit);
					cleanAction(transUnit);
				}

				try {
					if (toXliff.save()) {
						// XLIFF に実際に保管が実行されました。
						fIsSaved = true;
						getResultInfo().setSuccessCount(getResultInfo().getSuccessCount());
					} else {
						getResultInfo().setSkipCount(getResultInfo().getSkipCount());
					}
				} catch (final IOException e) {
					throw new IllegalArgumentException(fMsg.getCoreE013(), e);
				}

				if (fIsSaved) {
					if (progress(fMsg.getCoreP012())) {
						return;
					}
				} else {
					if (progress(fMsg.getCoreP013())) {
						return;
					}
				}
			}

			/**
			 * 同じ tool-id で複数の alt-trans が存在する場合、最後の(もっとも末尾にある)ものだけを残し、他は削除します。
			 * blancoNLpackGenerator から仮で移植。
			 *
			 * @param transUnit
			 */
			private void cleanAltTrans(final BlancoXliffTransUnit transUnit) {
				// benten および omegat 翻訳分を削除
				for (int indexAltTrans = transUnit.getAltTransList().size() - 1; indexAltTrans >= 0; indexAltTrans--) {
					final BlancoXliffAltTrans altTrans = transUnit.getAltTransList().get(indexAltTrans);

					if (BlancoStringUtil.null2Blank(altTrans.getMatchQuality()).length() > 0) {
						// match-quality の記載があるもののみ除去対象とします。
						if ("benten".equals(BlancoStringUtil.null2Blank(altTrans.getToolId()))
								|| "omegat".equals(BlancoStringUtil.null2Blank(altTrans.getToolId()))) {
							// Benten または OmegaT が訳出したalt-trans は削除します。

							transUnit.getAltTransList().remove(indexAltTrans);
						}
					}
				}

				// tool-idが重複する alt-trans を検索。
				final Map<String, String> mapPrev = new HashMap<String, String>();
				final Map<String, BlancoXliffAltTrans> mapDup = new HashMap<String, BlancoXliffAltTrans>();
				for (int indexAltTrans = 0; indexAltTrans < transUnit.getAltTransList().size(); indexAltTrans++) {
					final BlancoXliffAltTrans altTrans = transUnit.getAltTransList().get(indexAltTrans);
					if (BlancoStringUtil.null2Blank(altTrans.getToolId()).length() == 0) {
						// tool-idが設定されていないものはスキップします。
						continue;
					}

					if (mapPrev.get(altTrans.getToolId()) != null) {
						// 重複しています。
						mapDup.put(mapPrev.get(altTrans.getToolId()), altTrans);
					}
					mapPrev.put(altTrans.getToolId(), String.valueOf(indexAltTrans));
				}

				// 重複分を削除
				for (int indexAltTrans = transUnit.getAltTransList().size() - 1; indexAltTrans >= 0; indexAltTrans--) {
					if (mapDup.get(String.valueOf(indexAltTrans)) != null) {
						// 重複していたものです。
						transUnit.getAltTransList().remove(indexAltTrans);
					}
				}
			}

			/**
			 * blancoNLpackGenerator から仮で移植。
			 * 
			 * @param transUnit
			 */
			private void cleanAction(final BlancoXliffTransUnit transUnit) {
				// blanconlpackgenerator.actionを検索。
				for (int indexNote = transUnit.getNoteList().size() - 1; indexNote >= 0; indexNote--) {
					final BlancoXliffNote note = transUnit.getNoteList().get(indexNote);
					if (BlancoStringUtil.null2Blank(note.getFrom()).length() == 0) {
						// fromが設定されていないものはスキップします。
						continue;
					}

					if (note.getFrom().equals("blanconlpackgenerator.action")) {
						transUnit.getNoteList().remove(indexNote);
					}
				}
			}
		}
		final MyTraverseDir traverseDir = new MyTraverseDir();
		traverseDir.execute(inXliff, new File(fInput.getTargetdir()));
		if (traverseDir.isSaved()) {
			getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
		} else {
			getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
		}
	}
}
