/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.ui.wizards;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;

import benten.core.BentenProjectNature;
import benten.twa.BentenTwaConstants;
import benten.twa.ui.wizards.messages.NewProjectCreationWizardMessages;
import benten.ui.preference.BentenProjectProperty;
import benten.ui.preference.BentenProjectProperty.ProjectProperty;

/**
 * 翻訳プロジェクト作成ウィザード。
 *
 * <UL>
 * <LI>このプラグインを Eclipse 上で有効化するためのプラグイン・クラスです。
 * </UL>
 *
 * @author YAMAMOTO Koji
 * @author KASHIHARA Shinji
 */
public class NewProjectCreationWizard extends Wizard implements INewWizard, IExecutableExtension {

	/**
	 * 翻訳プロジェクト作成ウィザードのためのメッセージ。
	 */
	protected static final NewProjectCreationWizardMessages fMsg = new NewProjectCreationWizardMessages();

	/** ウィザードのページ。 */
	private NewProjectCreationWizardPage page;

	/**
	 * コンストラクター。
	 */
	public NewProjectCreationWizard() {
		setWindowTitle(fMsg.getWindowTitle());
	}

	/**
	 * {@inheritDoc}
	 */
	public void init(final IWorkbench workbench, final IStructuredSelection selection) {
	}

	/**
	 * {@inheritDoc}
	 */
	public void setInitializationData(final IConfigurationElement config, final String propertyName, final Object data)
			throws CoreException {
	}

	@Override
	public void addPages() {
		page = new NewProjectCreationWizardPage("page"); //$NON-NLS-1$
		page.setTitle(fMsg.getWizardPageTitle());
		page.setDescription(fMsg.getWizardPageDescription());
		addPage(page);
	}

	@Override
	public boolean performFinish() {
		final IProject project = page.getProjectHandle();
		try {
			if (page.validateInput() == false) {
				return false;
			}

			createProject(project);
			project.refreshLocal(IResource.DEPTH_INFINITE, null);
			showPerspective();

		} catch (final CoreException e) {
			throw new IllegalStateException(e);
		}
		return true;
	}

	/**
	 * プロジェクトの作成。
	 * @param project プロジェクト
	 * @throws CoreException コア例外が発生した場合
	 */
	public void createProject(final IProject project) throws CoreException {

		if (!project.exists()) {
			project.create(null);
		}
		project.open(null);

		// 翻訳プロジェクト・ネーチャーの追加
		final IProjectDescription description = project.getDescription();
		if (!description.hasNature(BentenProjectNature.NATURE_ID)) {
			final String[] ids = description.getNatureIds();
			final String[] newIds = new String[ids.length + 1];
			System.arraycopy(ids, 0, newIds, 0, ids.length);
			newIds[ids.length] = BentenProjectNature.NATURE_ID;
			description.setNatureIds(newIds);
			project.setDescription(description, null);
		}

		// インポートされた HTML が格納されるデフォルト・ディレクトリは [/.benten/source] とします。
		// このディレクトリには、現在は html しか格納されませんが、
		// 将来は docx や txt があわせて格納されることを想定しています。
		createFolder(project, BentenTwaConstants.DEFAULT_DIR);
		createFolder(project, BentenTwaConstants.DEFAULT_SOURCE_DIR);
		createFolder(project, BentenTwaConstants.DEFAULT_TMP_DIR);

		createFolder(project, BentenTwaConstants.DEFAULT_XLIFF_DIR);
		createFolder(project, BentenTwaConstants.DEFAULT_TMX_DIR);
		createFolder(project, BentenTwaConstants.DEFAULT_XLIFF_FRAGMENT_DIR);
		createFolder(project, BentenTwaConstants.DEFAULT_TMX_FRAGMENT_DIR);

		final IPreferenceStore projectStore = BentenProjectProperty.getStore(project);
		projectStore.setValue(ProjectProperty.TRANS_SOURCE_LANG.name(), page.getTransSourceLang());
		projectStore.setValue(ProjectProperty.TRANS_TARGET_LANG.name(), page.getTransTargetLang());
	}

	/**
	 * フォルダーの作成。
	 * @param project プロジェクト。
	 * @param name フォルダーの名前。
	 * @throws CoreException 例外が発生した場合。
	 */
	private void createFolder(final IProject project, final String name) throws CoreException {
		final IFolder folder = project.getFolder(new Path(name));
		if (!folder.exists()) {
			folder.create(true, true, null);
		}
	}

	/**
	 * パースペクティブの表示。
	 */
	private void showPerspective() {
		final IWorkbench workbench = PlatformUI.getWorkbench();
		final IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		final IWorkbenchPage page = window.getActivePage();
		IAdaptable input;
		if (page != null)
			input = page.getInput();
		else
			input = ResourcesPlugin.getWorkspace().getRoot();
		try {
			workbench.showPerspective("benten.translationPerspective", window, input); //$NON-NLS-1$
		} catch (final WorkbenchException e) {
			throw new IllegalStateException(e);
		}
	}
}
