/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.ui.jobs;

import java.io.IOException;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.IHandler;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.HandlerUtil;
import org.xml.sax.SAXException;

import benten.twa.process.BentenProcessResultInfo;
import benten.twa.ui.dialogs.AbstractWorkflowDialog;

/**
 * 翻訳ワークフローで使用するハンドラー基底クラス。
 *
 * @author KASHIHARA Shinji
 */
public abstract class AbstractWorkflowHandler extends AbstractHandler implements IHandler {

	/** ジョブ・メッセージ */
	protected String fJobMessage;

	/** 進捗モニター */
	protected IProgressMonitor monitor = new NullProgressMonitor();

	/** 処理結果情報 */
	protected BentenProcessResultInfo resultInfo;

	/**
	 * ジョブ・メッセージをセット。
	 * @param jobMessage ジョブ・メッセージ
	 */
	protected void setJobMessage(final String jobMessage) {
		fJobMessage = jobMessage;
	}

	/*
	 * ジョブ {@link WorkflowJob} の実行。<br>
	 * このメソッドはメニューから呼び出されます。
	 */
	public Object execute(final ExecutionEvent event) throws ExecutionException {

		// エディターをすべて保管（引数 true: 未保管の場合はダイアログ表示）
		PlatformUI.getWorkbench().saveAllEditors(true);

		// 選択を取得
		final ISelection sel = HandlerUtil.getCurrentSelection(event);
		IStructuredSelection selection = StructuredSelection.EMPTY;
		if (sel instanceof IStructuredSelection) {
			selection = (IStructuredSelection) sel;
		}

		// ダイアログを作成 (テンプレート・メソッド)
		final Shell shell = HandlerUtil.getActiveShell(event);
		final AbstractWorkflowDialog dialog = (AbstractWorkflowDialog) createDialog(shell, selection);

		if (dialog != null && dialog.open() == Dialog.OK) {

			// ジョブ開始
			final WorkflowJob job = new WorkflowJob(fJobMessage);
			job.setStatus(dialog.getStatus());
			job.setHandler(this);
			job.schedule();
		}
		return null;
	}

	/**
	 * ダイアログを作成。
	 * @param shell シェル
	 * @param selection 選択
	 * @return ダイアログ
	 */
	abstract protected StatusDialog createDialog(Shell shell, IStructuredSelection selection);

	/**
	 * 処理の起動。<br>
	 * このメソッドはジョブ {@link WorkflowJob} から呼び出されます。
	 * @param iStatus ステータス
	 * @throws IOException 入出力例外が発生した場合
	 * @throws SAXException SAX 解析でエラーが発生した場合
	 */
	public void run(final IStatus iStatus) throws IOException, SAXException {

		// テンプレート・メソッド呼び出し
		processFile(iStatus);

		if (monitor.isCanceled()) {
			return;
		}
		monitor.done();
	}

	/**
	 * ファイルを処理するテンプレート・メソッド。
	 * @param iStatus ステータス (処理パラメーター)
	 * @throws IOException 入出力例外が発生した場合
	 * @throws SAXException SAX 解析でエラーが発生した場合
	 */
	abstract protected void processFile(IStatus iStatus) throws IOException, SAXException;

	/**
	 * キャンセルなしで正常終了した場合の処理。<br>
	 * デフォルトでは何も行いません。
	 * @param iStatus ステータス (処理パラメーター)
	 */
	protected void complete(final IStatus iStatus) {
		// 処理済みとしてマークを行う場合などに各ハンドラーでオーバーライド
	}

	/**
	 * この処理の実行結果情報を取得。
	 * @return 処理結果情報
	 */
	public BentenProcessResultInfo getResultInfo() {
		return resultInfo;
	}
}
