/*******************************************************************************
 * Copyright (c) 2001, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.validation.validator;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.mergedoc.pleiades.resource.Property;

import org.eclipse.core.filebuffers.ITextFileBuffer;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.content.IContentDescription;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.core.runtime.content.IContentTypeManager;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.wst.sse.core.StructuredModelManager;
import org.eclipse.wst.sse.core.internal.FileBufferModelManager;
import org.eclipse.wst.sse.core.internal.provisional.IModelManager;
import org.eclipse.wst.sse.core.internal.provisional.INodeNotifier;
import org.eclipse.wst.sse.core.internal.provisional.IStructuredModel;
import org.eclipse.wst.sse.core.internal.provisional.IndexedRegion;
import org.eclipse.wst.sse.core.internal.provisional.text.IStructuredDocument;
import org.eclipse.wst.sse.core.internal.provisional.text.IStructuredDocumentRegion;
import org.eclipse.wst.sse.core.utils.StringUtils;
import org.eclipse.wst.sse.ui.internal.reconcile.validator.ISourceValidator;
import org.eclipse.wst.validation.AbstractValidator;
import org.eclipse.wst.validation.ValidationEvent;
import org.eclipse.wst.validation.ValidationResult;
import org.eclipse.wst.validation.ValidationState;
import org.eclipse.wst.validation.internal.core.Message;
import org.eclipse.wst.validation.internal.core.ValidationException;
import org.eclipse.wst.validation.internal.operations.IWorkbenchContext;
import org.eclipse.wst.validation.internal.operations.LocalizedMessage;
import org.eclipse.wst.validation.internal.provisional.core.IMessage;
import org.eclipse.wst.validation.internal.provisional.core.IReporter;
import org.eclipse.wst.validation.internal.provisional.core.IValidationContext;
import org.eclipse.wst.validation.internal.provisional.core.IValidatorJob;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMElement;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMText;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

import benten.cat.validation.CatValidationPlugin;
import benten.core.dom.TransUnitDelegate;

/**
 * 翻訳バリデーター。
 *
 * <UL>
 * <LI>このクラスは拡張ポイント org.eclipse.wst.validation.validatorV2 の実装です。
 * <LI>検証ロジック以外のコードのほとんどは org.eclipse.wst.html.internal.validation.HTMLValidator 由来となっています。
 * <LI>validateTransUnit については、新規に作成したものです。
 * </UL>
 *
 * ★基本設計「翻訳支援機能: 翻訳バリデーター機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings("restriction")
public class TranslationValidator extends AbstractValidator implements IValidatorJob, ISourceValidator,
		IExecutableExtension {

	@Override
	public ValidationResult validate(final ValidationEvent event, final ValidationState state,
			final IProgressMonitor monitor) {
		return null;
	}

	@Override
	public void validationStarting(final IProject project, final ValidationState state, final IProgressMonitor monitor) {
	}

	@Override
	public void validationFinishing(final IProject project, final ValidationState state, final IProgressMonitor monitor) {
	}

	/**
	 * XLIFF コンテンツ・タイプ
	 */
	private static final String XLIFF_CONTENT_TYPE = "benten.core.xliffsource"; //$NON-NLS-1$

	/**
	 * 検証が必要か判定。
	 * @param file ファイル
	 * @return 必要な場合は true
	 */
	static boolean shouldValidate(final IFile file) {
		IResource resource = file;
		do {
			if (resource.isDerived() || resource.isTeamPrivateMember() || !resource.isAccessible()
					|| (resource.getName().charAt(0) == '.' && resource.getType() == IResource.FOLDER)) {
				return false;
			}
			resource = resource.getParent();
		} while ((resource.getType() & IResource.PROJECT) == 0);
		return true;
	}

	/** ドキュメント */
	private IDocument fDocument;

	/** その他サポートするコンテンツ・タイプの配列 */
	private IContentType[] fOtherSupportedContentTypes = null;

	/** 追加コンテンツ・タイプ id の配列 */
	private String[] fAdditionalContentTypesIDs;

	/** コンテンツ・タイプ */
	private final IContentType fContentType;

	/**
	 * コンストラクター。
	 */
	public TranslationValidator() {
		super();
		final IContentTypeManager fContentTypeManager = Platform.getContentTypeManager();
		fContentType = fContentTypeManager.getContentType(XLIFF_CONTENT_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	public void cleanup(final IReporter reporter) {
	}

	/**
	 * その他のサポートされるコンテンツ・タイプの取得。
	 * @return コンテンツ・タイプの配列
	 */
	private IContentType[] getOtherSupportedContentTypes() {
		if (fOtherSupportedContentTypes == null) {
			final List<IContentType> contentTypes = new ArrayList<IContentType>(3);
			if (fAdditionalContentTypesIDs != null) {
				for (int i = 0; i < fAdditionalContentTypesIDs.length; i++) {
					final IContentType type = Platform.getContentTypeManager().getContentType(
							fAdditionalContentTypesIDs[i]);
					if (type != null) {
						contentTypes.add(type);
					}
				}
			}
			fOtherSupportedContentTypes = contentTypes.toArray(new IContentType[contentTypes.size()]);
		}
		return fOtherSupportedContentTypes;
	}

	/**
	 * モデルの取得。
	 * @param project プロジェクト
	 * @param file ファイル
	 * @return モデル
	 */
	private IDOMModel getModel(final IProject project, final IFile file) {
		if (project == null || file == null)
			return null;
		if (!file.exists())
			return null;
		if (!canHandle(file))
			return null;

		IStructuredModel model = null;
		final IModelManager manager = StructuredModelManager.getModelManager();
		try {
			file.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor());
		} catch (final CoreException e) {
			CatValidationPlugin.getDefault().log(e);
		}
		try {
			try {
				model = manager.getModelForRead(file);
			} catch (final UnsupportedEncodingException ex) {
				// retry ignoring META charset for invalid META charset
				// specification
				// recreate input stream, because it is already partially read

				// 以下のコードは FindBugs により「無駄な String の new」と報告されますが、これはそのまま残すこととします。
				model = manager.getModelForRead(file, new String(), null);
			}
		} catch (final UnsupportedEncodingException ex) {
		} catch (final IOException ex) {
		} catch (final CoreException e) {
			CatValidationPlugin.getDefault().log(e);
		}

		if (model == null)
			return null;
		if (!(model instanceof IDOMModel)) {
			releaseModel(model);
			return null;
		}
		return (IDOMModel) model;
	}

	/**
	 * レポーターの取得。
	 * @param reporter レポーター
	 * @param file ファイル
	 * @param model モデル
	 * @return レポーター
	 */
	private TranslationValidationReporter getReporter(final IReporter reporter, final IFile file, final IDOMModel model) {
		return new TranslationValidationReporter(this, reporter, file, model);
	}

	/**
	 * 処理可能か判定。
	 * @param file ファイル
	 * @return 可能な場合は true
	 */
	private boolean canHandle(final IFile file) {
		boolean result = false;
		if (file != null) {
			try {
				final IContentDescription contentDescription = file.getContentDescription();
				if (contentDescription != null) {
					final IContentType fileContentType = contentDescription.getContentType();
					if (fileContentType.isKindOf(fContentType)) {
						result = true;
					} else {
						final IContentType[] otherTypes = getOtherSupportedContentTypes();
						for (int i = 0; i < otherTypes.length; i++) {
							result = result || fileContentType.isKindOf(otherTypes[i]);
						}
					}
				} else if (fContentType != null) {
					result = fContentType.isAssociatedWith(file.getName());
				}
			} catch (final CoreException e) {
				// should be rare, but will ignore to avoid logging "encoding
				// exceptions" and the like here.
				// Logger.logException(e);
			}
		}
		return result;
	}

	/**
	 * モデルのリリース。
	 * @param model モデル
	 */
	private void releaseModel(final IStructuredModel model) {
		if (model != null)
			model.releaseFromRead();
	}

	/**
	 * {@inheritDoc}
	 */
	public void validate(final IValidationContext helper, final IReporter reporter) {
		if (helper == null)
			return;
		if ((reporter != null) && (reporter.isCancelled() == true)) {
			throw new OperationCanceledException();
		}
		final String[] deltaArray = helper.getURIs();
		if (deltaArray != null && deltaArray.length > 0) {
			validateDelta(helper, reporter);
		} else {
			validateFull(helper, reporter);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void validate(final IRegion dirtyRegion, final IValidationContext helper, final IReporter reporter) {
		if (helper == null || fDocument == null)
			return;

		if ((reporter != null) && (reporter.isCancelled() == true)) {
			throw new OperationCanceledException();
		}

		final IStructuredModel model = StructuredModelManager.getModelManager().getExistingModelForRead(fDocument);
		if (model == null)
			return; // error

		try {

			IDOMDocument document = null;
			if (model instanceof IDOMModel) {
				document = ((IDOMModel) model).getDocument();
			}

			if (document == null) {
				model.releaseFromRead();
				return; //ignore
			}

			IPath filePath = null;
			IFile file = null;

			final ITextFileBuffer fb = FileBufferModelManager.getInstance().getBuffer(fDocument);
			if (fb != null) {
				filePath = fb.getLocation();

				if (filePath.segmentCount() > 1) {
					file = ResourcesPlugin.getWorkspace().getRoot().getFile(filePath);
					if (!file.isAccessible()) {
						file = null;
					}
				}
			} else {
				filePath = new Path(model.getId());
			}

			// this will be the wrong region if it's Text (instead of Element)
			// we don't know how to validate Text
			IndexedRegion ir = getCoveringNode(dirtyRegion); //  model.getIndexedRegion(dirtyRegion.getOffset());
			if (ir instanceof Text) {
				while (ir != null && ir instanceof Text) {
					// it's assumed that this gets the IndexedRegion to
					// the right of the end offset
					ir = model.getIndexedRegion(ir.getEndOffset());
				}
			}

			if (ir instanceof INodeNotifier) {

				if (reporter != null) {
					TranslationValidationReporter rep = null;
					rep = getReporter(reporter, file, (IDOMModel) model);
					rep.clear();

					final Message mess = new LocalizedMessage(IMessage.LOW_SEVERITY, filePath.toString().substring(1));
					reporter.displaySubtask(this, mess);
				}
				//validate(ir);
			}
		} finally {
			if (model != null)
				model.releaseFromRead();
		}
	}

	/**
	 * カバーするノードの取得
	 * @param dirtyRegion 編集中の領域
	 */
	private IndexedRegion getCoveringNode(final IRegion dirtyRegion) {

		IndexedRegion largestRegion = null;
		if (fDocument instanceof IStructuredDocument) {
			final IStructuredDocumentRegion[] regions = ((IStructuredDocument) fDocument).getStructuredDocumentRegions(
					dirtyRegion.getOffset(), dirtyRegion.getLength());
			largestRegion = getLargest(regions);
		}
		return largestRegion;
	}

	/**
	 * 最大の領域を取得。
	 * @param sdRegions 構造化ドキュメント領域
	 * @return 最大の領域
	 */
	private IndexedRegion getLargest(final IStructuredDocumentRegion[] sdRegions) {

		if (sdRegions == null || sdRegions.length == 0)
			return null;

		IndexedRegion currentLargest = getCorrespondingNode(sdRegions[0]);
		for (int i = 0; i < sdRegions.length; i++) {
			if (!sdRegions[i].isDeleted()) {
				final IndexedRegion corresponding = getCorrespondingNode(sdRegions[i]);

				if (currentLargest instanceof Text)
					currentLargest = corresponding;

				if (corresponding != null) {
					if (!(corresponding instanceof Text)) {
						if (corresponding.getStartOffset() <= currentLargest.getStartOffset()
								&& corresponding.getEndOffset() >= currentLargest.getEndOffset())
							currentLargest = corresponding;
					}
				}

			}
		}
		return currentLargest;
	}

	/**
	 * 対応するノードの取得。
	 * @param sdRegion 構造化ドキュメント領域
	 * @return 対応するノード
	 */
	private IndexedRegion getCorrespondingNode(final IStructuredDocumentRegion sdRegion) {
		final IStructuredModel sModel = StructuredModelManager.getModelManager().getExistingModelForRead(fDocument);
		IndexedRegion indexedRegion = null;
		try {
			if (sModel != null)
				indexedRegion = sModel.getIndexedRegion(sdRegion.getStart());
		} finally {
			if (sModel != null)
				sModel.releaseFromRead();
		}
		return indexedRegion;
	}

	/**
	 * {@inheritDoc}
	 */
	public void connect(final IDocument document) {
		fDocument = document;
	}

	/**
	 * {@inheritDoc}
	 */
	public void disconnect(final IDocument document) {
		fDocument = null;
	}

	/**
	 * 検証の実施。
	 * @param reporter レポーター。
	 * @param file ファイル。
	 * @param model モデル。
	 * @oaram doMadoMakeMarkers マーカーを作成する場合は true。
	 * @throws CoreException Eclipse 例外が発生した場合。
	 */
	private void validate(final IReporter reporter, final IFile file, final IDOMModel model, final boolean doMakeMarkers)
			throws CoreException {
		if (file == null || model == null)
			return; // error
		final IDOMDocument document = model.getDocument();
		if (document == null)
			return; // error
		final TranslationValidationReporter rep = new TranslationValidationReporter(this, reporter, file, model);

		// 検証
		for (final TransUnitDelegate unit : TransUnitDelegate.listOf(document)) {
			validateTransUnit(rep, unit);
		}

		// 問題マーカー作成。
		// 注意）差分検証 (ファイル編集時など) のときのみマーカーを作成する。
		// 		 フォルダー右クリックからの検証で追加するとエラーが 2 倍になってしまう。
		//		 フォルダー右クリックからの検証では reporter に格納したメッセージを後で
		//		 自動的にマーカーに追加してくれる。
		if (doMakeMarkers) {
			file.deleteMarkers(null, false, IResource.DEPTH_ZERO);
			makeMarkers(rep.getMessages());
			rep.clear();
		}
	}

	/**
	 * マーカーの作成。
	 * @param list メッセージ・リスト
	 */
	private void makeMarkers(final List<IMessage> list) {
		// org.eclipse.wst.validation.internal.MarkerManager#makeMarkers のコピー

		for (final IMessage message : list) {
			IResource res = null;
			Object target = message.getTargetObject();
			if (target != null && target instanceof IResource)
				res = (IResource) target;
			if (res == null) {
				target = message.getAttribute(IMessage.TargetResource);
				if (target != null && target instanceof IResource)
					res = (IResource) target;
			}
			if (res != null) {
				try {
					String id = message.getMarkerId();
					if (id == null)
						id = "benten.cat.validation.problem"; //$NON-NLS-1$
					final IMarker marker = res.createMarker(id);
					marker.setAttributes(message.getAttributes());
					marker.setAttribute(IMarker.MESSAGE, message.getText());
					int markerSeverity = IMarker.SEVERITY_INFO;
					final int sev = message.getSeverity();
					if ((sev & IMessage.HIGH_SEVERITY) != 0)
						markerSeverity = IMarker.SEVERITY_ERROR;
					else if ((sev & IMessage.NORMAL_SEVERITY) != 0)
						markerSeverity = IMarker.SEVERITY_WARNING;
					marker.setAttribute(IMarker.SEVERITY, markerSeverity);
					marker.setAttribute(IMarker.LINE_NUMBER, message.getLineNumber());
					marker.setAttribute(IMarker.CHAR_START, message.getOffset());
					marker.setAttribute(IMarker.CHAR_END, message.getOffset() + message.getLength());
				} catch (final CoreException e) {
					CatValidationPlugin.getDefault().log(e);
				}
			}
		}
	}

	/**
	 * コンテナーの検証。
	 * @param helper ヘルパー
	 * @param reporter レポーター
	 * @param container コンテナー
	 */
	private void validateContainer(final IValidationContext helper, final IReporter reporter, final IContainer container) {
		try {
			final IResource[] resourceArray = container.members(false);
			for (int i = 0; i < resourceArray.length; i++) {
				final IResource resource = resourceArray[i];
				if (resource == null || reporter.isCancelled())
					continue;
				if (resource instanceof IFile) {
					final Message message = new LocalizedMessage(IMessage.LOW_SEVERITY, resource.getFullPath()
							.toString().substring(1));
					reporter.displaySubtask(this, message);
					validateFile(helper, reporter, (IFile) resource, false);
				} else if (resource instanceof IContainer) {
					validateContainer(helper, reporter, (IContainer) resource);
				}
			}
		} catch (final CoreException ex) {
		}
	}

	/**
	 * 差分の検証。
	 * @param helper ヘルパー
	 * @param reporter レポーター
	 */
	private void validateDelta(final IValidationContext helper, final IReporter reporter) {
		final String[] deltaArray = helper.getURIs();
		for (int i = 0; i < deltaArray.length; i++) {
			final String delta = deltaArray[i];
			if (delta == null)
				continue;

			if (reporter != null) {
				final Message message = new LocalizedMessage(IMessage.LOW_SEVERITY, delta.substring(1));
				reporter.displaySubtask(this, message);
			}

			final IResource resource = getResource(delta);
			if (resource == null || !(resource instanceof IFile))
				continue;

			// 最後の引数はマーカーを作成するかどうか。true で作成。
			// validateDelta (差分検証) では明示的にマーカーを作成する必要あり。
			// 逆にファイル検証では自動的にマーカーは作成される。
			validateFile(helper, reporter, (IFile) resource, true);
		}
	}

	/**
	 * ファイルの検証。
	 * @param helper ヘルパー
	 * @param reporter レポーター
	 * @param file ファイル
	 * @oaram doMadoMakeMarkers マーカーを作成する場合は true
	 */
	private void validateFile(final IValidationContext helper, final IReporter reporter, final IFile file,
			final boolean doMakeMarkers) {
		if ((reporter != null) && (reporter.isCancelled() == true)) {
			throw new OperationCanceledException();
		}
		if (!shouldValidate(file)) {
			return;
		}
		final IDOMModel model = getModel(file.getProject(), file);
		if (model == null)
			return;

		try {
			validate(reporter, file, model, doMakeMarkers);
		} catch (final CoreException e) {
			CatValidationPlugin.getDefault().log(e);
		} finally {
			releaseModel(model);
		}
	}

	/**
	 * 完全検証。
	 * @param helper ヘルパー
	 * @param reporter レポーター
	 */
	private void validateFull(final IValidationContext helper, final IReporter reporter) {
		IProject project = null;
		final String[] fileDelta = helper.getURIs();
		if (helper instanceof IWorkbenchContext) {
			final IWorkbenchContext wbHelper = (IWorkbenchContext) helper;
			project = wbHelper.getProject();
		} else if (fileDelta.length > 0) {
			// won't work for project validation (b/c nothing in file delta)
			project = getResource(fileDelta[0]).getProject();
		}
		if (project == null)
			return;
		validateContainer(helper, reporter, project);
	}

	/**
	 * リソースの取得。
	 * @param delta 差分
	 * @return リソース
	 */
	private IResource getResource(final String delta) {
		return ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(delta));
	}

	/**
	 * {@inheritDoc}
	 */
	public ISchedulingRule getSchedulingRule(final IValidationContext helper) {
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	public IStatus validateInJob(final IValidationContext helper, final IReporter reporter) throws ValidationException {
		// Exception catching was removed, see
		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=123600
		final IStatus status = Status.OK_STATUS;
		validate(helper, reporter);
		return status;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setInitializationData(final IConfigurationElement config, final String propertyName, final Object data)
			throws CoreException {
		fAdditionalContentTypesIDs = new String[0];
		if (data != null) {
			if (data instanceof String && data.toString().length() > 0) {
				fAdditionalContentTypesIDs = StringUtils.unpack(data.toString());
			}
		}
	}

	@Override
	public ValidationResult validate(final IResource resource, final int kind, final ValidationState state,
			final IProgressMonitor monitor) {
		if (resource.getType() != IResource.FILE)
			return null;
		final ValidationResult result = new ValidationResult();
		final IReporter reporter = result.getReporter(monitor);
		validateFile(null, reporter, (IFile) resource, false);
		return result;
	}

	//-----------------------------------------------------------------------------
	// ここから XLIFF 固有の検証コード

	/**
	 * 翻訳単位の検証。
	 * @param unit 翻訳単位
	 * @param reporter レポーター
	 */
	protected void validateTransUnit(final TranslationValidationReporter reporter, final TransUnitDelegate unit) {
		final String source = unit.getSource();
		final String target = unit.getTarget();
		final Node targetNode = unit.getTargetNode();

		// trans-unit に target 要素がある場合のみ検証
		if (targetNode != null) {
			final int targetOffset = getTargetStartOffset(targetNode);
			final int targetTextOffset = getTargetTextStartOffset(targetNode) + target.indexOf(target.trim());
			validateTarget(reporter, source, target, targetOffset, targetTextOffset);
		}
	}

	/**
	 * ターゲットの開始オフセットを取得。
	 * @param element 要素
	 * @return ターゲットの開始オフセット
	 */
	private int getTargetStartOffset(final Node element) {
		return ((IDOMElement) element).getStartOffset();
	}

	/**
	 * ターゲット・テキストの開始オフセットを取得。
	 * @param element 要素
	 * @return ターゲット・テキストの開始オフセット
	 */
	private int getTargetTextStartOffset(final Node element) {
		final IDOMText textNode = (IDOMText) element.getFirstChild();
		if (textNode == null) {
			return getTargetStartOffset(element);
		}
		return textNode.getStartOffset();
	}

	/**
	 * 翻訳単位の検証。
	 * @param reporter レポーター
	 * @param source ソース
	 * @param target ターゲット
	 * @param targetOffset ターゲット・オフセット
	 * @param targetTextOffset ターゲット・テキスト・オフセット
	 */
	protected void validateTarget(final TranslationValidationReporter reporter, final String source,
			final String target, final int targetOffset, final int targetTextOffset) {

		final String s = source.trim();
		final String t = target.trim();

		//---------------------------------------------------------------------
		// 翻訳忘れチェック
		//---------------------------------------------------------------------

		if (!"".equals(s) && "".equals(t)) { //$NON-NLS-1$ //$NON-NLS-2$
			reporter.addInfo(targetOffset, 0, "NotTranslated"); //$NON-NLS-1$
			// 翻訳忘れの場合、以降のチェックは行わない
			return;
		}

		if (s.equals(t)) {
			reporter.addInfo(targetOffset, 0, "NotTranslated"); //$NON-NLS-1$
			// 以降のチェックは行わない
			return;
		}

		//---------------------------------------------------------------------
		// 末尾の同一性チェック
		//---------------------------------------------------------------------

		// 末尾の「...」チェック
		if (s.endsWith("...") && !t.endsWith("...")) { //$NON-NLS-1$ //$NON-NLS-2$
			// 「...」がその他複数を示す場合は「、、、」があれば OK
			if (!t.contains("、、、")) { //$NON-NLS-1$
				reporter.addWarning(targetOffset, 0, "ThreeFullStopEllipsisNeeded"); //$NON-NLS-1$
			}
		}
		// 末尾の「..」チェック
		// ただし、以下のチェックでは、翻訳に「...」がある場合はスルーされてしまう。
		else if (s.endsWith("..") && !t.endsWith("..")) { //$NON-NLS-1$ //$NON-NLS-2$
		}

		// 末尾の「.」チェック
		else if (!s.endsWith("..") && s.endsWith(".") && !t.endsWith("。")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			// ・1 単語の場合はチェックしない
			// ・途中に「:」が含まれる場合はチェックしない
			// 例）Incompatible\ file\ format.\ Workspace\ was\ saved\ with\ an\ incompatible\ version\:\ {0}.
			if (s.contains(" ") && !s.contains(":")) { //$NON-NLS-1$ //$NON-NLS-2$
				reporter.addWarning(targetOffset, 0, "KutenNeeded"); //$NON-NLS-1$
			}
		}

		// 末尾の「:」チェック
		else if (s.endsWith(":") && !t.endsWith(":")) { //$NON-NLS-1$ //$NON-NLS-2$
			reporter.addWarning(targetOffset, 0, "SemiColonNeeded"); //$NON-NLS-1$
		}

		//---------------------------------------------------------------------
		// 英数字隣接空白チェック
		//---------------------------------------------------------------------

		final String IGNORE_CHAR = "\\p{ASCII}（）「」、。…"; //$NON-NLS-1$
		Matcher mat = Pattern.compile("[^" + IGNORE_CHAR + "\\(]\\p{Alnum}").matcher(t); //$NON-NLS-1$ //$NON-NLS-2$

		while (mat.find()) {
			reporter.addWarning(targetTextOffset + mat.start(), mat.end() - mat.start(), "AdjacentSpaceCheck"); //$NON-NLS-1$
		}
		mat = Pattern.compile("\\p{Alnum}[^" + IGNORE_CHAR + "\\)]").matcher(t); //$NON-NLS-1$ //$NON-NLS-2$
		while (mat.find()) {
			reporter.addWarning(targetTextOffset + mat.start(), mat.end() - mat.start(), "AdjacentSpaceCheck"); //$NON-NLS-1$
		}

		//---------------------------------------------------------------------
		// 埋め込み引数の出現数チェック
		//---------------------------------------------------------------------

		// {0} などの埋め込み数チェック
		mat = Pattern.compile("\\{[0-9]\\}").matcher(s); //$NON-NLS-1$
		while (mat.find()) {
			final String group = mat.group();
			if (!t.contains(group)) {
				final String[] params = new String[] { mat.group() };
				reporter.addWarning(targetOffset, 0, "IlligalVariableParameter", params); //$NON-NLS-1$
			}
		}

		// 改行数チェック
		final int eCount = source.length() - source.replace("\n", "").length(); //$NON-NLS-1$ //$NON-NLS-2$
		final int jCount = target.length() - target.replace("\n", "").length(); //$NON-NLS-1$ //$NON-NLS-2$
		if (eCount != jCount && eCount < 50) {
			final String[] params = new String[] { Integer.toString(eCount), Integer.toString(jCount) };
			reporter.addWarning(targetOffset, 0, "UnmatchedLineDelimiter", params); //$NON-NLS-1$
		}

		//---------------------------------------------------------------------
		// 用語チェック
		//---------------------------------------------------------------------

		for (final Property term : CatValidationPlugin.vTermProp) {

			final String ng = term.key;
			final String ok = term.value;

			if (t.contains(ng)) {
				if (ok.equals("")) { //$NON-NLS-1$
					final String[] params = new String[] { ng };
					reporter.addWarning(targetOffset, 0, "TermValidation", params); //$NON-NLS-1$
				} else {
					final String[] params = new String[] { ng, ok };
					reporter.addWarning(targetOffset, 0, "TermValidation2", params); //$NON-NLS-1$
				}
			}
		}

		//---------------------------------------------------------------------
		// 対訳正規表現チェック
		//---------------------------------------------------------------------

		for (final Property trans : CatValidationPlugin.vTransProp) {

			final String enPart = trans.key;
			final String jaPart = trans.value;

			if (s.matches(enPart) && !t.matches(jaPart)) {
				final String[] params = new String[2];
				params[0] = enPart.replaceAll("(\\(.+?\\)|\\.\\*)", "").replace("\\s", " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				params[1] = jaPart.replaceAll("(\\(\\?s\\)|\\.\\*\\?(\\(|)|(|\\))\\.\\*)", ""); //$NON-NLS-1$ //$NON-NLS-2$
				reporter.addWarning(targetOffset, 0, "TranslationValidation", params); //$NON-NLS-1$
			}
		}

		for (final Property trans : CatValidationPlugin.vTransReverseProp) {

			final String jaPart = trans.key;
			final String enPart = trans.value;

			if (t.matches(jaPart) && !s.matches(enPart)) {
				final String[] params = new String[2];
				params[0] = enPart.replaceAll("(\\(.+?\\)|\\.\\*)", "").replace("\\s", " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				params[1] = jaPart.replaceAll("(\\(\\?s\\)|\\.\\*\\?(\\(|)|(|\\))\\.\\*)", ""); //$NON-NLS-1$ //$NON-NLS-2$
				reporter.addWarning(targetOffset, 0, "TranslationValidation2", params); //$NON-NLS-1$
			}
		}

		//---------------------------------------------------------------------
		// 翻訳禁止
		//---------------------------------------------------------------------

		if (CatValidationPlugin.vForbiddenTransProp.keySet().contains(s)) {
			reporter.addWarning(targetOffset, 0, "ForbiddenTranslation", s); //$NON-NLS-1$
		}
	}
}
