/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.model;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import benten.core.io.Files;
import benten.core.model.messages.BentenCoreModelMessages;
import blanco.commons.util.BlancoStringUtil;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.BlancoXliffUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * XLIFF モデルとファイルを紐付けて保持するクラス。
 * @author KASHIHARA Shinji
 */
public class BentenXliff {
	/**
	 * Benten コア、モデルのためのメッセージ。
	 */
	protected static final BentenCoreModelMessages fMsg = new BentenCoreModelMessages();

	private final static String[] stateItems = { // Benten 追加定義
	"", "x-reviewed", //$NON-NLS-1$ //$NON-NLS-2$
			// XLIFF Version 1.2 の事前定義
			"final", "needs-adaptation", "needs-l10n", "needs-review-adaptation", "needs-review-l10n", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			"needs-review-translation", "needs-translation", "new", "signed-off", "translated" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$

	/**
	 * state 項目配列の取得。
	 * @return state 項目配列
	 */
	public static String[] getStateItems() {
		final String[] newItems = new String[stateItems.length];
		System.arraycopy(stateItems, 0, newItems, 0, stateItems.length);
		return newItems;
	}

	/**
	 * 指定されたファイルを BentenXliff としてロード。
	 * @param file ファイル
	 * @return BentenXliff
	 */
	public static BentenXliff loadInstance(final File file) {
		final BlancoXliff xliff = new BlancoXliffParser().parse(file);
		if (BlancoStringUtil.null2Blank(xliff.getVersion()).startsWith("1.2") == false) {
			throw new IllegalArgumentException(fMsg.getBentenXliffMsg001(file.getName(), xliff.getVersion()));
		}
		return new BentenXliff(file, xliff);
	}

	/**
	 * 指定されたファイルの新規 BentenXliff インスタンスを作成。
	 * @param file ファイル
	 * @return BentenXliff
	 */
	public static BentenXliff newInstance(final File file) {
		return new BentenXliff(file, new BlancoXliff());
	}

	/** 処理対象ファイル */
	private final File fFile;
	/** 処理対象 XLIFF オブジェクト。 */
	private final BlancoXliff fBlancoXliff;

	private BentenXliff(final File file, final BlancoXliff blancoXliff) {
		fFile = file;
		fBlancoXliff = blancoXliff;
	}

	/**
	 * BlancoXliffFile リストの取得。
	 * @return BlancoXliffFile リスト
	 */
	public List<BlancoXliffFile> getFileList() {
		return fBlancoXliff.getFileList();
	}

	/**
	 * BlancoXliffFile の作成。<br>
	 * 作成された BlancoXliffFile はこのインスタンスに追加されます。
	 * @return BlancoXliffFile
	 */
	public BlancoXliffFile createBlancoXliffFile() {
		final BlancoXliffFile file = new BlancoXliffFile();
		getFileList().add(file);
		return file;
	}

	/**
	 * BlancoXliffTransUnit リストの取得。
	 * @return BlancoXliffTransUnit リスト
	 */
	public List<BlancoXliffTransUnit> getAllTransUnitList() {
		final List<BlancoXliffTransUnit> list = new LinkedList<BlancoXliffTransUnit>();
		for (final BlancoXliffFile file : getFileList()) {
			list.addAll(file.getBody().getTransUnitList());
		}
		return list;
	}

	/**
	 * id を指定して BlancoXliffTransUnit を取得。
	 * @param id id
	 * @return BlancoXliffTransUnit
	 */
	public BlancoXliffTransUnit getTransUnit(final String id) {
		return BlancoXliffUtil.getTransUnit(fBlancoXliff, id);
	}

	/**
	 * ファイルへ保管。
	 * @return 実際に保存した場合には true。ファイルの内容に変化が無いので保存しなかった場合には false。
	 * @throws IOException 入出力例外が発生した場合
	 */
	public boolean save() throws IOException {
		if (BlancoXliffUtil.saveXliffInstance(fBlancoXliff, fFile.getParentFile(), Files.baseName(fFile)) != 0) {
			// 保存しました。
			return true;
		}

		// 保存処理はおこないませんでした。
		return false;
	}
}
