/**
 *	@file	events.js
 *
 *	Alpha から呼び出されるイベントハンドラの定義が含まれています。
 *	Alpha の設定を変更するにはイベントハンドラを書き換えます。
 *
 *	@author	exeal
 *	@date	2005.04.17
 */


// 現在、文書タイプの定義を幾つか用意していますが、これらの定義は正しくないかもしれません。
// 何か文書タイプを作ったら連絡してね。

Array.prototype.forEach = function(pred) {
	for(var i = 0; i < this.length; ++i)
		pred(this[i]);
};

Function.prototype.bind1st = function(x) {
	var self = this;
	return function(y) {
		return self(x, y);
	};
};

Function.prototype.bind2nd = function(y) {
	var self = this;
	return function(x) {
		return self(x, y);
	};
};


/**
 *	Alpha 起動時に呼び出される
 *	@param args : Arguments	コマンドライン
 */
function OnApplicationInitialized(args) {
	// 文書タイプの登録
	var conf = Ambient.configurations;
	conf.addDocumentType("ReadMe", "readme.txt", "", true);	// お遊び (^^;
	conf.addDocumentType("マクロスクリプト構成", "macros.xml", "", true);
	conf.addDocumentType("テキスト", "*.txt;*.wtx");
	conf.addDocumentType("JavaScript", "*.js", "WScript $F");
	conf.addDocumentType("VBScript", "*.vbs", "WScript $F");
	conf.addDocumentType("Perl", "*.pl;*.pm;*.cgi;*.pls", "perl $F");
	conf.addDocumentType("Ruby", "*.rb", "ruby $F");
	conf.addDocumentType("Python", "*.py;*.pys", "python $F");
	conf.addDocumentType("Haskell", "*.has;*.hash;*.hs;*.hls", "C:\\hugs98\\hugs.exe $F");
	conf.addDocumentType("Windows Script Host", "*.wsf;*.wsh", "WScript $F");
	conf.addDocumentType("Windows Script Component", "*.wsc", "WScript $F");
	conf.addDocumentType("XHTML", "*.htm;*.html", "\"C:\\Program Files\\Internet Explorer\\iexplore.exe\" $F");
	conf.addDocumentType("RSS", "*.rdf", "\"C:\\Program Files\\Internet Explorer\\iexplore.exe\" $F");
	conf.addDocumentType("XML", "*.xml", "\"C:\\Program Files\\Internet Explorer\\iexplore.exe\" $F");
	conf.addDocumentType("XML Schema", "*.xsd", "\"C:\\Program Files\\Internet Explorer\\iexplore.exe\" $F");
	conf.addDocumentType("XSL", "*.xsl", "\"C:\\Program Files\\Internet Explorer\\iexplore.exe\" $F");

	// キー割り当て
	var keys = conf.keyboardSchemes("basic");
//	assignWinStandardKeys(keys);
//	keys.save("default");
	keys.load("default");
//	setAinuInput(keys);	// アイヌ文字の割り当て
	conf.apply();

	// ウィンドウの初期状態。
	// 起動時のウィンドウの位置やサイズを変える場合は
	// コメントを外し、値を指定してください
//	Ambient.left = 0;
//	Ambient.top = 0;

	loadAbbreviations();	// 短縮形の読み込み
}

/// アプリケーション終了時に呼び出される
function OnApplicationTerminating() {
	// ここでウィンドウのサイズや開いているファイルの一覧を
	// テキストファイルに保存するなどすれば
	// 次回起動時に復元できる

	saveAbbreviations();	// 短縮形の保存
}

/**
 *	文書タイプ適用時に呼び出される
 *	@param typeName : string	タイプ名
 *	@param editor : TextEditor	適用先テキストエディタ
 */
function OnApplyDocumentType(typeName, editor) {
	var lexer = editor.lexer;
	var pref = editor.preferences;
	var cookies = new Object;

	pref.reset();
//	pref.showHandOnLink = true;	// リンクテキスト上でハンドカーソルを表示

	// NOTE:
	// cookies は注釈 (コメント) とキーワードに名前を付けるのに使ってます。
	// Lexer::AddXXXX の戻り値のコレクションで、
	// 例えば定数を表すキーワードの場合は cookies.constant で
	// クッキー値にアクセスできます。
	// (この管理方法は一例であり、別の書き方をしても構いません)

	pref.tokenDecoration("link").borderStyle = BorderType.UnderlineSolid;
	pref.tokenDecoration("link").borderColor = 0x3333FF;

	// default
	if(typeName == "default") {
	}

	// ReadMe
	else if(typeName == "ReadMe") {
		lexer.tokenEnabled("otherQuotation") = false;
		pref.tokenDecoration("normal").color = 0x666666;
		cookies.bracket = lexer.addSinglelineAnnotation("[", "]", AnnotationRestriction.OnlyHeadOfLine);
		pref.tokenDecoration("annotation_".concat(cookies.bracket)).color = 0xFFFFFF;
		pref.tokenDecoration("annotation_".concat(cookies.bracket)).backgroundColor = 0x666666;
		pref.tokenDecoration("annotation_".concat(cookies.bracket)).boldFont = true;
		cookies.productName = lexer.addKeywords("Alpha");
		pref.tokenDecoration("keyword_".concat(cookies.productName)).borderStyle = BorderType.UnderlineWaved;
		pref.tokenDecoration("doubleQuotation").backgroundColor = 0xEEEEEE;
		pref.tokenDecoration("link").borderStyle = BorderType.UnderlineDotted;
		pref.tokenDecoration("link").borderColor = 0x666666;
	}

	// マクロスクリプト構成
	else if(typeName == "マクロスクリプト構成") {
		applyXML(editor, cookies);
		cookies.xmlElement = lexer.addKeywords("description macro macros script");
		cookies.xmlAttribute = lexer.addKeywords("language name src");
	}

	// テキスト
	else if(typeName == "テキスト") {
		pref.TabWidth = 8;
		cookies.quotation = lexer.addSinglelineAnnotation(">", "", AnnotationRestriction.OnlyHeadOfLine);
		pref.tokenDecoration("annotation_".concat(cookies.quotation)).Color = 0x990000;
		cookies.additional = lexer.addSinglelineAnnotation("#", "", AnnotationRestriction.OnlyHeadOfLine);
		pref.tokenDecoration("annotation_".concat(cookies.additional)).Color = 0x3366FF;
	}

	// JavaScript 1.5
	else if(typeName == "JavaScript") {
		cookies.singlelineDocumentationComment = lexer.addSinglelineAnnotation("///");
		cookies.multilineDocumentationComment = lexer.addMultilineAnnotation("/**", "*/");
		cookies.singlelineComment = lexer.addSinglelineAnnotation("//");
		cookies.multilineComment = lexer.addMultilineAnnotation("/*", "*/");
		cookies.entity = lexer.addKeywords("function var");
		cookies.control = lexer.addKeywords("break case continue default do else for if return switch while with");
		cookies.operator = lexer.addKeywords("delete in instanceof new typeof void");
		cookies.exception = lexer.addKeywords("catch throw try");
		cookies.constant = lexer.addKeywords("false Infinity null this true undefined");
		cookies.futureReserved = lexer.addKeywords("abstract boolean byte char class double enum extends final float goto implements int interface long native package private protected public short static super synchronized throws transient volatile");
		cookies.msExtended = lexer.addKeywords("@cc_on @if @set debugger");
		cookies.nsExtended = lexer.addKeywords("const export get import set");
		lexer.setOperators("++ -- + - ~ ! * / % << >> >>> < > <= >= == != === !== & ^ | && || ? : = *= /= %= += -= <<= >>= >>>= &= ^= |= , ( ) . [ ]");
		lexer.setAdditionalAlphabets("@_$");
		lexer.setBrackets("([{");
	}

	// VBScript 5.6
	else if(typeName == "VBScript") {
		lexer.backSolidusEscapeEnabled = false;
		lexer.caseSensitive = false;
		cookies.singlelineDocumentationComment = lexer.addSinglelineAnnotation("\'\'\'");
		cookies.singlelineComment = lexer.addSinglelineAnnotation("\'");
		cookies.vbscriptPragma = lexer.addSinglelineAnnotation("Option ", "", AnnotationRestriction.OnlyStartOfLine);
//		lexer.addSinglelineAnnotation("Rem", "", 1);
		cookies.entity = lexer.addKeywords("Class Const Dim Function Get Private Property Public ReDim Set Sub");
		cookies.attribute = lexer.addKeywords("ByRef ByVal");
		cookies.control = lexer.addKeywords("Case Default Do Each Else ElseIf End Error Exit For GoTo If In Loop Next On Resume Select Step To Then Wend While");
		cookies.operator = lexer.addKeywords("And Eqv Imp Is Mod New Not Or Xor");
		cookies.command = lexer.addKeywords("Call Execute ExecuteGlobal Let Raise Randomize Stop");
		cookies.constant = lexer.addKeywords("Empty False Me Null Nothing True");
		lexer.setOperators("+ - ^ * / \\ & = < <= > >= <> . ( )");
		lexer.setAdditionalAlphabets("_");
		lexer.setBrackets("(");
		lexer.tokenEnabled("singleQuotation") = false;
	}

	// Perl
	else if(typeName == "Perl") {
		cookies.perlRubyCommandLine = lexer.addSinglelineAnnotation("#!", AnnotationRestriction.OnlyStartOfLine);
		cookies.singlelineComment = lexer.addSinglelineAnnotation("# ");
		cookies.multilineComment = lexer.addMultilineAnnotation("=", "=cut", AnnotationRestriction.OnlyStartOfLine);
		cookies.entity = lexer.addKeywords("local my package sub");
		cookies.control = lexer.addKeywords("continue elsif else for foreach goto if last next redo return unless until while");
		cookies.operator = lexer.addKeywords("and cmp eq ge gt le lt ne not or x xor");
		cookies.command = lexer.addKeywords("format no require use");
		lexer.setOperators("-> ++ -- ** ! ~ \\ + - =~ !~ * / % . << >> < > <= >= == != <=> & | ^ && || .. ... ? : = += -= *= /= %= |= &= ^= <<= >>= **= ||= &&= .= x= , => ( ) [ ]");
		lexer.setAdditionalAlphabets("_$@%");
		lexer.setBrackets("([{");
	}

	// Ruby
	else if(typeName == "Ruby") {
		cookies.perlRubyCommandLine = lexer.addSinglelineAnnotation("#!", AnnotationRestriction.OnlyStartOfLine);
		cookies.singlelineComment = lexer.addSinglelineAnnotation("# ");
		cookies.multilineComment = lexer.addMultilineAnnotation("=begin", "=end", AnnotationRestriction.OnlyStartOfLine);
		cookies.entity = lexer.addKeywords("alias class def defined? module undef");
		cookies.control = lexer.addKeywords("BEGIN begin break case do else elsif END end for if in next redo retry return then unless until when while");
		cookies.operator = lexer.addKeywords("and not or");
		cookies.command = lexer.addKeywords("super yield");
		cookies.exception = lexer.addKeywords("ensure raise rescue");
		cookies.constant = lexer.addKeywords("__FILE__ __LINE__ false nil self true");
		lexer.setOperators(":: . ** - + ! ~ * / % << >> & | ^ > >= < <= <=> == === != =~ !~ && || .. ... ? : = += -= *= /= %= |= &= ^= <<= >>= **= ||= &&= => ( ) [ ]");
		lexer.setAdditionalAlphabets("@_!?$");
		lexer.setBrackets("([");
	}

	// Python
	else if(typeName == "Python") {
		cookies.singlelineComment = lexer.addSinglelineAnnotation("#");
		cookies.multilineComment = lexer.addMultilineAnnotation("\"\"\"", "\"\"\"");
		cookies.multilineComment2 = lexer.addMultilineAnnotation("\'\'\'", "\'\'\'");
		cookies.entity = lexer.addKeywords("class def global lambda");
		cookies.control = lexer.addKeywords("break continue elif else for if return while");
		cookies.operator = lexer.addKeywords("and in is not or");
		cookies.command = lexer.addKeywords("assert del exec from future import pass print yield");
		cookies.exception = lexer.addKeywords("except finally raise try");
		cookies.constant = lexer.addKeywords("Ellipse False None NotImplemented True");
		lexer.setOperators("+ - * ** / // % << >> & | ^ ~ < > <= >= == != <> ( ) [ ] , : . ` = ; += -= *= /= //= %= &= |= ^= >>= <<= **=");
		lexer.setAdditionalAlphabets("_");
		lexer.setBrackets("([{");
	}

	// Haskell
	else if(typeName == "Haskell") {
		cookies.singlelineComment = lexer.addSinglelineAnnotation("--");
	}

	// Windows Script Host
	else if(typeName == "Windows Script Host") {
		applyXML(editor, cookies);
		cookies.wshJobDecl = lexer.addSinglelineAnnotation("<?job", "?>");
		cookies.xmlElement = lexer.addKeywords("description example job named object package reference resource runtime script unnamed usage");
		cookies.xmlAttribute = lexer.addKeywords("classid debug error guid helpstring id language many name object progid required src type version");
	}

	// Windows Script Component
	else if(typeName == "Windows Script Component") {
		applyXML(editor, cookies);
		cookies.wshJobDecl = lexer.addSinglelineAnnotation("<?component", "?>");
		cookies.xmlElement = lexer.addKeywords("comment component event get implements method object package parameter property public put reference registration resource script");
		cookies.xmlAttribute = lexer.addKeywords("classid debug default description dispid error events guid id internalName language name object progid remotable type version");
	}

	// XHTML 1.0 Strict (0)
	// XHTML 1.0 Transitional (1)
	// XHTML 1.0 Frameset (2)
	// XHTML 1.1 + Ruby (3; default)
	// XHTML Basic (4)
	else if(typeName == "XHTML") {
		var version = 3;
		var buffer = editor.buffer;
		var cLines = Math.min(buffer.lineCount, 3);
		var iLine = 0;
		var iDoctypeLine = -1;

		if(buffer.Line(iLine).match(/<!DOCTYPE\s+/)
				|| ((cLines > 1) && buffer.Line(++iLine).match(/<!DOCTYPE\s+/)))
			iDoctypeLine = iLine;
		if(iDoctypeLine != -1) {
			var content = "";
			for(; iLine < cLines; ++iLine) {
				var line = buffer.Line(iLine);
				var iHome = (iLine == iDoctypeLine) ? RegExp.lastIndex : 0;
				var iEndDelimiter = line.indexOf(">", iHome);
				if(iEndDelimiter != -1) {
					content += line.substring(iHome, iEndDelimiter);
					break;
				}
				content += line.substring(iHome) + "\n";
			}
			if(content.match(/html\s+PUBLIC\s+[\"\']-\/\/W3C\/\/DTD\s+XHTML\s+1\.0(\s+Strict)?\/\/EN[\"\']/))
				version = 0;
			else if(content.match(/html\s+PUBLIC\s+[\"\']-\/\/W3C\/\/DTD\s+XHTML\s+1\.0\s+Transitional\/\/EN[\"\']/))
				version = 1;
			else if(content.match(/html\s+PUBLIC\s+[\"\']-\/\/W3C\/\/DTD\s+XHTML\s+1\.0\s+Frameset\/\/EN[\"\']/))
				version = 2;
			else if(content.match(/html\s+PUBLIC\s+[\"\']-\/\/W3C\/\/DTD\s+XHTML\s+1\.1\/\/EN[\"\']/))
				version = 3;
//			else if(content.match(/html\s+PUBLIC\s+[\"\']-\/\/W3C\/\/DTD\s+XHTML\s+Basic\s+1\.0\/\/EN[\"\']/))
//				version = 4;
		}
		applyXML(editor, cookies);
/*		cookies.xhtml1Structure = lexer.addKeywords("body head html title");
		cookies.xhtml1Text = lexer.addKeywords("abbr acronym address blockquote br cite code dfn div em h1 h2 h3 h4 h5 h6 kbd p pre q samp span strong var");
		cookies.xhtml1Hypertext = lexer.addKeywords("a");
		cookies.xhtml1List = lexer.addKeywords("dd dl dt li ol ul");
		cookies.xhtml1Applet = lexer.addKeywords("applet");
		cookies.xhtml1TextExtension = lexer.addKeywords("b bdo big del hr i insert small sub sup tt");
		cookies.xhtml1Forms = lexer.addKeywords("button fieldset form input label legend optgroup option select textarea");
		cookies.xhtml1Table = lexer.addKeywords("caption col colgroup table tbody td tfoot th thead tr");
		cookies.xhtml1Image = lexer.addKeywords("img");
		cookies.xhtml1ClientSideImageMap = lexer.addKeywords("area map");
		cookies.xhtml1Object = lexer.addKeywords("object param");
		cookies.xhtml1Frames = lexer.addKeywords("frame frameset noframes");
		cookies.xhtml1IFrame = lexer.addKeywords("iframe");
		cookies.xhtml1Metainformation = lexer.addKeywords("meta");
		cookies.xhtml1Scripting = lexer.addKeywords("noscript script");
		cookies.xhtml1StyleSheet = lexer.addKeywords("style");
		cookies.xhtml1Link = lexer.addKeywords("link");
		cookies.xhtml1Base = lexer.addKeywords("base");
		cookies.xhtml1Ruby = lexer.addKeywords("rb rbc rp rt rtc ruby");
		cookies.xhtml1Legacy = lexer.addKeywords("basefont center dir font isindex menu s strike u");
*/
		cookies.xmlElement = lexer.addKeywords("a abbr acronym address applet area base bdo big blockquote body br button caption cite code col colgroup dd del dfn div dl dt em fieldset form frame frameset h1 h2 h3 h4 h5 h6 head hr html i iframe img input ins kbd label legend li link map meta noframe noscript object ol optgroup option p param pre q rb rbc rp rt rtc ruby samp script select small span strong style sub sup table tbody td textarea tfoot th thead title tr tt ul var");
		cookies.xmlAttribute = lexer.addKeywords("abbr accept accept-charset accesskey action align alt archive axis border cellpadding cellspacing charset checked cite class classid codebase codetype cols colspan content coords data datetime declare defer dir disabled enctype for frame headers height href hreflang ismap http-equiv id label long maxlength media method multiple name nohref onblur onchange onclick ondblclick onfocus onkeydown onkeypress onkeyup onload onmousedown onmousemove onmouseout onmouseover onmouseup onreset onselect onsubmit onunload rbspan readonly rel rev rows rowspan rules scheme scope selected shape size span src standby summary tabindex title type usemap valign value valuetype version width xml:lang xml:space xmlns");
	}

	// RSS 1.0 (without extension modules)
	else if(typeName == "RSS") {
		applyXML(editor, cookies);
		cookies.xmlElement = lexer.addKeywords("channel description image item items link name rdf:li rdf:RDF rdf:Seq textinput title url");
		cookies.xmlAttribute = lexer.addKeywords("rdf:about rdf:resource xml:lang xmlns xmlns:rdf");
	}

	// XML
	else if(typeName == "XML")
		applyXML(editor, cookies);

	// XML Schema (based on XML Schema Part 0: Primer Second Edition <http://www.w3.org/TR/2004/REC-xmlschema-0-20041028/>)
	else if(typeName == "XML Schema") {
		applyXML(editor, cookies);
		var elements = "all annotation any anyAttribute appinfo attribute attributeGroup choice complexContent complexType documentation element enumeration extension field group import include key keyref length list maxInclusive maxLength minInclusive minLength pattern redefine restriction schema selector sequence simpleContent simpleType union unique";
		cookies.xmlElement = lexer.addKeywords(elements + " " + elements.replace(/\w+/g, "xsd:$&"));
		cookies.xmlAttribute = lexer.addKeywords("abstract attributeFormDefault base block blockDefault default elementFormDefault final finalDefault fixed form itemType memberTypes maxOccurs minOccurs mixed name namespace xsi:noNamespaceSchemaLocation xsi:nil nillable processContents ref schemaLocation xsi:schemaLocation substitutionGroup targetNamespace type xsi:type use xpath");
	}

	// XSL (based on XSL Transformations (XSLT) Version 1.0 <http://www.w3.org/TR/1999/REC-xslt-19991116>)
	else if(typeName == "XSL") {
		applyXML(editor, cookies);
		var elements = "apply-imports apply-templates attribute attribute-set call-template choose comment copy copy-of decimal-format element fallback for-each if import include key message namespace-alias number otherwise output param preserve-space processing-instruction sort strip-space stylesheet template text transform value-of variable when with-param";
		cookies.xmlElement = lexer.addKeywords(elements + " " + elements.replace(/[\w\-]+/g, "xsl:$&"));
		cookies.xmlAttribute = lexer.addKeywords("case-order cdata-section-elements count data-type decimal-separator digit disable-output-escaping doctype-public doctype-system elements encoding exclude-result-prefixes extension-element-prefixes format from grouping-separator grouping-size href id indent infinity lang letter-value level match media-type method minus-sign mode name namespace NaN omit-xml-declaration order pattern-separator percent per-mille priority result-prefix select standalone stylesheet-prefix terminate test use use-attribute-sets value version zero-digit");
	}

	pref.tokenDecoration("asciiControl").color = 0x808000;

	if(typeName == "default");
	else if(typeName == "ReadMe"
			|| typeName == "テキスト") {
		pref.selectEndOfLine = false;
		pref.showEndOfFile = false;
		pref.showEndOfLine = false;
		pref.showCurrentUnderline = false;
		pref.showWhiteSpaceAlternative = false;
		pref.thinCaret = true;
	} else {	// プログラミング用の設定
		// 雑多な設定
		pref.selectEndOfLine = true;
		pref.showEndOfFile = true;
		pref.showEndOfLine = true;
		pref.showCurrentUnderline = true;
		pref.showIndicatorMargin = true;
		pref.showLineNumber = true;
		pref.thinCaret = false;
		pref.lineNumberBorderStyle = BorderType.BorderDotted;
		pref.showWhiteSpaceAlternative = true;

		lexer.tokenEnabled("otherQuotation") = false;

//		pref.tokenDecoration("normal").color = 0x333333;
//		pref.tokenDecoration("normal").backgroundColor = 0xFFFFF9;

		// 空白類文字
		pref.tokenDecoration("whiteSpace").color = 0xD0E0E0;
		pref.tokenDecoration("tab").color = 0xD0E0E0;

		// コメント
		if("singlelineComment" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.singlelineComment)).color = 0x008000;
		if("singlelineComment2" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.singlelineComment2)).color = 0x008000;
		if("multilineComment" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.multilineComment)).color = 0x008000;
		if("multilineComment2" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.multilineComment2)).color = 0x008000;
		if("singlelineDocumentationComment" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.singlelineDocumentationComment)).color = 0x408000;
		if("multilineDocumentationComment" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.multilineDocumentationComment)).color = 0x408000;

		// その他の注釈
		if("vbscriptPragma" in cookies) {
			pref.tokenDecoration("annotation_".concat(cookies.vbscriptPragma)).color = 0xFFFFFF;
			pref.tokenDecoration("annotation_".concat(cookies.vbscriptPragma)).backgroundColor = 0x666666;
			pref.tokenDecoration("annotation_".concat(cookies.vbscriptPragma)).boldFont = true;
		}
		if("perlRubyCommandLine" in cookies) {
			pref.tokenDecoration("annotation_".concat(cookies.perlRubyCommandLine)).color = 0x999999;
//			pref.tokenDecoration("annotation_".concat(cookies.perlRubyCommandLine)).backgroundColor = ;
		}
		if("xmlDecl" in cookies) {
			pref.tokenDecoration("annotation_".concat(cookies.xmlDecl)).color = 0xFFFFFF;
			pref.tokenDecoration("annotation_".concat(cookies.xmlDecl)).backgroundColor = 0x666666;
		}
		if("xmlCDataSec" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.xmlCDataSec)).backgroundColor = 0xFFFFF0;
		if("xmlDocType" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.xmlDocType)).color = 0xFF8000;
		if("wshJobDecl" in cookies)
			pref.tokenDecoration("annotation_".concat(cookies.wshJobDecl)).color = 0x808000;

		// キーワード
		if("entity" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.entity)).color = 0x0000FF;
		if("control" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.control)).color = 0x0000FF;
		if("operator" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.operator)).color = 0x800080;
		if("attribute" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.attribute)).color = 0x0000FF;
		if("exception" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.exception)).color = 0x0000FF;
		if("constant" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.constant)).color = 0x800000;
		if("command" in cookies)
			pref.tokenDecoration("keyword_".concat(cookies.command)).color = 0x0000FF;
		if("xmlElement" in cookies) {
			pref.tokenDecoration("keyword_".concat(cookies.xmlElement)).color = 0x800000;
			pref.tokenDecoration("keyword_".concat(cookies.xmlAttribute)).color = 0x008080;
		}
		if("futureReserved" in cookies) {
			pref.tokenDecoration("keyword_".concat(cookies.futureReserved)).color = 0x0000FF;
			pref.tokenDecoration("keyword_".concat(cookies.futureReserved)).borderColor = 0xCC0000;
			pref.tokenDecoration("keyword_".concat(cookies.futureReserved)).borderStyle = BorderType.UnderlineWaved;
		}
		if("msExtended" in cookies) {
			pref.tokenDecoration("keyword_".concat(cookies.msExtended)).color
			= pref.tokenDecoration("keyword_".concat(cookies.nsExtended)).color = 0x0000FF;
			pref.tokenDecoration("keyword_".concat(cookies.msExtended)).borderColor
			= pref.tokenDecoration("keyword_".concat(cookies.nsExtended)).borderColor = 0xCC0000;
			pref.tokenDecoration("keyword_".concat(cookies.msExtended)).borderStyle
			= pref.tokenDecoration("keyword_".concat(cookies.nsExtended)).borderStyle = BorderType.UnderlineWaved;
		}

		// その他
		pref.tokenDecoration("number").color = 0x800000;
		pref.tokenDecoration("operator").color = 0x666600;
		pref.tokenDecoration("singleQuotation").color = 0x000066;
		pref.tokenDecoration("singleQuotation").backgroundColor = 0xF6FFF6;
		pref.tokenDecoration("doubleQuotation").color = 0x000066;
		pref.tokenDecoration("doubleQuotation").backgroundColor = 0xF9F9FF;
//		pref.tokenDecoration("indicatorMargin").color = 0x808080;
//		pref.tokenDecoration("indicatorMargin").backgroundColor = 0xECE9D8;
		pref.tokenDecoration("lineNumber").color = 0x008080;
		pref.tokenDecoration("emphaticLineNumber").color = 0x008080;
		pref.tokenDecoration("emphaticLineNumber").boldFont = true;
		pref.tokenDecoration("matchBrackets").color = 0x990000;
		pref.tokenDecoration("matchBrackets").boldFont = true;
//		pref.tokenDecoration("matchBrackets").borderStyle = BorderType.BorderSolid;
//		pref.tokenDecoration("matchBrackets").borderColor = 0x3366CC;
		pref.tokenDecoration("endOfLine").color = 0x669999;
		pref.tokenDecoration("endOfFile").color = 0x008080;
//		pref.tokenDecoration("endOfFile").color = 0x006699;
//		pref.tokenDecoration("endOfFile").backgroundColor = 0x99CCFF;

		// XML 用の追加定義
		if(typeName == "マクロスクリプト構成"
				|| typeName == "XHTML" || typeName == "XML" || typeName == "RSS"
				|| typeName == "XML Schema" || typeName == "XSL"
				|| typeName == "Windows Script Host" || typeName == "Windows Script Component") {
			pref.tokenDecoration("operator").color = 0x3333FF;
			pref.tabWidth = 2;
		}
		// Unicode アルファベットの使用
		if(typeName == "VBScript" || typeName == "Perl" || typeName == "Ruby")
			lexer.unicodeAlphabetsEnabled = false;
	}
}

/// Alpha のパスを返す
function getBinaryPath() {
	var path = Ambient.fullName;
	return path.substring(0, path.lastIndexOf("\\") + 1);
}

/**
 *	XML 字句規則の適用
 *	@param editor : TextEditor	テキストエディタ
 *	@param cookies : Object		クッキーのコレクション
 */
function applyXML(editor, cookies) {
	var lexer = editor.lexer;
	cookies.xmlDecl = lexer.addMultilineAnnotation("<?xml", "?>");
	cookies.xmlDocType = lexer.addMultilineAnnotation("<!DOCTYPE", ">");
	cookies.xmlCDataSec = lexer.addMultilineAnnotation("<![CDATA[", "]]>");
	cookies.multilineComment = lexer.addMultilineAnnotation("<!--", "-->");
	lexer.setOperators("< > = /");
	lexer.setAdditionalAlphabets(".-_:");
	lexer.setBrackets("<");
	lexer.tokenEnabled("number") = false;
	editor.preferences.matchBracketScanLines = 8;
}

/// 短縮形の読み込み
function loadAbbreviations() {
	try {
		var fso = new ActiveXObject("Scripting.FileSystemObject");
		WScript.loadConstants(fso);

		var abbrs = Ambient.abbreviations;
		var s = fso.openTextFile(getBinaryPath() + "abbreviations",
					IOMode.ForReading, false, Tristate.TristateUseDefault);

		while(!s.atEndOfStream) {
			var key = s.readLine();
			var value = s.readLine();
			if(key.length == 0 || value.length == 0)
				break;
			abbrs.register(key, value);
		}
		s.close();
	} catch(e) {
	}
}

/// 短縮形の保存
function saveAbbreviations() {
	try {
		var fso = new ActiveXObject("Scripting.FileSystemObject");
		var abbrs = new Enumerator(Ambient.abbreviations);
		var s = fso.openTextFile(getBinaryPath() + "abbreviations",
					IOMode.ForWriting, true, Tristate.TristateUseDefault);

		for(abbrs.moveFirst(); !abbrs.atEnd(); abbrs.moveNext()) {
			var abbr = abbrs.item();
			s.writeLine(abbr);
			s.writeLine(Ambient.abbreviations.expand(abbr));
		}
		s.close();
	} catch(e) {
	}
}

/// アイヌ文字をキーボードから入力できるようにする。
/// キー割り当ては ROM さんの "ainu_input 1.10" によった (http://hp.vector.co.jp/authors/VA038316/)
function setAinuInput(keys) {
	var ainuKeyMap = [
		"€", "C-S-4",
		"セ゚", "C-M-p",
		"ツ゚", "C-M-z",
		"ト゚", "C-M-s",
		"ㇰ", "C-S-h",
		"ㇱ", "C-S-d",
		"ㇲ", "C-S-r",
		"ㇳ", "C-S-s",
		"ㇴ", "C-S-1",
		"ㇵ", "C-S-f",
		"ㇶ", "C-S-v",
		"ㇷ", "C-S-2",
		"ㇷ゚", "C-M-2",
		"ㇸ", "C-S-^",
		"ㇹ", "C-S--",
		"ㇺ", "C-S-]",
		"ㇻ", "C-S-o",
		"ㇼ", "C-S-l",
		"ㇽ", "C-S-.",
		"ㇾ", "C-S-;",
		"ㇿ", "C-S-\\"
	];
	function input(str) {
		Ambient.activeBuffer.editor.inputChar(str);
	}
	for(var i = 0; i < ainuKeyMap.length; i += 2)
		keys.assign(ainuKeyMap[i + 1], input.bind1st(ainuKeyMap[i]));
}

/**
 *	Windows の標準的なキー割り当てをする
 *	@param keys : KeyboardScheme	キーボードスキーム
 */
function assignWinStandardKeys(keys) {
	var keymap = [
		// ファイル
		"Ctrl+N", 1000,
		"Ctrl+O", 1002,
		"Ctrl+F4", 1003,
		"Ctrl+S", 1005,
		"Ctrl+Shift+S", 1006,
		"Alt+F4", 1010,
		// 編集
		"Del", 2000,
		"Bs", 2001,
		"Ctrl+Del", 2002,
		"Ctrl+Bs", 2003,
		"Shift+Del", 2004,
		"Ctrl+Shift+L", 2004,
		"Ctrl+Enter", 2005,
		"Enter", 2006,
		"Ctrl+Z", 2007,
		"Alt+Bs", 2007,
		"Ctrl+Y", 2008,
		"Ctrl+Shift+Z", 2008,
		"Shift+Alt+Bs", 2008,
		"Ctrl+X", 2009,
		"Shift+Del", 2009,
		"Ctrl+C", 2010,
		"Ctrl+Ins", 2010,
		"Ctrl+V", 2011,
		"Shift+Ins", 2011,
		"Tab", 2012,
		"Shift+Tab", 2013,
		"Shift+Alt+X", 2016,
		"Alt+X", 2017,
		"Ctrl+Shift+@", 2018,
		"Ctrl+Shift+7", 2019,
		"Ctrl+^", 2020,
		"Ctrl+Shift+^", 2021,
		"Ctrl+:", 2022,
		"Ctrl+,", 2023,
		"Ins", 2025,
		"Ctrl+Sp", 2026,
		"Alt+Right", 2026,
		"Ctrl+Home", 2027,
		"Ctrl+End", 2028,
		"End", 2030,
		"Right", 2031,
		"Left", 2032,
		"Ctrl+Right", 2035,
		"Ctrl+Left", 2036,
		"Down", 2037,
		"Up", 2038,
		"PgDn", 2039,
		"PgUp", 2040,
		"Ctrl+Shift+Home", 2041,
		"Ctrl+Shift+End", 2042,
		"Shift+End", 2044,
		"Shift+Right", 2045,
		"Shift+Left", 2046,
		"Ctrl+Shift+Right", 2049,
		"Ctrl+Shift+Left", 2050,
		"Shift+Down", 2051,
		"Shift+Up", 2052,
		"Shift+PgDn", 2053,
		"Shift+PgUp", 2054,
		"Ctrl+A", 2055,
		"Ctrl+W", 2056,
		"Esc", 2057,
		"Ctrl+Down", 2060,
		"Ctrl+Up", 2061,
		"Ctrl+PgDn", 2062,
		"Ctrl+PgUp", 2063,
		"Ctrl+Shift+-", 2068,
		"Ctrl+-", 2069,
		"Shift+Alt+Right", 2070,
		"Shift+Alt+Left", 2071,
		"Shift+Alt+Down", 2072,
		"Shift+Alt+Up", 2073,
		"Shift+Alt+End", 2074,
		"Shift+Alt+Home", 2075,
		"Ctrl+Shift+Alt+Right", 2076,
		"Ctrl+Shift+Alt+Left", 2077,
		"Ctrl+Shift+V", 2080,
		"Ctrl+Shift+Ins", 2080,
		"Ctrl+U", 2081,
		"Ctrl+Shift+U", 2082,
		"Shift+Alt+T", 2083,
		"Ctrl+T", 2084,
		"Ctrl+Shift+T", 2085,
		"Alt+-", 2087,
		"Alt+U", 2088,
		"Alt+^", 2089,
		"Home", 2094,
		"Shift+Home", 2096,
		// 検索
		"Ctrl+F", 3000,
		"F3", 3001,
		"Shift+F3", 3002,
		"Ctrl+H", 3003,
		"Ctrl+G", 3008,
		"Ctrl+F2", 3009,
		"F2", 3010,
		"Shift+F2", 3011,
		"Ctrl+Shift+F2", 3012,
		"Alt+F2", 3013,
		"Ctrl+[", 3014,
		"Ctrl+]", 3014,
		"Ctrl+Shift+[", 3015,
		"Ctrl+Shift+]", 3015,
		"Ctrl+I", 3016,
		"Ctrl+Shift+I", 3017,
		"Ctrl+Alt+I", 3018,
		"Ctrl+Shift+Alt+I", 3019,
		"Ctrl+M F", 3020,
		"Ctrl+M R", 3021,
		// 表示
		"Alt+F5", 4005,
		"Ctrl+Tab", 4007,
		"Ctrl+Shift+Tab", 4008,
		"F6", 4014,
		"Shift+F6", 4015,
		// マクロ
		"Ctrl+Shift+R", 5000,
		"Ctrl+Shift+P", 5001,
		// ツール
		"F5", 6003,
		"Shift+F5", 6004,
		// スクリプト
		"Ctrl+Alt+U", 11300,
		"Ctrl+Shift+Alt+U", 11301,
		"F8", 11303
	];
	for(var i = 0; i < keymap.length; i += 2)
		keys.assign(keymap[i], keymap[i + 1]);
}
