module QMMM_core
  !! This module contains data structures and general options regarding
  !! QM/MM calculations using SIESTA as the QM code.
  use precision, only: dp

  implicit none
  public :: doing_QMMM
  public :: set_QMMM
  public :: set_QMMM_cutoff
  public :: set_QMMM_coulomb_type

  public :: QMMM_uses_external_driver
  public :: QMMM_density_rcut
  public :: QMMM_cutoff
  public :: QMMM_coulomb_type
  public :: QMMM_does_pcpot

  public :: QMMM_initialize
  public :: QMMM_end

  ! These constants are used for reference in QMMM_coulomb.
  integer, public, parameter :: COULOMB_EWALD       = 1
  integer, public, parameter :: COULOMB_REAL_CUTOFF = 2

  private
  logical :: QMMM_on = .false.
    !! Whether we are doing a QM/MM calculation.

  logical :: QMMM_driver = .false.
    !! Whether we are using an external MM program as driver.

  logical :: get_pcpot = .true.
    !! Whether we are calculating the QMMM potential. This is used for debugging
    !! and when interacting with machine-learned electronic densities.

  real(dp) :: rho_rcut = 1.0e-6_dp
    !! Used to avoid summing density points that are too close to
    !! partial charges; if not, summations might diverge.

  real(dp) :: QMMM_cut = 20.0_dp
    !! Cut-off for real-space QM/MM interactions.

  integer  :: QMMM_coulomb = COULOMB_EWALD
    !! Sets the algorithm for Coulomb interactions:
    !!   1 - Ewald.
    !!   2 - Direct summation with cut-off scheme.
contains

  function doing_QMMM() result( areWe )
    !! Returns whether we are doing a QMMM calculation.
    implicit none
    logical :: areWe

    areWe = QMMM_on
  end function doing_QMMM

  subroutine set_QMMM( QMMM_input )
    !! Sets whether we are doing a QMMM calculation.
    implicit none
    logical, intent(in) :: QMMM_input

    QMMM_on = QMMM_input
  end subroutine set_QMMM

  function QMMM_uses_external_driver() result( areWe )
    !! Returns whether we are using an external MM driver for MD.
    implicit none
    logical :: areWe

    areWe = QMMM_driver
  end function QMMM_uses_external_driver

  function QMMM_does_pcpot() result( doCalc )
    !! Returns whether we are calculating the electrostatic potential
    !! generated by partial charges.
    implicit none
    logical :: doCalc

    doCalc = get_pcpot
  end function QMMM_does_pcpot

  function QMMM_density_rcut( ) result( d_rcut )
    !! Returns the distance for which a given point is too close
    !! to a density point.
    use precision, only : dp
    implicit none
    real(dp) :: d_rcut

    d_rcut = rho_rcut
  end function QMMM_density_rcut

  function QMMM_cutoff( ) result( d_rcut )
    !! Returns the cut-off distance for QM/MM real-space interactions.
    use precision, only : dp
    implicit none
    real(dp) :: d_rcut

    d_rcut = QMMM_cut
  end function QMMM_cutoff

  subroutine set_QMMM_cutoff( cutoff_in )
    !! Sets the cut-off distance for QM/MM real-space interactions.
    use precision, only : dp
    implicit none
    real(dp), intent(in) :: cutoff_in
      !! Cut-off distance in Bohr.

    QMMM_cut = cutoff_in
  end subroutine set_QMMM_cutoff

  function QMMM_coulomb_type( ) result( coul_type )
    !! Returns an integer indicating the current algorithm used of
    !! QM-MM Coulomb interaction. See QMMM_coulomb for possible values.
    integer :: coul_type

    coul_type = QMMM_coulomb
  end function QMMM_coulomb_type

  subroutine set_QMMM_coulomb_type( coulomb_in )
    !! Sets the type of algorithm for QM-MM Coulomb interactions.
    implicit none
    integer, intent(in) :: coulomb_in
      !! See QMMM_coulomb for the definition of this variable.

    QMMM_coulomb = coulomb_in
  end subroutine set_QMMM_coulomb_type

  subroutine QMMM_initialize( use_external_driver )
    !! Initializes the QMMM module and Ewald summation if necessary.
    use fdf           , only : fdf_get
    use precision     , only : dp
    use QMMM_interface, only : QMMM_communication_setup
    use QMMM_ewald_m  , only : QMMM_set_ewald

    implicit none
    logical, intent(in) :: use_external_driver

    if ( use_external_driver ) then
      QMMM_on     = .true.
      QMMM_driver = .true.
      call QMMM_communication_setup( )
    endif
    QMMM_on = fdf_get( 'QMMM.Enabled', QMMM_on )

    QMMM_coulomb = fdf_get( 'QMMM.CoulombType', QMMM_coulomb )
    QMMM_cut     = fdf_get( 'QMMM.Cutoff'     , QMMM_cut    , 'Bohr' )
    rho_rcut     = fdf_get( 'QMMM.DensityCut' , rho_rcut    , 'Bohr' )

    get_pcpot = fdf_get( 'QMMM.ComputePotential', get_pcpot )

    if ( QMMM_coulomb == COULOMB_EWALD ) call QMMM_set_ewald( )

  end subroutine QMMM_initialize

  subroutine QMMM_end( )
    !! Closes all interface pipes/sockets and deallocates arrays in
    !! structures.
    use QMMM_interface, only : QMMM_communication_close
    implicit none

    if ( QMMM_driver ) call QMMM_communication_close( )
  end subroutine QMMM_end
end module QMMM_core
