/*
 * Serial Studio
 * https://serial-studio.com/
 *
 * Copyright (C) 2020–2025 Alex Spataru
 *
 * This file is dual-licensed:
 *
 * - Under the GNU GPLv3 (or later) for builds that exclude Pro modules.
 * - Under the Serial Studio Commercial License for builds that include
 *   any Pro functionality.
 *
 * You must comply with the terms of one of these licenses, depending
 * on your use case.
 *
 * For GPL terms, see <https://www.gnu.org/licenses/gpl-3.0.html>
 * For commercial terms, see LICENSE_COMMERCIAL.md in the project root.
 *
 * SPDX-License-Identifier: GPL-3.0-only OR LicenseRef-SerialStudio-Commercial
 */

import QtQuick
import QtQuick.Effects
import QtQuick.Layouts
import QtQuick.Controls

import SerialStudio

import "../"

Item {
  id: root

  //
  // Widget data inputs
  //
  required property color color
  required property var windowRoot
  required property LEDPanelModel model

  //
  // Create scrollable grid view with LED states
  //
  Flickable {
    contentWidth: width
    anchors.fill: parent
    anchors.topMargin: 8
    anchors.leftMargin: 8
    anchors.bottomMargin: 8
    contentHeight: Math.max(grid.implicitHeight, height)

    ScrollBar.vertical: ScrollBar {
      id: scroll
    }

    GridLayout {
      id: grid
      rowSpacing: 4
      columnSpacing: 4
      width: parent.width - 8
      anchors.centerIn: parent
      anchors.horizontalCenterOffset: -4
      columns: {
        const itemHeight = 32 + 4
        const availableHeight = root.height - 16
        const itemsPerColumn = Math.floor(availableHeight / itemHeight)
        return (root.model.count <= itemsPerColumn) ? 1 : 2
      }

      Repeater {
        model: root.model.count
        delegate: Rectangle {
          border.width: 1
          Layout.fillWidth: true
          Layout.minimumHeight: 32
          Layout.maximumHeight: 32
          color: Cpp_ThemeManager.colors["widget_base"]
          Layout.alignment: Qt.AlignLeft | Qt.AlignVCenter
          border.color: Cpp_ThemeManager.colors["widget_border"]

          RowLayout {
            id: layout
            spacing: 12
            anchors.margins: 4
            anchors.leftMargin: 24
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.verticalCenter: parent.verticalCenter

            Rectangle {
              id: led
              readonly property color ledColor: root.model.colors[index]

              border.width: 1
              implicitWidth: 18
              implicitHeight: 18
              radius: implicitWidth / 2
              Layout.alignment: Qt.AlignVCenter
              border.color: Cpp_ThemeManager.colors["widget_border"]
              color: root.model.states[index] ? ledColor : Qt.darker(ledColor)

              Behavior on color {ColorAnimation{}}
            }

            Label {
              elide: Qt.ElideRight
              Layout.fillWidth: true
              text: root.model.titles[index]
              Layout.alignment: Qt.AlignVCenter
              font: Cpp_Misc_CommonFonts.monoFont
              horizontalAlignment: Label.AlignLeft
              Layout.maximumWidth: layout.width - 12 - 24
              color: Cpp_ThemeManager.colors["widget_text"]

              Behavior on color {ColorAnimation{}}
            }
          }

          MultiEffect {
            source: led
            width: led.width
            height: led.height
            x: layout.x + led.x
            y: layout.y + led.y

            blurEnabled: true
            blur: root.model.states[index] ? 1 : 0
            blurMax: root.model.states[index] ? 64 : 0
            brightness: root.model.states[index] ? 0.4 : 0
            saturation: root.model.states[index] ? 0.2 : 0
          }

          MultiEffect {
            source: led
            width: led.width
            height: led.height
            x: layout.x + led.x
            y: layout.y + led.y

            blurEnabled: true
            blur: root.model.states[index] ? 0.3 : 0
            blurMax: root.model.states[index] ? 64 : 0
            brightness: root.model.states[index] ? 0.4 : 0
            saturation: root.model.states[index] ? 0.2 : 0
          }
        }
      }
    }
  }
}
