/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <array>
#include <cassert>
#include <iostream>
#include <thread>

#include "../sh4lt/ipcs/reader.hpp"
#include "../sh4lt/logger/console.hpp"
#include "../sh4lt/writer.hpp"

using namespace sh4lt;

static std::atomic_int done(0);
static auto g_log = std::make_shared<logger::Console>();

static const std::string label("check-stress") ;
static const auto sockpath = ShType::get_path(label, ShType::default_group());

// a struct with contiguous data storage 
using Frame = struct frame_t {
  size_t count{0};
  std::array<int, 3> data{{3, 1, 4}};
  // no vector ! vector.data is contiguous, but not vector
};

auto reader() -> bool {
  { // creating one reader
    std::this_thread::sleep_for (std::chrono::milliseconds(10));
    Reader r(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(one reader) new data for client "
          //           << frame->count
          //           << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    assert(r);
    std::cout << "one more reader" << std::endl;
    std::this_thread::sleep_for (std::chrono::milliseconds(1000));
  }
  std::cout << "one less reader" << std::endl;
  { // creating five readers
    std::this_thread::sleep_for (std::chrono::milliseconds(10));
    Reader r1(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(1) new data for client "
          //           << frame->count
          //           << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    assert(r1);
    Reader r2(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(2) new data for client "
          //           << frame->count
          //           << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    assert(r2);
    Reader r3(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(3) new data for client "
          //           << frame->count
          //           << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    assert(r3);
    Reader r4(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(4) new data for client "
          //           << frame->count
          //           << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    assert(r4);
    Reader r5(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(5) new data for client "
          //           << frame->count
          //          << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    assert(r5);
    std::cout << "five more reader" << std::endl;
    std::this_thread::sleep_for (std::chrono::milliseconds(1000));
  }
  std::this_thread::sleep_for (std::chrono::milliseconds(100));
  done.store(1);
  std::cout << "done set to 1" << std::endl;
  return true;
}

auto main() -> int {
  using namespace sh4lt;

  {
    // direct access writer with one reader
    Writer w(ShType("application/x-check-sh4lt", label), sizeof(Frame), g_log);
    assert(w);
    // init
    {
      Frame frame;
      assert(w.copy_to_shm(&frame, sizeof(Frame), -1, -1));
    }
    Reader r(
        sockpath,
        [](void* /*data*/, size_t /*size*/, const sh4lt_time_info_t*) {
          // auto frame = static_cast<Frame *>(data);
          // std::cout << "(0) new data for client "
          //           << frame->count
          //           << " (size " << size << ")"
          //           << std::endl;
        },
        nullptr,
        nullptr,
        g_log.get());
    std::cout << "one reader" << std::endl;
    assert(r);
    auto reader_handle = std::async(std::launch::async, reader);
    while (1 != done.load()) {
      //  the following is locking the shared memory for writing
      auto access = w.get_one_write_access();
      assert(access);
      access->notify_clients(sizeof(Frame), -1, -1);
      auto frame = static_cast<Frame *>(access->get_mem());
      frame->count++;
    }
    assert(reader_handle.get());
  }
  std::this_thread::sleep_for (std::chrono::milliseconds(1000));
  std::cout << "fin" << std::endl;
  return 0;
}

