/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#ifndef SH4LT_WRITER_H_
#define SH4LT_WRITER_H_

#include <memory>
#include <string>

#include "./ipcs/sysv-sem.hpp"
#include "./ipcs/sysv-shm.hpp"
#include "./ipcs/unix-socket-protocol.hpp"
#include "./ipcs/unix-socket-server.hpp"
#include "./logger/logger.hpp"
#include "./shtype/shtype.hpp"
#include "./time.hpp"
#include "./utils/safe-bool-idiom.hpp"

namespace sh4lt {

class OneWriteAccess;
class Writer : public SafeBoolIdiom {
  friend OneWriteAccess;

 public:
  /**
   * Construct a Writer object.
   *
   * \param   data_desr             The ShType description for the frames to be transmitted.
   * \param   memsize               Initial size of the shared memory. Note the shared memory
   *                                can be resized at each frame.
   * \param   log                   Shared pointer holding a Log object where to write internal
   *                                logs.
   * \param   on_client_connect     Callback to be triggered when a follower connects.
   * \param   on_client_disconnect  Callback to be triggered when a follower disconnects.
   * \param   unix_permission       Permission to apply to the internal Unix socket, shared memory
   *                                and semaphore.
   */
  Writer(const ShType& data_descr,
         size_t memsize,
         logger::Logger::ptr log,
         UnixSocketProtocol::ServerSide::onClientConnect on_client_connect = nullptr,
         UnixSocketProtocol::ServerSide::onClientDisconnect on_client_disconnect = nullptr,
         mode_t unix_permission = UnixSocketServer::kDefaultPermission);

  /**
   * Destruct the Writer and releases resources.
   *
   */
  ~Writer() override = default;
  Writer(Writer&&) = delete;
  Writer() = delete;
  Writer(const Writer&) = delete;
  auto operator=(const Writer&) -> Writer& = delete;
  auto operator=(Writer&&) -> Writer& = delete;

  /**
   * Get currently allocated size of the shared memory used by the writer.
   *
   */
  [[nodiscard]] auto alloc_size() const -> size_t;

  /**
   * Get current number of followers.
   * \return the number of followers
   */
  [[nodiscard]] inline auto num_of_followers() const -> size_t { return srv_->num_of_clients(); };

  /**
   * Copy a frame of data to the sh4lt.
   *
   * \param data  Pointer to the begining of the frame.
   * \param size  Size of the frame to copy.
   * \param buffer_date Playhead date for the beginning of the buffer, in nanoseconds, or -1.
   * \param buffer_duration Buffer duration, in nanoseconds, or -1.
   *
   * \return Success of the copy to the shared memory
   *
   */
  auto copy_to_shm(const void* data, size_t size, int64_t buffer_date, int64_t buffer_duration) -> bool;

  /**
   * Provide direct access to the memory with lock. The locked/unlocked state of the shared
   * memory is synchronized with the life of the returned value.
   *
   * \return OneWriteAccess object in a unique pointer. Its destruction release the lock.
   *
   */
  auto get_one_write_access() -> std::unique_ptr<OneWriteAccess>;
  /**
   * Provide lock and resize simultaneously.
   * Same as the get_one_write_access method, but allows to resize the shared memory before
   * the new access
   *
   * \param new_size New size to be allocated.
   *
   * \return OneWriteAccess object in a unique pointer. Its destruction release the lock.
   *
   */
  auto get_one_write_access_resize(size_t new_size) -> std::unique_ptr<OneWriteAccess>;

  /**
   * Provide lock to the memory.
   * Pointer version of get_one_write_access.
   *
   * \return Pointer to the create OneWriteAccess.
   *
   */ 
  auto get_one_write_access_ptr() -> OneWriteAccess*;

  /**
   * Provide lock and resize simultaneously.
   * Pointer version of get_one_write_access_resize
   *
   * \return Pointer to the create OneWriteAccess.
   *
   */ 
  auto get_one_write_access_ptr_resize(size_t new_size) -> OneWriteAccess*;

 private:
  std::string path_{};
  UnixSocketProtocol::onConnectData connect_data_;
  UnixSocketProtocol::ServerSide proto_;
  std::unique_ptr<UnixSocketServer> srv_;
  std::unique_ptr<sysVShm> shm_;
  std::unique_ptr<sysVSem> sem_;
  logger::Logger::ptr log_;
  size_t alloc_size_;
  Time time_{};

  bool is_valid_{true};
  [[nodiscard]] auto is_valid() const -> bool final { return is_valid_; }
};

// see check-sh4lt
class OneWriteAccess {
  friend Writer;

 public:
  /**
   * Get the pointer to the sh4lt memory.
   *
   * \return The pointer to the sh4lt memory
   */
  auto get_mem() -> void* { return mem_; };

  /**
   * Resize the sh4lt with reallocation of a new uninitialized shared memory space.
   *
   * \note This reinitialize the memory. You probably want to apply writes after resizing.
   *
   * \param newsize Expected new size of the sh4lt memory.
   *
   * \return Allocated size, or 0 if resize failed. 
   */
  auto shm_resize(size_t newsize) -> size_t;

  /**
   * Notify sh4lt clients.
   *
   * \note This method must be called only once.
   *
   * \param size Size of the frame to be available for the clients.
   * \param buffer_date Playhead date for the beginning of the buffer, in nanoseconds, or -1.
   * \param buffer_duration Buffer duration, in nanoseconds, or -1.
   *
   * \return Number of notified clients.
   *
   */
  auto notify_clients(size_t size, int64_t buffer_date, int64_t buffer_duration) -> short;

  ~OneWriteAccess() = default;
  OneWriteAccess() = delete;
  OneWriteAccess(OneWriteAccess&&) = delete;
  OneWriteAccess(const OneWriteAccess&) = delete;
  auto operator=(const OneWriteAccess&) -> OneWriteAccess& = delete;
  auto operator=(OneWriteAccess&&) -> OneWriteAccess& = default;

 private:
  OneWriteAccess(
      Writer* writer, sysVSem* sem, void* mem, UnixSocketServer* srv, logger::Logger::ptr log);
  Writer* writer_;
  WriteLock wlock_;
  void* mem_;
  UnixSocketServer* srv_;
  logger::Logger::ptr log_;
  bool has_notified_{false};
};

}  // namespace sh4lt
#endif
