// SPDX-License-Identifier: Apache-2.0 AND MIT

/*
 * OQS OpenSSL 3 provider encoders
 *
 * Code strongly inspired by OpenSSL default provider.
 *
 */

#ifndef ENCODER_PROVIDER
#    error Macro ENCODER_PROVIDER undefined
#endif

#define ENCODER_STRUCTURE_type_specific_keypair   "type-specific"
#define ENCODER_STRUCTURE_type_specific_params    "type-specific"
#define ENCODER_STRUCTURE_type_specific           "type-specific"
#define ENCODER_STRUCTURE_type_specific_no_pub    "type-specific"
#define ENCODER_STRUCTURE_PKCS8                   "pkcs8"
#define ENCODER_STRUCTURE_SubjectPublicKeyInfo    "SubjectPublicKeyInfo"
#define ENCODER_STRUCTURE_PrivateKeyInfo          "PrivateKeyInfo"
#define ENCODER_STRUCTURE_EncryptedPrivateKeyInfo "EncryptedPrivateKeyInfo"
#define ENCODER_STRUCTURE_PKCS1                   "pkcs1"
#define ENCODER_STRUCTURE_PKCS3                   "pkcs3"

/* Arguments are prefixed with '_' to avoid build breaks on certain platforms */
#define ENCODER_TEXT(_name, _sym)                        \
    {_name, "provider=" ENCODER_PROVIDER ",output=text", \
     (oqs_##_sym##_to_text_encoder_functions)}
#define ENCODER(_name, _sym, _fips, _output)                  \
    {_name, "provider=" ENCODER_PROVIDER ",output=" #_output, \
     (oqs_##_sym##_to_##_output##_encoder_functions)}

#define ENCODER_w_structure(_name, _sym, _output, _structure) \
    {_name,                                                   \
     "provider=" ENCODER_PROVIDER ",output=" #_output         \
     ",structure=" ENCODER_STRUCTURE_##_structure,            \
     (oqs_##_sym##_to_##_structure##_##_output##_encoder_functions)}

/*
 * Entries for human text "encoders"
 */

/*
 * Entries for key type specific output formats.  The structure name on these
 * is the same as the key type name.  This allows us to say something like:
 *
 * To replace i2d_{TYPE}PrivateKey(), i2d_{TYPE}PublicKey() and
 * i2d_{TYPE}Params(), use OSSL_ENCODER functions with an OSSL_ENCODER_CTX
 * created like this:
 *
 * OSSL_ENCODER_CTX *ctx =
 *     OSSL_ENCODER_CTX_new_for_pkey(pkey, selection, "DER", "type-specific",
 *                                   NULL, NULL);
 *
 * To replace PEM_write_bio_{TYPE}PrivateKey(), PEM_write_bio_{TYPE}PublicKey()
 * and PEM_write_bio_{TYPE}Params(), use OSSL_ENCODER functions with an
 * OSSL_ENCODER_CTX created like this:
 *
 * OSSL_ENCODER_CTX *ctx =
 *     OSSL_ENCODER_CTX_new_for_pkey(pkey, selection, "PEM", "type-specific",
 *                                   NULL, NULL);
 *
 * We only implement those for which there are current i2d_ and PEM_write_bio
 * implementations.
 */

/*
 * Entries for PKCS#8 and SubjectPublicKeyInfo.
 * The "der" ones are added convenience for any user that wants to use
 * OSSL_ENCODER directly.
 * The "pem" ones also support PEM_write_bio_PrivateKey() and
 * PEM_write_bio_PUBKEY().
 */

///// OQS_TEMPLATE_FRAGMENT_MAKE_START
#ifdef OQS_KEM_ENCODERS


#ifdef OQS_ENABLE_KEM_frodokem_640_aes
ENCODER_w_structure("frodo640aes", frodo640aes, der, PrivateKeyInfo),
ENCODER_w_structure("frodo640aes", frodo640aes, pem, PrivateKeyInfo),
ENCODER_w_structure("frodo640aes", frodo640aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo640aes", frodo640aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo640aes", frodo640aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("frodo640aes", frodo640aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("frodo640aes", frodo640aes),
ENCODER_w_structure("p256_frodo640aes", p256_frodo640aes, der, PrivateKeyInfo),
ENCODER_w_structure("p256_frodo640aes", p256_frodo640aes, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_frodo640aes", p256_frodo640aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_frodo640aes", p256_frodo640aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_frodo640aes", p256_frodo640aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_frodo640aes", p256_frodo640aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_frodo640aes", p256_frodo640aes),
ENCODER_w_structure("x25519_frodo640aes", x25519_frodo640aes, der, PrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640aes", x25519_frodo640aes, pem, PrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640aes", x25519_frodo640aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640aes", x25519_frodo640aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640aes", x25519_frodo640aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x25519_frodo640aes", x25519_frodo640aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x25519_frodo640aes", x25519_frodo640aes),
#endif
#ifdef OQS_ENABLE_KEM_frodokem_640_shake
ENCODER_w_structure("frodo640shake", frodo640shake, der, PrivateKeyInfo),
ENCODER_w_structure("frodo640shake", frodo640shake, pem, PrivateKeyInfo),
ENCODER_w_structure("frodo640shake", frodo640shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo640shake", frodo640shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo640shake", frodo640shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("frodo640shake", frodo640shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("frodo640shake", frodo640shake),
ENCODER_w_structure("p256_frodo640shake", p256_frodo640shake, der, PrivateKeyInfo),
ENCODER_w_structure("p256_frodo640shake", p256_frodo640shake, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_frodo640shake", p256_frodo640shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_frodo640shake", p256_frodo640shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_frodo640shake", p256_frodo640shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_frodo640shake", p256_frodo640shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_frodo640shake", p256_frodo640shake),
ENCODER_w_structure("x25519_frodo640shake", x25519_frodo640shake, der, PrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640shake", x25519_frodo640shake, pem, PrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640shake", x25519_frodo640shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640shake", x25519_frodo640shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_frodo640shake", x25519_frodo640shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x25519_frodo640shake", x25519_frodo640shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x25519_frodo640shake", x25519_frodo640shake),
#endif
#ifdef OQS_ENABLE_KEM_frodokem_976_aes
ENCODER_w_structure("frodo976aes", frodo976aes, der, PrivateKeyInfo),
ENCODER_w_structure("frodo976aes", frodo976aes, pem, PrivateKeyInfo),
ENCODER_w_structure("frodo976aes", frodo976aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo976aes", frodo976aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo976aes", frodo976aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("frodo976aes", frodo976aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("frodo976aes", frodo976aes),
ENCODER_w_structure("p384_frodo976aes", p384_frodo976aes, der, PrivateKeyInfo),
ENCODER_w_structure("p384_frodo976aes", p384_frodo976aes, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_frodo976aes", p384_frodo976aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_frodo976aes", p384_frodo976aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_frodo976aes", p384_frodo976aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_frodo976aes", p384_frodo976aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_frodo976aes", p384_frodo976aes),
ENCODER_w_structure("x448_frodo976aes", x448_frodo976aes, der, PrivateKeyInfo),
ENCODER_w_structure("x448_frodo976aes", x448_frodo976aes, pem, PrivateKeyInfo),
ENCODER_w_structure("x448_frodo976aes", x448_frodo976aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_frodo976aes", x448_frodo976aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_frodo976aes", x448_frodo976aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x448_frodo976aes", x448_frodo976aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x448_frodo976aes", x448_frodo976aes),
#endif
#ifdef OQS_ENABLE_KEM_frodokem_976_shake
ENCODER_w_structure("frodo976shake", frodo976shake, der, PrivateKeyInfo),
ENCODER_w_structure("frodo976shake", frodo976shake, pem, PrivateKeyInfo),
ENCODER_w_structure("frodo976shake", frodo976shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo976shake", frodo976shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo976shake", frodo976shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("frodo976shake", frodo976shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("frodo976shake", frodo976shake),
ENCODER_w_structure("p384_frodo976shake", p384_frodo976shake, der, PrivateKeyInfo),
ENCODER_w_structure("p384_frodo976shake", p384_frodo976shake, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_frodo976shake", p384_frodo976shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_frodo976shake", p384_frodo976shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_frodo976shake", p384_frodo976shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_frodo976shake", p384_frodo976shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_frodo976shake", p384_frodo976shake),
ENCODER_w_structure("x448_frodo976shake", x448_frodo976shake, der, PrivateKeyInfo),
ENCODER_w_structure("x448_frodo976shake", x448_frodo976shake, pem, PrivateKeyInfo),
ENCODER_w_structure("x448_frodo976shake", x448_frodo976shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_frodo976shake", x448_frodo976shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_frodo976shake", x448_frodo976shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x448_frodo976shake", x448_frodo976shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x448_frodo976shake", x448_frodo976shake),
#endif
#ifdef OQS_ENABLE_KEM_frodokem_1344_aes
ENCODER_w_structure("frodo1344aes", frodo1344aes, der, PrivateKeyInfo),
ENCODER_w_structure("frodo1344aes", frodo1344aes, pem, PrivateKeyInfo),
ENCODER_w_structure("frodo1344aes", frodo1344aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo1344aes", frodo1344aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo1344aes", frodo1344aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("frodo1344aes", frodo1344aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("frodo1344aes", frodo1344aes),
ENCODER_w_structure("p521_frodo1344aes", p521_frodo1344aes, der, PrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344aes", p521_frodo1344aes, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344aes", p521_frodo1344aes, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344aes", p521_frodo1344aes, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344aes", p521_frodo1344aes, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_frodo1344aes", p521_frodo1344aes, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_frodo1344aes", p521_frodo1344aes),
#endif
#ifdef OQS_ENABLE_KEM_frodokem_1344_shake
ENCODER_w_structure("frodo1344shake", frodo1344shake, der, PrivateKeyInfo),
ENCODER_w_structure("frodo1344shake", frodo1344shake, pem, PrivateKeyInfo),
ENCODER_w_structure("frodo1344shake", frodo1344shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo1344shake", frodo1344shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("frodo1344shake", frodo1344shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("frodo1344shake", frodo1344shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("frodo1344shake", frodo1344shake),
ENCODER_w_structure("p521_frodo1344shake", p521_frodo1344shake, der, PrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344shake", p521_frodo1344shake, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344shake", p521_frodo1344shake, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344shake", p521_frodo1344shake, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_frodo1344shake", p521_frodo1344shake, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_frodo1344shake", p521_frodo1344shake, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_frodo1344shake", p521_frodo1344shake),
#endif
#ifdef OQS_ENABLE_KEM_ml_kem_512
ENCODER_w_structure("mlkem512", mlkem512, der, PrivateKeyInfo),
ENCODER_w_structure("mlkem512", mlkem512, pem, PrivateKeyInfo),
ENCODER_w_structure("mlkem512", mlkem512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mlkem512", mlkem512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mlkem512", mlkem512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mlkem512", mlkem512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mlkem512", mlkem512),
ENCODER_w_structure("p256_mlkem512", p256_mlkem512, der, PrivateKeyInfo),
ENCODER_w_structure("p256_mlkem512", p256_mlkem512, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_mlkem512", p256_mlkem512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mlkem512", p256_mlkem512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mlkem512", p256_mlkem512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_mlkem512", p256_mlkem512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_mlkem512", p256_mlkem512),
ENCODER_w_structure("x25519_mlkem512", x25519_mlkem512, der, PrivateKeyInfo),
ENCODER_w_structure("x25519_mlkem512", x25519_mlkem512, pem, PrivateKeyInfo),
ENCODER_w_structure("x25519_mlkem512", x25519_mlkem512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_mlkem512", x25519_mlkem512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_mlkem512", x25519_mlkem512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x25519_mlkem512", x25519_mlkem512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x25519_mlkem512", x25519_mlkem512),
#endif
#ifdef OQS_ENABLE_KEM_ml_kem_768
ENCODER_w_structure("mlkem768", mlkem768, der, PrivateKeyInfo),
ENCODER_w_structure("mlkem768", mlkem768, pem, PrivateKeyInfo),
ENCODER_w_structure("mlkem768", mlkem768, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mlkem768", mlkem768, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mlkem768", mlkem768, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mlkem768", mlkem768, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mlkem768", mlkem768),
ENCODER_w_structure("p384_mlkem768", p384_mlkem768, der, PrivateKeyInfo),
ENCODER_w_structure("p384_mlkem768", p384_mlkem768, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_mlkem768", p384_mlkem768, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_mlkem768", p384_mlkem768, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_mlkem768", p384_mlkem768, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_mlkem768", p384_mlkem768, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_mlkem768", p384_mlkem768),
ENCODER_w_structure("x448_mlkem768", x448_mlkem768, der, PrivateKeyInfo),
ENCODER_w_structure("x448_mlkem768", x448_mlkem768, pem, PrivateKeyInfo),
ENCODER_w_structure("x448_mlkem768", x448_mlkem768, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_mlkem768", x448_mlkem768, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_mlkem768", x448_mlkem768, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x448_mlkem768", x448_mlkem768, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x448_mlkem768", x448_mlkem768),
ENCODER_w_structure("X25519MLKEM768", X25519MLKEM768, der, PrivateKeyInfo),
ENCODER_w_structure("X25519MLKEM768", X25519MLKEM768, pem, PrivateKeyInfo),
ENCODER_w_structure("X25519MLKEM768", X25519MLKEM768, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("X25519MLKEM768", X25519MLKEM768, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("X25519MLKEM768", X25519MLKEM768, der, SubjectPublicKeyInfo),
ENCODER_w_structure("X25519MLKEM768", X25519MLKEM768, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("X25519MLKEM768", X25519MLKEM768),
ENCODER_w_structure("SecP256r1MLKEM768", SecP256r1MLKEM768, der, PrivateKeyInfo),
ENCODER_w_structure("SecP256r1MLKEM768", SecP256r1MLKEM768, pem, PrivateKeyInfo),
ENCODER_w_structure("SecP256r1MLKEM768", SecP256r1MLKEM768, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("SecP256r1MLKEM768", SecP256r1MLKEM768, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("SecP256r1MLKEM768", SecP256r1MLKEM768, der, SubjectPublicKeyInfo),
ENCODER_w_structure("SecP256r1MLKEM768", SecP256r1MLKEM768, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("SecP256r1MLKEM768", SecP256r1MLKEM768),
#endif
#ifdef OQS_ENABLE_KEM_ml_kem_1024
ENCODER_w_structure("mlkem1024", mlkem1024, der, PrivateKeyInfo),
ENCODER_w_structure("mlkem1024", mlkem1024, pem, PrivateKeyInfo),
ENCODER_w_structure("mlkem1024", mlkem1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mlkem1024", mlkem1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mlkem1024", mlkem1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mlkem1024", mlkem1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mlkem1024", mlkem1024),
ENCODER_w_structure("p521_mlkem1024", p521_mlkem1024, der, PrivateKeyInfo),
ENCODER_w_structure("p521_mlkem1024", p521_mlkem1024, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_mlkem1024", p521_mlkem1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_mlkem1024", p521_mlkem1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_mlkem1024", p521_mlkem1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_mlkem1024", p521_mlkem1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_mlkem1024", p521_mlkem1024),
ENCODER_w_structure("SecP384r1MLKEM1024", SecP384r1MLKEM1024, der, PrivateKeyInfo),
ENCODER_w_structure("SecP384r1MLKEM1024", SecP384r1MLKEM1024, pem, PrivateKeyInfo),
ENCODER_w_structure("SecP384r1MLKEM1024", SecP384r1MLKEM1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("SecP384r1MLKEM1024", SecP384r1MLKEM1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("SecP384r1MLKEM1024", SecP384r1MLKEM1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("SecP384r1MLKEM1024", SecP384r1MLKEM1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("SecP384r1MLKEM1024", SecP384r1MLKEM1024),
#endif
#ifdef OQS_ENABLE_KEM_bike_l1
ENCODER_w_structure("bikel1", bikel1, der, PrivateKeyInfo),
ENCODER_w_structure("bikel1", bikel1, pem, PrivateKeyInfo),
ENCODER_w_structure("bikel1", bikel1, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("bikel1", bikel1, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("bikel1", bikel1, der, SubjectPublicKeyInfo),
ENCODER_w_structure("bikel1", bikel1, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("bikel1", bikel1),
ENCODER_w_structure("p256_bikel1", p256_bikel1, der, PrivateKeyInfo),
ENCODER_w_structure("p256_bikel1", p256_bikel1, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_bikel1", p256_bikel1, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_bikel1", p256_bikel1, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_bikel1", p256_bikel1, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_bikel1", p256_bikel1, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_bikel1", p256_bikel1),
ENCODER_w_structure("x25519_bikel1", x25519_bikel1, der, PrivateKeyInfo),
ENCODER_w_structure("x25519_bikel1", x25519_bikel1, pem, PrivateKeyInfo),
ENCODER_w_structure("x25519_bikel1", x25519_bikel1, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_bikel1", x25519_bikel1, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x25519_bikel1", x25519_bikel1, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x25519_bikel1", x25519_bikel1, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x25519_bikel1", x25519_bikel1),
#endif
#ifdef OQS_ENABLE_KEM_bike_l3
ENCODER_w_structure("bikel3", bikel3, der, PrivateKeyInfo),
ENCODER_w_structure("bikel3", bikel3, pem, PrivateKeyInfo),
ENCODER_w_structure("bikel3", bikel3, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("bikel3", bikel3, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("bikel3", bikel3, der, SubjectPublicKeyInfo),
ENCODER_w_structure("bikel3", bikel3, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("bikel3", bikel3),
ENCODER_w_structure("p384_bikel3", p384_bikel3, der, PrivateKeyInfo),
ENCODER_w_structure("p384_bikel3", p384_bikel3, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_bikel3", p384_bikel3, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_bikel3", p384_bikel3, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_bikel3", p384_bikel3, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_bikel3", p384_bikel3, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_bikel3", p384_bikel3),
ENCODER_w_structure("x448_bikel3", x448_bikel3, der, PrivateKeyInfo),
ENCODER_w_structure("x448_bikel3", x448_bikel3, pem, PrivateKeyInfo),
ENCODER_w_structure("x448_bikel3", x448_bikel3, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_bikel3", x448_bikel3, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("x448_bikel3", x448_bikel3, der, SubjectPublicKeyInfo),
ENCODER_w_structure("x448_bikel3", x448_bikel3, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("x448_bikel3", x448_bikel3),
#endif
#ifdef OQS_ENABLE_KEM_bike_l5
ENCODER_w_structure("bikel5", bikel5, der, PrivateKeyInfo),
ENCODER_w_structure("bikel5", bikel5, pem, PrivateKeyInfo),
ENCODER_w_structure("bikel5", bikel5, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("bikel5", bikel5, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("bikel5", bikel5, der, SubjectPublicKeyInfo),
ENCODER_w_structure("bikel5", bikel5, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("bikel5", bikel5),
ENCODER_w_structure("p521_bikel5", p521_bikel5, der, PrivateKeyInfo),
ENCODER_w_structure("p521_bikel5", p521_bikel5, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_bikel5", p521_bikel5, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_bikel5", p521_bikel5, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_bikel5", p521_bikel5, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_bikel5", p521_bikel5, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_bikel5", p521_bikel5),
#endif

#endif /* OQS_KEM_ENCODERS */


#ifdef OQS_ENABLE_SIG_ml_dsa_44
ENCODER_w_structure("mldsa44", mldsa44, der, PrivateKeyInfo),
ENCODER_w_structure("mldsa44", mldsa44, pem, PrivateKeyInfo),
ENCODER_w_structure("mldsa44", mldsa44, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mldsa44", mldsa44, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mldsa44", mldsa44, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mldsa44", mldsa44, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mldsa44", mldsa44),
ENCODER_w_structure("p256_mldsa44", p256_mldsa44, der, PrivateKeyInfo),
ENCODER_w_structure("p256_mldsa44", p256_mldsa44, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_mldsa44", p256_mldsa44, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mldsa44", p256_mldsa44, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mldsa44", p256_mldsa44, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_mldsa44", p256_mldsa44, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_mldsa44", p256_mldsa44),
ENCODER_w_structure("rsa3072_mldsa44", rsa3072_mldsa44, der, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_mldsa44", rsa3072_mldsa44, pem, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_mldsa44", rsa3072_mldsa44, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_mldsa44", rsa3072_mldsa44, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_mldsa44", rsa3072_mldsa44, der, SubjectPublicKeyInfo),
ENCODER_w_structure("rsa3072_mldsa44", rsa3072_mldsa44, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("rsa3072_mldsa44", rsa3072_mldsa44),
#endif
#ifdef OQS_ENABLE_SIG_ml_dsa_65
ENCODER_w_structure("mldsa65", mldsa65, der, PrivateKeyInfo),
ENCODER_w_structure("mldsa65", mldsa65, pem, PrivateKeyInfo),
ENCODER_w_structure("mldsa65", mldsa65, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mldsa65", mldsa65, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mldsa65", mldsa65, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mldsa65", mldsa65, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mldsa65", mldsa65),
ENCODER_w_structure("p384_mldsa65", p384_mldsa65, der, PrivateKeyInfo),
ENCODER_w_structure("p384_mldsa65", p384_mldsa65, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_mldsa65", p384_mldsa65, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_mldsa65", p384_mldsa65, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_mldsa65", p384_mldsa65, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_mldsa65", p384_mldsa65, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_mldsa65", p384_mldsa65),
#endif
#ifdef OQS_ENABLE_SIG_ml_dsa_87
ENCODER_w_structure("mldsa87", mldsa87, der, PrivateKeyInfo),
ENCODER_w_structure("mldsa87", mldsa87, pem, PrivateKeyInfo),
ENCODER_w_structure("mldsa87", mldsa87, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mldsa87", mldsa87, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mldsa87", mldsa87, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mldsa87", mldsa87, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mldsa87", mldsa87),
ENCODER_w_structure("p521_mldsa87", p521_mldsa87, der, PrivateKeyInfo),
ENCODER_w_structure("p521_mldsa87", p521_mldsa87, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_mldsa87", p521_mldsa87, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_mldsa87", p521_mldsa87, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_mldsa87", p521_mldsa87, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_mldsa87", p521_mldsa87, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_mldsa87", p521_mldsa87),
#endif
#ifdef OQS_ENABLE_SIG_falcon_512
ENCODER_w_structure("falcon512", falcon512, der, PrivateKeyInfo),
ENCODER_w_structure("falcon512", falcon512, pem, PrivateKeyInfo),
ENCODER_w_structure("falcon512", falcon512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falcon512", falcon512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falcon512", falcon512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("falcon512", falcon512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("falcon512", falcon512),
ENCODER_w_structure("p256_falcon512", p256_falcon512, der, PrivateKeyInfo),
ENCODER_w_structure("p256_falcon512", p256_falcon512, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_falcon512", p256_falcon512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_falcon512", p256_falcon512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_falcon512", p256_falcon512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_falcon512", p256_falcon512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_falcon512", p256_falcon512),
ENCODER_w_structure("rsa3072_falcon512", rsa3072_falcon512, der, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_falcon512", rsa3072_falcon512, pem, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_falcon512", rsa3072_falcon512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_falcon512", rsa3072_falcon512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_falcon512", rsa3072_falcon512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("rsa3072_falcon512", rsa3072_falcon512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("rsa3072_falcon512", rsa3072_falcon512),
#endif
#ifdef OQS_ENABLE_SIG_falcon_padded_512
ENCODER_w_structure("falconpadded512", falconpadded512, der, PrivateKeyInfo),
ENCODER_w_structure("falconpadded512", falconpadded512, pem, PrivateKeyInfo),
ENCODER_w_structure("falconpadded512", falconpadded512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falconpadded512", falconpadded512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falconpadded512", falconpadded512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("falconpadded512", falconpadded512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("falconpadded512", falconpadded512),
ENCODER_w_structure("p256_falconpadded512", p256_falconpadded512, der, PrivateKeyInfo),
ENCODER_w_structure("p256_falconpadded512", p256_falconpadded512, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_falconpadded512", p256_falconpadded512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_falconpadded512", p256_falconpadded512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_falconpadded512", p256_falconpadded512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_falconpadded512", p256_falconpadded512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_falconpadded512", p256_falconpadded512),
ENCODER_w_structure("rsa3072_falconpadded512", rsa3072_falconpadded512, der, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_falconpadded512", rsa3072_falconpadded512, pem, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_falconpadded512", rsa3072_falconpadded512, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_falconpadded512", rsa3072_falconpadded512, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_falconpadded512", rsa3072_falconpadded512, der, SubjectPublicKeyInfo),
ENCODER_w_structure("rsa3072_falconpadded512", rsa3072_falconpadded512, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("rsa3072_falconpadded512", rsa3072_falconpadded512),
#endif
#ifdef OQS_ENABLE_SIG_falcon_1024
ENCODER_w_structure("falcon1024", falcon1024, der, PrivateKeyInfo),
ENCODER_w_structure("falcon1024", falcon1024, pem, PrivateKeyInfo),
ENCODER_w_structure("falcon1024", falcon1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falcon1024", falcon1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falcon1024", falcon1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("falcon1024", falcon1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("falcon1024", falcon1024),
ENCODER_w_structure("p521_falcon1024", p521_falcon1024, der, PrivateKeyInfo),
ENCODER_w_structure("p521_falcon1024", p521_falcon1024, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_falcon1024", p521_falcon1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_falcon1024", p521_falcon1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_falcon1024", p521_falcon1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_falcon1024", p521_falcon1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_falcon1024", p521_falcon1024),
#endif
#ifdef OQS_ENABLE_SIG_falcon_padded_1024
ENCODER_w_structure("falconpadded1024", falconpadded1024, der, PrivateKeyInfo),
ENCODER_w_structure("falconpadded1024", falconpadded1024, pem, PrivateKeyInfo),
ENCODER_w_structure("falconpadded1024", falconpadded1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falconpadded1024", falconpadded1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("falconpadded1024", falconpadded1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("falconpadded1024", falconpadded1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("falconpadded1024", falconpadded1024),
ENCODER_w_structure("p521_falconpadded1024", p521_falconpadded1024, der, PrivateKeyInfo),
ENCODER_w_structure("p521_falconpadded1024", p521_falconpadded1024, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_falconpadded1024", p521_falconpadded1024, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_falconpadded1024", p521_falconpadded1024, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_falconpadded1024", p521_falconpadded1024, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_falconpadded1024", p521_falconpadded1024, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_falconpadded1024", p521_falconpadded1024),
#endif
#ifdef OQS_ENABLE_SIG_sphincs_sha2_128f_simple
ENCODER_w_structure("sphincssha2128fsimple", sphincssha2128fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("sphincssha2128fsimple", sphincssha2128fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("sphincssha2128fsimple", sphincssha2128fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincssha2128fsimple", sphincssha2128fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincssha2128fsimple", sphincssha2128fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("sphincssha2128fsimple", sphincssha2128fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("sphincssha2128fsimple", sphincssha2128fsimple),
ENCODER_w_structure("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_sphincssha2128fsimple", p256_sphincssha2128fsimple),
ENCODER_w_structure("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("rsa3072_sphincssha2128fsimple", rsa3072_sphincssha2128fsimple),
#endif
#ifdef OQS_ENABLE_SIG_sphincs_sha2_128s_simple
ENCODER_w_structure("sphincssha2128ssimple", sphincssha2128ssimple, der, PrivateKeyInfo),
ENCODER_w_structure("sphincssha2128ssimple", sphincssha2128ssimple, pem, PrivateKeyInfo),
ENCODER_w_structure("sphincssha2128ssimple", sphincssha2128ssimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincssha2128ssimple", sphincssha2128ssimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincssha2128ssimple", sphincssha2128ssimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("sphincssha2128ssimple", sphincssha2128ssimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("sphincssha2128ssimple", sphincssha2128ssimple),
ENCODER_w_structure("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple, der, PrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_sphincssha2128ssimple", p256_sphincssha2128ssimple),
ENCODER_w_structure("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple, der, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple, pem, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("rsa3072_sphincssha2128ssimple", rsa3072_sphincssha2128ssimple),
#endif
#ifdef OQS_ENABLE_SIG_sphincs_sha2_192f_simple
ENCODER_w_structure("sphincssha2192fsimple", sphincssha2192fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("sphincssha2192fsimple", sphincssha2192fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("sphincssha2192fsimple", sphincssha2192fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincssha2192fsimple", sphincssha2192fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincssha2192fsimple", sphincssha2192fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("sphincssha2192fsimple", sphincssha2192fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("sphincssha2192fsimple", sphincssha2192fsimple),
ENCODER_w_structure("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_sphincssha2192fsimple", p384_sphincssha2192fsimple),
#endif
#ifdef OQS_ENABLE_SIG_sphincs_shake_128f_simple
ENCODER_w_structure("sphincsshake128fsimple", sphincsshake128fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("sphincsshake128fsimple", sphincsshake128fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("sphincsshake128fsimple", sphincsshake128fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincsshake128fsimple", sphincsshake128fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("sphincsshake128fsimple", sphincsshake128fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("sphincsshake128fsimple", sphincsshake128fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("sphincsshake128fsimple", sphincsshake128fsimple),
ENCODER_w_structure("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_sphincsshake128fsimple", p256_sphincsshake128fsimple),
ENCODER_w_structure("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple, der, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple, pem, PrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple, der, SubjectPublicKeyInfo),
ENCODER_w_structure("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("rsa3072_sphincsshake128fsimple", rsa3072_sphincsshake128fsimple),
#endif
#ifdef OQS_ENABLE_SIG_mayo_1
ENCODER_w_structure("mayo1", mayo1, der, PrivateKeyInfo),
ENCODER_w_structure("mayo1", mayo1, pem, PrivateKeyInfo),
ENCODER_w_structure("mayo1", mayo1, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo1", mayo1, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo1", mayo1, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mayo1", mayo1, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mayo1", mayo1),
ENCODER_w_structure("p256_mayo1", p256_mayo1, der, PrivateKeyInfo),
ENCODER_w_structure("p256_mayo1", p256_mayo1, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_mayo1", p256_mayo1, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mayo1", p256_mayo1, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mayo1", p256_mayo1, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_mayo1", p256_mayo1, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_mayo1", p256_mayo1),
#endif
#ifdef OQS_ENABLE_SIG_mayo_2
ENCODER_w_structure("mayo2", mayo2, der, PrivateKeyInfo),
ENCODER_w_structure("mayo2", mayo2, pem, PrivateKeyInfo),
ENCODER_w_structure("mayo2", mayo2, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo2", mayo2, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo2", mayo2, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mayo2", mayo2, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mayo2", mayo2),
ENCODER_w_structure("p256_mayo2", p256_mayo2, der, PrivateKeyInfo),
ENCODER_w_structure("p256_mayo2", p256_mayo2, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_mayo2", p256_mayo2, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mayo2", p256_mayo2, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_mayo2", p256_mayo2, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_mayo2", p256_mayo2, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_mayo2", p256_mayo2),
#endif
#ifdef OQS_ENABLE_SIG_mayo_3
ENCODER_w_structure("mayo3", mayo3, der, PrivateKeyInfo),
ENCODER_w_structure("mayo3", mayo3, pem, PrivateKeyInfo),
ENCODER_w_structure("mayo3", mayo3, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo3", mayo3, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo3", mayo3, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mayo3", mayo3, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mayo3", mayo3),
ENCODER_w_structure("p384_mayo3", p384_mayo3, der, PrivateKeyInfo),
ENCODER_w_structure("p384_mayo3", p384_mayo3, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_mayo3", p384_mayo3, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_mayo3", p384_mayo3, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_mayo3", p384_mayo3, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_mayo3", p384_mayo3, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_mayo3", p384_mayo3),
#endif
#ifdef OQS_ENABLE_SIG_mayo_5
ENCODER_w_structure("mayo5", mayo5, der, PrivateKeyInfo),
ENCODER_w_structure("mayo5", mayo5, pem, PrivateKeyInfo),
ENCODER_w_structure("mayo5", mayo5, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo5", mayo5, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("mayo5", mayo5, der, SubjectPublicKeyInfo),
ENCODER_w_structure("mayo5", mayo5, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("mayo5", mayo5),
ENCODER_w_structure("p521_mayo5", p521_mayo5, der, PrivateKeyInfo),
ENCODER_w_structure("p521_mayo5", p521_mayo5, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_mayo5", p521_mayo5, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_mayo5", p521_mayo5, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_mayo5", p521_mayo5, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_mayo5", p521_mayo5, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_mayo5", p521_mayo5),
#endif
#ifdef OQS_ENABLE_SIG_cross_rsdp_128_balanced
ENCODER_w_structure("CROSSrsdp128balanced", CROSSrsdp128balanced, der, PrivateKeyInfo),
ENCODER_w_structure("CROSSrsdp128balanced", CROSSrsdp128balanced, pem, PrivateKeyInfo),
ENCODER_w_structure("CROSSrsdp128balanced", CROSSrsdp128balanced, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("CROSSrsdp128balanced", CROSSrsdp128balanced, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("CROSSrsdp128balanced", CROSSrsdp128balanced, der, SubjectPublicKeyInfo),
ENCODER_w_structure("CROSSrsdp128balanced", CROSSrsdp128balanced, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("CROSSrsdp128balanced", CROSSrsdp128balanced),
#endif
#ifdef OQS_ENABLE_SIG_uov_ov_Is_pkc
ENCODER_w_structure("OV_Is_pkc", OV_Is_pkc, der, PrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc", OV_Is_pkc, pem, PrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc", OV_Is_pkc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc", OV_Is_pkc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc", OV_Is_pkc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("OV_Is_pkc", OV_Is_pkc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("OV_Is_pkc", OV_Is_pkc),
ENCODER_w_structure("p256_OV_Is_pkc", p256_OV_Is_pkc, der, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc", p256_OV_Is_pkc, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc", p256_OV_Is_pkc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc", p256_OV_Is_pkc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc", p256_OV_Is_pkc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc", p256_OV_Is_pkc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_OV_Is_pkc", p256_OV_Is_pkc),
#endif
#ifdef OQS_ENABLE_SIG_uov_ov_Ip_pkc
ENCODER_w_structure("OV_Ip_pkc", OV_Ip_pkc, der, PrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc", OV_Ip_pkc, pem, PrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc", OV_Ip_pkc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc", OV_Ip_pkc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc", OV_Ip_pkc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("OV_Ip_pkc", OV_Ip_pkc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("OV_Ip_pkc", OV_Ip_pkc),
ENCODER_w_structure("p256_OV_Ip_pkc", p256_OV_Ip_pkc, der, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc", p256_OV_Ip_pkc, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc", p256_OV_Ip_pkc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc", p256_OV_Ip_pkc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc", p256_OV_Ip_pkc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc", p256_OV_Ip_pkc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_OV_Ip_pkc", p256_OV_Ip_pkc),
#endif
#ifdef OQS_ENABLE_SIG_uov_ov_Is_pkc_skc
ENCODER_w_structure("OV_Is_pkc_skc", OV_Is_pkc_skc, der, PrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc_skc", OV_Is_pkc_skc, pem, PrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc_skc", OV_Is_pkc_skc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc_skc", OV_Is_pkc_skc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Is_pkc_skc", OV_Is_pkc_skc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("OV_Is_pkc_skc", OV_Is_pkc_skc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("OV_Is_pkc_skc", OV_Is_pkc_skc),
ENCODER_w_structure("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc, der, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_OV_Is_pkc_skc", p256_OV_Is_pkc_skc),
#endif
#ifdef OQS_ENABLE_SIG_uov_ov_Ip_pkc_skc
ENCODER_w_structure("OV_Ip_pkc_skc", OV_Ip_pkc_skc, der, PrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc_skc", OV_Ip_pkc_skc, pem, PrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc_skc", OV_Ip_pkc_skc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc_skc", OV_Ip_pkc_skc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("OV_Ip_pkc_skc", OV_Ip_pkc_skc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("OV_Ip_pkc_skc", OV_Ip_pkc_skc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("OV_Ip_pkc_skc", OV_Ip_pkc_skc),
ENCODER_w_structure("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc, der, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_OV_Ip_pkc_skc", p256_OV_Ip_pkc_skc),
#endif
#ifdef OQS_ENABLE_SIG_snova_SNOVA_24_5_4
ENCODER_w_structure("snova2454", snova2454, der, PrivateKeyInfo),
ENCODER_w_structure("snova2454", snova2454, pem, PrivateKeyInfo),
ENCODER_w_structure("snova2454", snova2454, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2454", snova2454, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2454", snova2454, der, SubjectPublicKeyInfo),
ENCODER_w_structure("snova2454", snova2454, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("snova2454", snova2454),
ENCODER_w_structure("p256_snova2454", p256_snova2454, der, PrivateKeyInfo),
ENCODER_w_structure("p256_snova2454", p256_snova2454, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_snova2454", p256_snova2454, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_snova2454", p256_snova2454, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_snova2454", p256_snova2454, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_snova2454", p256_snova2454, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_snova2454", p256_snova2454),
#endif
#ifdef OQS_ENABLE_SIG_snova_SNOVA_24_5_4_esk
ENCODER_w_structure("snova2454esk", snova2454esk, der, PrivateKeyInfo),
ENCODER_w_structure("snova2454esk", snova2454esk, pem, PrivateKeyInfo),
ENCODER_w_structure("snova2454esk", snova2454esk, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2454esk", snova2454esk, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2454esk", snova2454esk, der, SubjectPublicKeyInfo),
ENCODER_w_structure("snova2454esk", snova2454esk, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("snova2454esk", snova2454esk),
ENCODER_w_structure("p256_snova2454esk", p256_snova2454esk, der, PrivateKeyInfo),
ENCODER_w_structure("p256_snova2454esk", p256_snova2454esk, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_snova2454esk", p256_snova2454esk, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_snova2454esk", p256_snova2454esk, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_snova2454esk", p256_snova2454esk, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_snova2454esk", p256_snova2454esk, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_snova2454esk", p256_snova2454esk),
#endif
#ifdef OQS_ENABLE_SIG_snova_SNOVA_37_17_2
ENCODER_w_structure("snova37172", snova37172, der, PrivateKeyInfo),
ENCODER_w_structure("snova37172", snova37172, pem, PrivateKeyInfo),
ENCODER_w_structure("snova37172", snova37172, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova37172", snova37172, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova37172", snova37172, der, SubjectPublicKeyInfo),
ENCODER_w_structure("snova37172", snova37172, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("snova37172", snova37172),
ENCODER_w_structure("p256_snova37172", p256_snova37172, der, PrivateKeyInfo),
ENCODER_w_structure("p256_snova37172", p256_snova37172, pem, PrivateKeyInfo),
ENCODER_w_structure("p256_snova37172", p256_snova37172, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_snova37172", p256_snova37172, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p256_snova37172", p256_snova37172, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p256_snova37172", p256_snova37172, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p256_snova37172", p256_snova37172),
#endif
#ifdef OQS_ENABLE_SIG_snova_SNOVA_24_5_5
ENCODER_w_structure("snova2455", snova2455, der, PrivateKeyInfo),
ENCODER_w_structure("snova2455", snova2455, pem, PrivateKeyInfo),
ENCODER_w_structure("snova2455", snova2455, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2455", snova2455, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2455", snova2455, der, SubjectPublicKeyInfo),
ENCODER_w_structure("snova2455", snova2455, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("snova2455", snova2455),
ENCODER_w_structure("p384_snova2455", p384_snova2455, der, PrivateKeyInfo),
ENCODER_w_structure("p384_snova2455", p384_snova2455, pem, PrivateKeyInfo),
ENCODER_w_structure("p384_snova2455", p384_snova2455, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_snova2455", p384_snova2455, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p384_snova2455", p384_snova2455, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p384_snova2455", p384_snova2455, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p384_snova2455", p384_snova2455),
#endif
#ifdef OQS_ENABLE_SIG_snova_SNOVA_29_6_5
ENCODER_w_structure("snova2965", snova2965, der, PrivateKeyInfo),
ENCODER_w_structure("snova2965", snova2965, pem, PrivateKeyInfo),
ENCODER_w_structure("snova2965", snova2965, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2965", snova2965, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("snova2965", snova2965, der, SubjectPublicKeyInfo),
ENCODER_w_structure("snova2965", snova2965, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("snova2965", snova2965),
ENCODER_w_structure("p521_snova2965", p521_snova2965, der, PrivateKeyInfo),
ENCODER_w_structure("p521_snova2965", p521_snova2965, pem, PrivateKeyInfo),
ENCODER_w_structure("p521_snova2965", p521_snova2965, der, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_snova2965", p521_snova2965, pem, EncryptedPrivateKeyInfo),
ENCODER_w_structure("p521_snova2965", p521_snova2965, der, SubjectPublicKeyInfo),
ENCODER_w_structure("p521_snova2965", p521_snova2965, pem, SubjectPublicKeyInfo),
ENCODER_TEXT("p521_snova2965", p521_snova2965),
#endif
///// OQS_TEMPLATE_FRAGMENT_MAKE_END
