//go:build !integration

package merge

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.uber.org/mock/gomock"

	gitlab "gitlab.com/gitlab-org/api/client-go"
	gitlabtesting "gitlab.com/gitlab-org/api/client-go/testing"

	"gitlab.com/gitlab-org/cli/internal/testing/cmdtest"
)

func TestMrMerge(t *testing.T) {
	type testCase struct {
		name        string
		cli         string
		expectedOut string
		wantErr     bool
		wantStderr  string
		setupMock   func(tc *gitlabtesting.TestClient)
	}

	mergedMR := &gitlab.MergeRequest{
		BasicMergeRequest: gitlab.BasicMergeRequest{
			ID:        190608322,
			IID:       123,
			ProjectID: 37777023,
			Title:     "foo",
			State:     "merged",
			WebURL:    "https://gitlab.com/OWNER/REPO/-/merge_requests/123",
		},
	}

	testCases := []testCase{
		{
			name: "Merge MR by ID without pipeline",
			cli:  "123",
			// Note: The test verifies the merge flow works correctly
			// The "No pipeline running" warning appears when Pipeline is nil
			// Note: SourceBranch and Pipeline appear empty when using mock client
			// This test verifies the merge flow works; other tests cover pipeline scenarios
			expectedOut: "! No pipeline running on \n✓ Merged!\nhttps://gitlab.com/OWNER/REPO/-/merge_requests/123\n",
			setupMock: func(tc *gitlabtesting.TestClient) {
				getMR := &gitlab.MergeRequest{
					BasicMergeRequest: gitlab.BasicMergeRequest{
						ID:                  190608322,
						IID:                 123,
						ProjectID:           37777023,
						Title:               "foo",
						State:               "opened",
						SourceBranch:        "1-issue-20",
						WebURL:              "https://gitlab.com/OWNER/REPO/-/merge_requests/123",
						DetailedMergeStatus: "mergeable",
					},
					User: gitlab.MergeRequestUser{
						CanMerge: true,
					},
				}
				tc.MockMergeRequests.EXPECT().
					GetMergeRequest("OWNER/REPO", int64(123), gomock.Any()).
					Return(getMR, nil, nil)
				tc.MockMergeRequests.EXPECT().
					AcceptMergeRequest("OWNER/REPO", int64(123), gomock.Any()).
					Return(mergedMR, nil, nil)
			},
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			// GIVEN
			testClient := gitlabtesting.NewTestClient(t)
			tc.setupMock(testClient)
			exec := cmdtest.SetupCmdForTest(
				t,
				NewCmdMerge,
				false,
				cmdtest.WithGitLabClient(testClient.Client),
			)

			// WHEN
			out, err := exec(tc.cli)

			// THEN
			if tc.wantErr {
				require.Error(t, err)
				assert.Contains(t, err.Error(), tc.wantStderr)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, tc.expectedOut, out.OutBuf.String())
			assert.Empty(t, out.ErrBuf.String())
		})
	}
}
