# pylint: disable=wrong-spelling-in-comment

from . import pylonC_defs  # pylint: disable=unused-import
from .pylonC_defs import CPylonDeviceInfo_t, CPylonInterfaceInfo_t, CPylonGrabResult_t, PylonGrabResult_t  # pylint: disable=unused-import
from .pylonC_defs import EGenApiNodeType, EGenApiAccessMode, EGenApiVisibility, EGenApiNameSpace, EGenApiRepresentation, PYLONC_ACCESS  # pylint: disable=unused-import
from .pylonC_defs import drGENAPI_ERR, drPYLONC_ERR
from .pylonC_defs import define_functions

from ...core.utils import ctypes_wrap, py3
from ...core.devio.comm_backend import DeviceError
from ..utils import load_lib

import ctypes
import functools


class BaslerError(DeviceError):
    """Generic Basler error"""
class BaslerLibError(BaslerError):
    """Generic Basler library error"""
    def __init__(self, func, code, lib=None):
        self.func=func
        self.code=code
        if code in drPYLONC_ERR:
            self.name=drPYLONC_ERR[code]
        else:
            self.name=drGENAPI_ERR.get(code,"N/A")
        self.desc=""
        try:
            if lib is not None:
                self.desc=py3.as_str(lib.GenApiGetLastErrorMessage())
        except BaslerLibError:
            pass
        self.msg="function '{}' raised error 0x{:08X}({}): {}".format(func,code%0x100000000,self.name,self.desc)
        super().__init__(self.msg)
def errcheck(passing=None, lib=None):
    """
    Build an error checking function.

    Return a function which checks return codes of Basler library functions.
    `passing` is a list specifying which return codes are acceptable (by default only 0, which is success code, is acceptable).
    """
    if passing is None:
        passing={0}
    def errchecker(result, func, arguments):  # pylint: disable=unused-argument
        if result not in passing:
            raise BaslerLibError(func.__name__,result,lib=lib)
        return result
    return errchecker


def wrap_string_func(func, buffarg="pBuf", lenarg="pBufLen"):
    @functools.wraps(func)
    def wrapped(*args, **kwargs):
        kwargs[buffarg]=None
        kwargs[lenarg]=0
        sz=func(*args,**kwargs)
        msg=ctypes.create_string_buffer(sz)
        kwargs[buffarg]=ctypes.addressof(msg)
        kwargs[lenarg]=sz
        func(*args,**kwargs)
        return ctypes.string_at(msg)
    return wrapped

class pylonCLib:
    def __init__(self):
        self._initialized=False

    def initlib(self):
        if self._initialized:
            return
        error_message="The library is automatically supplied with Basler Pylon software\n"+load_lib.par_error_message.format("basler_pylon")
        self.lib=None
        for vmaj in range(15,-1,-1):
            for vmin in range(15,-1,-1):
                try:
                    self.lib=load_lib.load_lib("PylonC_v{}_{}.dll".format(vmaj,vmin),locations=("parameter/basler_pylon","global"),error_message=error_message,call_conv="stdcall")
                except OSError:
                    pass
        if self.lib is None:
            self.lib=load_lib.load_lib("PylonC_vX_Y.dll",locations=("parameter/basler_pylon","global"),error_message=error_message,call_conv="stdcall")

        lib=self.lib
        define_functions(lib)

        self.errcheck=errcheck(lib=self)
        wrapper=ctypes_wrap.CFunctionWrapper(errcheck=errcheck(lib=self),default_rvals="pointer")

        #  HRESULT GenApiGetLastErrorMessage(ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiGetLastErrorMessage_lib=wrapper(lib.GenApiGetLastErrorMessage, args="all")
        self.GenApiGetLastErrorMessage=wrap_string_func(self.GenApiGetLastErrorMessage_lib)
        #  HRESULT GenApiGetLastErrorDetail(ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiGetLastErrorDetail_lib=wrapper(lib.GenApiGetLastErrorDetail, args="all")
        self.GenApiGetLastErrorDetail=wrap_string_func(self.GenApiGetLastErrorDetail_lib)


        #  HRESULT PylonInitialize()
        self.PylonInitialize=wrapper(lib.PylonInitialize)
        #  HRESULT PylonTerminate()
        self.PylonTerminate=wrapper(lib.PylonTerminate)
        #  HRESULT PylonEnumerateInterfaces(ctypes.POINTER(ctypes.c_size_t) numInterfaces)
        self.PylonEnumerateInterfaces=wrapper(lib.PylonEnumerateInterfaces)
        #  HRESULT PylonEnumerateInterfacesByDeviceClass(ctypes.c_char_p pDeviceClass, ctypes.POINTER(ctypes.c_size_t) numInterfaces)
        self.PylonEnumerateInterfacesByDeviceClass=wrapper(lib.PylonEnumerateInterfacesByDeviceClass)
        #  HRESULT PylonCreateInterfaceByIndex(ctypes.c_size_t index, ctypes.POINTER(PYLON_INTERFACE_HANDLE) phIf)
        self.PylonCreateInterfaceByIndex=wrapper(lib.PylonCreateInterfaceByIndex)
        #  HRESULT PylonDestroyInterface(PYLON_INTERFACE_HANDLE hIf)
        self.PylonDestroyInterface=wrapper(lib.PylonDestroyInterface)
        #  HRESULT PylonInterfaceOpen(PYLON_INTERFACE_HANDLE hIf)
        self.PylonInterfaceOpen=wrapper(lib.PylonInterfaceOpen)
        #  HRESULT PylonInterfaceClose(PYLON_INTERFACE_HANDLE hIf)
        self.PylonInterfaceClose=wrapper(lib.PylonInterfaceClose)
        #  HRESULT PylonInterfaceIsOpen(PYLON_INTERFACE_HANDLE hIf, ctypes.POINTER(ctypes.c_ubyte) pOpen)
        self.PylonInterfaceIsOpen=wrapper(lib.PylonInterfaceIsOpen)
        #  HRESULT PylonGetInterfaceInfo(ctypes.c_size_t index, ctypes.POINTER(PylonInterfaceInfo_t) pIfInfo)
        self.PylonGetInterfaceInfo=wrapper(lib.PylonGetInterfaceInfo,
            argprep={"pIfInfo":CPylonInterfaceInfo_t.prep_struct}, rconv={"pIfInfo":CPylonInterfaceInfo_t.tup_struct})
        #  HRESULT PylonInterfaceGetInterfaceInfo(PYLON_INTERFACE_HANDLE hIf, ctypes.POINTER(PylonInterfaceInfo_t) pIfInfo)
        self.PylonInterfaceGetInterfaceInfo=wrapper(lib.PylonInterfaceGetInterfaceInfo,
            argprep={"pIfInfo":CPylonInterfaceInfo_t.prep_struct}, rconv={"pIfInfo":CPylonInterfaceInfo_t.tup_struct})

        #  HRESULT PylonEnumerateDevices(ctypes.POINTER(ctypes.c_size_t) numDevices)
        self.PylonEnumerateDevices=wrapper(lib.PylonEnumerateDevices)
        #  HRESULT PylonGetDeviceInfo(ctypes.c_size_t index, ctypes.POINTER(PylonDeviceInfo_t) pDi)
        self.PylonGetDeviceInfo=wrapper(lib.PylonGetDeviceInfo,
            argprep={"pDi":CPylonDeviceInfo_t.prep_struct}, rconv={"pDi":CPylonDeviceInfo_t.tup_struct})
        #  HRESULT PylonDeviceGetDeviceInfo(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(PylonDeviceInfo_t) pDeviceInfo)
        self.PylonDeviceGetDeviceInfo=wrapper(lib.PylonDeviceGetDeviceInfo,
            argprep={"pDeviceInfo":CPylonDeviceInfo_t.prep_struct}, rconv={"pDeviceInfo":CPylonDeviceInfo_t.tup_struct})
        #  HRESULT PylonGetDeviceInfoHandle(ctypes.c_size_t index, ctypes.POINTER(PYLON_DEVICE_INFO_HANDLE) phDi)
        self.PylonGetDeviceInfoHandle=wrapper(lib.PylonGetDeviceInfoHandle)
        #  HRESULT PylonDeviceGetDeviceInfoHandle(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(PYLON_DEVICE_INFO_HANDLE) phDi)
        self.PylonDeviceGetDeviceInfoHandle=wrapper(lib.PylonDeviceGetDeviceInfoHandle)
        #  HRESULT PylonDeviceInfoGetNumProperties(PYLON_DEVICE_INFO_HANDLE hDi, ctypes.POINTER(ctypes.c_size_t) numProperties)
        self.PylonDeviceInfoGetNumProperties=wrapper(lib.PylonDeviceInfoGetNumProperties)
        #  HRESULT PylonDeviceInfoGetPropertyName(PYLON_DEVICE_INFO_HANDLE hDi, ctypes.c_size_t index, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.PylonDeviceInfoGetPropertyName_lib=wrapper(lib.PylonDeviceInfoGetPropertyName, args="all")
        self.PylonDeviceInfoGetPropertyName=wrap_string_func(self.PylonDeviceInfoGetPropertyName_lib,buffarg="pName")
        #  HRESULT PylonDeviceInfoGetPropertyValueByName(PYLON_DEVICE_INFO_HANDLE hDi, ctypes.c_char_p pName, ctypes.c_char_p pValue, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.PylonDeviceInfoGetPropertyValueByName_lib=wrapper(lib.PylonDeviceInfoGetPropertyValueByName, args="all")
        self.PylonDeviceInfoGetPropertyValueByName=wrap_string_func(self.PylonDeviceInfoGetPropertyValueByName_lib,buffarg="pValue")
        #  HRESULT PylonDeviceInfoGetPropertyValueByIndex(PYLON_DEVICE_INFO_HANDLE hDi, ctypes.c_size_t index, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.PylonDeviceInfoGetPropertyValueByIndex_lib=wrapper(lib.PylonDeviceInfoGetPropertyValueByIndex, args="all")
        self.PylonDeviceInfoGetPropertyValueByIndex=wrap_string_func(self.PylonDeviceInfoGetPropertyValueByIndex_lib)
        
        #  HRESULT PylonCreateDeviceByIndex(ctypes.c_size_t index, ctypes.POINTER(PYLON_DEVICE_HANDLE) phDev)
        self.PylonCreateDeviceByIndex=wrapper(lib.PylonCreateDeviceByIndex)
        #  HRESULT PylonDestroyDevice(PYLON_DEVICE_HANDLE hDev)
        self.PylonDestroyDevice=wrapper(lib.PylonDestroyDevice)
        #  HRESULT PylonIsDeviceAccessible(ctypes.c_size_t index, ctypes.c_int accessMode, ctypes.POINTER(ctypes.c_ubyte) pIsAccessible)
        self.PylonIsDeviceAccessible=wrapper(lib.PylonIsDeviceAccessible)
        #  HRESULT PylonDeviceOpen(PYLON_DEVICE_HANDLE hDev, ctypes.c_int accessMode)
        self.PylonDeviceOpen=wrapper(lib.PylonDeviceOpen)
        #  HRESULT PylonDeviceClose(PYLON_DEVICE_HANDLE hDev)
        self.PylonDeviceClose=wrapper(lib.PylonDeviceClose)
        #  HRESULT PylonDeviceIsOpen(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(ctypes.c_ubyte) pOpen)
        self.PylonDeviceIsOpen=wrapper(lib.PylonDeviceIsOpen)
        #  HRESULT PylonDeviceAccessMode(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(ctypes.c_int) pAccessMode)
        self.PylonDeviceAccessMode=wrapper(lib.PylonDeviceAccessMode)


        #  HRESULT GenApiNodeMapGetNumNodes(NODEMAP_HANDLE hMap, ctypes.POINTER(ctypes.c_size_t) pValue)
        self.GenApiNodeMapGetNumNodes=wrapper(lib.GenApiNodeMapGetNumNodes)
        #  HRESULT GenApiNodeMapGetNode(NODEMAP_HANDLE hMap, ctypes.c_char_p pName, ctypes.POINTER(NODE_HANDLE) phNode)
        self.GenApiNodeMapGetNode=wrapper(lib.GenApiNodeMapGetNode)
        #  HRESULT GenApiNodeMapGetNodeByIndex(NODEMAP_HANDLE hMap, ctypes.c_size_t index, ctypes.POINTER(NODE_HANDLE) phNode)
        self.GenApiNodeMapGetNodeByIndex=wrapper(lib.GenApiNodeMapGetNodeByIndex)
        #  HRESULT GenApiNodeGetName(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiNodeGetName_lib=wrapper(lib.GenApiNodeGetName, args="all")
        self.GenApiNodeGetName=wrap_string_func(self.GenApiNodeGetName_lib)
        #  HRESULT GenApiCategoryGetNumFeatures(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_size_t) pValue)
        self.GenApiCategoryGetNumFeatures=wrapper(lib.GenApiCategoryGetNumFeatures)
        #  HRESULT GenApiCategoryGetFeatureByIndex(NODE_HANDLE hNode, ctypes.c_size_t index, ctypes.POINTER(NODE_HANDLE) phEntry)
        self.GenApiCategoryGetFeatureByIndex=wrapper(lib.GenApiCategoryGetFeatureByIndex)
        #  HRESULT GenApiNodeGetAlias(NODE_HANDLE hNode, ctypes.POINTER(NODE_HANDLE) phNode)
        self.GenApiNodeGetAlias=wrapper(lib.GenApiNodeGetAlias)
        #  HRESULT GenApiNodeGetAccessMode(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pAccessMode)
        self.GenApiNodeGetAccessMode=wrapper(lib.GenApiNodeGetAccessMode)
        #  HRESULT GenApiNodeImposeAccessMode(NODE_HANDLE hNode, ctypes.c_int imposedAccessMode)
        self.GenApiNodeImposeAccessMode=wrapper(lib.GenApiNodeImposeAccessMode)
        #  HRESULT GenApiNodeGetVisibility(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pVisibility)
        self.GenApiNodeGetVisibility=wrapper(lib.GenApiNodeGetVisibility)
        #  HRESULT GenApiNodeImposeVisibility(NODE_HANDLE hNode, ctypes.c_int imposedVisibility)
        self.GenApiNodeImposeVisibility=wrapper(lib.GenApiNodeImposeVisibility)
        #  HRESULT GenApiNodeGetCachingMode(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pCachingMode)
        self.GenApiNodeGetCachingMode=wrapper(lib.GenApiNodeGetCachingMode)
        #  HRESULT GenApiNodeGetToolTip(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiNodeGetToolTip_lib=wrapper(lib.GenApiNodeGetToolTip, args="all")
        self.GenApiNodeGetToolTip=wrap_string_func(self.GenApiNodeGetToolTip_lib)
        #  HRESULT GenApiNodeGetDescription(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiNodeGetDescription_lib=wrapper(lib.GenApiNodeGetDescription, args="all")
        self.GenApiNodeGetDescription=wrap_string_func(self.GenApiNodeGetDescription_lib)
        #  HRESULT GenApiNodeGetDisplayName(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiNodeGetDisplayName_lib=wrapper(lib.GenApiNodeGetDisplayName, args="all")
        self.GenApiNodeGetDisplayName=wrap_string_func(self.GenApiNodeGetDisplayName_lib)
        #  HRESULT GenApiNodeGetType(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pType)
        self.GenApiNodeGetType=wrapper(lib.GenApiNodeGetType)
        #  HRESULT GenApiNodeGetPollingTime(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int64) pPollingTime)
        self.GenApiNodeGetPollingTime=wrapper(lib.GenApiNodeGetPollingTime)
        #  HRESULT GenApiNodeGetPollingTimeInt32(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int32) pollingTime)
        self.GenApiNodeGetPollingTimeInt32=wrapper(lib.GenApiNodeGetPollingTimeInt32)
        #  HRESULT GenApiNodeIsImplemented(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_ubyte) pResult)
        self.GenApiNodeIsImplemented=wrapper(lib.GenApiNodeIsImplemented)
        #  HRESULT GenApiNodeIsReadable(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_ubyte) pResult)
        self.GenApiNodeIsReadable=wrapper(lib.GenApiNodeIsReadable)
        #  HRESULT GenApiNodeIsWritable(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_ubyte) pResult)
        self.GenApiNodeIsWritable=wrapper(lib.GenApiNodeIsWritable)
        #  HRESULT GenApiNodeIsAvailable(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_ubyte) pResult)
        self.GenApiNodeIsAvailable=wrapper(lib.GenApiNodeIsAvailable)
        #  HRESULT GenApiNodeToString(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiNodeToString_lib=wrapper(lib.GenApiNodeToString, args="all")
        self.GenApiNodeToString=wrap_string_func(self.GenApiNodeToString_lib)
        #  HRESULT GenApiNodeToStringEx(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiNodeToStringEx_lib=wrapper(lib.GenApiNodeToStringEx, args="all")
        self.GenApiNodeToStringEx=wrap_string_func(self.GenApiNodeToStringEx_lib)
        #  HRESULT GenApiNodeFromString(NODE_HANDLE hNode, ctypes.c_char_p pString)
        self.GenApiNodeFromString=wrapper(lib.GenApiNodeFromString)
        #  HRESULT GenApiNodeFromStringEx(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.c_char_p pString)
        self.GenApiNodeFromStringEx=wrapper(lib.GenApiNodeFromStringEx)

        #  HRESULT GenApiIntegerSetValue(NODE_HANDLE hNode, ctypes.c_int64 value)
        self.GenApiIntegerSetValue=wrapper(lib.GenApiIntegerSetValue)
        #  HRESULT GenApiIntegerSetValueEx(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.c_int64 value)
        self.GenApiIntegerSetValueEx=wrapper(lib.GenApiIntegerSetValueEx)
        #  HRESULT GenApiIntegerGetValue(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int64) pValue)
        self.GenApiIntegerGetValue=wrapper(lib.GenApiIntegerGetValue)
        #  HRESULT GenApiIntegerGetValueEx(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.POINTER(ctypes.c_int64) pValue)
        self.GenApiIntegerGetValueEx=wrapper(lib.GenApiIntegerGetValueEx)
        #  HRESULT GenApiIntegerGetMin(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int64) pValue)
        self.GenApiIntegerGetMin=wrapper(lib.GenApiIntegerGetMin)
        #  HRESULT GenApiIntegerGetMax(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int64) pValue)
        self.GenApiIntegerGetMax=wrapper(lib.GenApiIntegerGetMax)
        #  HRESULT GenApiIntegerGetInc(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int64) pValue)
        self.GenApiIntegerGetInc=wrapper(lib.GenApiIntegerGetInc)
        #  HRESULT GenApiIntegerGetRepresentation(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pValue)
        self.GenApiIntegerGetRepresentation=wrapper(lib.GenApiIntegerGetRepresentation)

        #  HRESULT GenApiIntegerSetValueInt32(NODE_HANDLE hNode, ctypes.c_int32 value)
        self.GenApiIntegerSetValueInt32=wrapper(lib.GenApiIntegerSetValueInt32)
        #  HRESULT GenApiIntegerSetValueExInt32(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.c_int32 value)
        self.GenApiIntegerSetValueExInt32=wrapper(lib.GenApiIntegerSetValueExInt32)
        #  HRESULT GenApiIntegerGetValueInt32(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int32) pValue)
        self.GenApiIntegerGetValueInt32=wrapper(lib.GenApiIntegerGetValueInt32)
        #  HRESULT GenApiIntegerGetValueExInt32(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.POINTER(ctypes.c_int32) pValue)
        self.GenApiIntegerGetValueExInt32=wrapper(lib.GenApiIntegerGetValueExInt32)
        #  HRESULT GenApiIntegerGetMinInt32(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int32) pValue)
        self.GenApiIntegerGetMinInt32=wrapper(lib.GenApiIntegerGetMinInt32)
        #  HRESULT GenApiIntegerGetMaxInt32(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int32) pValue)
        self.GenApiIntegerGetMaxInt32=wrapper(lib.GenApiIntegerGetMaxInt32)
        #  HRESULT GenApiIntegerGetIncInt32(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int32) pValue)
        self.GenApiIntegerGetIncInt32=wrapper(lib.GenApiIntegerGetIncInt32)

        #  HRESULT GenApiBooleanSetValue(NODE_HANDLE hNode, ctypes.c_ubyte value)
        self.GenApiBooleanSetValue=wrapper(lib.GenApiBooleanSetValue)
        #  HRESULT GenApiBooleanGetValue(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_ubyte) pValue)
        self.GenApiBooleanGetValue=wrapper(lib.GenApiBooleanGetValue)
        
        #  HRESULT GenApiFloatSetValue(NODE_HANDLE hNode, ctypes.c_double value)
        self.GenApiFloatSetValue=wrapper(lib.GenApiFloatSetValue)
        #  HRESULT GenApiFloatSetValueEx(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.c_double value)
        self.GenApiFloatSetValueEx=wrapper(lib.GenApiFloatSetValueEx)
        #  HRESULT GenApiFloatGetValue(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_double) pValue)
        self.GenApiFloatGetValue=wrapper(lib.GenApiFloatGetValue)
        #  HRESULT GenApiFloatGetValueEx(NODE_HANDLE hNode, ctypes.c_ubyte verify, ctypes.POINTER(ctypes.c_double) pValue)
        self.GenApiFloatGetValueEx=wrapper(lib.GenApiFloatGetValueEx)
        #  HRESULT GenApiFloatGetMin(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_double) pValue)
        self.GenApiFloatGetMin=wrapper(lib.GenApiFloatGetMin)
        #  HRESULT GenApiFloatGetMax(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_double) pValue)
        self.GenApiFloatGetMax=wrapper(lib.GenApiFloatGetMax)
        #  HRESULT GenApiFloatGetRepresentation(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pValue)
        self.GenApiFloatGetRepresentation=wrapper(lib.GenApiFloatGetRepresentation)
        #  HRESULT GenApiFloatGetUnit(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiFloatGetUnit_lib=wrapper(lib.GenApiFloatGetUnit, args="all")
        self.GenApiFloatGetUnit=wrap_string_func(self.GenApiFloatGetUnit_lib)
        
        #  HRESULT GenApiEnumerationGetNumEntries(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_size_t) pValue)
        self.GenApiEnumerationGetNumEntries=wrapper(lib.GenApiEnumerationGetNumEntries)
        #  HRESULT GenApiEnumerationGetEntryByIndex(NODE_HANDLE hNode, ctypes.c_size_t index, ctypes.POINTER(NODE_HANDLE) pEntry)
        self.GenApiEnumerationGetEntryByIndex=wrapper(lib.GenApiEnumerationGetEntryByIndex)
        #  HRESULT GenApiEnumerationGetEntryByName(NODE_HANDLE hNode, ctypes.c_char_p pName, ctypes.POINTER(NODE_HANDLE) pEntry)
        self.GenApiEnumerationGetEntryByName=wrapper(lib.GenApiEnumerationGetEntryByName)
        #  HRESULT GenApiEnumerationEntryGetValue(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pValue)
        self.GenApiEnumerationEntryGetValue=wrapper(lib.GenApiEnumerationEntryGetValue)
        #  HRESULT GenApiEnumerationEntryGetSymbolic(NODE_HANDLE hNode, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.GenApiEnumerationEntryGetSymbolic_lib=wrapper(lib.GenApiEnumerationEntryGetSymbolic, args="all")
        self.GenApiEnumerationEntryGetSymbolic=wrap_string_func(self.GenApiEnumerationEntryGetSymbolic_lib)
        
        #  HRESULT GenApiCommandExecute(NODE_HANDLE hNode)
        self.GenApiCommandExecute=wrapper(lib.GenApiCommandExecute)
        #  HRESULT GenApiCommandIsDone(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_ubyte) pValue)
        self.GenApiCommandIsDone=wrapper(lib.GenApiCommandIsDone)
        

        #  HRESULT PylonInterfaceGetNodeMap(PYLON_INTERFACE_HANDLE hIf, ctypes.POINTER(NODEMAP_HANDLE) phMap)
        self.PylonInterfaceGetNodeMap=wrapper(lib.PylonInterfaceGetNodeMap)
        #  HRESULT PylonDeviceGetNodeMap(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(NODEMAP_HANDLE) phMap)
        self.PylonDeviceGetNodeMap=wrapper(lib.PylonDeviceGetNodeMap)
        #  HRESULT PylonDeviceGetTLNodeMap(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(NODEMAP_HANDLE) phMap)
        self.PylonDeviceGetTLNodeMap=wrapper(lib.PylonDeviceGetTLNodeMap)
        #  HRESULT PylonStreamGrabberGetNodeMap(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(NODEMAP_HANDLE) phMap)
        self.PylonStreamGrabberGetNodeMap=wrapper(lib.PylonStreamGrabberGetNodeMap)
        #  HRESULT PylonEventGrabberGetNodeMap(PYLON_EVENTGRABBER_HANDLE hEvg, ctypes.POINTER(NODEMAP_HANDLE) phMap)
        self.PylonEventGrabberGetNodeMap=wrapper(lib.PylonEventGrabberGetNodeMap)
        #  HRESULT PylonDeviceFeatureFromString(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.c_char_p pValue)
        self.PylonDeviceFeatureFromString=wrapper(lib.PylonDeviceFeatureFromString)
        #  HRESULT PylonDeviceFeatureToString(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.c_char_p pBuf, ctypes.POINTER(ctypes.c_size_t) pBufLen)
        self.PylonDeviceFeatureToString_lib=wrapper(lib.PylonDeviceFeatureToString, args="all")
        self.PylonDeviceFeatureToString=wrap_string_func(self.PylonDeviceFeatureToString_lib)
        #  ctypes.c_ubyte PylonDeviceFeatureIsImplemented(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName)
        self.PylonDeviceFeatureIsImplemented=wrapper(lib.PylonDeviceFeatureIsImplemented)
        #  ctypes.c_ubyte PylonDeviceFeatureIsAvailable(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName)
        self.PylonDeviceFeatureIsAvailable=wrapper(lib.PylonDeviceFeatureIsAvailable)
        #  ctypes.c_ubyte PylonDeviceFeatureIsReadable(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName)
        self.PylonDeviceFeatureIsReadable=wrapper(lib.PylonDeviceFeatureIsReadable)
        #  ctypes.c_ubyte PylonDeviceFeatureIsWritable(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName)
        self.PylonDeviceFeatureIsWritable=wrapper(lib.PylonDeviceFeatureIsWritable)
        #  HRESULT PylonDeviceFeatureGetAccessMode(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_int) pResult)
        self.PylonDeviceFeatureGetAccessMode=wrapper(lib.PylonDeviceFeatureGetAccessMode)
        #  HRESULT PylonDeviceSetIntegerFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.c_int64 value)
        self.PylonDeviceSetIntegerFeature=wrapper(lib.PylonDeviceSetIntegerFeature)
        #  HRESULT PylonDeviceGetIntegerFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_int64) pValue)
        self.PylonDeviceGetIntegerFeature=wrapper(lib.PylonDeviceGetIntegerFeature)
        #  HRESULT PylonDeviceGetIntegerFeatureMin(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_int64) pValue)
        self.PylonDeviceGetIntegerFeatureMin=wrapper(lib.PylonDeviceGetIntegerFeatureMin)
        #  HRESULT PylonDeviceGetIntegerFeatureMax(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_int64) pValue)
        self.PylonDeviceGetIntegerFeatureMax=wrapper(lib.PylonDeviceGetIntegerFeatureMax)
        #  HRESULT PylonDeviceGetIntegerFeatureInc(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_int64) pValue)
        self.PylonDeviceGetIntegerFeatureInc=wrapper(lib.PylonDeviceGetIntegerFeatureInc)
        #  HRESULT PylonDeviceSetIntegerFeatureInt32(PYLON_DEVICE_HANDLE dev, ctypes.c_char_p name, ctypes.c_int32 value)
        self.PylonDeviceSetIntegerFeatureInt32=wrapper(lib.PylonDeviceSetIntegerFeatureInt32)
        #  HRESULT PylonDeviceGetIntegerFeatureInt32(PYLON_DEVICE_HANDLE dev, ctypes.c_char_p name, ctypes.POINTER(ctypes.c_int32) value)
        self.PylonDeviceGetIntegerFeatureInt32=wrapper(lib.PylonDeviceGetIntegerFeatureInt32)
        #  HRESULT PylonDeviceGetIntegerFeatureMinInt32(PYLON_DEVICE_HANDLE dev, ctypes.c_char_p name, ctypes.POINTER(ctypes.c_int32) value)
        self.PylonDeviceGetIntegerFeatureMinInt32=wrapper(lib.PylonDeviceGetIntegerFeatureMinInt32)
        #  HRESULT PylonDeviceGetIntegerFeatureMaxInt32(PYLON_DEVICE_HANDLE dev, ctypes.c_char_p name, ctypes.POINTER(ctypes.c_int32) value)
        self.PylonDeviceGetIntegerFeatureMaxInt32=wrapper(lib.PylonDeviceGetIntegerFeatureMaxInt32)
        #  HRESULT PylonDeviceGetIntegerFeatureIncInt32(PYLON_DEVICE_HANDLE dev, ctypes.c_char_p name, ctypes.POINTER(ctypes.c_int32) value)
        self.PylonDeviceGetIntegerFeatureIncInt32=wrapper(lib.PylonDeviceGetIntegerFeatureIncInt32)
        #  HRESULT PylonDeviceSetFloatFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.c_double value)
        self.PylonDeviceSetFloatFeature=wrapper(lib.PylonDeviceSetFloatFeature)
        #  HRESULT PylonDeviceGetFloatFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_double) pValue)
        self.PylonDeviceGetFloatFeature=wrapper(lib.PylonDeviceGetFloatFeature)
        #  HRESULT PylonDeviceGetFloatFeatureMin(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_double) pValue)
        self.PylonDeviceGetFloatFeatureMin=wrapper(lib.PylonDeviceGetFloatFeatureMin)
        #  HRESULT PylonDeviceGetFloatFeatureMax(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_double) pValue)
        self.PylonDeviceGetFloatFeatureMax=wrapper(lib.PylonDeviceGetFloatFeatureMax)
        #  HRESULT PylonDeviceSetBooleanFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.c_ubyte value)
        self.PylonDeviceSetBooleanFeature=wrapper(lib.PylonDeviceSetBooleanFeature)
        #  HRESULT PylonDeviceGetBooleanFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_ubyte) pValue)
        self.PylonDeviceGetBooleanFeature=wrapper(lib.PylonDeviceGetBooleanFeature)
        #  HRESULT PylonDeviceExecuteCommandFeature(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName)
        self.PylonDeviceExecuteCommandFeature=wrapper(lib.PylonDeviceExecuteCommandFeature)
        #  HRESULT PylonDeviceIsCommandDone(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pName, ctypes.POINTER(ctypes.c_ubyte) pResult)
        self.PylonDeviceIsCommandDone=wrapper(lib.PylonDeviceIsCommandDone)

        #  HRESULT PylonDeviceGetNumStreamGrabberChannels(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(ctypes.c_size_t) pNumChannels)
        self.PylonDeviceGetNumStreamGrabberChannels=wrapper(lib.PylonDeviceGetNumStreamGrabberChannels)
        #  HRESULT PylonDeviceGetStreamGrabber(PYLON_DEVICE_HANDLE hDev, ctypes.c_size_t index, ctypes.POINTER(PYLON_STREAMGRABBER_HANDLE) phStg)
        self.PylonDeviceGetStreamGrabber=wrapper(lib.PylonDeviceGetStreamGrabber)
        #  HRESULT PylonStreamGrabberOpen(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberOpen=wrapper(lib.PylonStreamGrabberOpen)
        #  HRESULT PylonStreamGrabberClose(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberClose=wrapper(lib.PylonStreamGrabberClose)
        #  HRESULT PylonStreamGrabberIsOpen(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(ctypes.c_ubyte) pOpen)
        self.PylonStreamGrabberIsOpen=wrapper(lib.PylonStreamGrabberIsOpen)
        #  HRESULT PylonStreamGrabberGetWaitObject(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(PYLON_WAITOBJECT_HANDLE) phWobj)
        self.PylonStreamGrabberGetWaitObject=wrapper(lib.PylonStreamGrabberGetWaitObject)
        #  HRESULT PylonStreamGrabberSetMaxNumBuffer(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.c_size_t numBuffers)
        self.PylonStreamGrabberSetMaxNumBuffer=wrapper(lib.PylonStreamGrabberSetMaxNumBuffer)
        #  HRESULT PylonStreamGrabberGetMaxNumBuffer(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(ctypes.c_size_t) pNumBuffers)
        self.PylonStreamGrabberGetMaxNumBuffer=wrapper(lib.PylonStreamGrabberGetMaxNumBuffer)
        #  HRESULT PylonStreamGrabberSetMaxBufferSize(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.c_size_t maxSize)
        self.PylonStreamGrabberSetMaxBufferSize=wrapper(lib.PylonStreamGrabberSetMaxBufferSize)
        #  HRESULT PylonStreamGrabberGetMaxBufferSize(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(ctypes.c_size_t) pMaxSize)
        self.PylonStreamGrabberGetMaxBufferSize=wrapper(lib.PylonStreamGrabberGetMaxBufferSize)
        #  HRESULT PylonStreamGrabberGetPayloadSize(PYLON_DEVICE_HANDLE hDev, PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(ctypes.c_size_t) payloadsize)
        self.PylonStreamGrabberGetPayloadSize=wrapper(lib.PylonStreamGrabberGetPayloadSize)
        #  HRESULT PylonStreamGrabberRegisterBuffer(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.c_void_p pBuffer, ctypes.c_size_t BufLen, ctypes.POINTER(PYLON_STREAMBUFFER_HANDLE) phBuf)
        self.PylonStreamGrabberRegisterBuffer=wrapper(lib.PylonStreamGrabberRegisterBuffer)
        #  HRESULT PylonStreamGrabberDeregisterBuffer(PYLON_STREAMGRABBER_HANDLE hStg, PYLON_STREAMBUFFER_HANDLE hBuf)
        self.PylonStreamGrabberDeregisterBuffer=wrapper(lib.PylonStreamGrabberDeregisterBuffer)
        #  HRESULT PylonStreamGrabberPrepareGrab(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberPrepareGrab=wrapper(lib.PylonStreamGrabberPrepareGrab)
        #  HRESULT PylonStreamGrabberIsStartAndStopStreamingMandatory(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(ctypes.c_ubyte) pMandatory)
        self.PylonStreamGrabberIsStartAndStopStreamingMandatory=wrapper(lib.PylonStreamGrabberIsStartAndStopStreamingMandatory)
        #  HRESULT PylonStreamGrabberStartStreamingIfMandatory(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberStartStreamingIfMandatory=wrapper(lib.PylonStreamGrabberStartStreamingIfMandatory)
        #  HRESULT PylonStreamGrabberStopStreamingIfMandatory(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberStopStreamingIfMandatory=wrapper(lib.PylonStreamGrabberStopStreamingIfMandatory)
        #  HRESULT PylonStreamGrabberFinishGrab(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberFinishGrab=wrapper(lib.PylonStreamGrabberFinishGrab)
        #  HRESULT PylonStreamGrabberFlushBuffersToOutput(PYLON_STREAMGRABBER_HANDLE hStg)
        self.PylonStreamGrabberFlushBuffersToOutput=wrapper(lib.PylonStreamGrabberFlushBuffersToOutput)
        # #  HRESULT PylonStreamGrabberCancelGrab(PYLON_STREAMGRABBER_HANDLE hStg)
        # self.PylonStreamGrabberCancelGrab=wrapper(lib.PylonStreamGrabberCancelGrab)
        #  HRESULT PylonStreamGrabberRetrieveResult(PYLON_STREAMGRABBER_HANDLE hStg, ctypes.POINTER(PylonGrabResult_t) pGrabResult, ctypes.POINTER(ctypes.c_ubyte) pReady)
        self.PylonStreamGrabberRetrieveResult=wrapper(lib.PylonStreamGrabberRetrieveResult,
            argprep={"pGrabResult":CPylonGrabResult_t.prep_struct}, rconv={"pGrabResult":CPylonGrabResult_t.tup_struct})
        #  HRESULT PylonStreamGrabberQueueBuffer(PYLON_STREAMGRABBER_HANDLE hStg, PYLON_STREAMBUFFER_HANDLE hBuf, ctypes.c_void_p pContext)
        self.PylonStreamGrabberQueueBuffer=wrapper(lib.PylonStreamGrabberQueueBuffer)

        #  HRESULT PylonWaitObjectIsValid(PYLON_WAITOBJECT_HANDLE hWobj, ctypes.POINTER(ctypes.c_ubyte) pValid)
        self.PylonWaitObjectIsValid=wrapper(lib.PylonWaitObjectIsValid)
        #  HRESULT PylonWaitObjectWait(PYLON_WAITOBJECT_HANDLE hWobj, ctypes.c_uint32 timeout, ctypes.POINTER(ctypes.c_ubyte) pResult)
        self.PylonWaitObjectWait=wrapper(lib.PylonWaitObjectWait)
        #  HRESULT PylonWaitObjectWaitEx(PYLON_WAITOBJECT_HANDLE hWobj, ctypes.c_uint32 timeout, ctypes.c_ubyte alertable, ctypes.POINTER(ctypes.c_int) pWaitResult)
        self.PylonWaitObjectWaitEx=wrapper(lib.PylonWaitObjectWaitEx)
        #  HRESULT PylonWaitObjectCreate(ctypes.POINTER(PYLON_WAITOBJECT_HANDLE) phWobj)
        self.PylonWaitObjectCreate=wrapper(lib.PylonWaitObjectCreate)
        #  HRESULT PylonWaitObjectDestroy(PYLON_WAITOBJECT_HANDLE hWobj)
        self.PylonWaitObjectDestroy=wrapper(lib.PylonWaitObjectDestroy)
        #  HRESULT PylonWaitObjectSignal(PYLON_WAITOBJECT_HANDLE hWobj)
        self.PylonWaitObjectSignal=wrapper(lib.PylonWaitObjectSignal)
        #  HRESULT PylonWaitObjectReset(PYLON_WAITOBJECT_HANDLE hWobj)
        self.PylonWaitObjectReset=wrapper(lib.PylonWaitObjectReset)
        #  HRESULT PylonWaitObjectFromW32(HANDLE hW32, ctypes.c_ubyte duplicate, ctypes.POINTER(PYLON_WAITOBJECT_HANDLE) phWobj)
        self.PylonWaitObjectFromW32=wrapper(lib.PylonWaitObjectFromW32)
        #  HRESULT PylonWaitObjectGetW32Handle(PYLON_WAITOBJECT_HANDLE hWobj, PHANDLE phW32)
        self.PylonWaitObjectGetW32Handle=wrapper(lib.PylonWaitObjectGetW32Handle)
        


        self._initialized=True



        # #  HRESULT PylonSetProperty(ctypes.c_int propertyId, ctypes.c_void_p pData, ctypes.c_size_t size)
        # self.PylonSetProperty=wrapper(lib.PylonSetProperty)
        # #  HRESULT PylonGetProperty(ctypes.c_int propertyId, ctypes.c_void_p pData, ctypes.POINTER(ctypes.c_size_t) pSize)
        # self.PylonGetProperty=wrapper(lib.PylonGetProperty)



        # #  HRESULT PylonDeviceGetEventGrabber(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(PYLON_EVENTGRABBER_HANDLE) phEvg)
        # self.PylonDeviceGetEventGrabber=wrapper(lib.PylonDeviceGetEventGrabber)
        # #  HRESULT PylonDeviceCreateChunkParser(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(PYLON_CHUNKPARSER_HANDLE) phChp)
        # self.PylonDeviceCreateChunkParser=wrapper(lib.PylonDeviceCreateChunkParser)
        # #  HRESULT PylonDeviceDestroyChunkParser(PYLON_DEVICE_HANDLE hDev, PYLON_CHUNKPARSER_HANDLE hChp)
        # self.PylonDeviceDestroyChunkParser=wrapper(lib.PylonDeviceDestroyChunkParser)
        # #  HRESULT PylonDeviceCreateEventAdapter(PYLON_DEVICE_HANDLE hDev, ctypes.POINTER(PYLON_EVENTADAPTER_HANDLE) phEva)
        # self.PylonDeviceCreateEventAdapter=wrapper(lib.PylonDeviceCreateEventAdapter)
        # #  HRESULT PylonDeviceDestroyEventAdapter(PYLON_DEVICE_HANDLE hDev, PYLON_EVENTADAPTER_HANDLE hEva)
        # self.PylonDeviceDestroyEventAdapter=wrapper(lib.PylonDeviceDestroyEventAdapter)
        # #  HRESULT PylonDeviceRegisterRemovalCallback(PYLON_DEVICE_HANDLE hDev, ctypes.c_void_p pCbFunction, ctypes.POINTER(PYLON_DEVICECALLBACK_HANDLE) phCb)
        # self.PylonDeviceRegisterRemovalCallback=wrapper(lib.PylonDeviceRegisterRemovalCallback)
        # #  HRESULT PylonDeviceDeregisterRemovalCallback(PYLON_DEVICE_HANDLE hDev, PYLON_DEVICECALLBACK_HANDLE hCb)
        # self.PylonDeviceDeregisterRemovalCallback=wrapper(lib.PylonDeviceDeregisterRemovalCallback)
        # #  HRESULT PylonDeviceGrabSingleFrame(PYLON_DEVICE_HANDLE hDev, ctypes.c_size_t channel, ctypes.c_void_p pBuffer, ctypes.c_size_t bufferSize, ctypes.POINTER(PylonGrabResult_t) pGrabResult, ctypes.POINTER(ctypes.c_ubyte) pReady, ctypes.c_uint32 timeout)
        # self.PylonDeviceGrabSingleFrame=wrapper(lib.PylonDeviceGrabSingleFrame)
        # #  HRESULT PylonRTThreadGetPriorityCapabilities(ctypes.POINTER(ctypes.c_int32) pPriorityMin, ctypes.POINTER(ctypes.c_int32) pPriorityMax)
        # self.PylonRTThreadGetPriorityCapabilities=wrapper(lib.PylonRTThreadGetPriorityCapabilities)
        # #  HRESULT PylonRTThreadSetPriority(HANDLE hThread, ctypes.c_int32 priority)
        # self.PylonRTThreadSetPriority=wrapper(lib.PylonRTThreadSetPriority)
        # #  HRESULT PylonRTThreadGetPriority(HANDLE hThread, ctypes.POINTER(ctypes.c_int32) pPriority)
        # self.PylonRTThreadGetPriority=wrapper(lib.PylonRTThreadGetPriority)
        # #  HRESULT PylonWaitObjectsCreate(ctypes.POINTER(PYLON_WAITOBJECTS_HANDLE) phWos)
        # self.PylonWaitObjectsCreate=wrapper(lib.PylonWaitObjectsCreate)
        # #  HRESULT PylonWaitObjectsDestroy(PYLON_WAITOBJECTS_HANDLE hWos)
        # self.PylonWaitObjectsDestroy=wrapper(lib.PylonWaitObjectsDestroy)
        # #  HRESULT PylonWaitObjectsAdd(PYLON_WAITOBJECTS_HANDLE hWos, PYLON_WAITOBJECT_HANDLE hWobj, ctypes.POINTER(ctypes.c_size_t) pIndex)
        # self.PylonWaitObjectsAdd=wrapper(lib.PylonWaitObjectsAdd)
        # #  HRESULT PylonWaitObjectsAddMany(PYLON_WAITOBJECTS_HANDLE hWos, ctypes.c_size_t numWaitObjects, ...)
        # self.PylonWaitObjectsAddMany=wrapper(lib.PylonWaitObjectsAddMany)
        # #  HRESULT PylonWaitObjectsRemoveAll(PYLON_WAITOBJECTS_HANDLE hWos)
        # self.PylonWaitObjectsRemoveAll=wrapper(lib.PylonWaitObjectsRemoveAll)
        # #  HRESULT PylonWaitObjectsWaitForAll(PYLON_WAITOBJECTS_HANDLE hWos, ctypes.c_uint32 timeout, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonWaitObjectsWaitForAll=wrapper(lib.PylonWaitObjectsWaitForAll)
        # #  HRESULT PylonWaitObjectsWaitForAny(PYLON_WAITOBJECTS_HANDLE hWos, ctypes.c_uint32 timeout, ctypes.POINTER(ctypes.c_size_t) pIndex, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonWaitObjectsWaitForAny=wrapper(lib.PylonWaitObjectsWaitForAny)
        # #  HRESULT PylonWaitObjectsWaitForAny(PYLON_WAITOBJECTS_HANDLE hWos, ctypes.c_uint32 timeout, ctypes.POINTER(ctypes.c_size_t) pIndex, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonWaitObjectsWaitForAny=wrapper(lib.PylonWaitObjectsWaitForAny)
        # #  HRESULT PylonWaitObjectsWaitForAllEx(PYLON_WAITOBJECTS_HANDLE hWos, ctypes.c_uint32 timeout, ctypes.c_ubyte alertable, ctypes.POINTER(ctypes.c_int) pWaitResult)
        # self.PylonWaitObjectsWaitForAllEx=wrapper(lib.PylonWaitObjectsWaitForAllEx)
        # #  HRESULT PylonWaitObjectsWaitForAnyEx(PYLON_WAITOBJECTS_HANDLE hWos, ctypes.c_uint32 timeout, ctypes.POINTER(ctypes.c_size_t) pIndex, ctypes.c_ubyte alertable, ctypes.POINTER(ctypes.c_int) pWaitResult)
        # self.PylonWaitObjectsWaitForAnyEx=wrapper(lib.PylonWaitObjectsWaitForAnyEx)
        # #  HRESULT PylonEventGrabberOpen(PYLON_EVENTGRABBER_HANDLE hEvg)
        # self.PylonEventGrabberOpen=wrapper(lib.PylonEventGrabberOpen)
        # #  HRESULT PylonEventGrabberClose(PYLON_EVENTGRABBER_HANDLE hEvg)
        # self.PylonEventGrabberClose=wrapper(lib.PylonEventGrabberClose)
        # #  HRESULT PylonEventGrabberIsOpen(PYLON_EVENTGRABBER_HANDLE hEvg, ctypes.POINTER(ctypes.c_ubyte) pOpen)
        # self.PylonEventGrabberIsOpen=wrapper(lib.PylonEventGrabberIsOpen)
        # #  HRESULT PylonEventGrabberRetrieveEvent(PYLON_EVENTGRABBER_HANDLE hEvg, ctypes.POINTER(PylonEventResult_t) pEventResult, ctypes.POINTER(ctypes.c_ubyte) pReady)
        # self.PylonEventGrabberRetrieveEvent=wrapper(lib.PylonEventGrabberRetrieveEvent)
        # #  HRESULT PylonEventGrabberGetWaitObject(PYLON_EVENTGRABBER_HANDLE hEvg, ctypes.POINTER(PYLON_WAITOBJECT_HANDLE) phWobj)
        # self.PylonEventGrabberGetWaitObject=wrapper(lib.PylonEventGrabberGetWaitObject)
        # #  HRESULT PylonEventGrabberGetNumBuffers(PYLON_EVENTGRABBER_HANDLE hEvg, ctypes.POINTER(ctypes.c_size_t) pNumBuffers)
        # self.PylonEventGrabberGetNumBuffers=wrapper(lib.PylonEventGrabberGetNumBuffers)
        # #  HRESULT PylonEventGrabberSetNumBuffers(PYLON_EVENTGRABBER_HANDLE hEvg, ctypes.c_size_t numBuffers)
        # self.PylonEventGrabberSetNumBuffers=wrapper(lib.PylonEventGrabberSetNumBuffers)
        # #  HRESULT PylonEventAdapterDeliverMessage(PYLON_EVENTADAPTER_HANDLE hEva, ctypes.POINTER(PylonEventResult_t) pEventResult)
        # self.PylonEventAdapterDeliverMessage=wrapper(lib.PylonEventAdapterDeliverMessage)
        # #  HRESULT PylonChunkParserAttachBuffer(PYLON_CHUNKPARSER_HANDLE hChp, ctypes.c_void_p pBuffer, ctypes.c_size_t BufLen)
        # self.PylonChunkParserAttachBuffer=wrapper(lib.PylonChunkParserAttachBuffer)
        # #  HRESULT PylonChunkParserDetachBuffer(PYLON_CHUNKPARSER_HANDLE hChp)
        # self.PylonChunkParserDetachBuffer=wrapper(lib.PylonChunkParserDetachBuffer)
        # #  HRESULT PylonChunkParserUpdateBuffer(PYLON_CHUNKPARSER_HANDLE hChp, ctypes.c_void_p pBuffer)
        # self.PylonChunkParserUpdateBuffer=wrapper(lib.PylonChunkParserUpdateBuffer)
        # #  HRESULT PylonChunkParserHasCRC(PYLON_CHUNKPARSER_HANDLE hChp, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonChunkParserHasCRC=wrapper(lib.PylonChunkParserHasCRC)
        # #  HRESULT PylonChunkParserCheckCRC(PYLON_CHUNKPARSER_HANDLE hChp, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonChunkParserCheckCRC=wrapper(lib.PylonChunkParserCheckCRC)
        # #  HRESULT PylonPixelFormatConverterCreate(PYLON_DEVICE_HANDLE hDev, ctypes.c_int outAlign, ctypes.POINTER(PYLON_FORMAT_CONVERTER_HANDLE) phConv)
        # self.PylonPixelFormatConverterCreate=wrapper(lib.PylonPixelFormatConverterCreate)
        # #  HRESULT PylonPixelFormatConverterConvert(PYLON_FORMAT_CONVERTER_HANDLE hConv, ctypes.c_void_p targetBuffer, ctypes.c_size_t targetBufferSize, ctypes.c_void_p sourceBuffer, ctypes.c_size_t sourceBufferSize)
        # self.PylonPixelFormatConverterConvert=wrapper(lib.PylonPixelFormatConverterConvert)
        # #  HRESULT PylonPixelFormatConverterGetOutputBufferSize(PYLON_FORMAT_CONVERTER_HANDLE hConv, ctypes.POINTER(ctypes.c_size_t) pBufSiz)
        # self.PylonPixelFormatConverterGetOutputBufferSize=wrapper(lib.PylonPixelFormatConverterGetOutputBufferSize)
        # #  HRESULT PylonPixelFormatConverterDestroy(PYLON_FORMAT_CONVERTER_HANDLE hConv)
        # self.PylonPixelFormatConverterDestroy=wrapper(lib.PylonPixelFormatConverterDestroy)
        # #  HRESULT PylonBitsPerPixel(ctypes.c_int pixelType, ctypes.POINTER(ctypes.c_int) pResult)
        # self.PylonBitsPerPixel=wrapper(lib.PylonBitsPerPixel)
        # #  HRESULT PylonIsMono(ctypes.c_int pixelType, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonIsMono=wrapper(lib.PylonIsMono)
        # #  HRESULT PylonIsBayer(ctypes.c_int pixelType, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.PylonIsBayer=wrapper(lib.PylonIsBayer)
        # #  HRESULT PylonPixelTypeFromString(ctypes.c_char_p pString, ctypes.POINTER(ctypes.c_int) pPixelType)
        # self.PylonPixelTypeFromString=wrapper(lib.PylonPixelTypeFromString)
        # #  HRESULT PylonDevicePortRead(PYLON_DEVICE_HANDLE hDev, ctypes.c_void_p pBuffer, ctypes.c_int64 Address, ctypes.c_size_t Length)
        # self.PylonDevicePortRead=wrapper(lib.PylonDevicePortRead)
        # #  HRESULT PylonDevicePortWrite(PYLON_DEVICE_HANDLE hDev, ctypes.c_void_p pBuffer, ctypes.c_int64 Address, ctypes.c_size_t Length)
        # self.PylonDevicePortWrite=wrapper(lib.PylonDevicePortWrite)
        # #  HRESULT PylonFeaturePersistenceSave(NODEMAP_HANDLE hMap, ctypes.c_char_p pFileName)
        # self.PylonFeaturePersistenceSave=wrapper(lib.PylonFeaturePersistenceSave)
        # #  HRESULT PylonFeaturePersistenceLoad(NODEMAP_HANDLE hMap, ctypes.c_char_p pFileName, ctypes.c_ubyte verify)
        # self.PylonFeaturePersistenceLoad=wrapper(lib.PylonFeaturePersistenceLoad)
        # #  HRESULT PylonFeaturePersistenceSaveToString(NODEMAP_HANDLE hMap, ctypes.c_char_p pFeatures, ctypes.POINTER(ctypes.c_size_t) pFeaturesLen)
        # self.PylonFeaturePersistenceSaveToString=wrapper(lib.PylonFeaturePersistenceSaveToString)
        # #  HRESULT PylonFeaturePersistenceLoadFromString(NODEMAP_HANDLE hMap, ctypes.c_char_p pFeatures, ctypes.c_ubyte verify)
        # self.PylonFeaturePersistenceLoadFromString=wrapper(lib.PylonFeaturePersistenceLoadFromString)
        # #  HRESULT PylonImagePersistenceSave(ctypes.c_int imageFileFormat, ctypes.c_char_p pFilename, ctypes.c_void_p pBuffer, ctypes.c_size_t bufferSize, ctypes.c_int pixelType, ctypes.c_uint32 width, ctypes.c_uint32 height, ctypes.c_size_t paddingX, ctypes.c_int orientation, ctypes.POINTER(PylonImagePersistenceOptions_t) pOptions)
        # self.PylonImagePersistenceSave=wrapper(lib.PylonImagePersistenceSave)
        # #  HRESULT PylonImagePersistenceLoad(ctypes.c_char_p pFilename, ctypes.c_void_p pBuffer, ctypes.POINTER(ctypes.c_size_t) pBufferSize, ctypes.POINTER(ctypes.c_int) pPixelType, ctypes.POINTER(ctypes.c_uint32) pWidth, ctypes.POINTER(ctypes.c_uint32) pHeight, ctypes.POINTER(ctypes.c_size_t) pPaddingX, ctypes.POINTER(ctypes.c_int) pOrientation)
        # self.PylonImagePersistenceLoad=wrapper(lib.PylonImagePersistenceLoad)
        # #  HRESULT PylonImageDecompressorCreate(ctypes.POINTER(PYLON_IMAGE_DECOMPRESSOR_HANDLE) phDecompressor)
        # self.PylonImageDecompressorCreate=wrapper(lib.PylonImageDecompressorCreate)
        # #  HRESULT PylonImageDecompressorDestroy(PYLON_IMAGE_DECOMPRESSOR_HANDLE hDecompressor)
        # self.PylonImageDecompressorDestroy=wrapper(lib.PylonImageDecompressorDestroy)
        # #  HRESULT PylonImageDecompressorSetCompressionDescriptor(PYLON_IMAGE_DECOMPRESSOR_HANDLE hDecompressor, ctypes.c_void_p pCompressionDescriptor, ctypes.c_size_t sizeCompressionDescriptor)
        # self.PylonImageDecompressorSetCompressionDescriptor=wrapper(lib.PylonImageDecompressorSetCompressionDescriptor)
        # #  HRESULT PylonImageDecompressorGetCompressionInfo(ctypes.c_void_p pPayload, ctypes.c_size_t payloadSize, ctypes.POINTER(PylonCompressionInfo_t) pCompressionInfo)
        # self.PylonImageDecompressorGetCompressionInfo=wrapper(lib.PylonImageDecompressorGetCompressionInfo)
        # #  HRESULT PylonImageDecompressorDecompressImage(PYLON_IMAGE_DECOMPRESSOR_HANDLE hDecompressor, ctypes.c_void_p pOutputBuffer, ctypes.POINTER(ctypes.c_size_t) pOutputBufferSize, ctypes.c_void_p pPayload, ctypes.c_size_t payloadSize, ctypes.POINTER(PylonCompressionInfo_t) pCompressionInfo)
        # self.PylonImageDecompressorDecompressImage=wrapper(lib.PylonImageDecompressorDecompressImage)
        



        # #  HRESULT PylonGigEEnumerateAllDevices(ctypes.POINTER(ctypes.c_size_t) numDevices)
        # self.PylonGigEEnumerateAllDevices=wrapper(lib.PylonGigEEnumerateAllDevices)
        # #  HRESULT PylonGigEForceIp(ctypes.c_char_p pMacAddress, ctypes.c_char_p pIpAddress, ctypes.c_char_p pSubnetMask, ctypes.c_char_p pDefaultGateway)
        # self.PylonGigEForceIp=wrapper(lib.PylonGigEForceIp)
        # #  HRESULT PylonGigERestartIpConfiguration(ctypes.c_char_p pMacAddress)
        # self.PylonGigERestartIpConfiguration=wrapper(lib.PylonGigERestartIpConfiguration)
        # #  HRESULT PylonGigEChangeIpConfiguration(PYLON_DEVICE_HANDLE hDev, ctypes.c_ubyte EnablePersistentIp, ctypes.c_ubyte EnableDhcp)
        # self.PylonGigEChangeIpConfiguration=wrapper(lib.PylonGigEChangeIpConfiguration)
        # #  HRESULT PylonGigESetPersistentIpAddress(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pIpAddress, ctypes.c_char_p pSubnetMask, ctypes.c_char_p pDefaultGateway)
        # self.PylonGigESetPersistentIpAddress=wrapper(lib.PylonGigESetPersistentIpAddress)
        # #  HRESULT PylonGigEAnnounceRemoteDevice(ctypes.c_char_p pIpAddress)
        # self.PylonGigEAnnounceRemoteDevice=wrapper(lib.PylonGigEAnnounceRemoteDevice)
        # #  HRESULT PylonGigERenounceRemoteDevice(ctypes.c_char_p pIpAddress, ctypes.POINTER(ctypes.c_ubyte) pFound)
        # self.PylonGigERenounceRemoteDevice=wrapper(lib.PylonGigERenounceRemoteDevice)
        # #  HRESULT PylonGigEGetPersistentIpAddress(PYLON_DEVICE_HANDLE hDev, ctypes.c_char_p pIpAddress, ctypes.POINTER(ctypes.c_size_t) pIpAddressLen, ctypes.c_char_p pSubnetMask, ctypes.POINTER(ctypes.c_size_t) pSubnetMaskLen, ctypes.c_char_p pDefaultGateway, ctypes.POINTER(ctypes.c_size_t) pDefaultGatewayLen)
        # self.PylonGigEGetPersistentIpAddress=wrapper(lib.PylonGigEGetPersistentIpAddress)
        # #  HRESULT PylonGigEBroadcastIpConfiguration(ctypes.c_char_p pMacAddress, ctypes.c_ubyte EnablePersistentIp, ctypes.c_ubyte EnableDHCP, ctypes.c_char_p pIpAddress, ctypes.c_char_p pSubnetMask, ctypes.c_char_p pDefaultGateway, ctypes.c_char_p pUserdefinedName, ctypes.POINTER(ctypes.c_ubyte) pRetval)
        # self.PylonGigEBroadcastIpConfiguration=wrapper(lib.PylonGigEBroadcastIpConfiguration)
        # #  HRESULT PylonGigEIssueActionCommand(ctypes.c_uint32 deviceKey, ctypes.c_uint32 groupKey, ctypes.c_uint32 groupMask, ctypes.c_char_p pBroadcastAddress, ctypes.c_uint32 timeoutMs, ctypes.POINTER(ctypes.c_uint32) pNumResults, ctypes.POINTER(PylonGigEActionCommandResult_t) pResults)
        # self.PylonGigEIssueActionCommand=wrapper(lib.PylonGigEIssueActionCommand)
        # #  HRESULT PylonGigEIssueScheduledActionCommand(ctypes.c_uint32 deviceKey, ctypes.c_uint32 groupKey, ctypes.c_uint32 groupMask, ctypes.c_uint64 actiontimeNs, ctypes.c_char_p pBroadcastAddress, ctypes.c_uint32 timeoutMs, ctypes.POINTER(ctypes.c_uint32) pNumResults, ctypes.POINTER(PylonGigEActionCommandResult_t) pResults)
        # self.PylonGigEIssueScheduledActionCommand=wrapper(lib.PylonGigEIssueScheduledActionCommand)
        # #  HRESULT PylonCreateDeviceFromDirectShowID(ctypes.c_int id, ctypes.POINTER(PYLON_DEVICE_HANDLE) phDev)
        # self.PylonCreateDeviceFromDirectShowID=wrapper(lib.PylonCreateDeviceFromDirectShowID)
        




        ### Skipping ###
        # #  HRESULT GenApiNodeMapPoll(NODEMAP_HANDLE hMap, ctypes.c_int64 timestamp)
        # self.GenApiNodeMapPoll=wrapper(lib.GenApiNodeMapPoll)
        # #  HRESULT GenApiNodeGetNameSpace(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int) pNamespace)
        # self.GenApiNodeGetNameSpace=wrapper(lib.GenApiNodeGetNameSpace)
        # #  HRESULT GenApiNodeInvalidateNode(NODE_HANDLE hNode)
        # self.GenApiNodeInvalidateNode=wrapper(lib.GenApiNodeInvalidateNode)
        # #  HRESULT GenApiNodeRegisterCallback(NODE_HANDLE hNode, ctypes.c_void_p pCbFunction, ctypes.POINTER(NODE_CALLBACK_HANDLE) phCb)
        # self.GenApiNodeRegisterCallback=wrapper(lib.GenApiNodeRegisterCallback)
        # #  HRESULT GenApiNodeDeregisterCallback(NODE_HANDLE hNode, NODE_CALLBACK_HANDLE hCb)
        # self.GenApiNodeDeregisterCallback=wrapper(lib.GenApiNodeDeregisterCallback)
        # #  HRESULT GenApiSelectorGetNumSelectingFeatures(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_size_t) pValue)
        # self.GenApiSelectorGetNumSelectingFeatures=wrapper(lib.GenApiSelectorGetNumSelectingFeatures)
        # #  HRESULT GenApiSelectorGetSelectingFeatureByIndex(NODE_HANDLE hNode, ctypes.c_size_t index, ctypes.POINTER(NODE_HANDLE) phNode)
        # self.GenApiSelectorGetSelectingFeatureByIndex=wrapper(lib.GenApiSelectorGetSelectingFeatureByIndex)
        # #  HRESULT GenApiSelectorGetNumSelectedFeatures(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_size_t) pValue)
        # self.GenApiSelectorGetNumSelectedFeatures=wrapper(lib.GenApiSelectorGetNumSelectedFeatures)
        # #  HRESULT GenApiSelectorGetSelectedFeatureByIndex(NODE_HANDLE hNode, ctypes.c_size_t index, ctypes.POINTER(NODE_HANDLE) phNode)
        # self.GenApiSelectorGetSelectedFeatureByIndex=wrapper(lib.GenApiSelectorGetSelectedFeatureByIndex)
        # #  HRESULT GenApiRegisterGetValue(NODE_HANDLE hNode, ctypes.c_void_p pBuffer, ctypes.POINTER(ctypes.c_size_t) pLength)
        # self.GenApiRegisterGetValue=wrapper(lib.GenApiRegisterGetValue)
        # #  HRESULT GenApiRegisterSetValue(NODE_HANDLE hNode, ctypes.c_void_p pBuffer, ctypes.c_size_t length)
        # self.GenApiRegisterSetValue=wrapper(lib.GenApiRegisterSetValue)
        # #  HRESULT GenApiRegisterGetLength(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_size_t) pLength)
        # self.GenApiRegisterGetLength=wrapper(lib.GenApiRegisterGetLength)
        # #  HRESULT GenApiRegisterGetAddress(NODE_HANDLE hNode, ctypes.POINTER(ctypes.c_int64) pAddress)
        # self.GenApiRegisterGetAddress=wrapper(lib.GenApiRegisterGetAddress)
        # #  HRESULT GenApiPortRead(NODE_HANDLE hNode, ctypes.c_void_p pBuffer, ctypes.c_int64 Address, ctypes.c_size_t Length)
        # self.GenApiPortRead=wrapper(lib.GenApiPortRead)
        # #  HRESULT GenApiPortWrite(NODE_HANDLE hNode, ctypes.c_void_p pBuffer, ctypes.c_int64 Address, ctypes.c_size_t Length)
        # self.GenApiPortWrite=wrapper(lib.GenApiPortWrite)
        # #  HRESULT GenApiNodeMapPollInt32(NODEMAP_HANDLE hMap, ctypes.c_int32 timestamp)
        # self.GenApiNodeMapPollInt32=wrapper(lib.GenApiNodeMapPollInt32)


        ### Not required ###
        # #  HRESULT GenApiFilesAreSupported(NODEMAP_HANDLE hMap, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.GenApiFilesAreSupported=wrapper(lib.GenApiFilesAreSupported)
        # #  HRESULT GenApiFileExists(NODEMAP_HANDLE hMap, ctypes.c_char_p pFileName, ctypes.POINTER(ctypes.c_ubyte) pResult)
        # self.GenApiFileExists=wrapper(lib.GenApiFileExists)
        # #  HRESULT GenApiFileOpen(NODEMAP_HANDLE hMap, ctypes.c_char_p pFileName, ctypes.c_int accessMode, ctypes.POINTER(GENAPI_FILE_HANDLE) phFile)
        # self.GenApiFileOpen=wrapper(lib.GenApiFileOpen)
        # #  HRESULT GenApiFileRead(GENAPI_FILE_HANDLE hFile, ctypes.c_void_p pBuffer, ctypes.POINTER(ctypes.c_size_t) pLength)
        # self.GenApiFileRead=wrapper(lib.GenApiFileRead)
        # #  HRESULT GenApiFileWrite(GENAPI_FILE_HANDLE hFile, ctypes.c_void_p pBuffer, ctypes.c_size_t length)
        # self.GenApiFileWrite=wrapper(lib.GenApiFileWrite)
        # #  HRESULT GenApiFileClose(GENAPI_FILE_HANDLE hFile)
        # self.GenApiFileClose=wrapper(lib.GenApiFileClose)
        # #  HRESULT PylonImageFormatConverterCreate(ctypes.POINTER(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE) phConv)
        # self.PylonImageFormatConverterCreate=wrapper(lib.PylonImageFormatConverterCreate)
        # #  HRESULT PylonImageFormatConverterGetNodeMap(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.POINTER(NODEMAP_HANDLE) phMap)
        # self.PylonImageFormatConverterGetNodeMap=wrapper(lib.PylonImageFormatConverterGetNodeMap)
        # #  HRESULT PylonImageFormatConverterSetOutputPixelFormat(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.c_int pixelType)
        # self.PylonImageFormatConverterSetOutputPixelFormat=wrapper(lib.PylonImageFormatConverterSetOutputPixelFormat)
        # #  HRESULT PylonImageFormatConverterGetOutputPixelFormat(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.POINTER(ctypes.c_int) pPixelType)
        # self.PylonImageFormatConverterGetOutputPixelFormat=wrapper(lib.PylonImageFormatConverterGetOutputPixelFormat)
        # #  HRESULT PylonImageFormatConverterSetOutputPaddingX(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.c_size_t paddingX)
        # self.PylonImageFormatConverterSetOutputPaddingX=wrapper(lib.PylonImageFormatConverterSetOutputPaddingX)
        # #  HRESULT PylonImageFormatConverterGetOutputPaddingX(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.POINTER(ctypes.c_size_t) pPaddingX)
        # self.PylonImageFormatConverterGetOutputPaddingX=wrapper(lib.PylonImageFormatConverterGetOutputPaddingX)
        # #  HRESULT PylonImageFormatConverterConvert(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.c_void_p targetBuffer, ctypes.c_size_t targetBufferSize, ctypes.c_void_p sourceBuffer, ctypes.c_size_t sourceBufferSize, ctypes.c_int sourcePixelType, ctypes.c_uint32 sourceWidth, ctypes.c_uint32 sourceHeight, ctypes.c_size_t sourcePaddingX, ctypes.c_int sourceOrientation)
        # self.PylonImageFormatConverterConvert=wrapper(lib.PylonImageFormatConverterConvert)
        # #  HRESULT PylonImageFormatConverterGetBufferSizeForConversion(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv, ctypes.c_int sourcePixelType, ctypes.c_uint32 sourceWidth, ctypes.c_uint32 sourceHeight, ctypes.POINTER(ctypes.c_size_t) pBufSize)
        # self.PylonImageFormatConverterGetBufferSizeForConversion=wrapper(lib.PylonImageFormatConverterGetBufferSizeForConversion)
        # #  HRESULT PylonImageFormatConverterDestroy(PYLON_IMAGE_FORMAT_CONVERTER_HANDLE hConv)
        # self.PylonImageFormatConverterDestroy=wrapper(lib.PylonImageFormatConverterDestroy)
        # #  HRESULT PylonAviWriterCreate(ctypes.POINTER(PYLON_AVI_WRITER_HANDLE) phWriter)
        # self.PylonAviWriterCreate=wrapper(lib.PylonAviWriterCreate)
        # #  HRESULT PylonAviWriterDestroy(PYLON_AVI_WRITER_HANDLE hWriter)
        # self.PylonAviWriterDestroy=wrapper(lib.PylonAviWriterDestroy)
        # #  HRESULT PylonAviWriterOpen(PYLON_AVI_WRITER_HANDLE hWriter, ctypes.c_char_p pFilename, ctypes.c_double framesPerSecondPlayback, ctypes.c_int pixelType, ctypes.c_uint32 width, ctypes.c_uint32 height, ctypes.c_int orientation, ctypes.c_void_p pCompressionOptions)
        # self.PylonAviWriterOpen=wrapper(lib.PylonAviWriterOpen)
        # #  HRESULT PylonAviWriterClose(PYLON_AVI_WRITER_HANDLE hWriter)
        # self.PylonAviWriterClose=wrapper(lib.PylonAviWriterClose)
        # #  HRESULT PylonAviWriterIsOpen(PYLON_AVI_WRITER_HANDLE hWriter, ctypes.POINTER(ctypes.c_ubyte) pIsOpen)
        # self.PylonAviWriterIsOpen=wrapper(lib.PylonAviWriterIsOpen)
        # #  HRESULT PylonAviWriterAdd(PYLON_AVI_WRITER_HANDLE hWriter, ctypes.c_void_p pBuffer, ctypes.c_size_t bufferSize, ctypes.c_int pixelType, ctypes.c_uint32 width, ctypes.c_uint32 height, ctypes.c_size_t paddingX, ctypes.c_int orientation, ctypes.c_int keyFrameSelection)
        # self.PylonAviWriterAdd=wrapper(lib.PylonAviWriterAdd)
        # #  HRESULT PylonAviWriterCanAddWithoutConversion(PYLON_AVI_WRITER_HANDLE hWriter, ctypes.c_int pixelType, ctypes.c_uint32 width, ctypes.c_uint32 height, ctypes.c_size_t paddingX, ctypes.c_int orientation, ctypes.POINTER(ctypes.c_ubyte) pCanAddWithoutConversion)
        # self.PylonAviWriterCanAddWithoutConversion=wrapper(lib.PylonAviWriterCanAddWithoutConversion)
        # #  HRESULT PylonAviWriterGetImageDataBytesWritten(PYLON_AVI_WRITER_HANDLE hWriter, ctypes.POINTER(ctypes.c_size_t) pImageDataBytesWritten)
        # self.PylonAviWriterGetImageDataBytesWritten=wrapper(lib.PylonAviWriterGetImageDataBytesWritten)
        # #  HRESULT PylonAviWriterGetCountOfAddedImages(PYLON_AVI_WRITER_HANDLE hWriter, ctypes.POINTER(ctypes.c_size_t) pCountOfAddedImages)
        # self.PylonAviWriterGetCountOfAddedImages=wrapper(lib.PylonAviWriterGetCountOfAddedImages)
        # #  HRESULT PylonImageWindowCreate(ctypes.c_size_t winIndex, ctypes.c_int x, ctypes.c_int y, ctypes.c_int nWidth, ctypes.c_int nHeight)
        # self.PylonImageWindowCreate=wrapper(lib.PylonImageWindowCreate)
        # #  HRESULT PylonImageWindowShow(ctypes.c_size_t winIndex, ctypes.c_int nShow)
        # self.PylonImageWindowShow=wrapper(lib.PylonImageWindowShow)
        # #  HRESULT PylonImageWindowHide(ctypes.c_size_t winIndex)
        # self.PylonImageWindowHide=wrapper(lib.PylonImageWindowHide)
        # #  HRESULT PylonImageWindowClose(ctypes.c_size_t winIndex)
        # self.PylonImageWindowClose=wrapper(lib.PylonImageWindowClose)
        # #  HRESULT PylonImageWindowGetWindowHandle(ctypes.c_size_t winIndex, ctypes.POINTER(HWND) phWindow)
        # self.PylonImageWindowGetWindowHandle=wrapper(lib.PylonImageWindowGetWindowHandle)
        # #  HRESULT PylonImageWindowSetImage(ctypes.c_size_t winIndex, ctypes.c_void_p buffer, ctypes.c_size_t bufferSize, ctypes.c_int pixelType, ctypes.c_uint32 width, ctypes.c_uint32 height, ctypes.c_size_t paddingX, ctypes.c_int orientation)
        # self.PylonImageWindowSetImage=wrapper(lib.PylonImageWindowSetImage)
        # #  HRESULT PylonImageWindowDisplayImage(ctypes.c_size_t winIndex, ctypes.c_void_p buffer, ctypes.c_size_t bufferSize, ctypes.c_int pixelType, ctypes.c_uint32 width, ctypes.c_uint32 height, ctypes.c_size_t paddingX, ctypes.c_int orientation)
        # self.PylonImageWindowDisplayImage=wrapper(lib.PylonImageWindowDisplayImage)
        # #  HRESULT PylonImageWindowDisplayImageGrabResult(ctypes.c_size_t winIndex, ctypes.POINTER(PylonGrabResult_t) grabResult)
        # self.PylonImageWindowDisplayImageGrabResult=wrapper(lib.PylonImageWindowDisplayImageGrabResult)
        

    _branch_node_name="_branch"
    def collect_nodes(self, node, pfx="", tree=None, add_branch=True):
        if tree is None:
            tree={}
        name=py3.as_str(self.GenApiNodeGetName(node))
        if pfx:
            name=pfx+"/"+name
        if self.GenApiNodeGetType(node)==EGenApiNodeType.Category:
            if add_branch:
                tree[name+"/"+self._branch_node_name]=node
            for i in range(self.GenApiCategoryGetNumFeatures(node)):
                self.collect_nodes(self.GenApiCategoryGetFeatureByIndex(node,i),pfx=name,tree=tree,add_branch=add_branch)
        else:
            tree[name]=node
        return tree

wlib=pylonCLib()