%feature("docstring") OT::GaussianProcessConditionalCovariance
"Conditional covariance post processing of a Gaussian Process Regression result.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.

Parameters
----------
gprResult :  :class:`~openturns.experimental.GaussianProcessRegressionResult`
    The result class of a Gaussian process regression.


Notes
-----
The class provides services around conditional covariance of a Gaussian Process Regression


Examples
--------
Create the model :math:`\\cM: \\Rset \\mapsto \\Rset` and the samples:

>>> import openturns as ot
>>> from openturns.experimental import GaussianProcessRegression
>>> from openturns.experimental import GaussianProcessConditionalCovariance
>>> trend = ot.SymbolicFunction(['x'],  ['1'])
>>> sampleX = [[1.0], [2.0], [3.0], [4.0], [5.0], [6.0]]
>>> sampleY = trend(sampleX)

Create the algorithm:

>>> covarianceModel = ot.SquaredExponential([1.0])
>>> covarianceModel.setActiveParameter([])

>>> algo = GaussianProcessRegression(sampleX, sampleY, covarianceModel, trend)
>>> algo.run()
>>> result = algo.getResult()
>>> condCov = GaussianProcessConditionalCovariance(result)
>>> c = condCov([1.1])

"


// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessConditionalCovariance::getConditionalMean
"Compute the conditional mean of the Gaussian process on a point or a sample of points.

Parameters
----------
x : sequence of float
    The point :math:`\\vect{x}` where the conditional mean of the output has to be evaluated.
sampleX : 2-d sequence of float
     The sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)` where the conditional mean of the output has to be evaluated (*M* can be equal to 1).

Returns
-------
condMean : :class:`~openturns.Point`
    The conditional mean :math:`\\Expect{\\vect{Y}(\\omega, \\vect{x})\\, | \\,  \\cC}` at point :math:`\\vect{x}`.
    Or the conditional mean matrix at the sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)`:

    .. math::

        \\left(
          \\begin{array}{l}
            \\Expect{\\vect{Y}(\\omega, \\vect{\\xi}_1)\\, | \\,  \\cC}\\\\
            \\dots  \\\\
            \\Expect{\\vect{Y}(\\omega, \\vect{\\xi}_M)\\, | \\,  \\cC}
          \\end{array}
        \\right)

"


// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessConditionalCovariance::getConditionalCovariance
"Compute the conditional covariance of the Gaussian process on a point (or several points).

Parameters
----------
x : sequence of float
    The point :math:`\\vect{x}` where the conditional covariance of the output has to be evaluated.
sampleX : 2-d sequence of float
     The sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)` where the conditional covariance of the output has to be evaluated (*M* can be equal to 1).

Returns
-------
condCov : :class:`~openturns.CovarianceMatrix`
    The conditional covariance :math:`\\Cov{\\vect{Y}(\\omega, \\vect{x})\\, | \\,  \\cC}` at point :math:`\\vect{x}`.
    Or the conditional covariance matrix at the sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)`:

    .. math::

        \\left(
          \\begin{array}{lcl}
            \\Sigma_{11} & \\dots & \\Sigma_{1M} \\\\
            \\dots  \\\\
            \\Sigma_{M1} & \\dots & \\Sigma_{MM}
          \\end{array}
        \\right)

    where :math:`\\Sigma_{ij} = \\Cov{\\vect{Y}(\\omega, \\vect{\\xi}_i), \\vect{Y}(\\omega, \\vect{\\xi}_j)\\, | \\,  \\cC}`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessConditionalCovariance::getConditionalMarginalCovariance
"Compute the conditional covariance of the Gaussian process on a point (or several points).

Parameters
----------
x : sequence of float
    The point :math:`\\vect{x}` where the conditional marginal covariance of the output has to be evaluated.
sampleX : 2-d sequence of float
     The sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)` where the conditional marginal covariance of the output has to be evaluated (*M* can be equal to 1).

Returns
-------
condCov : :class:`~openturns.CovarianceMatrix`
    The conditional covariance :math:`\\Cov{\\vect{Y}(\\omega, \\vect{x})\\, | \\,  \\cC}` at point :math:`\\vect{x}`.

condCov : :class:`~openturns.CovarianceMatrixCollection`
    The collection of conditional covariance matrices :math:`\\Cov{\\vect{Y}(\\omega, \\vect{\\xi})\\, | \\,  \\cC}` at
    each point of the sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)`:

Notes
-----
In case input parameter is a of type :class:`~openturns.Sample`, each element of the collection corresponds to the conditional
covariance with respect to the input learning set (pointwise evaluation of the getConditionalCovariance)."
// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessConditionalCovariance::getConditionalMarginalVariance
"Compute the conditional variance of the Gaussian process on a point (or several points).

Parameters
----------
x : sequence of float
    The point :math:`\\vect{x}` where the conditional variance of the output has to be evaluated.
sampleX : 2-d sequence of float
     The sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)` where the conditional variance of the output has to be evaluated (*M* can be equal to 1).
marginalIndex : int
    Marginal of interest (for multiple outputs).
    Default value is 0
marginalIndices : sequence of int
    Marginals of interest (for multiple outputs).

Returns
-------
var : float
      Variance of interest.
      float if one point (x) and one marginal of interest (x, marginalIndex)

varPoint : sequence of float
    The marginal variances


Notes
-----
In case of fourth usage, the sequence of float is given as the concatenation of marginal variances 
for each point in sampleX."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessConditionalCovariance::operator()
"Compute the conditional Gaussian distribution on a new point / sample conditionally to the observed paths.

Parameters
----------
pt : sequence of float
    The point :math:`\\vect{x}` where the conditional distribution of the output has to be evaluated.
sampleX : 2-d sequence of float
     The sample :math:`(\\vect{\\xi}_1, \\dots, \\vect{\\xi}_M)` where the conditional distribution of the output has to be evaluated (*M* can be equal to 1).

Returns
-------
condDist : :class:`~openturns.Normal`
    The conditional Gaussian distribution.

Notes
-----
The conditional distribution :math:`\\cN(\\Expect{\\vect{Y}}, \\Cov{\\vect{Y}})` has respectively conditional mean and covariance functions 
implemented in :meth:`getConditionalMean` (respectively :meth:`getConditionalCovariance`)."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessConditionalCovariance::getMetaModel
"Accessor to the metamodel.

Returns
-------
metaModel : :class:`~openturns.Function`
    The meta model :math:`\\tilde{\\cM}: \\Rset^d \\rightarrow \\Rset^p`.
"

