%feature("docstring") OT::WhittleFactory
"Whittle estimator of a scalar ARMA Gaussian process.

Available constructors:
    WhittleFactory()

    WhittleFactory(*p, q, invert*)

    WhittleFactory(*indP, indQ, invertible*)


Parameters
----------
p : int
    Order of the AR part of the :math:`ARMA(p,q)` process of dimension :math:`d`.
q : int
    Order of the MA part of the :math:`ARMA(p,q)` process of dimension :math:`d`.
invertible : bool, optional
    Restrict the estimation to invertible ARMA processes.

    By default: True.
indP : :class:`~openturns.Indices`
    All the :math:`p` orders that will be investigated.
    Care: not yet implemented.
indQ : :class:`~openturns.Indices`
    All the :math:`p` orders that will be investigated.
    Care: not yet implemented.

Notes
-----
We suppose here that the white noise is normal with zero mean and  variance :math:`\\sigma^2`. It implies that the ARMA process estimated is normal.

For each order :math:`(p,q)`, the estimation of the coefficients :math:`(a_k)_{1\\leq k\\leq p}`, :math:`(b_k)_{1\\leq k\\leq q}` and the variance :math:`\\sigma^2` is done using the Whittle estimator which is based on the maximization of the likelihood function in the frequency domain. 

The principle is detailed hereafter for the case of a time series : in the case of a process sample, the estimator is similar except for the periodogram which is computed differently.

Let :math:`(t_i, \\vect{X}_i)_{0\\leq i \\leq n-1}` be a multivariate time series of dimension :math:`d` from an :math:`ARMA(p,q)` process. 

The spectral density function of the :math:`ARMA(p,q)` process writes :

.. math::

    f(\\lambda, \\vect{\\theta}, \\sigma^2) = \\frac{\\sigma^2}{2 \\pi} \\frac{|1 + b_1 \\exp(-i \\lambda) + \\ldots + b_q \\exp(-i q \\lambda)|^2}{|1 + a_1 \\exp(-i \\lambda) + \\ldots + a_p \\exp(-i p \\lambda)|^2} = \\frac{\\sigma^2}{2 \\pi} g(\\lambda, \\vect{\\theta})

where  :math:`\\vect{\\theta} = (a_1, a_2,...,a_p,b_1,...,b_q)` and :math:`\\lambda` is the frequency value.

The Whittle log-likelihood writes :

.. math::

    \\log L_w(\\vect{\\theta}, \\sigma^2) = - \\sum_{j=1}^{m} \\log f(\\lambda_j,  \\vect{\\theta}, \\sigma^2) - \\frac{1}{2 \\pi} \\sum_{j=1}^{m} \\frac{I(\\lambda_j)}{f(\\lambda_j,  \\vect{\\theta}, \\sigma^2)}

where :

- :math:`I` is the non parametric estimate of the spectral density, expressed in the Fourier space (frequencies in :math:`[0,2\\pi]` instead of :math:`[-f_{max}, f_{max}]`). OpenTURNS uses by default the Welch estimator.

- :math:`\\lambda_j` is the :math:`j-th` Fourier frequency, :math:`\\lambda_j = \\frac{2 \\pi j}{n}`, :math:`j=1 \\ldots m` with :math:`m` the largest integer  :math:`\\leq \\frac{n-1}{2}`.


We estimate the :math:`(p+q+1)` scalar coefficients by maximizing the log-likelihood function. The corresponding equations lead to the following relation :

.. math::

    \\sigma^{2*} = \\frac{1}{m} \\sum_{j=1}^{m} \\frac{I(\\lambda_j)}{g(\\lambda_j, \\vect{\\theta}^{*})}

where :math:`\\vect{\\theta}^{*}` maximizes :

.. math::

    \\log L_w(\\vect{\\theta}) = m (\\log(2 \\pi) - 1) - m \\log\\left( \\frac{1}{m} \\sum_{j=1}^{m} \\frac{I(\\lambda_j)}{g(\\lambda_j, \\vect{\\theta})}\\right) - \\sum_{j=1}^{m} g(\\lambda_j, \\vect{\\theta})

The Whitle estimation requires that :

- the determinant of the eigenvalues of the companion matrix associated to the polynomial  :math:`1 + a_1X + \\dots + a_pX^p` are  outside the unit disc, which guarantees the stationarity of the process;

- the determinant of the eigenvalues of the companion matrix associated to the polynomial  :math:`1 + ba_1X + \\dots + b_qX^q` are outside the unit disc, which guarantees the invertibility of the process.


The criteria *AIC*, :math:`AIC_c` (corrected AIC) and *BIC* are evaluated to help the model selection:

.. math::
    :nowrap:

    \\begin{eqnarray*}
        AIC_c  & = &  -2\\log L_w + 2(p + q + 1)\\frac{m}{m - p - q - 2}\\\\
        AIC & = & -2\\log L_w + 2(p + q + 1)\\\\
        BIC & = & -2\\log L_w + 2(p + q + 1)\\log(p + q + 1)
    \\end{eqnarray*}

where :math:`m` is half the number of points of the time grid of the process sample (if the data are a process sample) or in a block of the time series (if the data are a time series).

The *BIC* criterion leads to a model that gives a better prediction. The *AIC* criterion selects the best model that fits the given data. The :math:`AIC_c` criterion improves the previous one by penalizing a too high order that would artificially fit to the data.


Examples
--------
Create a time series from a scalar ARMA(4,2) and a normal white noise:

>>> import openturns as ot
>>> myTimeGrid = ot.RegularGrid(0.0, 0.1, 100)
>>> myWhiteNoise = ot.WhiteNoise(ot.Triangular(-1.0, 0.0, 1.0), myTimeGrid)
>>> myARCoef = ot.ARMACoefficients([0.4, 0.3, 0.2, 0.1])
>>> myMACoef = ot.ARMACoefficients([0.4, 0.3])
>>> myARMAProcess = ot.ARMA(myARCoef, myMACoef, myWhiteNoise)
>>> myTimeSeries = myARMAProcess.getRealization()
>>> myProcessSample = myARMAProcess.getSample(10)

Estimate the ARMA process specifying the orders:

>>> myFactory_42 = ot.WhittleFactory(4, 2)

Check the default SpectralModelFactory:

>>> print(myFactory_42.getSpectralModelFactory())  # doctest: +SKIP

Set a particular spectral model: WelchFactory as SpectralModelFactory with the Hann filtering window:

>>> myFilteringWindow = ot.Hann()
>>> mySpectralFactory = ot.WelchFactory(myFilteringWindow, 4, 0)
>>> myFactory_42.setSpectralModelFactory(mySpectralFactory)
>>> print(myFactory_42.getSpectralModelFactory())  # doctest: +SKIP

Estimate the ARMA process specifying a range for the orders:

p = [1, 2, 4] and q = [4,5,6]:

>>> pIndices = [1, 2, 4]
>>> qIndices = [4, 5, 6]
>>> myFactory_Range = ot.WhittleFactory(pIndices, qIndices)

To get the quantified AICc, AIC and BIC criteria:

>>> myARMA_42, myCriterion = myFactory_42.buildWithCriteria(ot.TimeSeries(myTimeSeries))
>>> AICc, AIC, BIC = myCriterion[0:3]"

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::build
"Estimate the ARMA process.

Available usages:
    build(*myTimeSeries*)

    build(*myProcessSample*)

Parameters
----------
myTimeSeries : :class:`~openturns.TimeSeries`
    One realization of the  process.
myProcessSample : :class:`~openturns.ProcessSample`
    Several realizations of the  process.

Returns
-------
myARMA : :class:`~openturns.ARMA`
    The  process estimated with the Whittle estimator.

Notes
-----
The model selection is made using the spectral density built using the given data and theoretical spectral density of the ARMA process.

The best ARMA process is selected according to the corrected AIC criterion.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::buildWithCriteria
"Estimate the ARMA process.

Available usages:
    buildWithCriteria(*myTimeSeries*)

    buildWithCriteria(*myProcessSample*)

Parameters
----------
myTimeSeries : :class:`~openturns.TimeSeries`
    One realization of the  process.
myProcessSample : :class:`~openturns.ProcessSample`
    Several realizations of the  process.

Returns
-------
myARMA : :class:`~openturns.ARMA`
    The  process estimated with the Whittle estimator.
criterion : :class:`~openturns.Point`
    Result of the evaluation of the AICc, AIC and BIC criteria

Notes
-----
The model selection is made using the spectral density built using the given data and theoretical spectral density of the ARMA process.

The best ARMA process is selected according to the corrected AIC criterion."

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::getSpectralModelFactory
"Accessor to the spectral factory.

Returns
-------
initARCoeff : :class:`~openturns.SpectralModelFactory`
    The spectral factory used to estimate the spectral density based on the data.
"
// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::setSpectralModelFactory
"Accessor to the spectral factory.

Parameters
----------
spectralModelFact : :class:`~openturns.SpectralModelFactory`
    The spectral factory used to estimate the spectral density based on the data.
"
// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::enableHistory
"Activate the history of all the estimated models.

Notes
-----
Activate the history mechanism which is the trace of all the tested models and their associated information criteria.

By default, the history mechanism is activated.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::disableHistory
"Deactivate the history of all the estimated models.

Notes
-----
Deactivate the history mechanism which is the trace of all the tested models and their associated information criteria.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::WhittleFactory::isHistoryEnabled
"Check whether the history mechanism is activated.

Returns
-------
histMec : bool
    Check whether the history mechanism is activated.

    By default, the history mechanism is activated.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::WhittleFactory::isHistoryEnabled
"Check whether the history mechanism is activated.

Returns
-------
histMec : bool
    Check whether the history mechanism is activated.

    By default, the history mechanism is activated.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::WhittleFactory::clearHistory
"Clear the history of the factory.

Notes
-----
Clear the history of the factory.
"
// ---------------------------------------------------------------------
%feature("docstring") OT::WhittleFactory::getHistory
"Check whether the history mechanism is activated.

Returns
-------
histMec : a list of :class:`~openturns.WhittleFactoryState`
    Returns the collection of all the states that have been built for the estimation.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::setVerbose
"Accessor to the verbose mode.

Parameters
----------
verboseMode : bool
    Set the verbose mode while both the exploration of the possible models and the optimization steps.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::getVerbose
"Accessor to the verbose mode.

Returns
-------
verboseMode : bool
    Get the verbose mode while both the exploration of the possible models and the optimization steps.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::setStartingPoints
"Accessor to the starting points for the optimization step.

Parameters
----------
startPointsList : a list of :class:`~openturns.Point`
    Starting points for the optimization step, for each pair of orders that will be tested.
"
// ---------------------------------------------------------------------

%feature("docstring") OT::WhittleFactory::getStartingPoints
"Accessor to the starting points for the optimization step.

Returns
-------
startPointsList : a list of :class:`~openturns.Point`
    Starting points for the optimization step, for each pair of orders that will be tested.
"
