# (c) 1992-2024 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 


# This TCL script creates a qsys simulation system which includes all the OpenCL simulation board BFM's
# It can generate with and without the kernel system
#
# Default settings: Simulation board only, single 2GB memory bank
#  GLB_MEM_AV_ADDRESS_WIDTH : if defined, sets global memory address width. Default is 2GB
#  USE_SNOOP                : if 1, kernel system can use snoop port in memory bank divider. Depends on BSP
#  USE_CLK2X                : if 1, a clock2x signal is present in the board spec and we are safe to generate that here.
#  HOST_TO_DEV_READ_WIDTH   : set to 0 unless board supports host channel
#  DEV_TO_HOST_WRITE_WIDTH  : set to 0 unless board supports host channel
#  INCL_KERNEL_SYSTEM       : if defined, it means OpenCL invoked qsys on kernel_system.tcl to create a kernel_system.qsys
#                           : expected to be obsoleted in future release
#  GLB_MEM_AV_WAITREQUEST_ALLOWANCE : if defined, sets the wait request allowance value on memory side and use S10 Kernel Interface.
#                                     Depends on device. Default is 0.
# TODO's
# - Host-device memory transfers via Memory bank divider

package require -exact qsys 17.0

proc assert {cond {msg "assertion failed"}} {
  if {![uplevel 1 expr $cond]} {error $msg}
}

proc instantiate_dpi_bfm { bfm_inst_name bfm_shortname bfm_type } {
  global quartus_pro
  if {$quartus_pro == 1} {
    add_component ${bfm_inst_name} ${bfm_shortname}.ip ${bfm_type} ${bfm_shortname}
  } else {
    add_instance ${bfm_inst_name} ${bfm_type}
  }
}

proc get_report_component_name { component } {
  set instances [get_composed_instances ks]
  set the_inst ""
  foreach inst $instances {
    if { [string first "_sys" $inst] >= 0 } {
      set the_inst $inst
    }
  }
  if { $the_inst == "" } {
    return ${component}
  }
  set report_names [get_composed_instance_assignment ks $the_inst ipa.report.names]
  set mangled_names [get_composed_instance_assignment ks $the_inst ipa.mangled.names]
  set ndx [lsearch ${mangled_names} ${component}]
  if { $ndx >= 0 } {
    return [lindex ${report_names} $ndx]
  }
  return ${component}
}

proc get_cosim_component_name { component } {
  global quartus_pro
  if {$quartus_pro == 1} {
    load_component ${component}
    set cosim_component [get_component_assignment ipa.cosim.names]
  } else {
    set cosim_component [get_composed_instance_assignment ${component}_inst ${component}_internal_inst ipa.cosim.names]
  }
  return $cosim_component
}

proc dpi_bfm_set_param { instance_name param param_value args } {
  global quartus_pro
  if {$quartus_pro == 1} {
    load_component $instance_name
    set_component_parameter_value $param $param_value
    foreach {name value} $args {
      set_component_parameter_value $name $value
    }
    save_component
  } else {
    set_instance_parameter_value $instance_name $param $param_value
    foreach {name value} $args {
      set_instance_parameter_value $instance_name $name $value
    }
  }
}

proc instantiate_dpi_bfm_and_connect_to_stream_sink_call { ks interface } {
  # add streaming sink BFM
  set instance_name sso_${interface}_inst
  set instance_shortname sso_${interface}

  instantiate_dpi_bfm ${instance_name} ${instance_shortname} aoc_sim_stream_source_dpi_bfm

  add_connection clk_rst.clock      ${instance_name}.clock
  add_connection clk_rst.clock2x    ${instance_name}.clock2x
  add_connection clk_rst.reset      ${instance_name}.reset

  dpi_bfm_set_param $instance_name \
    COMPONENT_NAME "ks" \
    INTERFACE_NAME "\$do" \
    INTERFACE_MANGLED_NAME $interface \
    USE_DATA_PORT 0

  add_connection   ${instance_name}.source  $ks.$interface
}

proc instantiate_dpi_bfm_and_connect_to_implicit_stream_sink { component interface } {
  set instance_name ss_${interface}_inst
  instantiate_dpi_bfm ${instance_name} sso_${component}_${interface} aoc_sim_stream_source_dpi_bfm

  add_connection clk_rst.clock         $instance_name.clock
  add_connection clk_rst.clock2x       $instance_name.clock2x
  add_connection clk_rst.reset         $instance_name.reset

  set portlist [get_instance_interface_ports ${component} $interface]
  set stream_width [get_instance_interface_port_property ${component} ${interface} ${interface}_data WIDTH]

  set stream_dir [get_instance_interface_port_property ${component} ${interface} ${interface}_data DIRECTION]

  # TODO: should COMPONENT_NAME be just ks (current implementation) or the actual component name?
  #       Note that setting it to ks will lead to it not being logged in the hld report.html
  dpi_bfm_set_param $instance_name FORCE_STREAM_CONDUIT 1 STREAM_DATAWIDTH $stream_width INTERFACE_MANGLED_NAME $interface INTERFACE_NAME "\$data" COMPONENT_NAME $component

  add_connection ss_${interface}_inst.source_data ${component}.${interface}
}

proc instantiate_dpi_bfm_and_connect_to_stream_sink { ks interface component } {
  global external_pipe
  # add streaming sink BFM
  set symbolsPerBeat [get_instance_interface_parameter_value $ks $interface symbolsPerBeat]
  set stream_width [get_instance_interface_port_property $ks ${interface} ${interface}_data WIDTH]
  set firstSymbolInHighOrderBits [get_instance_interface_parameter_value $ks $interface firstSymbolInHighOrderBits]

  set instance_name sso_${interface}_inst
  set instance_shortname sso_${interface}

  instantiate_dpi_bfm ${instance_name} ${instance_shortname} aoc_sim_stream_source_dpi_bfm

  add_connection clk_rst.clock      ${instance_name}.clock
  add_connection clk_rst.clock2x    ${instance_name}.clock2x
  add_connection clk_rst.reset      ${instance_name}.reset

  if { [string compare $component ""] } {
    set component_report_name [get_report_component_name $component]
    dpi_bfm_set_param $instance_name COMPONENT_NAME $external_pipe
  }
  dpi_bfm_set_param $instance_name \
    INTERFACE_NAME $interface \
    INTERFACE_MANGLED_NAME $interface \
    STREAM_DATAWIDTH $stream_width \
    FIRST_SYMBOL_IN_HIGH_ORDER_BITS $firstSymbolInHighOrderBits

  set stream_ports [get_instance_interface_ports $ks ${interface}]
  if { [lsearch $stream_ports ${interface}_startofpacket] >= 0 } {
    dpi_bfm_set_param $instance_name USES_PACKETS 1
    if { [lsearch $stream_ports ${interface}_empty] >= 0 } {
      set stream_bitspersymbol [get_instance_interface_parameter_value $ks ${interface} dataBitsPerSymbol]
      set empty_width [expr int(ceil(log($stream_width / $stream_bitspersymbol) / log(2)))]
      dpi_bfm_set_param $instance_name EMPTY_WIDTH $empty_width
    }
  }

  if {$stream_width <= 4096} {
    set stream_bitspersymbol [get_instance_interface_parameter_value $ks ${interface} dataBitsPerSymbol]
    set firstsymbolinhighorderbits [get_instance_interface_parameter_value $ks ${interface} firstSymbolInHighOrderBits]
    dpi_bfm_set_param $instance_name \
      STREAM_BITSPERSYMBOL $stream_bitspersymbol \
      FIRST_SYMBOL_IN_HIGH_ORDER_BITS $firstsymbolinhighorderbits
  }

  add_connection   ${instance_name}.source  $ks.$interface
}

proc instantiate_dpi_bfm_and_connect_to_stream_source_return { ks interface } {
  # add streaming sink BFM
  set instance_name ssi_${interface}_inst
  set instance_shortname ssi_${interface}

  instantiate_dpi_bfm ${instance_name} ${instance_shortname} aoc_sim_stream_sink_dpi_bfm

  add_connection clk_rst.clock      ${instance_name}.clock
  add_connection clk_rst.clock2x    ${instance_name}.clock2x
  add_connection clk_rst.reset      ${instance_name}.reset

  # component name is set to ks for now as this call interface has no associated component
  dpi_bfm_set_param $instance_name \
    COMPONENT_NAME "ks" \
    INTERFACE_NAME "\$return" \
    INTERFACE_MANGLED_NAME $interface \
    USE_DATA_PORT 0

  add_connection   $ks.$interface  ${instance_name}.sink
}

proc instantiate_dpi_bfm_and_connect_to_stream_source { ks interface component } {
  global external_pipe
  set symbolsPerBeat [get_instance_interface_parameter_value $ks $interface symbolsPerBeat]
  set stream_width [get_instance_interface_port_property $ks ${interface} ${interface}_data WIDTH]
  set firstSymbolInHighOrderBits [get_instance_interface_parameter_value $ks $interface firstSymbolInHighOrderBits]
  # add streaming sink BFM
  set instance_name ssi_${interface}_inst
  set instance_shortname ssi_${interface}

  instantiate_dpi_bfm ${instance_name} ${instance_shortname} aoc_sim_stream_sink_dpi_bfm

  add_connection clk_rst.clock      ${instance_name}.clock
  add_connection clk_rst.clock2x    ${instance_name}.clock2x
  add_connection clk_rst.reset      ${instance_name}.reset

  if { [string compare $component ""] } {
    set component_report_name [get_report_component_name $component]
    dpi_bfm_set_param $instance_name COMPONENT_NAME $external_pipe
  }
   dpi_bfm_set_param $instance_name \
    INTERFACE_NAME $interface \
    INTERFACE_MANGLED_NAME $interface \
    STREAM_DATAWIDTH $stream_width \
    FIRST_SYMBOL_IN_HIGH_ORDER_BITS $firstSymbolInHighOrderBits

  set stream_ports [get_instance_interface_ports $ks ${interface}]
  if { [lsearch $stream_ports ${interface}_startofpacket] >= 0 } {
    dpi_bfm_set_param $instance_name USES_PACKETS 1
    if { [lsearch $stream_ports ${interface}_empty] >= 0 } {
      set stream_bitspersymbol [get_instance_interface_parameter_value $ks ${interface} dataBitsPerSymbol]
      set empty_width [expr int(ceil(log($stream_width / $stream_bitspersymbol) / log(2)))]
      dpi_bfm_set_param $instance_name EMPTY_WIDTH $empty_width
    }
  }

  if {$stream_width <= 4096} {
    set stream_bitspersymbol [get_instance_interface_parameter_value $ks ${interface} dataBitsPerSymbol]
    set firstsymbolinhighorderbits [get_instance_interface_parameter_value $ks ${interface} firstSymbolInHighOrderBits]
    dpi_bfm_set_param $instance_name \
      STREAM_BITSPERSYMBOL $stream_bitspersymbol \
      FIRST_SYMBOL_IN_HIGH_ORDER_BITS $firstsymbolinhighorderbits
  }

  add_connection   $ks.$interface  ${instance_name}.sink
}

proc instantiate_dpi_bfm_and_connect_to_avalon_host { component interface index } {
  set readwrite_mode "rw"

  set cosim_component "avs"
  set instance_name osm_${index}_inst
  instantiate_dpi_bfm $instance_name mm_agent_${component}_${interface} aoc_sim_mm_slave_dpi_bfm

  set AV_ADDRESS_W [get_instance_interface_port_property ${component} ${interface} ${interface}_address WIDTH]

  # For IP Authoring flow we cannot just assume that the readwrite_mode is
  # "rw", this needs to be read from the board spec. Currently, the compiler
  # will put this info in the port_name string, the name comes to this
  # function from input string "interface"
  if { [regexp {mem([0-9]+)_(.*)} $interface host_string host_id readwrite_mode] } {
    regexp {mem([0-9]+)_(.*)} $interface host_string host_id readwrite_mode
  }

  # symbol width in bits, symbol is one byte
  set AV_SYMBOL_W 8
  set USE_WAIT_REQUEST 0
  set USE_BURSTCOUNT 0
  set USE_READ_DATA_VALID 0
  set AV_MAX_PENDING_READS 0
  set USE_READ 1
  set USE_READ_DATA 1
  set USE_WRITE 1
  set USE_WRITE_DATA 1

  if {$readwrite_mode == "w"} {
    set AV_DATA_W [get_instance_interface_port_property ${component} ${interface} ${interface}_writedata WIDTH]
    set USE_READ 0
    set USE_READ_DATA 0
  } else {
    set AV_DATA_W [get_instance_interface_port_property ${component} ${interface} ${interface}_readdata WIDTH]
    if {$readwrite_mode == "r"} {
      set USE_WRITE 0
      set USE_WRITE_DATA 0
    }

    # only used if the interface is variable latency
    set AV_FIX_READ_LATENCY [get_instance_interface_parameter_value ${component} $interface readLatency]
    dpi_bfm_set_param $instance_name AV_FIX_READ_LATENCY $AV_FIX_READ_LATENCY
  }
  set AV_NUMSYMBOLS [ expr {$AV_DATA_W / $AV_SYMBOL_W}]
  set AV_NUMSYMBOLS_REM [ expr {$AV_DATA_W % $AV_SYMBOL_W}]
  assert {$AV_NUMSYMBOLS>0 && $AV_NUMSYMBOLS_REM==0} "AV_DATA_W must be a multiple of AV_SYMBOL_W=$AV_SYMBOL_W"

  set AV_BYTEENABLE_W [get_instance_interface_port_property ${component} ${interface} ${interface}_byteenable WIDTH]
  assert {$AV_NUMSYMBOLS==$AV_BYTEENABLE_W} "AV_NUMSYMBOLS=$AV_NUMSYMBOLS must be equal to the width of the byteenable signal ($AV_BYTEENABLE_W)"

  set ADDRESS_UNITS [get_instance_interface_parameter_value ${component} $interface addressUnits]

  set portArr [get_instance_interface_ports ${component} ${interface}]
  set burstcountPort ${interface}_burstcount
  set waitrequestPort ${interface}_waitrequest
  set readdatavalidPort ${interface}_readdatavalid
  foreach elem $portArr {
    if {$elem == $burstcountPort} {
      set AV_BURSTCOUNT_W [get_instance_interface_port_property ${component} ${interface} ${interface}_burstcount WIDTH]
      dpi_bfm_set_param $instance_name AV_BURSTCOUNT_W $AV_BURSTCOUNT_W
      set USE_BURSTCOUNT 1
    } elseif {$elem == $waitrequestPort} {
      set USE_WAIT_REQUEST 1
    } elseif {$elem == $readdatavalidPort && $USE_READ == 1} {
      # Need to check $USE_READ here when the target memory location has write-only property,
      # and the compiler decides to optimize the memory interface out on the device image.
      # When that happens, the memory interface on the kernel system will be marked as write-only,
      # but be exported from an altera_avalon_mm_bridge instance, which has a readdatavalid port
      # as part of the interface.
      set USE_READ_DATA_VALID 1
      set AV_MAX_PENDING_READS 256
    }
  }

  dpi_bfm_set_param $instance_name \
    INTERFACE_ID [expr $index] \
    COMPONENT_NAME $cosim_component \
    AV_ADDRESS_W $AV_ADDRESS_W \
    AV_NUMSYMBOLS $AV_NUMSYMBOLS \
    AV_SYMBOL_W $AV_SYMBOL_W \
    ADDRESS_UNITS $ADDRESS_UNITS \
    USE_WAIT_REQUEST $USE_WAIT_REQUEST \
    USE_BURSTCOUNT $USE_BURSTCOUNT \
    USE_READ_DATA_VALID $USE_READ_DATA_VALID \
    USE_READ $USE_READ \
    USE_READ_DATA $USE_READ_DATA \
    USE_WRITE $USE_WRITE \
    USE_WRITE_DATA $USE_WRITE_DATA \
    USE_READ_DATA_VALID $USE_READ_DATA_VALID \
    AV_MAX_PENDING_READS $AV_MAX_PENDING_READS

  add_connection clk_rst.clock         $instance_name.clock
  add_connection clk_rst.reset         $instance_name.reset
  add_connection ${component}.${interface} $instance_name.s0

}

proc instantiate_dpi_bfm_and_connect_to_avalon_agent { component interface index } {
  set instance_name avmm_agent_${index}_inst
  instantiate_dpi_bfm $instance_name mm_host_${component}_${interface} aoc_sim_mm_master_dpi_bfm

  set KI_AV_ADDRESS_W [get_instance_interface_port_property ${component} ${interface} ${interface}_address WIDTH]
  # symbol width in bits, symbol is one byte
  set KI_AV_SYMBOL_W 8


  # set everything to 0 by default
  set USE_READ 0
  set USE_READ_DATA 0
  set USE_READ_DATA_VALID 0
  set USE_WRITE 0
  set USE_WRITE_DATA 0
  set KI_USE_WAIT_REQUEST 0
  set MBD_USE_BURST_COUNT 0
  set IS_MAIN_CONTROLLER 0

  set portArr [get_instance_interface_ports ${component} ${interface}]
  set readPort ${interface}_read
  set writePort ${interface}_write
  set waitrequestPort ${interface}_waitrequest
  set readdatavalidPort ${interface}_readdatavalid
  foreach elem $portArr {
    if {$elem == $readPort} {
      set USE_READ 1
      set USE_READ_DATA 1
      set AV_DATA_W [get_instance_interface_port_property ${component} ${interface} ${interface}_readdata WIDTH]
    } elseif {$elem == $writePort} {
      set USE_WRITE 1
      set USE_WRITE_DATA 1
      set AV_DATA_W [get_instance_interface_port_property ${component} ${interface} ${interface}_writedata WIDTH]
    } elseif {$elem == $waitrequestPort} {
      set KI_USE_WAIT_REQUEST 1
    } elseif {$elem == $readdatavalidPort} {
      set USE_READ_DATA_VALID 1
    }
  }

  assert {$AV_DATA_W>0} "AV_DATA_W must be non-zero"
  set KI_AV_NUMSYMBOLS [ expr {$AV_DATA_W / $KI_AV_SYMBOL_W} ]
  set KI_AV_NUMSYMBOLS_REM [ expr {$AV_DATA_W % $KI_AV_SYMBOL_W} ]
  assert {$KI_AV_NUMSYMBOLS>0 && $KI_AV_NUMSYMBOLS_REM==0} "AV_DATA_W must be a multiple of KI_AV_SYMBOL_W=$KI_AV_SYMBOL_W"

  set AV_BYTEENABLE_W [get_instance_interface_port_property ${component} ${interface} ${interface}_byteenable WIDTH]
  assert {$KI_AV_NUMSYMBOLS==$AV_BYTEENABLE_W} "KI_AV_NUMSYMBOLS=$KI_AV_NUMSYMBOLS must be equal to the width of the byteenable signal ($AV_BYTEENABLE_W)"

  set ADDRESS_UNITS [get_instance_interface_parameter_value ${component} $interface addressUnits]
  set AV_FIX_READ_LATENCY [get_instance_interface_parameter_value ${component} ${interface} readLatency]

  # Tighten timing from default
  set COMMAND_INITIAL_WAIT_CYCLES 1000
  set COMMAND_WAIT_CYCLES          200
  set COMMAND_READ_WRITE_CYCLES      4

  # set NUM_COMPONENTS_WITH_CSR to 1 for input stall signal
  set NUM_COMPONENTS_WITH_CSR 1

  dpi_bfm_set_param $instance_name \
    COMPONENT_NAME $interface \
    IS_MAIN_CONTROLLER $IS_MAIN_CONTROLLER \
    KI_AV_ADDRESS_W $KI_AV_ADDRESS_W \
    KI_AV_NUMSYMBOLS $KI_AV_NUMSYMBOLS \
    KI_AV_SYMBOL_W $KI_AV_SYMBOL_W \
    ADDRESS_UNITS $ADDRESS_UNITS \
    KI_USE_WAIT_REQUEST $KI_USE_WAIT_REQUEST \
    USE_READ_DATA_VALID $USE_READ_DATA_VALID \
    USE_READ $USE_READ \
    USE_READ_DATA $USE_READ_DATA \
    USE_WRITE $USE_WRITE \
    USE_WRITE_DATA $USE_WRITE_DATA \
    USE_READ_DATA_VALID $USE_READ_DATA_VALID \
    COMMAND_INITIAL_WAIT_CYCLES $COMMAND_INITIAL_WAIT_CYCLES \
    COMMAND_WAIT_CYCLES $COMMAND_WAIT_CYCLES \
    COMMAND_READ_WRITE_CYCLES $COMMAND_READ_WRITE_CYCLES \
    NUM_COMPONENTS_WITH_CSR $NUM_COMPONENTS_WITH_CSR \
    AV_FIX_READ_LATENCY $AV_FIX_READ_LATENCY \
    MBD_USE_BURST_COUNT $MBD_USE_BURST_COUNT

  add_connection clk_rst.clock         $instance_name.clock
  add_connection clk_rst.reset         $instance_name.reset
  add_connection $instance_name.m_ki ${component}.${interface}

}

set the_component_list ""
set the_agent_interfaces ""
set the_host_iface_list ""
set the_agent_iface_list ""
set the_stream_intf_mapping ""
set the_host_pipe_mapping ""
set in_sycl_mode 0
if [info exists component_list] {
  set the_component_list $component_list
}
if [info exists num_agent_interface] {
  set the_agent_interfaces $num_agent_interface
}
if [info exists host_iface_list] {
  set the_host_iface_list $host_iface_list
}
if [info exists agent_iface_list] {
  set the_agent_iface_list $agent_iface_list
}
if [info exists stream_intf_mapping] {
  set the_stream_intf_mapping $stream_intf_mapping
}
if [info exists host_pipe_mapping] {
  set the_host_pipe_mapping $host_pipe_mapping
}
if [info exists sycl_mode] {
  set in_sycl_mode $sycl_mode
}

# Inputs from caller
set components [split $the_component_list ","]
set agent_interfaces [split $the_agent_interfaces ","]
set host_ifaces [split $the_host_iface_list ","]
set agent_ifaces [split $the_agent_iface_list ","]
set stream_intfs [split $the_stream_intf_mapping ","]
set host_pipes [split $the_host_pipe_mapping ","]

create_system mpsim

# Basic clock and reset for the simulation
# generate the whole system - put clock reset as part of testbench
add_instance clk_rst hls_sim_clock_reset
set_instance_parameter_value clk_rst RESET_CYCLE_HOLD    120

# Main OpenCL controller for the Kernel Interface and Memory Divider
add_instance oirq aoc_sim_main_dpi_controller


# Kernel Interface that controls the OpenCL kernels
if [info exists GLB_MEM_AV_WAITREQUEST_ALLOWANCE] {
  add_instance ki kernel_interface_s10
} else {
  add_instance ki kernel_interface
}

# OpenCL Simulator Interface main host BFM to control:
#   - Kernel Interface (will poll host process)
#   - Memory Bank Divider (will poll host-device memory transfers)
# Default values for USE_WAIT_REQUEST's, USE_READ_DATA_VALID, USE_BURSTCOUNT
# matches with associated interface. We only set values that is different
add_instance avmm_agent_cra_inst aoc_sim_mm_master_dpi_bfm
set_instance_parameter_value avmm_agent_cra_inst COMPONENT_NAME     "kernel_cra"
set_instance_parameter_value avmm_agent_cra_inst IS_MAIN_CONTROLLER  1
set_instance_parameter_value avmm_agent_cra_inst KI_AV_ADDRESS_W    14
set_instance_parameter_value avmm_agent_cra_inst KI_AV_SYMBOL_W      8
set_instance_parameter_value avmm_agent_cra_inst KI_AV_NUMSYMBOLS    4
set_instance_parameter_value avmm_agent_cra_inst MBD_AV_ADDRESS_W   31
set_instance_parameter_value avmm_agent_cra_inst MBD_AV_SYMBOL_W     8
set_instance_parameter_value avmm_agent_cra_inst MBD_AV_NUMSYMBOLS   4

# Tighten timing from default
set_instance_parameter_value avmm_agent_cra_inst COMMAND_INITIAL_WAIT_CYCLES 1000
set_instance_parameter_value avmm_agent_cra_inst COMMAND_WAIT_CYCLES          200
set_instance_parameter_value avmm_agent_cra_inst COMMAND_READ_WRITE_CYCLES      4

if { [llength $components] > 0 } {
  # For now, just report that all components have a CSR; this only affects the
  # width of the bus indicating when we've got all the CRA writes
  set_instance_parameter_value avmm_agent_cra_inst NUM_COMPONENTS_WITH_CSR [llength $components]
}

# Setup all connections for clock and resets to all the components and connections to kernel system
# The generated OpenCL kernels or the empty kernel system
add_instance ks kernel_system

# Set parameter for snoop port and connect it
if [info exists USE_SNOOP] {
  if ![info exists INCL_KERNEL_SYSTEM] {
    # Empty kernel system
    set_instance_parameter_value ks USE_SNOOP  $USE_SNOOP
    add_connection clk_rst.clock ks.cc_snoop_clk_clk
  } else {
    # 18.0 and 18.1 flow - use kernel_system.qsys
    add_connection clk_rst.clock ks.cc_snoop_clk
  }
}

# Connect the AOC main controller
add_connection clk_rst.clock oirq.clock
add_connection clk_rst.clock2x oirq.clock2x
add_connection clk_rst.reset oirq.reset
add_connection oirq.reset_ctrl clk_rst.reset_ctrl

# Connect all the OCL kernel system
if ![info exists INCL_KERNEL_SYSTEM] {
  add_connection clk_rst.clock ks.clock_reset_clk
  if [info exists USE_CLK2X] {
    add_connection clk_rst.clock2x ks.clock_reset2x_clk
  }
  add_connection clk_rst.reset ks.clock_reset_reset_reset_n
  add_connection ki.kernel_cra ks.kernel_cra
  add_connection ki.kernel_irq_from_kernel ks.kernel_irq_irq
  # Workaround for host channel support as empty kernel system is not dynamically generated
  set_instance_parameter_value ks HOST_TO_DEV_READ_WIDTH $HOST_TO_DEV_READ_WIDTH
  set_instance_parameter_value ks DEV_TO_HOST_WRITE_WIDTH $DEV_TO_HOST_WRITE_WIDTH
} else {
  add_connection clk_rst.clock ks.clock_reset
  if [info exists USE_CLK2X] {
    add_connection clk_rst.clock2x ks.clock_reset2x
  }
  add_connection clk_rst.reset ks.clock_reset_reset
  add_connection ki.kernel_cra ks.kernel_cra
  add_connection ki.kernel_irq_from_kernel ks.kernel_irq
}

# Connect the kernel interface (KI)
add_connection clk_rst.clock ki.clk
add_connection clk_rst.clock ki.kernel_clk
add_connection clk_rst.reset ki.reset
add_connection clk_rst.reset ki.sw_reset_in


# Connect the Simulator Interface BFM host
add_connection clk_rst.clock avmm_agent_cra_inst.clock
add_connection clk_rst.reset avmm_agent_cra_inst.reset

auto_assign_irqs avmm_agent_cra_inst

add_connection avmm_agent_cra_inst.m_ki ki.ctrl
add_connection oirq.kernel_interrupt ki.kernel_irq_to_host


if { [llength $components] > 0 } {
  # split_component_csr_ready
  instantiate_dpi_bfm split_component_csr_ready_inst sp_csr_ready avalon_split_multibit_conduit
  dpi_bfm_set_param split_component_csr_ready_inst multibit_width [llength $components]
  add_connection avmm_agent_cra_inst.done_writes_to_cra split_component_csr_ready_inst.in_conduit

  # concat_component_stall
  instantiate_dpi_bfm concat_component_stall_inst cct_comp_stall avalon_concatenate_singlebit_conduits
  dpi_bfm_set_param concat_component_stall_inst multibit_width [llength $components]
  add_connection avmm_agent_cra_inst.component_control_not_ready_in concat_component_stall_inst.out_conduit

  # master_done_fanout
  instantiate_dpi_bfm master_done_fanout_inst done_cfan avalon_conduit_fanout
  dpi_bfm_set_param master_done_fanout_inst numFanOut [llength $components]
  add_connection avmm_agent_cra_inst.sim_done master_done_fanout_inst.in_conduit
}


proc add_dpi_controller_for_component { component clean_component_name num_iface comp_has_call_iface comp_has_return_iface comp_has_return_stall num_implicit_iface num_agent_iface comp_has_agent_cra num_output_stream_iface} {

  set component_report_name [get_report_component_name $component]

  # Add component_dpi_controller instance  (one per component so component name must be in the inst name)
  set instance_name component_dpi_controller_${clean_component_name}_inst
  instantiate_dpi_bfm ${instance_name} dpic_${clean_component_name} aoc_sim_component_dpi_controller

  # Connect clock/reset to the component_dpi_controller instance
  add_connection clk_rst.clock       ${instance_name}.clock
  add_connection clk_rst.clock2x     ${instance_name}.clock2x
  add_connection clk_rst.reset       ${instance_name}.reset

  if { $comp_has_return_stall == 1 } {
    dpi_bfm_set_param $instance_name COMPONENT_HAS_RETURN_STALL 1
  }

  # Connect implicit input/output streams between the component_dpi_controller and the component
  if { $comp_has_call_iface == 1 } {
    dpi_bfm_set_param $instance_name COMPONENT_HAS_CALL_INTERFACE 1
    add_connection ${instance_name}.component_call ks.${component}_streaming_start
  }

  if { $comp_has_return_iface == 1 } {
    add_connection ks.${component}_streaming_done ${instance_name}.component_return
  }

  # must set COMPONENT_NAME parameter of the component_dpi_controller
  # set it to the original component name ($component) not ${component}_inst
  dpi_bfm_set_param $instance_name \
    COMPONENT_NAME $component_report_name \
    COMPONENT_MANGLED_NAME $component

  # add Avalon Conduit Fanout module for enable signal
  instantiate_dpi_bfm ${clean_component_name}_component_dpi_controller_enable_conduit_fanout_inst ${clean_component_name}_en_cfan avalon_conduit_fanout
  dpi_bfm_set_param ${clean_component_name}_component_dpi_controller_enable_conduit_fanout_inst numFanOut $num_iface
  add_connection $instance_name.dpi_control_enable  ${clean_component_name}_component_dpi_controller_enable_conduit_fanout_inst.in_conduit

  # add Avalon Conduit Fanout module for implicit stream ready signals
  if { $num_implicit_iface > 0 } {
    instantiate_dpi_bfm ${clean_component_name}_component_dpi_controller_implicit_ready_conduit_fanout_inst ${clean_component_name}_ir_cfan avalon_conduit_fanout
    dpi_bfm_set_param ${clean_component_name}_component_dpi_controller_implicit_ready_conduit_fanout_inst numFanOut $num_implicit_iface

    add_connection $instance_name.read_implicit_streams  ${clean_component_name}_component_dpi_controller_implicit_ready_conduit_fanout_inst.in_conduit
  }

  # Set parameter to ensure all the control signals are connected properly.
  if { $num_agent_iface > 0 } {
    dpi_bfm_set_param $instance_name COMPONENT_NUM_AGENTS 1
  }
}

set host_iface_ndx 0
proc get_host_index { host_iface_name } {
  set ndx 0
  global host_ifaces
  global host_iface_ndx
  foreach host $host_ifaces {
    if { [string equal $host $host_iface_name] == 1 } {
      return $ndx
    }
    incr ndx
  }
  set this_host_iface_ndx $host_iface_ndx
  puts "WARNING: Not able to determine expected host interface index for interface '$host_iface_name'; using $this_host_iface_ndx!\n"
  incr host_iface_ndx
  return $this_host_iface_ndx
}

set host_writeack_iface_ndx 0
proc get_host_writeack_index { host_writeack_iface_name } {
  set ndx 0
  global host_ifaces
  global host_writeack_iface_ndx
  foreach host $host_ifaces {
    if { [string equal ${host}_writeack $host_writeack_iface_name] == 1 } {
      return $ndx
    }
    incr ndx
  }
  set this_host_writeack_iface_ndx $host_writeack_iface_ndx
  puts "WARNING: Not able to determine expected host interface index for interface '$host_writeack_iface_name'; using $this_host_writeack_iface_ndx!\n"
  incr host_writeack_iface_ndx
  return $this_host_writeack_iface_ndx
}

set agent_iface_ndx 0
proc get_agent_index { agent_iface_name } {
  set ndx 0
  global agent_ifaces
  global agent_iface_ndx
  foreach agent $agent_ifaces {
    if { [string first $agent $agent_iface_name] >= 0 } {
      return $ndx
    }
    incr ndx
  }
  set this_agent_iface_ndx $agent_iface_ndx
  puts "WARNING: Not able to determine expected agent interface index for interface '$agent_iface_name'; using $this_agent_iface_ndx!\n"
  incr agent_iface_ndx
  return $this_agent_iface_ndx
}

proc get_stream_intf_component { interface_name } {
  global components
  global stream_intfs
  # First check if this is a streaming control interface
  foreach component $components {
    if {[string compare ${component}_streaming_start $interface_name] == 0} {
      return $component
    }
    if {[string compare ${component}_streaming_done $interface_name] == 0} {
      return $component
    }
  }
  # Then check if this is a streaming argument interface
  foreach {interface component} $stream_intfs {
    if { [string equal $interface_name $interface] } {
      return $component
    }
  }
  return ""
}

proc get_host_pipe_component { host_pipe_name } {
  global host_pipes
  foreach {pipe component} $host_pipes {
    if { [string equal $host_pipe_name $pipe] } {
      return $component
    }
  }
  return ""
}

# Generate a unique name by base_name followed by unique_suffix_int
proc generate_unique_name { components base_name } {
  set clean_name_unsafe 1
  # Start with an ascii 'A', increment until we find something safe
  set unique_suffix_int 65
  while {$clean_name_unsafe == 1} {
    set clean_name_unsafe 0
    set unique_suffix [format %c $unique_suffix_int]
    set clean_component_name $base_name$unique_suffix
    foreach c $components {
      if {[string compare $c $clean_component_name] == 0} {
        set clean_name_unsafe 1
        break
      }
    }
    incr unique_suffix_int
    if {$unique_suffix_int > 90} {
      # After 'Z', skip to 'a'
      set $unique_suffix_int 97
    }
    if { $unique_suffix_int > 122 } {
      error "Error: Unable to find suitable unique name."
    }
  }
  return $clean_component_name
}

set arr_index 0
set visited_interfaces []
set host_iface_writeack_map []
set external_pipe "external_pipe"

# Connect interfaces associated with each component/kernel
foreach component $components {
  set num_iface 0
  set num_implicit_iface 0
  set num_output_stream_iface 0
  set comp_has_call_iface 0
  set comp_has_return_iface 0
  set comp_has_return_stall 0
  set comp_has_agent_cra 0

  set num_agent_iface [lindex $agent_interfaces $arr_index]

  # Generate a unique, clean component name if necessary
  set clean_component_name $component
  # For Windows the maximum length we allow for component name is 50 since:
  # The maximum length for any path on Windows is 260;
  # Component name exists for both the directory and file, while the path above the ip directory also takes some length;
  # The useful information about the component is almost always presented in the first 50 characters.
  if { [string equal [lindex $tcl_platform(os) 0] "Windows"] } {
    if { [string length $clean_component_name] > 50} {
      set short_component_name [string range $clean_component_name 0 48]
      set clean_component_name [generate_unique_name $components $short_component_name]
    }
  }
  # Component name ending with "_" is unsafe.
  if { [string equal [string range $clean_component_name end end] "_"] } {
    set clean_component_name [generate_unique_name $components $clean_component_name]
  }

  foreach interface [get_instance_interfaces ks] {
    # skip if not matching this $component
    # for host pipes, their names don't include component name, therefore we use compiler
    # passed in host-pipe-to-component mapping to determine the component they belong to
    set is_component_stream_intf [expr [string compare $component [get_stream_intf_component $interface]] == 0]
    set is_component_host_pipe [expr [string compare $component [get_host_pipe_component $interface]] == 0]
    if {$is_component_stream_intf == 0 && $is_component_host_pipe == 0} {
      continue
    }
    lappend visited_interfaces $interface
    set class_name [get_instance_interface_property ks $interface CLASS_NAME]

    set portlist [get_instance_interface_ports ks $interface]
    set numports [llength $portlist]

    if [string compare $interface cc_snoop]!=0 {
      # dont connect BFM to snoop port
      if { $class_name=="avalon_host" || $class_name=="avalon_master"} {
        set my_host_index [get_host_index $interface]
        instantiate_dpi_bfm_and_connect_to_avalon_host ks $interface $my_host_index
        set  num_iface [expr $num_iface + 1]
      } elseif { ($class_name=="avalon_agent" || $class_name=="avalon_slave") && [regexp {avs_(.*)} $interface] } {
        set my_agent_index [get_agent_index $interface]
        instantiate_dpi_bfm_and_connect_to_avalon_agent ks $interface $my_agent_index
        set  num_iface [expr $num_iface + 1]
      } elseif { [string compare $class_name avalon_streaming_sink]==0 } {
        instantiate_dpi_bfm_and_connect_to_stream_sink ks $interface $component
        set  num_iface [expr $num_iface + 1]
      } elseif { [string compare $class_name avalon_streaming_source]==0 } {
        instantiate_dpi_bfm_and_connect_to_stream_source ks $interface $component
        set  num_iface [expr $num_iface + 1]
      } elseif { $class_name == "conduit_end" && $numports == 2 && [string first "_streaming_start" $interface] > 0 } {
        set comp_has_call_iface 1
      } elseif { $class_name == "conduit_end" && $numports == 2 && [string first "_streaming_done" $interface] > 0 } {
        set comp_has_return_iface 1
        set comp_has_return_stall 1
      } elseif { $class_name == "conduit_end" && $numports == 1 } {
        set port_role [get_instance_interface_port_property ks $interface $portlist ROLE]
        if {[string compare $port_role writeack] == 0} {
          # Found writeack signal interface, match with one of the avalon host interfaces
          set my_host_writeack_index [get_host_writeack_index $interface]
          lappend host_iface_writeack_map osm_${my_host_writeack_index}_inst $interface
          set  num_iface [expr $num_iface + 1]
        } elseif {[string compare $port_role valid] == 0 && [string first "_streaming_done" $interface] != -1} {
          set comp_has_return_iface 1
        } else {
          set  num_iface [expr $num_iface + 1]
          set  num_implicit_iface [expr $num_implicit_iface + 1]
          instantiate_dpi_bfm_and_connect_to_implicit_stream_sink ks $interface
        }
      } else {
        puts "found unknown interface type: $class_name, name: $interface, numports: $numports";
      }
    }
  }

  if ($in_sycl_mode) {
    add_dpi_controller_for_component $component $clean_component_name $num_iface $comp_has_call_iface $comp_has_return_iface $comp_has_return_stall $num_implicit_iface $num_agent_iface $comp_has_agent_cra $num_output_stream_iface

    add_connection split_component_csr_ready_inst.out_conduit_${arr_index} component_dpi_controller_${clean_component_name}_inst.dpi_control_agents_ready
    add_connection concat_component_stall_inst.in_conduit_${arr_index} component_dpi_controller_${clean_component_name}_inst.master_stall
    add_connection master_done_fanout_inst.out_conduit_${arr_index} component_dpi_controller_${clean_component_name}_inst.master_done

    set num_iface 0
    set num_implicit_iface 0
    set num_output_stream_iface 0
    set num_non_cra_agent_iface 0
    foreach interface [get_instance_interfaces ks] {
      set is_component_stream_intf [expr [string compare $component [get_stream_intf_component $interface]] == 0]
      set is_component_host_pipe [expr [string compare $component [get_host_pipe_component $interface]] == 0]
      # skip if not matching this $component, do the same check for host pipes as above
      if {$is_component_stream_intf == 0 && $is_component_host_pipe == 0} {
        continue
      }
      set class_name [get_instance_interface_property ks $interface CLASS_NAME]

      set portlist [get_instance_interface_ports ks $interface]
      set numports [llength $portlist]

      if [string compare $interface cc_snoop]!=0 {
        if { $class_name=="avalon_host" || $class_name=="avalon_master"} {
          set  num_iface [expr $num_iface + 1]
        } elseif { ($class_name=="avalon_agent" || $class_name=="avalon_slave") && [regexp {avs_(.*)} $interface] } {
          set  num_iface [expr $num_iface + 1]
        } elseif { [string compare $class_name avalon_streaming_sink]==0 } {
          add_connection ${clean_component_name}_component_dpi_controller_enable_conduit_fanout_inst.out_conduit_$num_iface sso_${interface}_inst.dpi_control_enable
          set  num_iface [expr $num_iface + 1]
        } elseif { [string compare $class_name avalon_streaming_source]==0 } {
          add_connection ${clean_component_name}_component_dpi_controller_enable_conduit_fanout_inst.out_conduit_$num_iface ssi_${interface}_inst.dpi_control_enable
          set  num_iface [expr $num_iface + 1]
        } elseif { $class_name == "conduit_end" && $numports == 2 && [string first "_streaming_start" $interface] > 0 } {
          set comp_has_call_iface 1
        } elseif { $class_name == "conduit_end" && $numports == 2 && [string first "_streaming_done" $interface] > 0 } {
          set comp_has_return_iface 1
          set comp_has_return_stall 1
        } elseif { $class_name == "conduit_end" && $numports == 1} {
          set port_role [get_instance_interface_port_property ks $interface $portlist ROLE]
          if {[string compare $port_role writeack] == 0} {
            set  num_iface [expr $num_iface + 1]
          } elseif {[string compare $port_role valid] == 0 && [string first "_streaming_done" $interface] != -1} {
            set comp_has_return_iface 1
          } else {
            add_connection ${clean_component_name}_component_dpi_controller_enable_conduit_fanout_inst.out_conduit_$num_iface ss_${interface}_inst.dpi_control_enable
            add_connection ${clean_component_name}_component_dpi_controller_implicit_ready_conduit_fanout_inst.out_conduit_$num_implicit_iface ss_${interface}_inst.source_ready
            dpi_bfm_set_param ss_${interface}_inst USE_FORCE_POP 0
            set  num_iface [expr $num_iface + 1]
            set  num_implicit_iface [expr $num_implicit_iface + 1]
          }
        } else {
          puts "found unknown interface type: $class_name, name: $interface, numports: $numports";
        }
      }
    }
  }
  incr arr_index
}

# Connect interfaces not associated with a specific component/kernel
foreach interface [get_instance_interfaces ks] {
  # skip interfaces we've already connected
  if {[lsearch $visited_interfaces $interface] != -1} {
    continue
  }
  lappend visited_interfaces $interface
  set class_name [get_instance_interface_property ks $interface CLASS_NAME]

  set portlist [get_instance_interface_ports ks $interface]
  set numports [llength $portlist]

  if [string compare $interface cc_snoop]!=0 {
    #dont connect BFM to snoop port
    if { $class_name=="avalon_host" || $class_name=="avalon_master"} {
      set my_host_index [get_host_index $interface]
      instantiate_dpi_bfm_and_connect_to_avalon_host ks $interface $my_host_index
    } elseif { ($class_name=="avalon_agent" || $class_name=="avalon_slave") && [regexp {avs_(.*)} $interface] } {
      set my_agent_index [get_agent_index $interface]
      instantiate_dpi_bfm_and_connect_to_avalon_agent ks $interface $my_agent_index
    } elseif { [string compare $class_name avalon_streaming_sink]==0 } {
      instantiate_dpi_bfm_and_connect_to_stream_sink ks $interface ""
    } elseif { [string compare $class_name avalon_streaming_source]==0 } {
      instantiate_dpi_bfm_and_connect_to_stream_source ks $interface ""
    } elseif { $class_name == "conduit_end" && $numports == 2 && [string first "_streaming_start" $interface] > 0 } {
      instantiate_dpi_bfm_and_connect_to_stream_sink_call ks $interface
    } elseif { $class_name == "conduit_end" && $numports == 2 && [string first "_streaming_done" $interface] > 0 } {
      instantiate_dpi_bfm_and_connect_to_stream_source_return ks $interface
    } elseif { $class_name == "conduit_end" && $numports == 1} {
      set port_role [get_instance_interface_port_property ks $interface $portlist ROLE]
      if {[string compare $port_role writeack] == 0} {
        # Found writeack signal interface, match with one of the avalon host interfaces
        set my_host_writeack_index [get_host_writeack_index $interface]
        lappend host_iface_writeack_map osm_${my_host_writeack_index}_inst $interface
      } elseif {[string compare $port_role valid] == 0 && [string first "_streaming_done" $interface] != -1} {
        # TODO: skip for now
        #instantiate_dpi_bfm_and_connect_to_implicit_stream_source ks $interface
      } elseif {[string first "_streaming_stall_out" $interface] != -1} {
        # TODO: skip for now
        #instantiate_dpi_bfm_and_connect_to_implicit_stream_source ks $interface
      } else {
        instantiate_dpi_bfm_and_connect_to_implicit_stream_sink ks $interface
      }
    } else {
      #puts "found unknown interface type: $class_name, name: $interface, numports: $numports";
    }
  }
}

if { [llength $host_iface_writeack_map] > 0 } {
  foreach { slave_bfm_inst writeack_intf } $host_iface_writeack_map {
    dpi_bfm_set_param $slave_bfm_inst USE_WRITE_ACK 1
    add_connection $slave_bfm_inst.s0_writeack ks.$writeack_intf
  }
}

# Fanout stall signal from components from main master bfm to other master bfms for dedicated interfaces
if { [llength $agent_ifaces] > 0 } {
  instantiate_dpi_bfm master_bfm_stall_conduit_fanout_inst master_bfm_stall_cfan avalon_conduit_fanout
  dpi_bfm_set_param master_bfm_stall_conduit_fanout_inst numFanOut [llength $agent_ifaces]
  add_connection avmm_agent_cra_inst.component_control_not_ready_out master_bfm_stall_conduit_fanout_inst.in_conduit

  set agent_iface_ndx 0
  foreach agent_iface $agent_ifaces {
    add_connection master_bfm_stall_conduit_fanout_inst.out_conduit_$agent_iface_ndx avmm_agent_${agent_iface_ndx}_inst.component_control_not_ready_in
    incr agent_iface_ndx
  }
}

# Generate memory bank divider if we aren't using the default simulation flow,
# and may need to use front door loading for memory model
if ![info exists INCL_KERNEL_SYSTEM] {
  # Memory Bank Divider that handles multiple banks of memory.
  add_instance mbd acl_memory_bank_divider
  set_instance_parameter_value mbd NUM_BANKS 1
  set_instance_parameter_value mbd DATA_WIDTH 512
  # Connect the memory bank divider
  add_connection ki.kernel_reset mbd.kernel_reset
  # add_connection mbd.acl_bsp_memorg_host ki.acl_bsp_memorg_host0x018
  if [info exists USE_SNOOP] {
    add_connection mbd.acl_bsp_snoop ks.cc_snoop
  }
  # TODO: this is only for front door loading
  #add_connection mbd.bank1 osm0.s0
  # TODO: add more banks to support interleaving
  set_interface_property mbd_bank1 EXPORT_OF mbd.bank1
  add_connection clk_rst.clock mbd.clk
  add_connection clk_rst.clock mbd.kernel_clk
  add_connection clk_rst.reset mbd.reset
  set_instance_parameter_value mbd NUM_BANKS 1
  add_connection avmm_agent_cra_inst.m_mbd mbd.s
}

# And save the whole thing
if {$quartus_pro == 1} {
  sync_sysinfo_parameters
}
save_system mpsim
