%feature("docstring") OT::GaussianProcessRandomVector
"GaussianProcessRandom vector, a conditioned Gaussian process.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.

Parameters
----------
gprResult : :class:`~openturns.experimental.GaussianProcessRegressionResult`
    Structure that contains elements of computation of a Gaussian Process Regression algorithm
points : 1-d or 2-d sequence of float
    Sequence of values defining a :class:`~openturns.Point` or a :class:`~openturns.Sample`.

Notes
-----
GaussianProcessRandomVector helps to create Gaussian random vector, :math:`Y: \\Rset^n \\mapsto \\Rset^d`, with stationary covariance function  :math:`\\cC^{stat}: \\Rset^n \\mapsto \\cM_{d \\times d}(\\Rset)`, conditionally to some observations.

Let :math:`Y(x=x_1)=y_1,\\cdots,Y(x=x_n)=y_n` be the observations of the Gaussian process. We assume the same Gaussian prior as in the :class:`~openturns.experimental.GaussianProcessRegression`:

.. math::

    Y(\\vect{x}) = \\Tr{\\vect{f}(\\vect{x})} \\vect{\\beta} + Z(\\vect{x})

with :math:`\\Tr{\\vect{f}(\\vect{x})} \\vect{\\beta}` a general linear model, :math:`Z(\\vect{x})` a zero-mean Gaussian process with a stationary autocorrelation function :math:`\\cC^{stat}`:

.. math::

    \\mathbb{E}[Z(\\vect{x}), Z(\\vect{\\tilde{x}})] = \\sigma^2 \\cC^{stat}_{\\theta}(\\vect{x} - \\vect{\\tilde{x}})

The objective is to generate realizations of the random vector :math:`Y`, on new points :math:`\\vect{\\tilde{x}}`, conditionally to these observations. For that purpose, :class:`~openturns.experimental.GaussianProcessRegression` build such a prior and stores results in a :class:`~openturns.experimental.GaussianProcessRegressionResult` structure on a first step. This structure is given as input argument.

Then, in a second step, both the prior and the covariance on input points :math:`\\vect{\\tilde{x}}`, conditionally to the previous observations, are evaluated (respectively :math:`Y(\\vect{\\tilde{x}})` and :math:`\\cC^{stat}_{\\theta}(\\vect{\\tilde{x}})`).

Finally realizations are randomly generated by the Gaussian distribution :math:`\\cN ( Y(\\vect{\\tilde{x}}), \\cC^{stat}_{\\theta}(\\vect{\\tilde{x}}) )`

This class inherits from :class:`~openturns.UsualRandomVector`. Thus it stores the previous distribution and returns elements thanks to that distribution (realization, mean, covariance, sample...)

Examples
--------
Create the model :math:`\\cM: \\Rset \\mapsto \\Rset` and the samples:

>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> f = ot.SymbolicFunction(['x'],  ['x * sin(x)'])
>>> sampleX = [[1.0], [2.0], [3.0], [4.0], [5.0], [6.0], [7.0], [8.0]]
>>> sampleY = f(sampleX)

Create the algorithm:

>>> basis = ot.Basis([ot.SymbolicFunction(['x'], ['x']), ot.SymbolicFunction(['x'], ['x^2'])])
>>> covarianceModel = ot.SquaredExponential([1.0])
>>> covarianceModel.setActiveParameter([])
>>> fit_algo = otexp.GaussianProcessFitter(sampleX, sampleY, covarianceModel, basis)
>>> fit_algo.run()
>>> gpr_algo = otexp.GaussianProcessRegression(fit_algo.getResult())
>>> gpr_algo.run()

Get the results:

>>> gpr_result = gpr_algo.getResult()
>>> rvector = otexp.GaussianProcessRandomVector(gpr_result, [[0.0]])

Get a sample of the random vector:

>>> sample = rvector.getSample(5)"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRandomVector::getRealization
"Compute a realization of the conditional Gaussian process (conditional on the learning set).

The realization predicts the value on the given input *points*.

Returns
-------
realization : :class:`~openturns.Point`
    Sequence of values of the Gaussian process.

See also
--------
getSample"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRandomVector::getSample
"Compute a sample of realizations of the conditional Gaussian process (conditional on the learning set).

The realization predicts the value on the given input *points*.

Returns
-------
realizations : :class:`~openturns.Sample`
    2-d float sequence of values of the Gaussian process.

See also
--------
getRealization"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRandomVector::getGaussianProcessRegressionResult
"Return the Gaussian Process Regression result structure.

Returns
-------
gprResult : :class:`~openturns.experimental.GaussianProcessRegressionResult`
    The structure containing the elements of a Gaussian Process Regression."
