%feature("docstring") OT::MethodOfMomentsFactory
"Estimation by method of moments.

Parameters
----------
distribution : :class:`~openturns.Distribution`
    The distribution defining the parametric model to be adjusted to data.
momentsOrder : sequence of int
    The orders of moments to estimate (1 for mean, 2 for variance, etc)
bounds : :class:`~openturns.Interval`, optional
    Parameter bounds

Notes
-----
This method fits a distribution to data of dimension 1, using the method of moments.

Let :math:`(x_1, \\dots, x_n)` be
the sample, :math:`F_{\\vect{\\theta}}`
the  cumulative distribution function we want to fit to the sample,
and :math:`\\vect{\\theta} = (\\theta_1, \\dots, \\theta_{d_{\\theta}}) \\in \\Rset^{d_\\theta}`
its parameter vector
where :math:`{d_\\theta} \\in \\Nset` is the number of parameters of the parametric model.

We assume that
the :math:`d_\\theta` first moments of the distribution exist. Let :math:`(\\mu_1, \\dots, \\mu_{d_\\theta})`
be the mean and the :math:`(d_\\theta -1)`
first centered  moments of the parametric model. They can be
can be expressed as a function of the :math:`\\vect{\\theta}`:

.. math::

    \\mu_1 & =  \\Expect{X} = g_1(\\theta_1, \\dots, \\theta_{d_\\theta})  \\\\
    \\mu_k & =  \\Expect{(X - \\mu_1)^k} = g_k(\\theta_1, \\dots, \\theta_{d_\\theta}), \\quad 2 \\leq k \\leq d_\\theta.


Let :math:`(\\widehat{\\mu}_1, \\dots, \\widehat{\\mu}_{d_\\theta})` be the empirical mean and the
:math:`(d_\\theta -1)` first empirical centered  moments
evaluated on the sample :math:`(x_1, \\dots, x_n)`:

.. math::

    \\widehat{\\mu}_1 & = \\dfrac{1}{n} \\sum_{i=1}^n x_i \\\\
    \\widehat{\\mu}_k & = \\dfrac{1}{n} \\sum_{i=1}^n (x_i - \\widehat{\\mu}_1)^k, \\quad 2 \\leq k \\leq d_\\theta.

Then the estimator :math:`\\widehat{\\theta} = \\left(\\widehat{\\theta}_1,
\\dots, \\widehat{\\theta}_{d_\\theta}\\right)` built by the method of moments is solution of the
following nonlinear system:

.. math::
    :label: MethMomSyst

    \\widehat{\\mu}_1 & =  g_1(\\theta_1, \\dots, \\theta_{d_\\theta}) \\\\
    \\widehat{\\mu}_2 & = g_2(\\theta_1, \\dots, \\theta_{d_\\theta}) \\\\
    \\vdots &  \\\\
    \\widehat{\\mu}_{d_\\theta} & = g_{d_\\theta}(\\theta_1, \\dots, \\theta_{d_\\theta})

which is equivalent to the solution of the following optimization problem:

.. math::

    \\widehat{\\theta} = \\argmin_{\\vect{\\theta} \\in \\Rset^{d_\\theta}}  \\sum_{k=1}^{d_\\theta} \\left[ \\left( g_k(\\vect{\\theta}) \\right)^{1/k}-
    \\widehat{\\mu}_k^{1/k} \\right]^2


See also
--------
DistributionFactory

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal(0.9, 1.7)
>>> sample = distribution.getSample(10)
>>> orders = [1, 2]  # mean, variance
>>> factory = ot.MethodOfMomentsFactory(ot.Normal(), orders)
>>> inf_distribution = factory.build(sample)

With parameter bounds:

>>> bounds = ot.Interval([0.8, 1.6], [1.0, 1.8])
>>> factory = ot.MethodOfMomentsFactory(ot.Normal(), orders, bounds)
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setOptimizationAlgorithm
"Accessor to the solver.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getOptimizationAlgorithm
"Accessor to the solver.

Returns
-------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setOptimizationBounds
"Accessor to the optimization bounds.

Parameters
----------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getOptimizationBounds
"Accessor to the optimization bounds.

Returns
-------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setKnownParameter
"Accessor to the known parameters.

Parameters
----------
values : sequence of float
    Values of fixed parameters.
indices : sequence of int
    Indices of fixed parameters.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> sample = distribution.getSample(10)
>>> orders = [3, 4]  # skewness, kurtosis
>>> factory = ot.MethodOfMomentsFactory(ot.Beta(), orders)
>>> # set (a,b) out of (r, t, a, b)
>>> factory.setKnownParameter([-1.0, 1.0], [2, 3])
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getKnownParameterValues
"Accessor to the known parameters indices.

Returns
-------
values : :class:`~openturns.Point`
    Values of fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getKnownParameterIndices
"Accessor to the known parameters indices.

Returns
-------
indices : :class:`~openturns.Indices`
    Indices of fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::buildFromMoments
"Build from moments.

Parameters
----------
moments : sequence of float
    Consists in the mean followed by consecutive central moments from order 2
    (variance), of total size at least the distribution parameter dimension.

Returns
-------
dist : :class:`~openturns.Distribution`
    Estimated distribution.

Notes
-----
Depending on the parametric model choosed, not all moments define a valid
distribution, so it should only used with empirical moments from the same model.

Examples
--------
>>> import openturns as ot
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> factory = ot.MethodOfMomentsFactory(ot.Beta(), [1, 2, 3, 4])
>>> cm = [distribution.getCentralMoment(i + 2)[0] for i in range(3)]
>>> moments = [distribution.getMean()[0]] + cm
>>> inf_distribution = factory.buildFromMoments(moments)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setMomentOrders
"Accessor to the moment orders.

Parameters
----------
momentsOrder : sequence of int
    The orders of moments to estimate (1 for mean, 2 for variance, etc)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getMomentOrders
"Accessor to the moment orders.

Returns
-------
momentsOrder : sequence of int
    The orders of moments to estimate (1 for mean, 2 for variance, etc)"
