﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lakeformation/LakeFormationRequest.h>
#include <aws/lakeformation/LakeFormation_EXPORTS.h>
#include <aws/lakeformation/model/ResourceShareType.h>

#include <utility>

namespace Aws {
namespace LakeFormation {
namespace Model {

/**
 */
class ListLFTagsRequest : public LakeFormationRequest {
 public:
  AWS_LAKEFORMATION_API ListLFTagsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListLFTags"; }

  AWS_LAKEFORMATION_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier for the Data Catalog. By default, the account ID. The Data
   * Catalog is the persistent metadata store. It contains database definitions,
   * table definitions, and other control information to manage your Lake Formation
   * environment. </p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  ListLFTagsRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If resource share type is <code>ALL</code>, returns both in-account LF-tags
   * and shared LF-tags that the requester has permission to view. If resource share
   * type is <code>FOREIGN</code>, returns all share LF-tags that the requester can
   * view. If no resource share type is passed, lists LF-tags in the given catalog ID
   * that the requester has permission to view.</p>
   */
  inline ResourceShareType GetResourceShareType() const { return m_resourceShareType; }
  inline bool ResourceShareTypeHasBeenSet() const { return m_resourceShareTypeHasBeenSet; }
  inline void SetResourceShareType(ResourceShareType value) {
    m_resourceShareTypeHasBeenSet = true;
    m_resourceShareType = value;
  }
  inline ListLFTagsRequest& WithResourceShareType(ResourceShareType value) {
    SetResourceShareType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListLFTagsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A continuation token, if this is not the first call to retrieve this
   * list.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListLFTagsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;

  ResourceShareType m_resourceShareType{ResourceShareType::NOT_SET};

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_catalogIdHasBeenSet = false;
  bool m_resourceShareTypeHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace LakeFormation
}  // namespace Aws
