/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *               2003, 2004 Dirk Ziegelmeier <dziegel@gmx.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <kconfig.h>
#include <kdebug.h>

#include "cfgdata.h"

struct grab_resolutions_s grab_resolutions[] = {
        { 160, 120, "160 x 120 (1/4 NTSC)" },
        { 192, 144, "192 x 144 (1/4 PAL)"  },
        { 320, 240, "320 x 240 (1/2 NTSC)" },
        { 384, 288, "384 x 288 (1/2 PAL)"  },
        { 400, 300, "400 x 300 (default)"  },
        { 640, 480, "640 x 480 (NTSC)"     },
        { 768, 576, "768 x 576 (PAL)"      },
        {  -1,  -1, ""                     },
};

static ConfigData& defaultCfg = *ConfigData::createDefaultConfig();

const ConfigData& ConfigData::getDefaultConfig()
{
    return defaultCfg;
}

ConfigData* ConfigData::createDefaultConfig()
{
    return new ConfigData(0L, 0L);
}

ConfigData::ConfigData(KConfig* backendCfg, KConfig* clientCfg)
    : _backendCfg(backendCfg),
      _clientCfg(clientCfg)
{
    defaults();
}

ConfigData::~ConfigData()
{
}

ConfigData const &ConfigData::operator=(ConfigData const &other)
{
    if (this != &other) {
        // General
        // maxResolution = other.maxResolution;
        channelNumberingStart = other.channelNumberingStart;
        firstTime             = other.firstTime;
        showSelectedOnly      = other.showSelectedOnly;
        maxKeypressInterval   = other.maxKeypressInterval;
        mouseWheelUpIsChUp    = other.mouseWheelUpIsChUp;

        // Device
        prevDev = other.prevDev;

        // Picture
        hue              = other.hue;
        colour           = other.colour;
        brightness       = other.brightness;
        whiteness        = other.whiteness;
        contrast         = other.contrast;
        // autoScaleEnabled = other.autoScaleEnabled;
        
        // Sound
        autoMute           = other.autoMute;
        volumeIncrement    = other.volumeIncrement;
        volumeRestoreDelay = other.volumeRestoreDelay;

        // Video
        fixAR           = other.fixAR;
        ARmode          = other.ARmode;
        snapshotPath    = other.snapshotPath;
        snapshotFormat  = other.snapshotFormat;
        snapshotQuality = other.snapshotQuality;
        snapshotRes     = other.snapshotRes;

        // Channel
        channelFile        = other.channelFile;
        setChannelOnSelect = other.setChannelOnSelect;
        lastChannel        = other.lastChannel;
        
        kdDebug() << "ConfigData::operator = : called..." << endl;
    } // if
    return(*this);
} // operator=


void ConfigData::defaults()
{
    // General
    firstTime = true;
    // maxResolution = false;
    channelNumberingStart = 2;
    showSelectedOnly = true;
    maxKeypressInterval = 2000;
    mouseWheelUpIsChUp = true;

    // Device
    prevDev = QString::null;

    // Picture
    brightness = hue = colour = contrast = whiteness = 32768;
    // autoScaleEnabled = true;

    // Channels
    channelFile        = QString::null;
    lastChannel        = -1;
    setChannelOnSelect = false;

    // Sound
    autoMute = false;
    volumeIncrement = 5;
    volumeRestoreDelay = 30;
    volumeMuted = false;
    volumeLeft = 50;
    volumeRight = 50;

    // Video
    fixAR = true;
    ARmode = AR_BEST_FIT;
    snapshotPath = "$HOME/";
    snapshotFormat = "PNG";
    snapshotQuality = -1;
    snapshotRes = 4;
}

int ConfigData::loadBackendSettings()
{
    // none so far
    return 0;
}

int ConfigData::loadClientSettings()
{
    if (!_clientCfg)
        return -1;

    const ConfigData& def = getDefaultConfig();

    _clientCfg->setGroup("General");
    firstTime = _clientCfg->readBoolEntry("First Time", def.firstTime);
    // maxResolution = _clientCfg->readBoolEntry("Capture at Max Res", def.maxResolution);
    channelNumberingStart = _clientCfg->readNumEntry ("Channel Numbering Start", def.channelNumberingStart);
    showSelectedOnly = _clientCfg->readBoolEntry("Show Only Selected Channels", def.showSelectedOnly);
    maxKeypressInterval = _clientCfg->readNumEntry ("Max Keypress Interval", def.maxKeypressInterval);
    mouseWheelUpIsChUp = _clientCfg->readBoolEntry ("MouseWheelUpIsChUp", def.mouseWheelUpIsChUp);
    // autoScaleEnabled = _clientCfg->readBoolEntry("Auto Scale Brightness", def.autoScaleEnabled);
    snapshotPath = _clientCfg->readPathEntry("Snapshot Path", def.snapshotPath);
    snapshotFormat = _clientCfg->readEntry("Snapshot Format", def.snapshotFormat);
    snapshotQuality = _clientCfg->readNumEntry("Snapshot Quality", def.snapshotQuality);
    snapshotRes = _clientCfg->readNumEntry("Snapshot Resolution", def.snapshotRes);
    autoMute = _clientCfg->readBoolEntry("Auto Mute", def.autoMute);
    volumeRestoreDelay = _clientCfg->readNumEntry ("Volume Restore Delay", def.volumeRestoreDelay);
    setChannelOnSelect = _clientCfg->readBoolEntry("SetChannelOnSelect", def.setChannelOnSelect);
    prevDev = _clientCfg->readEntry("Previous Device", def.prevDev);
    fixAR = _clientCfg->readBoolEntry("Fix Aspect Ratio", def.fixAR);
    ARmode = _clientCfg->readNumEntry("Aspect Ratio Mode", def.ARmode);
    volumeIncrement = _clientCfg->readNumEntry ("Volume Increment", def.volumeIncrement);
    
    _clientCfg->setGroup("Volume");    
    volumeMuted = _clientCfg->readBoolEntry("Volume Muted", def.volumeMuted);
    volumeLeft  = _clientCfg->readNumEntry("Volume Left", def.volumeLeft);
    volumeRight = _clientCfg->readNumEntry("Volume Right", def.volumeRight);

    // Make sure the volume has sane values (the user may still need his ears)
    if ( (volumeLeft > 100) || (volumeLeft < 0) )
        volumeLeft = def.volumeLeft;

    if ( (volumeRight > 100) || (volumeRight < 0) )
        volumeRight = def.volumeRight;

    return 0;
}

int ConfigData::load()
{
    int rc1, rc2;

    kdDebug() << "CfgData: load()" << endl;

    rc1 = loadBackendSettings();
    rc2 = loadClientSettings();

    if (rc1 || rc2) {
        return -1;
    } else {
        return 0;
    }
}

int ConfigData::loadDeviceSettings(const QString& dev)
{
    if (!_clientCfg || dev.isEmpty())
        return -1;

    const ConfigData& def = getDefaultConfig();

    kdDebug() << "CfgData: Loading settings for device: " << dev << endl;
    _clientCfg->setGroup(dev);
    lastChannel  = _clientCfg->readNumEntry("Previous Channel", def.lastChannel);
    channelFile  = _clientCfg->readPathEntry("Channel File", def.channelFile);

    return 0;
}

int ConfigData::saveDeviceSettings(const QString& dev)
{
    if (!_clientCfg || dev.isEmpty())
        return -1;

    kdDebug() << "CfgData: Saving settings for device: " << dev << endl;
    _clientCfg->setGroup(dev);
    _clientCfg->writeEntry("Previous Channel", lastChannel);
    _clientCfg->writePathEntry("Channel File", channelFile);
    return 0;
}

int ConfigData::loadPictureSettings(const QString& dev, const QString& key)
{
    if (!_clientCfg || dev.isEmpty() || key.isEmpty())
        return -1;

    _clientCfg->setGroup(dev);
    if (!_clientCfg->hasKey(key + " Whiteness")) {
        const ConfigData& def = getDefaultConfig();

        brightness = def.brightness;
        hue = def.hue;
        colour = def.colour;
        contrast = def.contrast;
        whiteness = def.whiteness;

        return -1;
    }

    kdDebug() << "CfgData: Loading picture settings for device: " << dev << " with key: " << key << endl;

    whiteness    = _clientCfg->readNumEntry(key + " Whiteness",  0);
    colour       = _clientCfg->readNumEntry(key + " Colour",     0);
    contrast     = _clientCfg->readNumEntry(key + " Contrast",   0);
    hue          = _clientCfg->readNumEntry(key + " Hue",        0);
    brightness   = _clientCfg->readNumEntry(key + " Brightness", 0);
    return 0;
}

int ConfigData::savePictureSettings(const QString& dev, const QString& key)
{
    if (!_clientCfg || dev.isEmpty() || key.isEmpty())
        return -1;

    kdDebug() << "CfgData: Saving picture settings for device: " << dev << " with key: " << key << endl;
    _clientCfg->setGroup(dev);
    _clientCfg->writeEntry(key + " Whiteness",  whiteness);
    _clientCfg->writeEntry(key + " Colour",     colour);
    _clientCfg->writeEntry(key + " Contrast",   contrast);
    _clientCfg->writeEntry(key + " Hue",        hue);
    _clientCfg->writeEntry(key + " Brightness", brightness);
    return 0;
}

int ConfigData::saveBackendSettings()
{
    // none so far
    return 0;
}

int ConfigData::saveClientSettings()
{
    if (!_clientCfg)
        return -1;

    _clientCfg->setGroup("General");
    _clientCfg->writeEntry("First Time", false);
    _clientCfg->writeEntry("Auto Mute", autoMute);
    // _clientCfg->writeEntry("Capture at Max Res", maxResolution);
    // _clientCfg->writeEntry("Auto Scale Brightness", autoScaleEnabled);
    _clientCfg->writeEntry("Max Keypress Interval", maxKeypressInterval);
    _clientCfg->writeEntry("MouseWheelUpIsChUp", mouseWheelUpIsChUp);
    _clientCfg->writeEntry("Show Only Selected Channels", showSelectedOnly);
    _clientCfg->writeEntry("Snapshot Path", snapshotPath);
    _clientCfg->writeEntry("Snapshot Format", snapshotFormat);
    _clientCfg->writeEntry("Snapshot Quality", snapshotQuality);
    _clientCfg->writeEntry("Snapshot Resolution", snapshotRes);
    _clientCfg->writeEntry("Channel Numbering Start", channelNumberingStart);
    _clientCfg->writeEntry("Volume Restore Delay", volumeRestoreDelay);
    _clientCfg->writeEntry("SetChannelOnSelect", setChannelOnSelect);
    _clientCfg->writeEntry("Previous Device", prevDev);
    _clientCfg->writeEntry("Fix Aspect Ratio", fixAR);
    _clientCfg->writeEntry("Aspect Ratio Mode", ARmode);
    _clientCfg->writeEntry("Volume Increment", volumeIncrement);

    _clientCfg->setGroup("Volume");
    _clientCfg->writeEntry("Volume Muted", volumeMuted);
    _clientCfg->writeEntry("Volume Left", volumeLeft);
    _clientCfg->writeEntry("Volume Right", volumeRight);

    _clientCfg->sync();
    return 0;
}

int ConfigData::save()
{
    int rc1, rc2;

    kdDebug() << "CfgData: save()" << endl;

    rc1 = saveBackendSettings();
    rc2 = saveClientSettings();

    if (rc1 || rc2) {
        return -1;
    } else {
        return 0;
    }
}
