/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */

Cu.import("resource://services-common/moz-kinto-client.js");

// set up what we need to make storage adapters
const Kinto = loadKinto();
const IceCatAdapter = Kinto.adapters.IceCatAdapter;
const kintoFilename = "kinto.sqlite";

let gIceCatAdapter = null;

function do_get_kinto_adapter() {
  if (gIceCatAdapter == null) {
    gIceCatAdapter = new IceCatAdapter("test");
  }
  return gIceCatAdapter;
}

function do_get_kinto_db() {
  let profile = do_get_profile();
  let kintoDB = profile.clone();
  kintoDB.append(kintoFilename);
  return kintoDB;
}

function cleanup_kinto() {
  add_test(function cleanup_kinto_files(){
    let kintoDB = do_get_kinto_db();
    // clean up the db
    kintoDB.remove(false);
    // force re-creation of the adapter
    gIceCatAdapter = null;
    run_next_test();
  });
}

function test_collection_operations() {
  add_task(function* test_kinto_clear() {
    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    yield adapter.clear();
    yield adapter.close();
  });

  // test creating new records... and getting them again
  add_task(function* test_kinto_create_new_get_existing() {
    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    let record = {id:"test-id", foo:"bar"};
    yield adapter.create(record);
    let newRecord = yield adapter.get("test-id");
    // ensure the record is the same as when it was added
    deepEqual(record, newRecord);
    yield adapter.close();
  });

  // test removing records
  add_task(function* test_kinto_create_new_get_existing() {
    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    // create a second record
    let record = {id:"test-id-2", foo:"baz"};
    yield adapter.create(record);
    let newRecord = yield adapter.get("test-id-2");
    deepEqual(record, newRecord);
    // delete the record
    let id = yield adapter.delete(record.id);
    // ensure the delete resolved with the record id
    do_check_eq(record.id, id);
    newRecord = yield adapter.get(record.id);
    // ... and ensure it's no longer there
    do_check_eq(newRecord, undefined);
    // ensure the other record still exists
    newRecord = yield adapter.get("test-id");
    do_check_neq(newRecord, undefined);
    yield adapter.close();
  });

  // test getting records that don't exist
  add_task(function* test_kinto_get_non_existant() {
    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    // Kinto expects adapters to either:
    let newRecord = yield adapter.get("missing-test-id");
    // resolve with an undefined record
    do_check_eq(newRecord, undefined);
    yield adapter.close();
  });

  // test updating records... and getting them again
  add_task(function* test_kinto_update_get_existing() {
    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    let originalRecord = {id:"test-id", foo:"bar"};
    let updatedRecord = {id:"test-id", foo:"baz"};
    yield adapter.clear();
    yield adapter.create(originalRecord);
    yield adapter.update(updatedRecord);
    // ensure the record exists
    let newRecord = yield adapter.get("test-id");
    // ensure the record is the same as when it was added
    deepEqual(updatedRecord, newRecord);
    yield adapter.close();
  });

  // test listing records
  add_task(function* test_kinto_list() {
    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    let originalRecord = {id:"test-id-1", foo:"bar"};
    let records = yield adapter.list();
    do_check_eq(records.length, 1);
    yield adapter.create(originalRecord);
    records = yield adapter.list();
    do_check_eq(records.length, 2);
    yield adapter.close();
  });

  // test save and get last modified
  add_task(function* test_kinto_last_modified() {
    const initialValue = 0;
    const intendedValue = 12345678;

    let adapter = do_get_kinto_adapter();
    yield adapter.open();
    let lastModified = yield adapter.getLastModified();
    do_check_eq(lastModified, initialValue);
    let result = yield adapter.saveLastModified(intendedValue);
    do_check_eq(result, intendedValue);
    lastModified = yield adapter.getLastModified();
    do_check_eq(lastModified, intendedValue);

    // test saveLastModified parses values correctly
    result = yield adapter.saveLastModified(" " + intendedValue + " blah");
    // should resolve with the parsed int
    do_check_eq(result, intendedValue);
    // and should have saved correctly
    lastModified = yield adapter.getLastModified();
    do_check_eq(lastModified, intendedValue);
    yield adapter.close();
  });
}

// test kinto db setup and operations in various scenarios
// test from scratch - no current existing database
add_test(function test_db_creation() {
  add_test(function test_create_from_scratch() {
    // ensure the file does not exist in the profile
    let kintoDB = do_get_kinto_db();
    do_check_false(kintoDB.exists());
    run_next_test();
  });

  test_collection_operations();

  cleanup_kinto();
  run_next_test();
});

// this is the closest we can get to a schema version upgrade at v1 - test an
// existing database
add_test(function test_creation_from_empty_db() {
  add_test(function test_create_from_empty_db() {
    // place an empty kinto db file in the profile
    let profile = do_get_profile();
    let kintoDB = do_get_kinto_db();

    let emptyDB = do_get_file("test_storage_adapter/empty.sqlite");
    emptyDB.copyTo(profile,kintoFilename);

    run_next_test();
  });

  test_collection_operations();

  cleanup_kinto();
  run_next_test();
});

function run_test() {
  run_next_test();
}
