/* ws-queries.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "ws-queries.h"
#include "workspace-page.h"
#include <string.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libgnomedb/libgnomedb.h>
#include "query-druid.h"
#include "query-editor.h"
#include "query-params-editor.h"

/* 
 * Main static functions 
 */
static void ws_queries_class_init (WsQueriesClass * class);
static void ws_queries_init (WsQueries *ws);
static void ws_queries_dispose (GObject *object);
static void ws_queries_finalize (GObject *object);

/* WorkspacePage interface */
static void         ws_queries_page_init       (WorkspacePageIface *iface);
static gchar       *ws_queries_get_name        (WorkspacePage *iface);
static gchar       *ws_queries_get_description (WorkspacePage *iface);
static GtkWidget   *ws_queries_get_sel_button    (WorkspacePage *iface);
static GtkWidget   *ws_queries_get_selector    (WorkspacePage *iface);
static GtkWidget   *ws_queries_get_work_area   (WorkspacePage *iface);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _WsQueriesPrivate
{
	Workspace      *wspace;
	GnomeDbDict    *dict;
	GtkWidget      *selector;
	GtkWidget      *work_area;

	GtkWidget      *notebook;
	GtkWidget      *query_info_nb;
	GtkWidget      *description;
	GtkWidget      *sql_editor;
	GtkWidget      *params_editor;
	GtkWidget      *query_add;
	GtkWidget      *query_edit;
	GtkWidget      *query_del;
	GtkWidget      *query_exec;	

	GObject        *sel_obj;
};

guint
ws_queries_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (WsQueriesClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) ws_queries_class_init,
			NULL,
			NULL,
			sizeof (WsQueries),
			0,
			(GInstanceInitFunc) ws_queries_init
		};

		static const GInterfaceInfo workspace_page_info = {
			(GInterfaceInitFunc) ws_queries_page_init,
			NULL,
			NULL
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WsQueries", &info, 0);
		g_type_add_interface_static (type, WORKSPACE_PAGE_TYPE, &workspace_page_info);
	}
	return type;
}

static void 
ws_queries_page_init (WorkspacePageIface *iface)
{
	iface->get_name = ws_queries_get_name;
	iface->get_description = ws_queries_get_description;
	iface->get_sel_button = ws_queries_get_sel_button;
	iface->get_selector = ws_queries_get_selector;
	iface->get_work_area = ws_queries_get_work_area;
}

static void
ws_queries_class_init (WsQueriesClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = ws_queries_dispose;
	object_class->finalize = ws_queries_finalize;
}

static void
ws_queries_init (WsQueries *ws)
{
	ws->priv = g_new0 (WsQueriesPrivate, 1);
}

static void ws_queries_initialize (WsQueries *ws);

/**
 * ws_queries_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new WsQueries object
 *
 * Returns: the new object
 */
GObject*
ws_queries_new (Workspace* wspace)
{
	GObject  *obj;
	WsQueries *ws;
	GnomeDbDict *dict;

	g_return_val_if_fail (wspace && IS_WORKSPACE (wspace), NULL);
	dict = gnome_db_server_get_dict (workspace_get_server (wspace));

	obj = g_object_new (WS_QUERIES_TYPE, NULL);
	ws = WS_QUERIES (obj);
	ws->priv->dict = dict;
	ws->priv->wspace = wspace;
	g_object_ref (G_OBJECT (dict));

	ws_queries_initialize (ws);

	return obj;
}


static void query_updated_cb (GnomeDbDict *dict, GnomeDbQuery *query, WsQueries *ws);
static void
ws_queries_dispose (GObject *object)
{
	WsQueries *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_QUERIES (object));

	ws = WS_QUERIES (object);
	if (ws->priv) {
		if (ws->priv->selector) {
			gtk_widget_destroy (ws->priv->selector);
			ws->priv->selector = NULL;
		}

		if (ws->priv->work_area) {
			gtk_widget_destroy (ws->priv->work_area);
			ws->priv->work_area = NULL;
		}

		if (ws->priv->dict) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ws->priv->dict),
							      G_CALLBACK (query_updated_cb), ws);

			g_object_unref (G_OBJECT (ws->priv->dict));
			ws->priv->dict = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
ws_queries_finalize (GObject   * object)
{
	WsQueries *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_QUERIES (object));

	ws = WS_QUERIES (object);
	if (ws->priv) {
		g_free (ws->priv);
		ws->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}



static void selector_selection_changed_cb (GnomeDbSelector *mgsel, GObject *sel_object, WsQueries *ws);
static void query_add_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_edit_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_del_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_exec_clicked_cb (GtkButton *button, WsQueries *ws);
static void
ws_queries_initialize (WsQueries *ws)
{
	GtkWidget *label, *vbox, *nb, *vbox2, *bbox, *button, *wid, *hbox, *vbox3, *vp;
	GtkWidget *nb2, *vbox4;
	gchar *str;

	/* Selector part */
	wid = gnome_db_selector_new (ws->priv->dict, NULL,
			       GNOME_DB_SELECTOR_QUERIES, GNOME_DB_SELECTOR_COLUMN_TYPE);
	ws->priv->selector = wid;
	g_signal_connect (G_OBJECT (ws->priv->selector), "selection_changed", 
			  G_CALLBACK (selector_selection_changed_cb), ws);

	/* WorkArea part */
	vbox = gtk_vbox_new (FALSE, 5);
	ws->priv->work_area = vbox;

	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ws->priv->notebook = nb;
	
	label = gtk_label_new (_("Please select a query from the list on the left,\n"
				 "or create a new one using the 'Add' button below."));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);

	vp = gtk_vpaned_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vp, NULL);
	gtk_widget_show (vp);
	gtk_paned_set_position (GTK_PANED (vp), 370);
	
	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add1 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s:</b>", _("Description"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ws->priv->description = label;

	nb2 = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb2), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb2), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox2), nb2, TRUE, TRUE, 0);
	gtk_widget_show (nb2);
	ws->priv->query_info_nb = nb2;

	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb2), vbox4, NULL);
	gtk_widget_show (vbox4);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s:</b>", _("SQL statement"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox4), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox4), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vbox3, TRUE, TRUE, 0);
	gtk_widget_show (vbox3);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<small>%s</small>", _("For information only, to edit, click on the \"Edit\" button."));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox3), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	wid = gnome_db_editor_new ();
        gnome_db_editor_set_editable (GNOME_DB_EDITOR (wid), FALSE);
        gnome_db_editor_set_highlight (GNOME_DB_EDITOR (wid), TRUE);
        gtk_box_pack_start (GTK_BOX (vbox3), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ws->priv->sql_editor = wid;



	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb2), vbox4, NULL);
	gtk_widget_show (vbox4);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s:</b>", _("Broken query"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox4), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox4), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	str = g_strdup_printf ("<span foreground=\"#ff6d6d\">%s</span>\n\n%s",
			       _("The query uses elements which do not exist anymore such as "
				 "table's fields which have been deleted."),
			       _("To repair the query, click on the \"Edit\" button, and remove "
				 "or change the parts which appear as striked out text."));
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add2 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);
	
	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s:</b>", _("Query parameters"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vbox3, TRUE, TRUE, 0);
	gtk_widget_show (vbox3);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<small>%s</small>", _("For information only, to edit, click on the \"Edit\" button."));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox3), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	wid = query_params_editor_new (NULL, 0);
        gtk_box_pack_start (GTK_BOX (vbox3), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ws->priv->params_editor = wid;

	/* action buttons */
	bbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show (bbox);

	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	ws->priv->query_add = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_add_clicked_cb), ws);

	button = gtk_button_new_with_label (_("Edit"));
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ws->priv->query_edit = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_edit_clicked_cb), ws);

	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ws->priv->query_del = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_del_clicked_cb), ws);

	button = gtk_button_new_from_stock (GTK_STOCK_INDEX);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_button_set_label (GTK_BUTTON (button), _("Execute"));
	ws->priv->query_exec = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_exec_clicked_cb), ws);
	
	g_signal_connect (G_OBJECT (ws->priv->dict), "query_updated",
			  G_CALLBACK (query_updated_cb), ws);
}

static void query_info_display_update (GnomeDbQuery *query, WsQueries *ws);
static void
selector_selection_changed_cb (GnomeDbSelector *mgsel, GObject *sel_object, WsQueries *ws)
{
	if (sel_object && !IS_GNOME_DB_QUERY (sel_object))
		return;
	
	ws->priv->sel_obj = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ws->priv->notebook), sel_object ? 1 : 0);
	gtk_widget_set_sensitive (ws->priv->query_exec, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ws->priv->query_del, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ws->priv->query_edit, sel_object ? TRUE : FALSE);
	query_info_display_update ((GnomeDbQuery*) sel_object, ws);
}

static void
query_updated_cb (GnomeDbDict *dict, GnomeDbQuery *query, WsQueries *ws)
{
	if ((GObject *)query == ws->priv->sel_obj)
		query_info_display_update (query, ws);
}

static void
dialog_exec_response_cb (GtkDialog *dlg, gint response, GObject *obj)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
}

static void
query_add_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GtkWidget *druid;

	druid = query_druid_new (ws->priv->dict);
	gtk_widget_show (druid);
	gnome_druid_construct_with_window (GNOME_DRUID (druid), _("New Query"), NULL, TRUE, NULL);
}

static void
query_edit_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GtkWidget *dlg, *editor;
	GObject *sel_query = ws->priv->sel_obj;
	gchar *title;

	if (!sel_query)
		return;

	editor = query_editor_new (GNOME_DB_QUERY (sel_query));
	gtk_widget_show (editor);
	title = g_strdup_printf (_("Edition of query '%s'"), gnome_db_base_get_name (GNOME_DB_BASE (sel_query)));
	dlg = gtk_dialog_new_with_buttons (title, NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	g_free (title);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), editor, TRUE, TRUE, 6);
	gtk_widget_set_size_request (dlg, 770, 570);
	gtk_widget_show (dlg);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (dialog_exec_response_cb), NULL);
}

static void
query_del_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GObject *sel_query = ws->priv->sel_obj;
	GtkWidget *dlg;

	if (!sel_query)
		return;

	dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
				      _("Do you want to delete the '%s' query?"), 
				      gnome_db_base_get_name (GNOME_DB_BASE (sel_query)));
	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_YES) 
		gnome_db_base_nullify (GNOME_DB_BASE (sel_query));

	gtk_widget_destroy (dlg);
}

/* extra action for the grid widgets */
static void extra_action_close_cb (GtkAction *action, GtkWidget *window);
static GtkActionEntry extra_actions[] = {
	{ "Close", GTK_STOCK_CLOSE, "_Close", NULL, "Close this window", G_CALLBACK (extra_action_close_cb)}
};
static const gchar *grid_actions =
	"<ui>"
	"  <toolbar  name='ToolBar'>"
	"    <toolitem action='Close'/>"
	"  </toolbar>"
	"</ui>";


static gint
fill_context_in_dialog (WsQueries *ws, GnomeDbDataSet *context)
{
	GtkWidget *dlg;
	GnomeDbBasicForm *sform;
	GtkWidget *parent_window;
	gint result;

	if (! gnome_db_data_set_needs_user_input (context))
		return GTK_RESPONSE_ACCEPT;
	
	parent_window = gtk_widget_get_toplevel (ws->priv->work_area);
	if (! GTK_WIDGET_TOPLEVEL (parent_window))
		parent_window = NULL;

	dlg = gnome_db_basic_form_new_in_dialog (ws->priv->dict, 
						 context, GTK_WINDOW (parent_window),
						 _("Values to be filled"), 
						 _("<big><b>Required values:</b></big>\n"
						   "<small>The following values are required to "
						   "execute the query.</small>"));
	sform = g_object_get_data (G_OBJECT (dlg), "form");
	gnome_db_basic_form_set_entries_auto_default (sform, TRUE);
	
	gtk_widget_show (dlg);
	result = gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);

	return result;
}

static void
query_exec_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GnomeDbQuery *query;
	GnomeDbDataSet *context;
	GtkWidget *parent_window;
	
	parent_window = gtk_widget_get_toplevel (ws->priv->work_area);
	if (! GTK_WIDGET_TOPLEVEL (parent_window))
		parent_window = NULL;
		
	query = (GnomeDbQuery *) ws->priv->sel_obj;

	if (gnome_db_query_is_modif_query (query)) {
		/* modification query */
		context = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (query));
		if (fill_context_in_dialog (ws, context) == GTK_RESPONSE_ACCEPT) 
			utility_query_execute_modif (query, context, TRUE, TRUE, TRUE, parent_window, NULL, NULL);
		g_object_unref (G_OBJECT (context));
	}
	else {
		/* SELECT query */
		if (gnome_db_query_is_select_query (query)) {
			/* parsed SELECT query */
			GtkWidget *grid;

			grid = gnome_db_grid_new_with_select_query (query, NULL); /* no modification possible */
			g_object_set (G_OBJECT (grid), "title_visible", FALSE, NULL);
			gnome_db_data_widget_global_show_actions (GNOME_DB_DATA_WIDGET (grid), TRUE);

			context = gnome_db_data_widget_get_params (GNOME_DB_DATA_WIDGET (grid));
			if (fill_context_in_dialog (ws, context) == GTK_RESPONSE_ACCEPT) {
				GtkWidget *window, *vbox, *toolbar;
				guint mode = GNOME_DB_ACTION_NAVIGATION_SCROLL | GNOME_DB_ACTION_NAVIGATION_ARROWS;
				gchar *str;
				GtkActionGroup *extra_group;
				GtkUIManager *ui;

				str = g_strdup_printf (_("Execution of query '%s'"), 
						       gnome_db_base_get_name (GNOME_DB_BASE (query)));
				window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
				gtk_window_set_title (GTK_WINDOW (window), str);
				g_free (str);				

				vbox = gtk_vbox_new (FALSE, 0);
				gtk_container_add (GTK_CONTAINER (window), vbox);
				gtk_widget_show (vbox);

				ui = gtk_ui_manager_new ();
				extra_group = gtk_action_group_new ("ExtraActions");
				gtk_action_group_add_actions (extra_group, extra_actions, 
							      G_N_ELEMENTS (extra_actions),
							      window);
				gtk_ui_manager_insert_action_group (ui, extra_group, 0);
				gtk_ui_manager_add_ui_from_string (ui, grid_actions, -1, NULL);
				toolbar = gtk_ui_manager_get_widget (ui, "/ToolBar");
				gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
				gtk_widget_show (toolbar);

				gtk_box_pack_start (GTK_BOX (vbox), grid, TRUE, TRUE, 0);
				gtk_widget_show (grid);
				gtk_widget_set_size_request (grid, 800, 300);
				
				gnome_db_data_widget_set_mode (GNOME_DB_DATA_WIDGET (grid), mode);
				gtk_widget_show (window);
			}
			else
				gtk_widget_destroy (grid);
		}
		else {
			/* unparsed query */
			context = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (query));			
			if (fill_context_in_dialog (ws, context) == GTK_RESPONSE_ACCEPT) {
				GtkWidget *dlg;
				gint result;
				gchar *msg;
				const gchar *sql;
				
				sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), context, 
								       GNOME_DB_RENDERER_EXTRA_PRETTY_SQL, NULL);
				msg = g_strdup_printf (_("<b><big>Execute the following query ?</big></b>\n"
							 "<small>This query can't be parsed: it may contain some "
							 "syntax or grammar which is dependant on the type of database "
							 "used, or may contain some error.</small>\n\n%s"), sql);
				dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
							      GTK_MESSAGE_QUESTION,
							      GTK_BUTTONS_YES_NO, msg);
				g_free (msg);
				gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dlg)->label), TRUE);
				result = gtk_dialog_run (GTK_DIALOG (dlg));
				gtk_widget_destroy (dlg);
				if (result == GTK_RESPONSE_YES) {
					GError *error = NULL;
					GdaDataModel *data;
					
					data = gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (ws->priv->dict),
										       sql, GNOME_DB_SERVER_QUERY_SQL, &error);
					if (error) {
						GtkWidget *dlg;
						gchar *message;
						
						message = g_strdup (error->message);
						g_error_free (error);
						dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
									      GTK_MESSAGE_ERROR,
									      GTK_BUTTONS_CLOSE,
									      message);
						g_free (message);
						gtk_dialog_run (GTK_DIALOG (dlg));
						gtk_widget_destroy (dlg);
					}
					else {
						if (data) {
							GtkWidget *grid, *window, *vbox, *toolbar;
							GtkActionGroup *extra_group;
							GtkUIManager *ui;

							grid = gnome_db_grid_new_with_gda_model (ws->priv->dict, data);
							gtk_widget_set_size_request (grid, 800, 300);
							g_object_unref (G_OBJECT (data));
							
							window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
							gtk_window_set_title (GTK_WINDOW (window), _("Execution results"));
							
							vbox = gtk_vbox_new (FALSE, 0);
							gtk_container_add (GTK_CONTAINER (window), vbox);
							gtk_widget_show (vbox);
							
							ui = gtk_ui_manager_new ();
							extra_group = gtk_action_group_new ("ExtraActions");
							gtk_action_group_add_actions (extra_group, extra_actions, 
										      G_N_ELEMENTS (extra_actions),
										      window);
							gtk_ui_manager_insert_action_group (ui, extra_group, 0);
							gtk_ui_manager_add_ui_from_string (ui, grid_actions, -1, NULL);
							toolbar = gtk_ui_manager_get_widget (ui, "/ToolBar");
							gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
							gtk_widget_show (toolbar);
							
							gtk_box_pack_start (GTK_BOX (vbox), grid, TRUE, TRUE, 0);
							gtk_widget_show (grid);
							gtk_widget_show (window);
						}
					}
				}
			}

		}
	}
}

static void
extra_action_close_cb (GtkAction *action, GtkWidget *window)
{
	gtk_widget_destroy (window);
}



static void
query_info_display_update (GnomeDbQuery *query, WsQueries *ws)
{
	const gchar *str = NULL;
	gchar *title = NULL;
	
	if (query)
		str = gnome_db_base_get_description (GNOME_DB_BASE (query));
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ws->priv->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ws->priv->description), "---");

	/* global title update */
	title = ws_queries_get_description (WORKSPACE_PAGE (ws));
	g_signal_emit_by_name (G_OBJECT (ws), "description_changed", title);
	g_free (title);

	/* Active / non active query */
	if (gnome_db_referer_is_active (GNOME_DB_REFERER (query)))
		gtk_notebook_set_current_page (GTK_NOTEBOOK (ws->priv->query_info_nb), 0);
	else
		gtk_notebook_set_current_page (GTK_NOTEBOOK (ws->priv->query_info_nb), 1);

	/* SQL version of the query */
	if (query) {
		gchar *sql;
		GError *error = NULL;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL,
						 GNOME_DB_RENDERER_EXTRA_PRETTY_SQL, &error);
		if (sql) {
			gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor), sql, -1);
			g_free (sql);
		}
		else {
			if (error) {
				str = error->message;
				gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor), str, -1);
				g_error_free (error);
			}
			else
				gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor),
							  _("Non reported error"), -1);
		}
	}
	else
		gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor), "", -1);

	/* query parameters */
	query_params_editor_set_query (QUERY_PARAMS_EDITOR (ws->priv->params_editor), query);
}



/* 
 * WorkspacePage interface implementation
 */
static gchar *
ws_queries_get_name (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	return g_strdup_printf (_("Queries"));
}

static gchar *
ws_queries_get_description (WorkspacePage *iface)
{
	gchar *str = NULL;
	WsQueries *ws;

	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	ws = WS_QUERIES (iface);
	if (ws->priv->sel_obj) {
		const gchar *cstr;
		 cstr =  gnome_db_base_get_name (GNOME_DB_BASE (ws->priv->sel_obj));
		 str = g_strdup_printf ("Query: <b>%s</b>", (cstr && *cstr) ? cstr : "---");
	}
	else
		str = g_strdup_printf ("<b>%s</b>", _("No query selected"));

	return str;
}

static GtkWidget *
ws_queries_get_sel_button (WorkspacePage *iface)
{
	GdkPixbuf *pixbuf;
	GtkWidget *button, *wid, *hbox;

	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	hbox = gtk_hbox_new (FALSE, 0);

	pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_QUERY);
	wid = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (pixbuf);
	gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	wid = gtk_label_new (_("Queries"));
	gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, FALSE, 5);
	gtk_widget_show (wid);

	button = gtk_toggle_button_new ();
	gtk_container_add (GTK_CONTAINER (button), hbox);
	gtk_widget_show (hbox);

	return button;
}

static GtkWidget *
ws_queries_get_selector (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	return WS_QUERIES (iface)->priv->selector;
}

static GtkWidget *
ws_queries_get_work_area (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	return WS_QUERIES (iface)->priv->work_area;
}
