/*
 * Java-Gnome Bindings Library
 *
 * * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome Team Members:
 *   Jean Van Wyk <jeanvanwyk@iname.com>
 *   Jeffrey S. Morgan <jeffrey.morgan@bristolwest.com>
 *   Dan Bornstein <danfuzz@milk.com>
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gtk.event;

import org.gnu.glib.EventType;
import org.gnu.gtk.DeleteType;
import org.gnu.gtk.MovementStep;

/**
 * This event is used to identify when an entry receives an event
 * 
 * @see EntryListener
 */
public class EntryEvent extends GtkEvent {

    /**
     * The starting position of the deleted text. Applies to the DELETE_TEXT
     * type of event.
     */
    private int startPosition;

    /**
     * The ending position of the deleted text. Applies to the DELETE_TEXT type
     * of event.
     */
    private int endPosition;

    /**
     * The text that was inserted into the widget. Applies to the INSERT_TEXT
     * and INSET_AT_CURSOR type of event.
     */
    private String text;

    /**
     * The position at which the next text was inserted. Applies to the
     * INSERT_TEXT type of event.
     */
    private int insertPosition;

    /**
     * deleteType is set when the DELETE_FROM_CURSOR event is invoked.
     */
    private DeleteType deleteType;

    /**
     * count is set when the DELETE_FROM_CURSOR or MOVE_CURSOR events are
     * invoked.
     */
    private int count;

    /**
     * movementStep is set when the MOVE_CURSOR event is invoked.
     */
    private MovementStep movementStep;

    /**
     * extendedSelection is set when the MOVE_CURSOR event is invoked.
     */
    private boolean extendedSelection;

    public static class Type extends EventType {
        private Type(int id, String name) {
            super(id, name);
        }

        /**
         * Indicates that the user has pressed the enter key while in the widget
         */
        public static final Type ACTIVATE = new Type(1, "ACTIVATE");

        /**
         * Indicates that the user has moved the cursor in the widget.
         */
        public static final Type MOVE_CURSOR = new Type(2, "MOVE_CURSOR");

        /**
         * Indicates that the user has inserted text at the cursor.
         */
        public static final Type INSERT_AT_CURSOR = new Type(3,
                "INSERT_AT_CURSOR");

        /**
         * Indicates that the user has deleted text starting at the cursor.
         */
        public static final Type DELETE_FROM_CURSOR = new Type(4,
                "DELETE_FROM_CURSOR");

        /**
         * Clipboard related events
         */
        public static final Type CUT_CLIPBOARD = new Type(5, "CUT_CLIPBOARD");

        public static final Type COPY_CLIPBOARD = new Type(6, "COPY_CLIPBOARD");

        public static final Type PASTE_CLIPBOARD = new Type(7,
                "PASTE_CLIPBOARD");

        public static final Type TOGGLE_OVERWRITE = new Type(8,
                "TOGGLE_OVERWRITE");

        /**
         * Indicates that the user has changed the contents of the widget.
         */
        public static final Type CHANGED = new Type(9, "CHANGED");

        /**
         * This type of event is emitted when the user deletes text from the
         * widget
         */
        public static final Type DELETE_TEXT = new Type(10, "DELETE_TEXT");

        /**
         * This type of event is emitted when the user inserts text into the
         * widget.
         */
        public static final Type INSERT_TEXT = new Type(11, "INSERT_TEXT");

    }

    /**
     * Constructor for EditableEvent.
     * 
     * @param source
     * @param type
     */
    public EntryEvent(Object source, EventType type) {
        super(source, type);
    }

    /**
     * @return True if the type of this event is the same as that stated.
     */
    public boolean isOfType(EntryEvent.Type aType) {
        return (type.getID() == aType.getID());
    }

    public void setText(String text) {
        this.text = text;
    }

    public String getText() {
        return text;
    }

    public void setDeleteType(DeleteType type) {
        deleteType = type;
    }

    public DeleteType getDeleteType() {
        return deleteType;
    }

    public void setCount(int count) {
        this.count = count;
    }

    public int getCount() {
        return count;
    }

    public void setMovementStep(MovementStep step) {
        movementStep = step;
    }

    public MovementStep getMovementStep() {
        return movementStep;
    }

    public void setExtendedSelection(boolean extendedSelection) {
        this.extendedSelection = extendedSelection;
    }

    public boolean getExtendedSelection() {
        return extendedSelection;
    }

    /**
     * @return Returns the endPosition.
     */
    public int getEndPosition() {
        return endPosition;
    }

    /**
     * @param endPosition
     *            The endPosition to set.
     */
    public void setEndPosition(int endPosition) {
        this.endPosition = endPosition;
    }

    /**
     * @return Returns the insertPosition.
     */
    public int getInsertPosition() {
        return insertPosition;
    }

    /**
     * @param insertPosition
     *            The insertPosition to set.
     */
    public void setInsertPosition(int insertPosition) {
        this.insertPosition = insertPosition;
    }

    /**
     * @return Returns the startPosition.
     */
    public int getStartPosition() {
        return startPosition;
    }

    /**
     * @param startPosition
     *            The startPosition to set.
     */
    public void setStartPosition(int startPosition) {
        this.startPosition = startPosition;
    }

}
