
#include <gtkmm/object.h>
#include <gtkmm/private/object_p.h>

/* $Id: object.ccg,v 1.23 2002/04/09 17:08:13 daniel Exp $ */

/* 
 *
 * Copyright 1998-1999 The Gtk-- Development Team
 * Copyright 2001      Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtkobject.h>
#include <gtk/gtksignal.h>
#include <glibmm/quark.h>
#include <gtkmm/main.h>

namespace Gtk
{

Object::Object(GtkObject *castitem)
  : Glib::Object((GObject*)castitem)
{

  referenced_ = true; //Not managed.
  gobject_disposed_ = false;

  if(gobject_)
  {
    //Glib::Object::Object has already store a pointer to this C++ instance in the underlying C instance,
    //and connected a callback which will, in turn, call our destroy_notify_(),
    //so that we know when GTK+ disposes of the underlying instance.

    // GTK+ objects are floating, by default. This means that the container widget controls their lifetime.
    // We'll change this:
    g_object_ref(gobject_);  //Increase ref by 1 so that it doesn't get deleted when we sink() it.
    gtk_object_sink((GtkObject*)gobject_); //Stops it from being floating - we will make this optional ( see Gtk::manage() ),
  }
}

Object::~Object()
{
  cpp_destruction_in_progress_ = true;

  // remove our hook.
  GtkObject* object = gobj();
  if (object)
  {
    if (!gobject_disposed_) //We can't do anything with the gobject_ if it's already been disposed.
    {
      if (referenced_) //unref is meaningless for manage()ed (floating) objects.
      {
        g_object_unref(object);
      }

      if (!gobject_disposed_) //If the C instance still isn't dead.
      {
        gtk_object_destroy(object); //Container widgets can respond to this.
      }
    }

    disconnect_cpp_wrapper();
    //If we are killing the C++ instance before the C instance, then this might lead to strange behaviour.
    //If this is a problem, then you'll have to use a managed() object, which will die only upon GTK+'s request.

    //Glib::Object::~Object() will not g_object_unref()ing it too. because gobject_ is now 0.
  }
}

void Object::disconnect_cpp_wrapper()
{
  //Prevent gtk vfuncs and default signal handlers from calling our instance methods:
  g_object_steal_qdata((GObject*)gobj(), Glib::quark_); //It will no longer be possible to get the C++ instance from the C instance.

  //Prevent C++ instance from using GTK+ object:
  gobject_ = 0;

  //TODO: Disconnect any signals, using gtk methods.
  //We'll have to keep a record of all the connections.
}

void Object::destroy_notify_()
{
  //Overriden.

  //Remember that it's been disposed (which only happens once):
  //This also stops us from destroying it again in the destructor when it calls destroy_().
  gobject_disposed_ = true;

  if(!cpp_destruction_in_progress_) //This function might have been called as a side-effect of destroy() when it called gtk_object_destroy().
  {
    if (!referenced_) //If it's manage()ed.
    {
      delete this; //Free the C++ instance.
    }
    else  //It's not managed, but the C gobject_ just died before the C++ instance..
    {
      gobject_ = 0;
    }
  }
}

void Object::destroy_()
{
  //Called from destructors.

  if ( !cpp_destruction_in_progress_ ) //see comment below.
  {
    //Prevent destroy_notify_() from running as a possible side-effect of gtk_object_destroy.
    //We can't predict whether destroy_notify_() will really be run, so we'll disconnect the C++ instance here.
    cpp_destruction_in_progress_ = true;

    //Prevent our *_Class callbacks from accessing the now half-destroyed C++ instance.
    //For instance, even using dynamic_cast<> on the half-destroyed C++ instance could segfault.
    GtkObject* gobject = gobj(); //gobj() won't work after disconnect_cpp_wrapper().
    disconnect_cpp_wrapper();

    //Tell the C instance to unref anything that it has.
    //This might call gtk_object_dispose, and therefore our destroy_notify_(). It might not,
    if(!gobject_disposed_) //Don't destroy it twice.
    {
      gtk_object_destroy(gobject); //cpp_destruction_in_progress prevents destroy_notify_() from deleting this
    }
  }
}

// This should indicate that this was a C++ wrapper generated
// by the translating API and not directly.  
// It must:
//   remove the reference to the object so that object lifetime is not 
//     increased
void Object::set_manage() 
{
  if (!referenced_) return; //It's already managed.

  // tell libsigc that the object is disposable
  SigC::ObjectBase::set_manage();

  // remove our reference
  if (gobject_->ref_count == 1)
  {
    // Cowardly refuse to remove last reference make floating instead
    GTK_OBJECT_SET_FLAGS(gobj(), GTK_FLOATING);
  }
  else
  {
    g_object_unref(gobj());
  }

  referenced_ = false;
}


/*
void Object::set_user_data(gpointer data)
{
  gtk_object_set_user_data(gobj(), data);
}

gpointer Object::get_user_data()
{
  return gtk_object_get_user_data(gobj());
}


void Object::weakref(GtkDestroyNotify notify,gpointer data)
{
  gtk_object_weakref(gobj(),notify,data);
}

void Object::weakunref(GtkDestroyNotify notify,gpointer data)
{
  gtk_object_weakunref(gobj(),notify,data);
}
*/


} // namespace Gtk


namespace
{

const Glib::SignalProxyInfo Object_signal_destroy_info =
{
  "destroy",
  (GCallback) &Glib::SignalProxyNormal::slot0_void_callback,
  (GCallback) &Glib::SignalProxyNormal::slot0_void_callback
};

} // anonymous namespace


namespace Glib
{

Gtk::Object* wrap(GtkObject* object, bool take_copy /* = false */)
{
  return dynamic_cast<Gtk::Object *> (Glib::wrap_auto ((GObject*)(object), take_copy));
}

} /* namespace Glib */

namespace Gtk
{


/* The *_Class implementation: */

GType Object_Class::get_type()
{
  if(!gtype_) // create the GType if necessary
  {
    // TODO: This is currently just optimized away, apparently with no harm.
    // Is it actually needed?
    // Make sure that the parent type has been created.
    CppClassParent::CppObjectType::get_type();

    // Create the wrapper type, with the same class/instance size as the base type.
    register_derived_type(gtk_object_get_type(), (GClassInitFunc) &class_init_function);

    // Add derived versions of interfaces, if the C type implements any interfaces:
  }

  return gtype_;
}

void Object_Class::class_init_function(BaseClassType* klass)
{
  CppClassParent::class_init_function((CppClassParent::BaseClassType*) klass);
  klass->destroy = &destroy_callback;
}


void Object_Class::destroy_callback(GtkObject* self)
{
  CppObjectType *const obj = dynamic_cast<CppObjectType*>(
      Glib::ObjectBase::_get_current_wrapper((GObject*)self));

  if(obj)
  {
    try
    {
      obj->on_destroy();
    }
    catch(...)
    {
      Glib::exception_handlers_invoke();
    }
  }
  else
  {
    BaseClassType *const base = static_cast<BaseClassType*>(
        g_type_class_peek_parent(G_OBJECT_GET_CLASS(self)) // Get the parent class of the object class.
);
    g_assert(base != 0);

    if(base->destroy)
      (*base->destroy)(self);
  }
}


Glib::ObjectBase* Object_Class::wrap_new(GObject* o)
{
  return manage(new Object((GtkObject*)(o)));
}


/* The implementation: */


Object::CppClassType Object::object_class_; //Initialize static member.

GType Object::get_type()
{
  return object_class_.get_type();
}

GType Object::get_base_type()
{
  return gtk_object_get_type();
}


Glib::SignalProxy0<void> Object::signal_destroy()
{
  return Glib::SignalProxy0<void>(this, &Object_signal_destroy_info);
}


Glib::PropertyProxy<void*> Object::property_user_data()
{
  return Glib::PropertyProxy<void*>(this, "user_data");
}


void Gtk::Object::on_destroy()
{
  BaseClassType *const base = static_cast<BaseClassType*>(
      g_type_class_peek_parent(G_OBJECT_GET_CLASS(gobject_)) // Get the parent class of the object class.
);
  g_assert(base != 0);

  if(base->destroy)
    (*base->destroy)(gobj());
}


} // namespace Gtk

