// -*- c++ -*-
#ifndef _GLIBMM_TIMEVAL_H
#define _GLIBMM_TIMEVAL_H

/* $Id: timeval.h,v 1.1 2002/03/12 21:56:32 daniel Exp $ */

/* timeval.h
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glib/gtypes.h>


namespace Glib
{

struct TimeVal : public GTimeVal
{
  inline TimeVal();
  inline TimeVal(long seconds, long microseconds);

  inline TimeVal(const GTimeVal& gtimeval);
  inline TimeVal& operator=(const GTimeVal& gtimeval);

  void assign_current_time();

  void subtract(const TimeVal& rhs);
  void add_milliseconds(long milliseconds);
  void add_microseconds(long microseconds);

  inline bool negative() const;
  inline bool valid() const;
};


inline
TimeVal::TimeVal()
{
  tv_sec  = 0;
  tv_usec = 0;
}

inline
TimeVal::TimeVal(long seconds, long microseconds)
{
  tv_sec  = seconds;
  tv_usec = microseconds;
}

inline
TimeVal::TimeVal(const GTimeVal& gtimeval)
{
  tv_sec  = gtimeval.tv_sec;
  tv_usec = gtimeval.tv_usec;
}

inline
TimeVal& TimeVal::operator=(const GTimeVal& gtimeval)
{
  tv_sec  = gtimeval.tv_sec;
  tv_usec = gtimeval.tv_usec;
  return *this;
}

inline
bool TimeVal::negative() const
{
  return (tv_sec < 0);
}

inline
bool TimeVal::valid() const
{
  return (tv_usec >= 0 && tv_usec < 1000000);
}


/** @relates Glib::TimeVal */
inline
bool operator==(const TimeVal& lhs, const TimeVal& rhs)
{
  return (lhs.tv_sec == rhs.tv_sec && lhs.tv_usec == rhs.tv_usec);
}

/** @relates Glib::TimeVal */
inline
bool operator!=(const TimeVal& lhs, const TimeVal& rhs)
{
  return (lhs.tv_sec != rhs.tv_sec || lhs.tv_usec != rhs.tv_usec);
}

/** @relates Glib::TimeVal */
inline
bool operator<(const TimeVal& lhs, const TimeVal& rhs)
{
  return ((lhs.tv_sec < rhs.tv_sec) ||
          (lhs.tv_sec == rhs.tv_sec && lhs.tv_usec < rhs.tv_usec));
}

/** @relates Glib::TimeVal */
inline
bool operator>(const TimeVal& lhs, const TimeVal& rhs)
{
  return ((lhs.tv_sec > rhs.tv_sec) ||
          (lhs.tv_sec == rhs.tv_sec && lhs.tv_usec > rhs.tv_usec));
}

/** @relates Glib::TimeVal */
inline
bool operator<=(const TimeVal& lhs, const TimeVal& rhs)
{
  return ((lhs.tv_sec < rhs.tv_sec) ||
          (lhs.tv_sec == rhs.tv_sec && lhs.tv_usec <= rhs.tv_usec));
}

/** @relates Glib::TimeVal */
inline
bool operator>=(const TimeVal& lhs, const TimeVal& rhs)
{
  return ((lhs.tv_sec > rhs.tv_sec) ||
          (lhs.tv_sec == rhs.tv_sec && lhs.tv_usec >= rhs.tv_usec));
}

} // namespace Glib


#endif /* _GLIBMM_TIMEVAL_H */

