// -*- c++ -*-
#ifndef _GLIBMM_SPAWN_H
#define _GLIBMM_SPAWN_H

/* $Id: spawn.h,v 1.2 2002/03/16 14:28:04 daniel Exp $ */

/* spawn.h
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glib/gspawn.h>
#include <string>

#include <sigc++/sigc++.h>
#include <glibmm/arrayhandle.h>
#include <glibmm/error.h>

#include <glibmmconfig.h>
GTKMM_USING_STD(string)


namespace Glib
{

/** @defgroup Spawn Spawning Processes
 * Process launching with fork()/exec().
 * @{
 */

/** Exception class for errors occuring when spawning processes.
 */
class SpawnError : public Glib::Error
{
public:
  explicit SpawnError(GError* gobject);
  GSpawnError code() const;
};


void spawn_async_with_pipes(const std::string& working_directory,
                            const Glib::ArrayHandle<std::string>& argv,
                            const Glib::ArrayHandle<std::string>& envp,
                            GSpawnFlags flags = GSpawnFlags(0),
                            const SigC::Slot0<void>& child_setup = SigC::Slot0<void>(),
                            int* child_pid = 0,
                            int* standard_input = 0,
                            int* standard_output = 0,
                            int* standard_error = 0);

void spawn_async_with_pipes(const std::string& working_directory,
                            const Glib::ArrayHandle<std::string>& argv,
                            GSpawnFlags flags = GSpawnFlags(0),
                            const SigC::Slot0<void>& child_setup = SigC::Slot0<void>(),
                            int* child_pid = 0,
                            int* standard_input = 0,
                            int* standard_output = 0,
                            int* standard_error = 0);

void spawn_async(const std::string& working_directory,
                 const Glib::ArrayHandle<std::string>& argv,
                 const Glib::ArrayHandle<std::string>& envp,
                 GSpawnFlags flags = GSpawnFlags(0),
                 const SigC::Slot0<void>& child_setup = SigC::Slot0<void>(),
                 int* child_pid = 0);

void spawn_async(const std::string& working_directory,
                 const Glib::ArrayHandle<std::string>& argv,
                 GSpawnFlags flags = GSpawnFlags(0),
                 const SigC::Slot0<void>& child_setup = SigC::Slot0<void>(),
                 int* child_pid = 0);

void spawn_sync(const std::string& working_directory,
                const Glib::ArrayHandle<std::string>& argv,
                const Glib::ArrayHandle<std::string>& envp,
                GSpawnFlags flags = GSpawnFlags(0),
                const SigC::Slot0<void>& child_setup = SigC::Slot0<void>(),
                std::string* standard_output = 0,
                std::string* standard_error = 0,
                int* exit_status = 0);

void spawn_sync(const std::string& working_directory,
                const Glib::ArrayHandle<std::string>& argv,
                GSpawnFlags flags = GSpawnFlags(0),
                const SigC::Slot0<void>& child_setup = SigC::Slot0<void>(),
                std::string* standard_output = 0,
                std::string* standard_error = 0,
                int* exit_status = 0);

void spawn_command_line_async(const std::string& command_line);

void spawn_command_line_sync(const std::string& command_line,
                             std::string* standard_output = 0,
                             std::string* standard_error = 0,
                             int* exit_status = 0);

/** @} group Spawn */

} // namespace Glib


/** Convenience operator.
 * @ingroup Spawn
 * @return <tt>static_cast<GSpawnFlags>(static_cast<int>(lhs) | static_cast<int>(rhs))</tt>
 */
inline
GSpawnFlags operator|(GSpawnFlags lhs, GSpawnFlags rhs)
{
  return static_cast<GSpawnFlags>(static_cast<int>(lhs) | static_cast<int>(rhs));
}


#endif /* _GLIBMM_SPAWN_H */

