// -*- c++ -*-
#ifndef _GLIBMM_DATE_H
#define _GLIBMM_DATE_H

/* $Id: date.h,v 1.2 2002/03/04 18:34:03 daniel Exp $ */

/* date.h
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

extern "C" { struct tm; }

#include <glib/gdate.h>
#include <glibmm/ustring.h>


namespace Glib
{

/** Julian calendar date.
 */
class Date
{
public:
  Date();
  Date(GDateDay day, GDateMonth month, GDateYear year);
  explicit Date(guint32 julian_day);

#ifndef DOXYGEN_SHOULD_SKIP_THIS
  explicit Date(const GDate& castitem);
#endif

  void clear();
  void set_parse (const Glib::ustring& str);
  void set_time  (GTime time);
  void set_month (GDateMonth month);
  void set_day   (GDateDay day);
  void set_year  (GDateYear year);
  void set_dmy   (GDateDay day, GDateMonth month, GDateYear year);
  void set_julian(guint32 julian_day);

  Date& add_days       (int n_days);
  Date& subtract_days  (int n_days);
  Date& add_months     (int n_months);
  Date& subtract_months(int n_months);
  Date& add_years      (int n_years);
  Date& subtract_years (int n_years);

  int days_between(const Date& rhs) const;
  int compare(const Date& rhs) const;

  Date& clamp(const Date& min_date, const Date& max_date);
  Date& order(Date& other);

  GDateWeekday get_weekday()             const;
  GDateMonth   get_month()               const;
  GDateYear    get_year()                const;
  GDateDay     get_day()                 const;
  guint32      get_julian()              const;
  unsigned int get_day_of_year()         const;
  unsigned int get_monday_week_of_year() const;
  unsigned int get_sunday_week_of_year() const;
  bool         is_first_of_month()       const;
  bool         is_last_of_month()        const;

  static guint8 get_days_in_month(GDateMonth month, GDateYear year);
  static guint8 get_monday_weeks_in_year(GDateYear year);
  static guint8 get_sunday_weeks_in_year(GDateYear year);
  static bool   is_leap_year(GDateYear year);

  /** Convert date to string.
   * @param format A format string as used by @c strftime(), in UTF-8
   *   encoding.  Only date formats are allowed, the result of time formats
   *   is undefined.
   * @return The formatted date string.
   * @throw ConvertError see Glib::ConvertError
   */
  Glib::ustring format_time(const Glib::ustring& format) const;

  void to_struct_tm(struct tm& dest) const;

  bool valid() const;
  static bool valid_day    (GDateDay day);
  static bool valid_month  (GDateMonth month);
  static bool valid_year   (GDateYear year);
  static bool valid_weekday(GDateWeekday weekday);
  static bool valid_julian (guint32 julian_day);
  static bool valid_dmy    (GDateDay day, GDateMonth month, GDateYear year);

#ifndef DOXYGEN_SHOULD_SKIP_THIS
  GDate*       gobj()       { return &gobject_; }
  const GDate* gobj() const { return &gobject_; }
#endif

private:
  GDate gobject_;
};


/** @relates Glib::Date */
inline bool operator==(const Date& lhs, const Date& rhs)
  { return (lhs.compare(rhs) == 0); }

/** @relates Glib::Date */
inline bool operator!=(const Date& lhs, const Date& rhs)
  { return (lhs.compare(rhs) != 0); }

/** @relates Glib::Date */
inline bool operator<(const Date& lhs, const Date& rhs)
  { return (lhs.compare(rhs) < 0); }

/** @relates Glib::Date */
inline bool operator>(const Date& lhs, const Date& rhs)
  { return (lhs.compare(rhs) > 0); }

/** @relates Glib::Date */
inline bool operator<=(const Date& lhs, const Date& rhs)
  { return (lhs.compare(rhs) <= 0); }

/** @relates Glib::Date */
inline bool operator>=(const Date& lhs, const Date& rhs)
  { return (lhs.compare(rhs) >= 0); }

} // namespace Glib


#endif /* _GLIBMM_DATE_H */

