// -*- c++ -*-
#ifndef _GLIBMM_CONTAINERHANDLE_SHARED_H
#define _GLIBMM_CONTAINERHANDLE_SHARED_H

/* $Id: containerhandle_shared.h,v 1.7 2002/03/12 21:56:32 daniel Exp $ */

/* containerhandle_shared.h
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <cstddef>
#include <algorithm>
#include <iterator>
#include <vector>
#include <deque>
#include <list>

#include <glib-object.h>
#include <glib/gmem.h>
#include <glibmm/refptr.h>
#include <glibmm/ustring.h>
#include <glibmm/wrap.h>

#include <glibmmconfig.h>
GTKMM_USING_STD(forward_iterator_tag)
GTKMM_USING_STD(random_access_iterator_tag)
GTKMM_USING_STD(distance)
GTKMM_USING_STD(copy)
GTKMM_USING_STD(vector)
GTKMM_USING_STD(deque)
GTKMM_USING_STD(list)


namespace Glib
{

/** @defgroup ContHandles Generic container converters
 */

/**
 * @ingroup ContHandles
 */
enum OwnershipType
{
  OWNERSHIP_NONE = 0,
  OWNERSHIP_SHALLOW,
  OWNERSHIP_DEEP
};


/** Utility class holding an iterator sequence.
 * @ingroup ContHandles
 * This can be used to initialize a Glib container handle (such as
 * Glib::ArrayHandle) with an iterator sequence.  Use the helper
 * function Glib::sequence() to create a Sequence<> object.
 */
template <class Iterator>
class Sequence
{
private:
  Iterator pbegin_;
  Iterator pend_;

public:
  Sequence(Iterator pbegin, Iterator pend)
    : pbegin_(pbegin), pend_(pend) {}

  Iterator begin() const { return pbegin_; }
  Iterator end()   const { return pend_;   }
  size_t   size()  const { return std::distance(pbegin_, pend_); }
};

/** Helper function to create a Glib::Sequence<> object, which
 * in turn can be used to initialize a container handle.
 * @ingroup ContHandles
 *
 * Usage example:
 * @c combo.set_popdown_strings(Glib::sequence(foo_begin, foo_end));
 */
template <class Iterator> inline
Sequence<Iterator> sequence(Iterator pbegin, Iterator pend)
{
  return Sequence<Iterator>(pbegin, pend);
}


namespace Container_Helpers
{

/** @defgroup ContHelpers Helper classes
 * @ingroup ContHandles
 */

/** Generic TypeTraits implementation.
 * @ingroup ContHelpers
 * This can be used if the C++ type is the same as the C type, or if implicit
 * conversions between the types are available.  Also, the types are required
 * to implement copy-by-value semantics.  (Ownership is just ignored.)
 */
template <class T>
struct TypeTraits
{
  typedef T CppType;
  typedef T CType;
  typedef T CTypeNonConst;

  static CType   to_c_type      (const CppType& item) { return item; }
  static CppType to_cpp_type    (const CType&   item) { return item; }
  static void    release_c_type (const CType&)        {}
};

/** Partial specialization for pointers to GtkObject instances.
 * @ingroup ContHelpers
 */
template <class T>
struct TypeTraits<T*>
{
  typedef T *                          CppType;
  typedef typename T::BaseObjectType * CType;
  typedef typename T::BaseObjectType * CTypeNonConst;

  static CType   to_c_type      (CppType ptr) { return Glib::unwrap(ptr);      }
  static CType   to_c_type      (CType   ptr) { return ptr;                    }
  static CppType to_cpp_type    (CType   ptr) { return Glib::wrap(ptr, false); }
  static void    release_c_type (CType   ptr) { g_object_unref(ptr);           }
};

/** Partial specialization for pointers to const GtkObject instances.
 * @ingroup ContHelpers
 */
template <class T>
struct TypeTraits<const T*>
{
  typedef const T *                          CppType;
  typedef const typename T::BaseObjectType * CType;
  typedef typename T::BaseObjectType *       CTypeNonConst;

  static CType   to_c_type      (CppType ptr) { return Glib::unwrap(ptr);      }
  static CType   to_c_type      (CType   ptr) { return ptr;                    }
  static CppType to_cpp_type    (CType   ptr) { return Glib::wrap(ptr, false); }
  static void    release_c_type (CType   ptr)
    { g_object_unref(const_cast<CTypeNonConst>(ptr)); }
};

/** Partial specialization for pointers to GObject instances.
 * @ingroup ContHelpers
 * The C++ type is always a Glib::RefPtr<>.
 */
template <class T>
struct TypeTraits< Glib::RefPtr<T> >
{
  typedef Glib::RefPtr<T>              CppType;
  typedef typename T::BaseObjectType * CType;
  typedef typename T::BaseObjectType * CTypeNonConst;

  static CType   to_c_type      (const CppType& ptr) { return Glib::unwrap(ptr);     }
  static CType   to_c_type      (CType          ptr) { return ptr;                   }
  static CppType to_cpp_type    (CType          ptr) { return Glib::wrap(ptr, true); }
  static void    release_c_type (CType          ptr) { g_object_unref(ptr);          }
};

/** Partial specialization for pointers to const GObject instances.
 * @ingroup ContHelpers
 * The C++ type is always a Glib::RefPtr<>.
 */
template <class T>
struct TypeTraits< Glib::RefPtr<const T> >
{
  typedef Glib::RefPtr<const T>              CppType;
  typedef const typename T::BaseObjectType * CType;
  typedef typename T::BaseObjectType *       CTypeNonConst;

  static CType   to_c_type      (const CppType& ptr) { return Glib::unwrap(ptr);     }
  static CType   to_c_type      (CType          ptr) { return ptr;                   }
  static CppType to_cpp_type    (CType          ptr) { return Glib::wrap(ptr, true); }
  static void    release_c_type (CType          ptr)
    { g_object_unref(const_cast<CTypeNonConst>(ptr)); }
};

/** Specialization for UTF-8 strings.
 * @ingroup ContHelpers
 * When converting from C++ to C, Glib::ustring will be accepted as well as
 * std::string and 'const char*'.  However, when converting to the C++ side,
 * the output type cannot be 'const char*'.
 */
template <>
struct TypeTraits<Glib::ustring>
{
  typedef Glib::ustring CppType;
  typedef const char *  CType;
  typedef char *        CTypeNonConst;

  static CType to_c_type (const Glib::ustring& str) { return str.c_str(); }
  static CType to_c_type (const std::string&   str) { return str.c_str(); }
  static CType to_c_type (CType                str) { return str;         }

  static CppType to_cpp_type(CType str)
    { return (str) ? Glib::ustring(str) : Glib::ustring(); }

  static void release_c_type(CType str)
    { g_free(const_cast<CTypeNonConst>(str)); }
};

/** Specialization for std::string.
 * @ingroup ContHelpers
 * When converting from C++ to C, std::string will be accepted as well as
 * 'const char*'.  However, when converting to the C++ side, the output type
 * cannot be 'const char*'.
 */
template <>
struct TypeTraits<std::string>
{
  typedef std::string   CppType;
  typedef const char *  CType;
  typedef char *        CTypeNonConst;

  static CType to_c_type (const std::string&   str) { return str.c_str(); }
  static CType to_c_type (const Glib::ustring& str) { return str.c_str(); }
  static CType to_c_type (CType                str) { return str;         }

  static CppType to_cpp_type(CType str)
    { return (str) ? std::string(str) : std::string(); }

  static void release_c_type(CType str)
    { g_free(const_cast<CTypeNonConst>(str)); }
};

} // namespace Container_Helpers

} // namespace Glib


#endif /* _GLIBMM_CONTAINERHANDLE_SHARED_H */

