/* icecast-method.c - Icecast streaming servers access method for the GNOME
   Virtual File System.

   Copyright (C) 1999-2001 Free Software Foundation

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Authors: 
        Bastien Nocera <hadess@hadess.net>
   	Ettore Perazzoli <ettore@comm2000.it>
   	Pavel Cisler <pavel@eazel.com>

   Based on XMMS, Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas,
        Thomas Nilsson and 4Front Technologies

   FIXME support proxies
 */

/* Define DEBUG to get more info on what it's doing */
#undef DEBUG

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include <libgnomevfs/gnome-vfs-mime.h>

#include <libgnomevfs/gnome-vfs-cancellation.h>
#include <libgnomevfs/gnome-vfs-context.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>

#ifdef DEBUG
#define ICE(x...) g_print(## x)
#else
#define ICE(x...)
#endif

/* Authentication stuff */

#define min(x,y) ((x)<(y)?(x):(y))
#define min3(x,y,z) (min(x,y)<(z)?min(x,y):(z))
#define min4(x,y,z,w) (min3(x,y,z)<(w)?min3(x,y,z):(w))

#define BASE64_LENGTH(len) (4 * (((len) + 2) / 3))

/* Encode the string S of length LENGTH to base64 format and place it
 * to STORE.  STORE will be 0-terminated, and must point to a writable
 * buffer of at least 1+BASE64_LENGTH(length) bytes.  */
static void base64_encode (const gchar *s, gchar *store, gint length)
{
	/* Conversion table.  */
	static gchar tbl[64] = {
		'A','B','C','D','E','F','G','H',
		'I','J','K','L','M','N','O','P',
		'Q','R','S','T','U','V','W','X',
		'Y','Z','a','b','c','d','e','f',
		'g','h','i','j','k','l','m','n',
		'o','p','q','r','s','t','u','v',
		'w','x','y','z','0','1','2','3',
		'4','5','6','7','8','9','+','/'
	};
	gint i;
	guchar *p = (guchar *)store;

	/* Transform the 3x8 bits to 4x6 bits, as required by base64.  */
	for (i = 0; i < length; i += 3)
	{
		*p++ = tbl[s[0] >> 2];
		*p++ = tbl[((s[0] & 3) << 4) + (s[1] >> 4)];
		*p++ = tbl[((s[1] & 0xf) << 2) + (s[2] >> 6)];
		*p++ = tbl[s[2] & 0x3f];
		s += 3;
	}
	/* Pad the result if necessary...  */
	if (i == length + 1)
		*(p - 1) = '=';
	else if (i == length + 2)
		*(p - 1) = *(p - 2) = '=';
	/* ...and zero-terminate it.  */
	*p = '\0';
}

/* Create the authentication header contents for the `Basic' scheme.
 * This is done by encoding the string `USER:PASS' in base64 and
 * prepending `HEADER: Basic ' to it.  */
static gchar *basic_authentication_encode (const gchar *user, const gchar *passwd, const gchar *header)
{
	gchar *t1, *t2, *res;
	gint len1 = strlen (user) + 1 + strlen (passwd);
	gint len2 = BASE64_LENGTH (len1);

	t1 = g_strdup_printf("%s:%s", user, passwd);
	t2 = g_malloc0(len2 + 1);
	base64_encode (t1, t2, len1);
	res = g_strdup_printf("%s: Basic %s\r\n", header, t2);
	g_free(t2);
	g_free(t1);

	return res;
}

/* Parses a URL of the type icecast://user:password@host:port/filename */

static void parse_url(const gchar * url, gchar ** user, gchar ** pass,
		gchar ** host, int *port, gchar ** filename)
{
	gchar *h, *p, *pt, *f, *temp, *ptr;

	temp = g_strdup(url);
	ptr = temp;

	/* remove the icecast:// if it's not already done */
	if (!strncasecmp("icecast://", ptr, 11))
		ptr += 11;

	/* find the username/password and the host */
	if ((h = strchr(ptr, '@')))
	{
		*h = '\0';
		p = strchr(ptr, ':');
		if (p != NULL && p < h)
		{
			*p = '\0';
			p++;
			*pass = g_strdup(p);
		} else {
			*pass = NULL;
		}

		*user = g_strdup(ptr);
		h++;
		ptr = h;
	} else {
		*user = NULL;
		*pass = NULL;
		h = ptr;
	}

	/* find the port */
	pt = strchr(ptr, ':');
	f = strchr(ptr, '/');
	if (pt != NULL && (f == NULL || pt < f))
	{
		*pt = '\0';
		*port = atoi(pt + 1);
	} else {
		if (f) *f = '\0';
		*port = 80;
	}

	*host = g_strdup(h);

	/* find the filename */
	if (f)
		*filename = g_strdup(f + 1);
	else
		*filename = NULL;

	g_free(temp);
}

static gboolean http_check_for_data(gint sock)
{
	fd_set set;
	struct timeval tv;
	gint ret;

	tv.tv_sec = 0;
	tv.tv_usec = 20000;
	FD_ZERO(&set);
	FD_SET(sock, &set);
	ret = select(sock + 1, &set, NULL, NULL, &tv);
	if (ret > 0)
		return TRUE;
	return FALSE;
}

static gint mpg123_http_read_line(gchar * buf, gint size, gint sock)
{
	gint i = 0;

	while (i < size - 1)
	{
		if (http_check_for_data(sock))
		{
			if (read(sock, buf + i, 1) <= 0)
				return -1;
			if (buf[i] == '\n')
				break;
			if (buf[i] != '\r')
				i++;
		}
	}
	buf[i] = '\0';
	return i;
}

typedef struct {
	GnomeVFSURI *uri;
	gint fd;
} FileHandle;

static FileHandle *
file_handle_new (GnomeVFSURI *uri,
		 gint fd)
{
	FileHandle *result;
	result = g_new (FileHandle, 1);

	result->uri = gnome_vfs_uri_ref (uri);
	result->fd = fd;

	return result;
}

static void
file_handle_destroy (FileHandle *handle)
{
	gnome_vfs_uri_unref (handle->uri);
	g_free (handle);
}

/* We don't care about the mode */

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	GnomeVFSToplevelURI *topuri;
	FileHandle *file_handle;
	gchar line[1024], *temp;
	gchar *url = NULL, *user, *pass, *host, *file;
	gchar *auth = NULL, *proxy_auth = NULL;
	gint cnt, error, err_len, port;
	gint sock;
	gboolean redirect;
	fd_set set;
	struct hostent *hp;
	struct sockaddr_in address;
	struct timeval tv;

	_GNOME_VFS_METHOD_PARAM_CHECK (method_handle != NULL);
	_GNOME_VFS_METHOD_PARAM_CHECK (uri != NULL);

	topuri = (GnomeVFSToplevelURI *)uri;

	if (topuri->host_name == NULL)
		return GNOME_VFS_ERROR_INVALID_HOST_NAME;

	user = g_strdup(topuri->user_name);
	pass = g_strdup(topuri->password);
	host = g_strdup(topuri->host_name);
	port = topuri->host_port;
	file = g_strdup(uri->text);

	do {
		redirect=FALSE;

		if (url != NULL)
		{
			g_strstrip(url);
			parse_url(url, &user, &pass, &host, &port, &file);
			g_free(url);
		}
		ICE("user: %s, pass: %s, host: %s, port: %d, filename: %s\n",
			user, pass, host, port, file);

		// FIXME support proxies
		//chost = mpg123_cfg.use_proxy ? mpg123_cfg.proxy_host : host;
		//cport = mpg123_cfg.use_proxy ? mpg123_cfg.proxy_port : port;
		//chost = host;
		//cport = port;

		sock = socket(AF_INET, SOCK_STREAM, 0);
		//fcntl(sock, F_SETFL, O_NONBLOCK);
		address.sin_family = AF_INET;

		/* Lookup Host */
		ICE("Looking up %s\n", host);

		if (!(hp = gethostbyname(host)))
		{
			ICE("Couldn't look up host %s\n", host);

			return GNOME_VFS_ERROR_HOST_NOT_FOUND;
		}

		memcpy(&address.sin_addr.s_addr, *(hp->h_addr_list),
				sizeof (address.sin_addr.s_addr));
		address.sin_port = g_htons(port);

		/* Connect to host */
		ICE("Connecting to %s:%d\n", host, port);

		if (connect(sock, (struct sockaddr *) &address,
					sizeof (struct sockaddr_in)) == -1)
		{
			if (errno != EINPROGRESS)
			{
				ICE("Couldn't connect to host %s\n", host);

				return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
			}
		}

		while (!gnome_vfs_context_check_cancellation(context))
		{
			tv.tv_sec = 0;
			tv.tv_usec = 10000;
			FD_ZERO(&set);
			FD_SET(sock, &set);

			if (select(sock + 1, NULL, &set, NULL, &tv) > 0)
			{
				err_len = sizeof (error);
				getsockopt(sock, SOL_SOCKET, SO_ERROR, &error, &err_len);
				if (error)
				{
					ICE("Couldn't connect to host %s\n", host);

					return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
				}
				break;
			}
		}

		if(user && pass)
			auth = basic_authentication_encode(user, pass,
					"Authorization");
//FIXME proxy settings
#if 0
		if (use_proxy)
		{
			file = g_strdup(url);
			if (use_auth && proxy_user && proxy_pass)
			{
				proxy_auth = basic_authentication_encode
					(proxy_user, proxy_pass,
					 "Proxy-Authorization");
			}
		}
#endif

		temp = g_strdup_printf("GET %s HTTP/1.0\r\n"
				"Host: %s\r\n"
				"User-Agent: RhythmBox/1.0\r\n"
				"%s%s\r\n",
				file, host,
				"", "");
				/* FIXME proxy_auth ? proxy_auth : "", auth ? auth : "");*/
		if(proxy_auth)
			g_free(proxy_auth);
		if(auth)
			g_free(auth);
		write(sock, temp, strlen(temp));
		g_free(temp);

		ICE("Connected: Waiting for reply\n");

		/* here comes the tricky part
		 * first read out of the server and status error check */
		while (!gnome_vfs_context_check_cancellation(context))
		{
			if (http_check_for_data(sock))
			{
				if (mpg123_http_read_line(line, 1024, sock))
				{
					gchar *status;

					status = strchr(line, ' ');
					if (status)
					{
						if (status[1] == '2')
							break;
						if(status[1] == '3' && status[2] == '0' && status[3] == '2')
							while(!gnome_vfs_context_check_cancellation(context))
							{
								if(http_check_for_data(sock))
								{
									if((cnt = mpg123_http_read_line(line, 1024, sock)) != -1)
									{
										if(!cnt)
											break;
										if(!strncmp(line, "Location:", 9))
										{
											url = g_strdup(line+10);
										}
									} else {
										ICE("Couldn't connect to host %s\nServer reported: %s\n", host, status);

										return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
									}
								}
							}
						redirect=TRUE;
						break;
					} else {
						ICE("Couldn't connect to host %s\nServer reported: %s\n", host, status);

						return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
					}
				} else {
					ICE("Couldn't connect to host %s\nServer\n", host);

					return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
				}
			}
		}

		/* check out the headers */
		while (!redirect)
		{
			if (http_check_for_data(sock))
			{
				if ((cnt = mpg123_http_read_line(line, 1024, sock)) != -1)
				{
					ICE("line: %s\n", line);

					if (!cnt)
						break;
#if 0
					if (!strncmp(line, "icy-name:", 9))
						icy_name = g_strdup(line + 9);
					else if (!strncmp(line, "x-audiocast-name:", 17))
						icy_name = g_strdup(line + 17);
					if (!strncmp(line, "icy-metaint:", 12))
						icy_metaint = atoi(line + 12);
#endif
				} else {
					return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
				}
			}
		}
		
		ICE("Checked the headers\n");

		if (redirect)
		{
			ICE("We have a redirect\n");

			close(sock);
			g_free(user);
			g_free(pass);
			g_free(host);
			g_free(file);
		}
	} while (redirect);

	ICE("Done opening socket #%d\n", sock);

	g_free(user);
	g_free(pass);
	g_free(host);
	g_free(file);

	file_handle = file_handle_new (uri, sock);
	*method_handle = (GnomeVFSMethodHandle *) file_handle;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)
{
	return GNOME_VFS_ERROR_NOT_PERMITTED;
}

static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)
{
	FileHandle *file_handle;
	gint close_retval;

	g_return_val_if_fail (method_handle != NULL, GNOME_VFS_ERROR_INTERNAL);

	file_handle = (FileHandle *) method_handle;

	do
		close_retval = close (file_handle->fd);
	while (close_retval != 0
	       && errno == EINTR
	       && ! gnome_vfs_context_check_cancellation (context));

	/* FIXME bugzilla.eazel.com 1163: Should do this even after a failure?  */
	file_handle_destroy (file_handle);

	if (close_retval != 0) {
		return gnome_vfs_result_from_errno ();
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	FileHandle *file_handle;
	gint read_val;

	ICE("vfs reading\n");

	g_return_val_if_fail (method_handle != NULL, GNOME_VFS_ERROR_INTERNAL);

	file_handle = (FileHandle *) method_handle;

	do {
		read_val = read(file_handle->fd, buffer, num_bytes);
	} while (!num_bytes);

	ICE("vfs reading socket #%d\n", file_handle->fd);

	if (read_val == -1) {
		*bytes_read = 0;
		return gnome_vfs_result_from_errno ();
	} else {
		*bytes_read = read_val;
		/* Getting 0 from read() means EOF! */

		if (read_val == 0) {
			return GNOME_VFS_ERROR_EOF;
		}
	}

#if 0
	do {
		read_val = read (file_handle->fd, buffer, num_bytes);
		if (num_bytes == 0)
			usleep(100);
	}
	while (read_val == -1
	       && errno == EINTR
	       && ! gnome_vfs_context_check_cancellation (context)
	       && num_bytes == 0);

	if (read_val == -1) {
		*bytes_read = 0;
		return gnome_vfs_result_from_errno ();
	} else {
		*bytes_read = read_val;

		/* Getting 0 from read() means EOF! */
		if (read_val == 0) {
			return GNOME_VFS_ERROR_EOF;
		}
	}
#endif
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)
{
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)
{
	g_return_val_if_fail (uri != NULL, FALSE);

	/* We are always on a remote filesystem.  */
	return FALSE;
}

static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open,
	do_create,		/* returns GNOME_VFS_ERROR_NOT_PERMITTED */
	do_close,
	do_read,
	NULL,			/* do_write */
	NULL,			/* do_seek */
	NULL, 			/* do_tell */
	NULL, 			/* do_truncate_handle */
	NULL, 			/* do_open_directory */
	NULL,			/* do_close_directory */
	NULL,			/* do_read_directory */
	do_get_file_info,	/* not supported */
	NULL,			/* do_get_file_info_from_handle */
	do_is_local,
	NULL,			/* do_make_directory */
	NULL,			/* do_remove_directory */
	NULL,			/* do_move */
	NULL,			/* do_unlink */
	NULL,			/** do_check_same_fs */
	NULL,			/* do_set_file_info */
	NULL,			/* do_truncate */
	NULL,			/* do_find_directory */
	NULL,			/* do_create_symbolic_link */
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	return &method;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
}
