/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <X11/keysym.h>

#include <gdk/gdkx.h>
#include <gnome.h>
#include <libgnome/gnome-util.h>

#include <libxklavier/xklavier_config.h>

#include "gswitchit_config.h"

#include "switchcuts.h"

#define CONFIG_PREFIX                       "/apps/" PACKAGE

#define CONFIG_GENERAL_PREFIX               CONFIG_PREFIX         "/General"

static const char CONFIG_KEY_BEEP[] = CONFIG_GENERAL_PREFIX "/beep";
static const char CONFIG_KEY_DEFAULT_GROUP[] =
  CONFIG_GENERAL_PREFIX "/defaultGroup";
static const char CONFIG_KEY_GROUP_PER_WINDOW[] =
  CONFIG_GENERAL_PREFIX "/groupPerWindow";
static const char CONFIG_KEY_HANDLE_INDICATORS[] =
  CONFIG_GENERAL_PREFIX "/handleIndicators";
static const char CONFIG_KEY_LAYOUT_NAMES_AS_GROUP_NAMES[] =
  CONFIG_GENERAL_PREFIX "/layoutNamesAsGroupNames";
static const char CONFIG_KEY_DEBUG_LEVEL[] =
  CONFIG_GENERAL_PREFIX "/debugLevel";
static const char CONFIG_KEY_SECONDARIES[] =
  CONFIG_GENERAL_PREFIX "/secondary";

static const char *appletConfigVars[] = {
  CONFIG_KEY_BEEP,
  CONFIG_KEY_DEFAULT_GROUP,
  CONFIG_KEY_GROUP_PER_WINDOW,
  CONFIG_KEY_HANDLE_INDICATORS,
  CONFIG_KEY_LAYOUT_NAMES_AS_GROUP_NAMES,
  CONFIG_KEY_DEBUG_LEVEL,
  CONFIG_KEY_SECONDARIES,
  NULL
};

static int gconfListenerIds[( sizeof( appletConfigVars ) /
                              sizeof( appletConfigVars[0] ) ) - 1 +
                            ( 2 * XkbNumKbdGroups )];

#define CONFIG_XKB_PREFIX                   CONFIG_PREFIX     "/XKB"
static const char CONFIG_KEY_XKB_OVERRIDE_SETTINGS[] =
  CONFIG_XKB_PREFIX "/overrideSettings";
static const char CONFIG_KEY_XKB_MODEL[] = CONFIG_XKB_PREFIX "/model";
static const char CONFIG_KEY_XKB_LAYOUTS[] = CONFIG_XKB_PREFIX "/layouts";
static const char CONFIG_KEY_XKB_OPTIONS[] = CONFIG_XKB_PREFIX "/options";
static const char CONFIG_KEY_XKB_SWITCHCUT_ID[] =
  CONFIG_XKB_PREFIX "/switchcutId";

static const char CONFIG_KEY_IMAGES_FMT[] = CONFIG_PREFIX "/Images/%d";
static const char CONFIG_KEY_COMMANDS_FMT[] = CONFIG_PREFIX "/Commands/%d";

const Switchcut switchcuts[] = {
#include "switchcuts.inc"
};

int total_switchcuts = sizeof( switchcuts ) / sizeof( switchcuts[0] );

static char *defaultFiles[XkbNumKbdGroups] = {
  PACKAGE "/us.xpm",
  PACKAGE "/ru.xpm",
  PACKAGE "/III.xpm",
  PACKAGE "/IV.xpm",
};

static void _GSwitchItConfigXkbOptionsAdd( GSwitchItConfig * config,
                                           const gchar * fullOptionName )
{
  config->xkbOptions =
    g_slist_append( config->xkbOptions, g_strdup( fullOptionName ) );
}

static void _GSwitchItConfigXkbLayoutsAdd( GSwitchItConfig * config,
                                           const gchar * fullLayoutName )
{
  config->xkbLayouts =
    g_slist_append( config->xkbLayouts, g_strdup( fullLayoutName ) );
}

static void _GSwitchItConfigXkbListReset( GSList ** plist )
{
  while( *plist != NULL )
  {
    GSList *p = *plist;
    *plist = ( *plist )->next;
    g_free( p->data );
    g_slist_free_1( p );
  }
}

static void _GSwitchItConfigXkbDoWithSettings( GSwitchItConfig * config,
                                               gboolean activate,
                                               const char *psFileName )
{
  int i;
  XkbDescModifierFunc fun = NULL;
  void *userData = NULL;

  XklConfigRec data;
  XklConfigRecInit( &data );
  data.model = strdup( config->xkbModel );

  data.numLayouts = data.numVariants =
    ( config->xkbLayouts == NULL ) ? 0 : g_slist_length( config->xkbLayouts );
  data.numOptions =
    ( config->xkbOptions == NULL ) ? 0 : g_slist_length( config->xkbOptions );

  XklDebug( 150, "Taking %d layouts\n", data.numLayouts );
  if( data.numLayouts != 0 )
  {
    GSList *theLayout = config->xkbLayouts;
    char **p1 = data.layouts =
      g_malloc0( ( sizeof( char * ) ) * data.numLayouts );
    char **p2 = data.variants =
      g_malloc0( ( sizeof( char * ) ) * data.numVariants );
    for( i = data.numLayouts; --i >= 0; )
    {
      char *layout, *variant;
      if( GSwitchItConfigXkbSplitItems
          ( theLayout->data, &layout, &variant ) && variant != NULL )
      {
        *p1 = g_strdup( layout );
        *p2 = g_strdup( variant );
      } else
      {
        *p1 = g_strdup( theLayout->data );
        *p2 = NULL;
      }
      XklDebug( 150, "Adding [%s]/%p and [%s]/%p\n", *p1, *p1, *p2, *p2 );
      p1++;
      p2++;
      theLayout = theLayout->next;
    }
  }

  if( data.numOptions != 0 )
  {
    GSList *theOption = config->xkbOptions;
    char **p = data.options =
      g_malloc0( ( sizeof( char * ) ) * data.numOptions );
    for( i = data.numOptions; --i >= 0; )
    {
      char *group, *option;
      if( GSwitchItConfigXkbSplitItems
          ( theOption->data, &group, &option ) && option != NULL )
        *( p++ ) = g_strdup( option );
      else
        XklDebug( 150, "Could not split [%s]\n", theOption->data );
      theOption = theOption->next;
    }
  }

  if( config->xkbOverrideSettings &&
      !XklMultipleLayoutsSupported(  ) &&
      ( config->xkbSwitchcutId >= 0 ) &&
      ( config->xkbSwitchcutId < total_switchcuts ) )
  {
    const Switchcut *sc = switchcuts + config->xkbSwitchcutId;
    fun = sc->fun;
    userData = sc->userData;
  }

  if( activate )
  {
    XklConfigActivate( &data, fun, userData );
  } else
  {
    char *home = getenv( "HOME" );
    char xkmFileName[PATH_MAX];
    char cmd[PATH_MAX * 2 + 20];
    int status;
    int writtenOk;
    g_snprintf( xkmFileName, sizeof( xkmFileName ),
                "%s/.gnome_private/xkbpreview.xkm", home );
    writtenOk = XklConfigWriteXKMFile( xkmFileName, &data, fun, userData );
    if( writtenOk )
    {
      g_snprintf( cmd, sizeof( cmd ),
                  "xkbprint %s %s", xkmFileName, psFileName );
      status = system( cmd );
      XklDebug( 100, "Res: [%d]\n", status );
      //unlink( xkmFileName );
    } else
    {
      XklDebug( 10, "Could not create XKM file!" );
    }
  }
  XklConfigRecDestroy( &data );
}

static Bool _GSwitchItConfigGetDescriptions( const char *layoutName,
                                             const char *variantName,
                                             char **layoutDescr,
                                             char **variantDescr )
{
  static XklConfigItem litem;
  static XklConfigItem vitem;

  layoutName = g_strdup( layoutName );

  g_snprintf( litem.name, sizeof litem.name, "%s", layoutName );
  *layoutDescr = XklConfigFindLayout( &litem ) ? litem.description : NULL;

  if( variantName != NULL )
  {
    variantName = g_strdup( variantName );
    g_snprintf( vitem.name, sizeof vitem.name, "%s", variantName );
    *variantDescr =
      XklConfigFindVariant( layoutName, &vitem ) ? vitem.description : NULL;
    g_free( ( char * ) variantName );
  }
  g_free( ( char * ) layoutName );
  return *layoutDescr != NULL;
}

void GSwitchItConfigFreeImages( GSwitchItConfig * config )
{
  int i;
  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    if( config->images[i] )
    {
      gdk_pixbuf_unref( config->images[i] );
      config->images[i] = NULL;
    }
  }
}

void GSwitchItConfigLoadImages( GSwitchItConfig * config )
{
  int i;

  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    if( config->imageFiles[i] != NULL )
    {
      GError *err = NULL;
      config->images[i] =
        gdk_pixbuf_new_from_file( config->imageFiles[i], &err );
      if( config->images[i] == NULL )
      {
        gnome_error_dialog( err->message );
        g_error_free( err );
      }
      XklDebug( 150,
                "Image %d[%s] loaded -> %p[%dx%d]\n",
                i, config->imageFiles[i], config->images[i],
                gdk_pixbuf_get_width( config->images[i] ),
                gdk_pixbuf_get_height( config->images[i] ) );
    }
  }
}

void GSwitchItConfigInit( GSwitchItConfig * config )
{
  GError *gerror = NULL;

  config->confClient = gconf_client_get_default(  );

  gconf_client_add_dir( config->confClient,
                        CONFIG_GENERAL_PREFIX,
                        GCONF_CLIENT_PRELOAD_NONE, &gerror );
  if( gerror != NULL )
  {
    g_error( "err1:%s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }
  gconf_client_add_dir( config->confClient,
                        CONFIG_PREFIX "/Images",
                        GCONF_CLIENT_PRELOAD_NONE, &gerror );
  if( gerror != NULL )
  {
    g_error( "err2:%s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }

  gconf_client_add_dir( config->confClient,
                        CONFIG_XKB_PREFIX,
                        GCONF_CLIENT_PRELOAD_NONE, &gerror );
  if( gerror != NULL )
  {
    g_error( "err3:%s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }
}

void GSwitchItConfigTerm( GSwitchItConfig * config )
{
  int i;
  char **pi, **pc;

  GSwitchItConfigFreeImages( config );

  pi = config->imageFiles;
  pc = config->commands;

  for( i = XkbNumKbdGroups; --i >= 0; pi++, pc++ )
  {
    if( *pi )
    {
      g_free( *pi );
      *pi = NULL;
    }
    if( *pc )
    {
      g_free( *pc );
      *pc = NULL;
    }
  }

  GSwitchItConfigXkbModelSet( config, NULL );

  GSwitchItConfigXkbLayoutsReset( config );

  g_object_unref( config->confClient );
  config->confClient = NULL;
}

void GSwitchItConfigLoadAppletParams( GSwitchItConfig * config )
{
  int i;
  GError *gerror = NULL;

  // optimize me
  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    gchar szi[( sizeof CONFIG_KEY_IMAGES_FMT ) + 3];
    gchar szc[( sizeof CONFIG_KEY_COMMANDS_FMT ) + 3];
    char **pi = config->imageFiles + i;
    char **pc = config->commands + i, *p;

    if( *pi )
      g_free( *pi );
    g_snprintf( szi, sizeof( szi ), CONFIG_KEY_IMAGES_FMT, i );
    *pi = gconf_client_get_string( config->confClient, szi, &gerror );
    if( *pi == NULL || gerror != NULL )
    {
      if( gerror != NULL )
      {
        g_warning( "Error reading configuration:%s\n", gerror->message );
        g_error_free( gerror );
        gerror = NULL;
      }
      // just take default value
      *pi = gnome_program_locate_file( NULL,
                                       GNOME_FILE_DOMAIN_PIXMAP,
                                       defaultFiles[i], FALSE, NULL );
    }

    if( *pc )
      g_free( *pc );
    g_snprintf( szc, sizeof( szc ), CONFIG_KEY_COMMANDS_FMT, i );
    *pc = gconf_client_get_string( config->confClient, szc, &gerror );
    if( *pc == NULL || gerror != NULL )
    {
      if( gerror != NULL )
      {
        g_warning( "Error reading configuration:%s\n", gerror->message );
        g_error_free( gerror );
        gerror = NULL;
      }
      *pc = NULL;
    }
  }

  config->secondaryGroupsMask = gconf_client_get_int( config->confClient,
                                                      CONFIG_KEY_SECONDARIES,
                                                      &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->secondaryGroupsMask = 0;
    g_error_free( gerror );
    gerror = NULL;
  }

  config->doBeep = gconf_client_get_bool( config->confClient,
                                          CONFIG_KEY_BEEP, &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->doBeep = FALSE;
    g_error_free( gerror );
    gerror = NULL;
  }

  config->groupPerApp = gconf_client_get_bool( config->confClient,
                                               CONFIG_KEY_GROUP_PER_WINDOW,
                                               &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->groupPerApp = FALSE;
    g_error_free( gerror );
    gerror = NULL;
  }

  config->handleIndicators = gconf_client_get_bool( config->confClient,
                                                    CONFIG_KEY_HANDLE_INDICATORS,
                                                    &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->handleIndicators = FALSE;
    g_error_free( gerror );
    gerror = NULL;
  }

  config->layoutNamesAsGroupNames = gconf_client_get_bool( config->confClient,
                                                           CONFIG_KEY_LAYOUT_NAMES_AS_GROUP_NAMES,
                                                           &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->layoutNamesAsGroupNames = TRUE;
    g_error_free( gerror );
    gerror = NULL;
  }

  config->debugLevel = gconf_client_get_int( config->confClient,
                                             CONFIG_KEY_DEBUG_LEVEL,
                                             &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->debugLevel = -1;
    g_error_free( gerror );
    gerror = NULL;
  }

  config->defaultGroup = gconf_client_get_int( config->confClient,
                                               CONFIG_KEY_DEFAULT_GROUP,
                                               &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->defaultGroup = -1;
    g_error_free( gerror );
    gerror = NULL;
  }

  if( config->defaultGroup < -1 || config->defaultGroup >= XkbNumKbdGroups )
    config->defaultGroup = -1;

  GSwitchItConfigFreeImages( config );
  GSwitchItConfigLoadImages( config );
}

void GSwitchItConfigSaveAppletParams( GSwitchItConfig * config )
{
  int i;
  GConfChangeSet *cs;
  GError *gerror = NULL;
  cs = gconf_change_set_new(  );

  //!! optimize me
  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    char **pi = config->imageFiles + i;
    char **pc = config->commands + i;
    if( *pi != NULL )
    {
      gchar buf[( sizeof CONFIG_KEY_IMAGES_FMT ) + 3];
      g_snprintf( buf, sizeof( buf ), CONFIG_KEY_IMAGES_FMT, i );
      gconf_change_set_set_string( cs, buf, *pi );
    }
    if( *pc != NULL )
    {
      gchar buf[( sizeof CONFIG_KEY_COMMANDS_FMT ) + 3];
      g_snprintf( buf, sizeof( buf ), CONFIG_KEY_COMMANDS_FMT, i );
      gconf_change_set_set_string( cs, buf, *pc );
    }
  }

  gconf_change_set_set_int( cs, CONFIG_KEY_SECONDARIES,
                            config->secondaryGroupsMask );
  gconf_change_set_set_bool( cs, CONFIG_KEY_BEEP, config->doBeep );
  gconf_change_set_set_bool( cs, CONFIG_KEY_GROUP_PER_WINDOW,
                             config->groupPerApp );
  gconf_change_set_set_bool( cs, CONFIG_KEY_HANDLE_INDICATORS,
                             config->handleIndicators );
  gconf_change_set_set_bool( cs, CONFIG_KEY_LAYOUT_NAMES_AS_GROUP_NAMES,
                             config->layoutNamesAsGroupNames );
  gconf_change_set_set_int( cs, CONFIG_KEY_DEBUG_LEVEL, config->debugLevel );
  gconf_change_set_set_int( cs, CONFIG_KEY_DEFAULT_GROUP,
                            config->defaultGroup );

  gconf_client_commit_change_set( config->confClient, cs, TRUE, &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error saving configuration: %s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }
  gconf_change_set_unref( cs );
}

void GSwitchItConfigLoadXkbParams( GSwitchItConfig * config )
{
  GError *gerror = NULL;
  gchar *pc;
  GSList *pl;

  config->xkbOverrideSettings = gconf_client_get_bool( config->confClient,
                                                       CONFIG_KEY_XKB_OVERRIDE_SETTINGS,
                                                       &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->xkbOverrideSettings = FALSE;
    g_error_free( gerror );
    gerror = NULL;
  }
  XklDebug( 150, "Loaded XKB override cmd: [%s]\n",
            config->xkbOverrideSettings ? "true" : "false" );

  pc =
    gconf_client_get_string( config->confClient, CONFIG_KEY_XKB_MODEL,
                             &gerror );
  if( pc == NULL || gerror != NULL )
  {
    if( gerror != NULL )
    {
      g_warning( "Error reading configuration:%s\n", gerror->message );
      g_error_free( gerror );
      gerror = NULL;
    }
    GSwitchItConfigXkbModelSet( config, NULL );
  } else
  {
    GSwitchItConfigXkbModelSet( config, pc );
  }
  XklDebug( 150, "Loaded XKB model: [%s]\n", config->xkbModel );

  GSwitchItConfigXkbLayoutsReset( config );

  pl =
    gconf_client_get_list( config->confClient,
                           CONFIG_KEY_XKB_LAYOUTS,
                           GCONF_VALUE_STRING, &gerror );
  if( pl == NULL || gerror != NULL )
  {
    if( gerror != NULL )
    {
      g_warning( "Error reading configuration:%s\n", gerror->message );
      g_error_free( gerror );
      gerror = NULL;
    }
  }

  while( pl != NULL )
  {
    XklDebug( 150, "Loaded XKB layout: [%s]\n", pl->data );
    _GSwitchItConfigXkbLayoutsAdd( config, pl->data );
    pl = pl->next;
  }

  GSwitchItConfigXkbOptionsReset( config );

  pl =
    gconf_client_get_list( config->confClient,
                           CONFIG_KEY_XKB_OPTIONS,
                           GCONF_VALUE_STRING, &gerror );
  if( pl == NULL || gerror != NULL )
  {
    if( gerror != NULL )
    {
      g_warning( "Error reading configuration:%s\n", gerror->message );
      g_error_free( gerror );
      gerror = NULL;
    }
  }

  while( pl != NULL )
  {
    XklDebug( 150, "Loaded XKB option: [%s]\n", pl->data );
    _GSwitchItConfigXkbOptionsAdd( config, ( const char * ) pl->data );
    pl = pl->next;
  }

  config->xkbSwitchcutId = gconf_client_get_int( config->confClient,
                                                 CONFIG_KEY_XKB_SWITCHCUT_ID,
                                                 &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    config->xkbSwitchcutId = 0;
    g_error_free( gerror );
    gerror = NULL;
  }

  if( config->xkbSwitchcutId < 0
      || config->xkbSwitchcutId >= total_switchcuts )
    config->xkbSwitchcutId = 0;
}

void GSwitchItConfigLoadXkbParamsFromServer( GSwitchItConfig * config )
{
  XklConfigRec data;
  XklConfigRecInit( &data );
  if( XklConfigGetFromBackup( &data ) )
  {
    int i;
    char **p, **p1;
    GSwitchItConfigXkbModelSet( config, data.model );
    XklDebug( 150, "Loaded XKB model: [%s]\n", data.model );

    GSwitchItConfigXkbLayoutsReset( config );
    p = data.layouts;
    p1 = data.variants;
    for( i = data.numLayouts; --i >= 0; )
    {
      if( *p1 == NULL || **p1 == '\0' )
      {
        XklDebug( 150, "Loaded XKB layout: [%s]\n", *p );
        _GSwitchItConfigXkbLayoutsAdd( config, *p );
      } else
      {
        char fullLayout[XKL_MAX_CI_NAME_LENGTH * 2];
        g_snprintf( fullLayout, sizeof( fullLayout ), "%s\t%s", *p, *p1 );
        XklDebug( 150, "Loaded XKB layout with variant: [%s]\n", fullLayout );
        _GSwitchItConfigXkbLayoutsAdd( config, fullLayout );
      }
      p++;
      p1++;
    }

    GSwitchItConfigXkbOptionsReset( config );
    p = data.options;
    for( i = data.numOptions; --i >= 0; )
    {
      char group[XKL_MAX_CI_NAME_LENGTH];
      char *delim = strchr( *p, ':' );
      int len;
      if( ( delim != NULL ) &&
          ( ( len = ( delim - *p ) ) < XKL_MAX_CI_NAME_LENGTH ) )
      {
        strncpy( group, *p, len );
        group[len] = 0;
        XklDebug( 150, "Loaded XKB option: [%s][%s]\n", group, *p );
        GSwitchItConfigXkbOptionsAdd( config, group, *p );
      }
      p++;
    }
  }
  XklConfigRecDestroy( &data );
}

void GSwitchItConfigSaveXkbParams( GSwitchItConfig * config )
{
  GConfChangeSet *cs;
  GError *gerror = NULL;
  cs = gconf_change_set_new(  );
  gconf_change_set_set_bool( cs, CONFIG_KEY_XKB_OVERRIDE_SETTINGS,
                             config->xkbOverrideSettings );
  XklDebug( 150, "Saved XKB override cmd: [%s]\n",
            config->xkbOverrideSettings ? "true" : "false" );

  if( config->xkbOverrideSettings )
  {
    GSList *pl;
    gconf_change_set_set_string( cs, CONFIG_KEY_XKB_MODEL, config->xkbModel );
    XklDebug( 150, "Saved XKB model: [%s]\n", config->xkbModel );

    pl = config->xkbLayouts;
    while( pl != NULL )
    {
      XklDebug( 150, "Saved XKB layout: [%s]\n", pl->data );
      pl = pl->next;
    }
    gconf_change_set_set_list( cs,
                               CONFIG_KEY_XKB_LAYOUTS,
                               GCONF_VALUE_STRING, config->xkbLayouts );
    pl = config->xkbOptions;
    while( pl != NULL )
    {
      XklDebug( 150, "Saved XKB option: [%s]\n", pl->data );
      pl = pl->next;
    }
    gconf_change_set_set_list( cs,
                               CONFIG_KEY_XKB_OPTIONS,
                               GCONF_VALUE_STRING, config->xkbOptions );

    gconf_change_set_set_int( cs, CONFIG_KEY_XKB_SWITCHCUT_ID,
                              config->xkbSwitchcutId );
  } else
    XklDebug( 150, "Not saving other params\n" );
  gconf_client_commit_change_set( config->confClient, cs, TRUE, &gerror );
  if( gerror != NULL )
  {
    g_warning( "Error saving configuration: %s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }
  gconf_change_set_unref( cs );
}

void GSwitchItConfigLockNextGroup( GSwitchItConfig * config )
{
  int group = XklGetNextGroup(  );
  XklLockGroup( group );
}

void GSwitchItConfigLockPrevGroup( GSwitchItConfig * config )
{
  int group = XklGetPrevGroup(  );
  XklLockGroup( group );
}

void GSwitchItConfigRestoreGroup( GSwitchItConfig * config )
{
  int group = XklGetRestoreGroup(  );
  XklLockGroup( group );
}

void GSwitchItConfigXkbModelSet( GSwitchItConfig * config,
                                 const gchar * modelName )
{
  if( config->xkbModel != NULL )
    g_free( config->xkbModel );
  config->xkbModel = ( modelName == NULL ) ? NULL : g_strdup( modelName );
}

void GSwitchItConfigXkbLayoutsAdd( GSwitchItConfig * config,
                                   const gchar * layoutName,
                                   const gchar * variantName )
{
  const char *merged;
  if( layoutName == NULL )
    return;
  merged = GSwitchItConfigXkbMergeItems( layoutName, variantName );
  if( merged == NULL )
    return;
  _GSwitchItConfigXkbLayoutsAdd( config, merged );
}

void GSwitchItConfigXkbLayoutsReset( GSwitchItConfig * config )
{
  _GSwitchItConfigXkbListReset( &config->xkbLayouts );
}

void GSwitchItConfigXkbOptionsReset( GSwitchItConfig * config )
{
  _GSwitchItConfigXkbListReset( &config->xkbOptions );
}

void GSwitchItConfigXkbOptionsAdd( GSwitchItConfig * config,
                                   const gchar * groupName,
                                   const gchar * optionName )
{
  const char *merged;
  if( groupName == NULL || optionName == NULL )
    return;
  merged = GSwitchItConfigXkbMergeItems( groupName, optionName );
  if( merged == NULL )
    return;
  _GSwitchItConfigXkbOptionsAdd( config, merged );
}

gboolean GSwitchItConfigXkbOptionsIsSet( GSwitchItConfig * config,
                                         const gchar * groupName,
                                         const gchar * optionName )
{
  const char *merged = GSwitchItConfigXkbMergeItems( groupName, optionName );
  if( merged == NULL )
    return False;

  return NULL != g_slist_find_custom( config->xkbOptions,
                                      ( gpointer )
                                      merged,
                                      ( GCompareFunc ) g_ascii_strcasecmp );
}

const char *GSwitchItConfigXkbMergeItems( const char *parent,
                                          const char *child )
{
  static char buffer[XKL_MAX_CI_NAME_LENGTH * 2 - 1];
  *buffer = '\0';
  if( parent != NULL )
  {
    if( strlen( parent ) >= XKL_MAX_CI_NAME_LENGTH )
      return NULL;
    strcat( buffer, parent );
  }
  if( child != NULL )
  {
    if( strlen( child ) >= XKL_MAX_CI_NAME_LENGTH )
      return NULL;
    strcat( buffer, "\t" );
    strcat( buffer, child );
  }
  return buffer;
}

gboolean GSwitchItConfigXkbSplitItems( const char *merged, char **parent,
                                       char **child )
{
  static char pbuffer[XKL_MAX_CI_NAME_LENGTH];
  static char cbuffer[XKL_MAX_CI_NAME_LENGTH];
  int plen, clen;
  const char *pos;
  *parent = *child = NULL;

  if( merged == NULL )
    return False;

  pos = strchr( merged, '\t' );
  if( pos == NULL )
  {
    plen = strlen( merged );
    clen = 0;
  } else
  {
    plen = pos - merged;
    clen = strlen( pos + 1 );
    if( clen >= XKL_MAX_CI_NAME_LENGTH )
      return False;
    strcpy( *child = cbuffer, pos + 1 );
  }
  if( plen >= XKL_MAX_CI_NAME_LENGTH )
    return False;
  memcpy( *parent = pbuffer, merged, plen );
  pbuffer[plen] = '\0';
  return True;
}

void GSwitchItConfigActivateXkbParams( GSwitchItConfig * config )
{
  _GSwitchItConfigXkbDoWithSettings( config, True, NULL );
}

void GSwitchItConfigXkbDumpSettings( GSwitchItConfig * config,
                                     const char *fileName )
{
  _GSwitchItConfigXkbDoWithSettings( config, False, fileName );
}

void GSwitchItConfigActivateAppletParams( GSwitchItConfig * config )
{
  XklSetGroupPerApp( config->groupPerApp );
  XklSetIndicatorsHandling( config->handleIndicators );
  XklSetSecondaryGroupsMask( config->secondaryGroupsMask );
  if( config->debugLevel != -1 )
    XklSetDebugLevel( config->debugLevel );
}

static void GSwitchItConfigAddListener( GSwitchItConfig * config,
                                        const gchar * key,
                                        GConfClientNotifyFunc func,
                                        gpointer userData, int *pid )
{
  GError *err = NULL;
  XklDebug( 150, "Listening to [%s]\n", key );
  *pid = gconf_client_notify_add( config->confClient,
                                  key, func, userData, NULL, &err );
  if( 0 == *pid )
  {
    XklDebug( 100, "Error listening for configuration: [%s]\n",
              err->message );
    g_error_free( err );
  }
}

static void GSwitchItConfigRemoveListener( GSwitchItConfig * config,
                                           int *pid )
{
  if( *pid != 0 )
  {
    gconf_client_notify_remove( config->confClient, *pid );
    *pid = 0;
  }
}

void GSwitchItConfigStartListenSettings( GSwitchItConfig * config,
                                         GConfClientNotifyFunc func,
                                         gpointer userData )
{
  int i;
  int *pid = gconfListenerIds;
  const char **pvar = appletConfigVars;

  memset( gconfListenerIds, 0, sizeof( gconfListenerIds ) );

  while( *pvar != NULL )
    GSwitchItConfigAddListener( config, *pvar++, func, userData, pid++ );

  for( i = XklGetNumGroups(  ); --i >= 0; )
  {
    gchar szi[( sizeof CONFIG_KEY_IMAGES_FMT ) + 3];
    gchar szc[( sizeof CONFIG_KEY_COMMANDS_FMT ) + 3];
    snprintf( szi, sizeof( szi ), CONFIG_KEY_IMAGES_FMT, i );
    snprintf( szc, sizeof( szc ), CONFIG_KEY_COMMANDS_FMT, i );
    GSwitchItConfigAddListener( config, szi, func, userData, pid++ );
    GSwitchItConfigAddListener( config, szc, func, userData, pid++ );
  }
}

void GSwitchItConfigStopListenSettings( GSwitchItConfig * config )
{
  int i;
  int *pid = gconfListenerIds;
  const char **pvar = appletConfigVars;

  while( *pvar++ != NULL )
    GSwitchItConfigRemoveListener( config, pid++ );

  for( i = XklGetNumGroups(  ); --i >= 0; )
  {
    GSwitchItConfigRemoveListener( config, pid++ );
    GSwitchItConfigRemoveListener( config, pid++ );
  }
}

Bool GSwitchItConfigGetDescriptions( const char *name,
                                     char **layoutDescr, char **variantDescr )
{
  char *layoutName = NULL, *variantName = NULL;
  if( !GSwitchItConfigXkbSplitItems( name, &layoutName, &variantName ) )
    return False;
  return _GSwitchItConfigGetDescriptions( layoutName, variantName,
                                          layoutDescr, variantDescr );
}

const char *GSwitchItConfigFormatFullLayout( const char *layoutDescr,
                                             const char *variantDescr )
{
  static char fullDescr[XKL_MAX_CI_DESC_LENGTH * 2];
  if( variantDescr == NULL )
    g_snprintf( fullDescr, sizeof( fullDescr ), "%s", layoutDescr );
  else
    g_snprintf( fullDescr, sizeof( fullDescr ), "%s %s", layoutDescr,
                variantDescr );
  return fullDescr;
}

void GSwitchItConfigLoadGroupDescriptions( GSwitchItConfig * config,
                                           GroupDescriptionsBuffer
                                           namesToFill )
{
  int i;
  const char **pNativeNames = XklGetGroupNames(  );
  char *pDest = ( char * ) namesToFill;
  // first fill with defaults
  for( i = XklGetNumGroups(  ); --i >= 0; pDest += sizeof( namesToFill[0] ) )
    g_snprintf( pDest, sizeof( namesToFill[0] ), "%s", *pNativeNames++ );

  pDest = ( char * ) namesToFill;
  if( XklMultipleLayoutsSupported(  ) && config->layoutNamesAsGroupNames )
  {
    XklConfigRec xklConfig;
    if( XklConfigGetFromServer( &xklConfig ) )
    {
      char **pl = xklConfig.layouts;
      char **pv = xklConfig.variants;
      for( i = xklConfig.numLayouts; --i >= 0;
           pDest += sizeof( namesToFill[0] ) )
      {
        char *lDescr;
        char *vDescr;
        if( _GSwitchItConfigGetDescriptions( *pl++, *pv++,
                                             &lDescr, &vDescr ) )
        {
          g_snprintf( pDest, sizeof( namesToFill[0] ), "%s",
                      GSwitchItConfigFormatFullLayout( lDescr, vDescr ) );
        }
      }
    }
  }
}
