/*
 *    WMAcpiLoad - A dockapp to monitor ACPI status
 *    Copyright (C) 2002      Thomas Nemeth <tnemeth@free.fr>
 *    Copyright (C) 2004-2005 Alan Carriou <cariou_alan@yahoo.fr>
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002       Mark Staggs <me@markstaggs.net>
 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>             /* for signal(), SIGCHLD */
#include <sys/wait.h>           /* for waitpid(), WNOHANG */
#include <dirent.h>             /* for lib_acpi.h include to work */
#include "lib_acpi.h"
#include "wmacpiload.h"
#include "dockapp.h"            /* TODO: should be useless here */
#include "draw.h"
#include "battery.h"            /* TODO: all three should be useless here */
#include "thermal.h"
#include "ac_adapter.h"
#include "main.h"               /* TODO: only private functions here. No need for a .h */

/************************************************************************
 * Global variables definitions
 ************************************************************************/

AcpiInfos cur_acpi_infos;

char *ignored_device;

unsigned int verbose;

int batteries_enabled;

/************************************************************************
 * Private variables definitions
 ************************************************************************/

/* Command launched every time that alarm is raised */
static char *notif_cmd;

/* time in seconds between updates */
static unsigned long int update_interval;

#if 0                           /* unused */
static char *suspend_cmd;
static char *standby_cmd;
#endif

/************************************************************************
 * Functions definitions
 ************************************************************************/

int
main(int argc, char **argv)
{
    XEvent event;
    state previous_state;
    DrawConfig drw = { 0 };

    drw.backlight = LIGHTOFF;

    init_global_variables();

    parse_arguments(argc, argv, &cur_acpi_infos, &drw);

    if (atexit(&acpi_cleanup) != 0) {
        fprintf(stderr,
                "warning: atexit(acpi_cleanup) registering failed.\n");
    }
    if (atexit(&dockapp_cleanup) != 0) {
        fprintf(stderr,
                "warning: atexit(dockapp_cleanup) registering failed.\n");
    }

    switch (acpi_exists()) {
    case 1:
        fprintf(stderr, "No ACPI support in kernel\n");
        exit(EXIT_FAILURE);
        break;
    case 2:
        fprintf(stderr, "ACPI is supported by the kernel, "
                "but we are not allowed to acces it.\n");
        exit(EXIT_FAILURE);
        break;
    }

    acpi_detect_devices(&cur_acpi_infos);

    /* Initialize X settings and open the (X)window */
    if (draw_init(&drw, argc, argv) != 0)
        exit(EXIT_FAILURE);

    /* Let our zombies children die in peace */
    if (signal(SIGCHLD, signal_handler) == SIG_ERR)
        fprintf(stderr, PACKAGE " : could not set signal handler. "
                "Processes run when the alarm is raised may "
                "not exit properly (and become zombies)\n");

    /* Get acpi information and displays them for the first time */
    backup_to_previous_state(&previous_state, &cur_acpi_infos);
    update(&cur_acpi_infos, True, &previous_state, &drw);

    for (;;) {
        if (dockapp_nextevent_or_timeout(&event, update_interval * 1000)) {
            switch (event.type) {
            case ButtonPress:
                switch (event.xbutton.button) {
                case 1:
                    /* Left click */
                    backup_to_previous_state(&previous_state,
                                             &cur_acpi_infos);
                    acpi_update_status(&cur_acpi_infos);
                    switch_light(&drw, &cur_acpi_infos);
                    break;
                case 2:
                    /* Middle click */
                    backup_to_previous_state(&previous_state,
                                             &cur_acpi_infos);
                    thermal_zone_next(&cur_acpi_infos);
                    update(&cur_acpi_infos, False, &previous_state, &drw);
                    break;
                case 3:
                    /* Right click */
                    if (cur_acpi_infos.bat) {
                        backup_to_previous_state(&previous_state,
                                                 &cur_acpi_infos);
                        battery_next(&cur_acpi_infos);
                        update(&cur_acpi_infos, False, &previous_state,
                               &drw);
                    }
                    break;
                }
                break;
            }
        } else {
            /* Time Out */
            update(&cur_acpi_infos, False, &previous_state, &drw);
        }
    }

    /* should never be reached */
    exit(EXIT_SUCCESS);
}

static void
signal_handler(int signo)
{
    switch (signo) {
    case SIGCHLD:
        /* Some child of ours has turned into a zombie; let it die in peace... */
        if (waitpid(-1, NULL, WNOHANG) == -1)
            fprintf(stderr,
                    PACKAGE
                    " : error during child process termination.\n");
        break;
    }
}

static void
init_global_variables(void)
{
    verbose = 0;

    update_interval = DEFAULT_UPDATE_TIME;
    batteries_enabled = 1;
    ignored_device = NULL;
    notif_cmd = NULL;

    cur_acpi_infos.ac = NULL;
    cur_acpi_infos.bat = NULL;
    cur_acpi_infos.bat_first = NULL;
    cur_acpi_infos.bat_alarm_level = DEFAULT_ALARM_BAT_LEVEL;
    cur_acpi_infos.thermal = NULL;
    cur_acpi_infos.thermal_first = NULL;
    cur_acpi_infos.thermal_alarm_level = DEFAULT_ALARM_TEMPERATURE;
}

static void
backup_to_previous_state(state * dest, AcpiInfos *src)
{
    if (src && dest) {
        if (src->thermal)
            dest->temp = src->thermal->temp;

        if (src->bat) {
            dest->bat_percentage = src->bat->percentage;
            dest->bat_plugged = src->bat->plugged;
            dest->bat_status = src->bat->status;
        }

        dest->AC_power = src->AC_power;
    }
}

static void
update(AcpiInfos *infos, int first_time, state * old, DrawConfig *d)
{
    static light previous_backlight;
    static Bool in_alarm_mode = False;

    acpi_update_status(infos);

    if (acpi_check_alarm(infos)) {
        if (!in_alarm_mode) {
            /* Turn alarm on */
            in_alarm_mode = True;
            previous_backlight = d->backlight;
            my_system(notif_cmd);
        }
        switch_light(d, infos);
        /* switch_light() redraws the full window, so we can return now */
        return;

    } else {
        if (in_alarm_mode) {
            /* Turn alarm off */
            in_alarm_mode = False;
            if (d->backlight != previous_backlight) {
                switch_light(d, infos);
                return;
            }
        }
    }

    if (first_time)
        draw_background(d, d->backlight);

    /* redraw digits and icons, if needed */
    if (infos->thermal && old->temp != infos->thermal->temp)
        draw_tempdigit(d, infos);

    if (infos->bat) {
        if (old->bat_percentage != infos->bat->percentage
            || old->bat_plugged != infos->bat->plugged) {
            draw_pcdigit(d, infos);
            draw_pcgraph(d, infos);
        }

        if (old->bat_status !=infos->bat->status
            || old->bat_plugged != infos->bat->plugged)
            draw_statusdigit(d, infos);
    } else {
        /* no battery slot: displays '--%' and '-' charging state */
        draw_pcdigit(d, infos);
        draw_statusdigit(d, infos);
    }

    if (old->AC_power != infos->AC_power)
        draw_power_icon(d, infos);

    /* show */
    dockapp_copy2window(d->pixmap);
}

static void
parse_arguments(int argc, char **argv, AcpiInfos *infos, DrawConfig *drw)
{
    long int integer;
    int i;
    char *endptr;

    /* Note that argc can be zero... */
    for (i = 1; i < argc; i++) {
        if (!strcmp(argv[i], "--help") || !strcmp(argv[i], "-h")) {
            print_help(PACKAGE);
            exit(EXIT_SUCCESS);

        } else if (!strcmp(argv[i], "--version") || !strcmp(argv[i], "-v")) {
            printf("%s version %s\n", PACKAGE, VERSION);
            exit(EXIT_SUCCESS);

        } else if (!strcmp(argv[i], "--display") || !strcmp(argv[i], "-d")) {
            drw->display_name = argv[i + 1];
            i++;

        } else if (!strcmp(argv[i], "--backlight")
                   || !strcmp(argv[i], "-bl")) {
            drw->backlight = LIGHTON;

        } else if (!strcmp(argv[i], "--light-color")
                   || !strcmp(argv[i], "-lc")) {
            drw->light_color = argv[i + 1];
            i++;

        } else if (!strcmp(argv[i], "--interval")
                   || !strcmp(argv[i], "-i")) {
            if (argc == i + 1) {
                fprintf(stderr,
                        "%s: error parsing argument for option %s\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            integer = strtol(argv[i + 1], &endptr, 10);
            if (*endptr != '\0') {
                fprintf(stderr,
                        "%s: error parsing argument for option %s\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            if (integer < 1) {
                fprintf(stderr,
                        "%s: argument %s must be superior or equal to 1\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            update_interval = (unsigned long int) integer;
            i++;

        } else if (!strcmp(argv[i], "--alarm") || !strcmp(argv[i], "-a")) {
            if (argc == i + 1) {
                fprintf(stderr,
                        "%s: error parsing argument for option %s\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            integer = strtol(argv[i + 1], &endptr, 10);
            if (*endptr != '\0') {
                fprintf(stderr,
                        "%s: error parsing argument for option %s\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            if (integer < 0 || integer > 100) {
                fprintf(stderr,
                        "%s: argument %s must be >=0 and <=100\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            infos->bat_alarm_level = integer;
            i++;

        } else if (!strcmp(argv[i], "--temperature")
                   || !strcmp(argv[i], "-t")) {
            temperature temp;

            if (argc == i + 1) {
                fprintf(stderr,
                        "%s: error parsing argument for option %s\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            temp = strtoul(argv[i + 1], NULL, 10);
            if (*endptr != '\0') {
                fprintf(stderr,
                        "%s: error parsing argument for option %s\n",
                        PACKAGE, argv[i]);
                exit(EXIT_FAILURE);
            }

            infos->thermal_alarm_level = temp;
            i++;

        } else if (!strcmp(argv[i], "--windowed")
                   || !strcmp(argv[i], "-w")) {
            dockapp_iswindowed = True;

        } else if (!strcmp(argv[i], "--broken-wm")
                   || !strcmp(argv[i], "-bw")) {
            dockapp_isbrokenwm = True;

        } else if (!strcmp(argv[i], "--notify") || !strcmp(argv[i], "-n")) {
            notif_cmd = argv[i + 1];
            i++;
        } else if (!strcmp(argv[i], "--ignored-device")
                   || !strcmp(argv[i], "-I")) {
            ignored_device = argv[i + 1];
            i++;

        } else if (!strcmp(argv[i], "--disable-batteries")
                   || !strcmp(argv[i], "-D")) {
            batteries_enabled = 0;

        } else if (!strcmp(argv[i], "--verbose") || !strcmp(argv[i], "-V")) {
            verbose++;

#if 0                           /* unused */
        } else if (!strcmp(argv[i], "--suspend") || !strcmp(argv[i], "-s")) {
            suspend_cmd = argv[i + 1];
            i++;
        } else if (!strcmp(argv[i], "--standby") || !strcmp(argv[i], "-S")) {
            standby_cmd = argv[i + 1];
            i++;
#endif
        } else {
            fprintf(stderr, "%s: unrecognized option '%s'\n", PACKAGE,
                    argv[i]);
            print_help(PACKAGE);
            exit(EXIT_FAILURE);
        }
    }
}

static void
print_help(char *prog)
{
    /*
     * This message is too long to be printf'ed at once by ANSI C89
     * compilers. 
     */
    printf
        ("Usage : %s [OPTIONS]\n"
         "%s - Window Maker acpi monitor dockapp\n"
         "  -a,  --alarm <number>          set low battery level when to raise alarm "
         "(%d is default)\n"
         "  -bl, --backlight               turn on back-light\n"
         "  -bw, --broken-wm               activate broken window manager fix\n"
         "  -d,  --display <string>        set display to use\n"
         "  -D,  --disable-batteries       disable battery detection and display\n",
         prog, prog, DEFAULT_UPDATE_TIME);
    printf
        ("  -h,  --help                    show this help text and exit\n"
         "  -i,  --interval <number>       set the number of secs between updates "
         "(%d is default)\n"
         "  -I,  --ignore-device <string>  ignore the device (e.g. unused "
         "battery slot)\n"
         "  -lc, --light-color <string>    set back-light color (rgb:6E/C6/3B is "
         "default)\n", DEFAULT_ALARM_BAT_LEVEL);
    printf
        ("  -n,  --notify <string>         set the command to launch when alarm is on\n"
         "  -t,  --temperature <number>    set the temperature over which alarm is raised "
         "(%d C is default)\n"
         "  -v,  --version                 show program version and exit\n"
         "  -V,  --verbose                 display debug messages on standard "
         "output\n"
         "  -w,  --windowed                run the application in windowed mode\n",
         DEFAULT_ALARM_TEMPERATURE);
#if 0                           /* unused */
    printf
        ("  -s,  --suspend <string>        set command for acpi suspend\n"
         "  -S,  --standby <string>        set command for acpi standby\n");
#endif
}

static int
my_system(char *cmd)
{
    extern char **environ;
    char *argv[4];
    pid_t pid;
    int res;

    res = 0;

    if (cmd == NULL) {
        res = 1;
    } else {
        pid = fork();

        if (pid < 0) {
            fprintf(stderr, "%s : Could not fork\n", PACKAGE);
            res = 2;

        } else if (pid == 0) {
            argv[0] = "sh";
            argv[1] = "-c";
            argv[2] = cmd;
            argv[3] = NULL;
            if (execve("/bin/sh", argv, environ) == -1) {
                fprintf(stderr,
                        "%s : Couldn't execute \"%s\"\n", PACKAGE, cmd);
                /* trying to exit without disturbing father */
                _Exit(EXIT_FAILURE);
            }
        }
    }
    return res;
}
