/* random.c  - random number generator
 *	Copyright (C) 1998, 2000, 2001 Free Software Foundation, Inc.
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 * 
 * This file is part of GPGLIB.
 *
 * GPGLIB is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GPGLIB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPGLIB; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ChangeLog:
 *
 * - Heavily modified to get rid of the log_xxx functions and other stuff
 *   that uses the console.
 * - Applied the XOR bug fix -ts 2002
 * - Removed Libgcrypt malloc stubs and use malloc, calloc and friends directly.
 */

/****************
 * This random number generator is modelled after the one described
 * in Peter Gutmann's Paper: "Software Generation of Practically
 * Strong Random Numbers".
 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <process.h>

#include "openpgp.h"
#include "md.h"


int gather_random_fast( void (*add)(const void*, size_t, int), int requester );

void _gpg_secure_random_alloc( void );
int  _gpg_quick_random_gen( int onoff );
int  _gpg_random_is_faked(void);
byte *_gpg_get_random_bits( size_t nbits, int level, int secure );
void _gpg_fast_random_poll( void );
int gather_random( void (*add)(const void*, size_t, int), int requester,
		  size_t length, int level );

#define fast_random_poll() _gpg_fast_random_poll ()

#define SIZEOF_UNSIGNED_LONG 4

#if SIZEOF_UNSIGNED_LONG == 8
  #define ADD_VALUE 0xa5a5a5a5a5a5a5a5
#elif SIZEOF_UNSIGNED_LONG == 4
  #define ADD_VALUE 0xa5a5a5a5
#else
  #error "weird size for an unsigned long"
#endif

#define BLOCKLEN  64   /* hash this amount of bytes */
#define DIGESTLEN 20   /* into a digest of this length (rmd160) */
/* poolblocks is the number of digests which make up the pool
 * and poolsize must be a multiple of the digest length
 * to make the AND operations faster, the size should also be
 * a multiple of u32
 */
#define POOLBLOCKS 30
#define POOLSIZE (POOLBLOCKS*DIGESTLEN)
#if (POOLSIZE % SIZEOF_UNSIGNED_LONG)
  #error Please make sure that poolsize is a multiple of u32
#endif
#define POOLWORDS (POOLSIZE / SIZEOF_UNSIGNED_LONG)


static int is_initialized;
#define MASK_LEVEL(a) do {if( a > 2 ) a = 2; else if( a < 0 ) a = 0; } while(0)
static char *rndpool;	/* allocated size is POOLSIZE+BLOCKLEN */
static char *keypool;	/* allocated size is POOLSIZE+BLOCKLEN */
static size_t pool_readpos;
static size_t pool_writepos;
static int pool_filled;
static int pool_balance;
static int just_mixed;
static int did_initial_extra_seeding;
static char *seed_file_name;
static int allow_seed_file_update;

static int secure_alloc;
static int quick_test;
static int faked_rng;


static byte *get_random_bytes( size_t nbytes, int level, int secure );
static void read_pool( byte *buffer, size_t length, int level );
static void add_randomness( const void *buffer, size_t length, int source );
static void random_poll(void);
static void read_random_source( int requester, size_t length, int level);
static int gather_faked( void (*add)(const void*, size_t, int), int requester,
						    size_t length, int level );

static struct {
    u32 mixrnd;
    u32 mixkey;
    u32 slowpolls;
    u32 fastpolls;
    u32 getbytes1;
    u32 ngetbytes1;
    u32 getbytes2;
    u32 ngetbytes2;
    u32 addbytes;
    u32 naddbytes;
} rndstats;

static void
initialize(void)
{
    /* The data buffer is allocated somewhat larger, so that
     * we can use this extra space (which is allocated in secure memory)
     * as a temporary hash buffer */
    rndpool = secure_alloc ? calloc(1,POOLSIZE+BLOCKLEN)
			   : calloc(1,POOLSIZE+BLOCKLEN);
    keypool = secure_alloc ? calloc(1,POOLSIZE+BLOCKLEN)
			   : calloc(1,POOLSIZE+BLOCKLEN);
    is_initialized = 1;
}

static void
burn_stack (int bytes)
{
    char buf[128];
    
    memset (buf, 0, sizeof buf);
    bytes -= sizeof buf;
    if (bytes > 0)
        burn_stack (bytes);
}

static void
random_err(const char *format, ...)
{
    char log[8192]; 
    va_list arg_ptr;

    va_start( arg_ptr, format );    
    _vsnprintf( log, sizeof log-1, format, arg_ptr );
    MessageBox( NULL, log, "Crypto RNG", MB_ICONERROR|MB_OK );
    va_end( arg_ptr );
} /* random_err */


void
_gpg_secure_random_alloc( void )
{
    secure_alloc = 1;
}


int
gpg_quick_random_gen( int onoff )
{
    int last;

    read_random_source(0,0,0); /* init */
    last = quick_test;
    if( onoff != -1 )
	quick_test = onoff;
    return faked_rng? 1 : last;
}


/****************
 * Fill the buffer with LENGTH bytes of cryptographically strong
 * random bytes. level 0 is not very strong, 1 is strong enough
 * for most usage, 2 is good for key generation stuff but may be very slow.
 */
void
gpg_randomize( byte *buffer, size_t length, int level )
{
    char *p = get_random_bytes( length, level, 1 );
    memcpy( buffer, p, length );
    free(p);
}


char
gpg_random_char( int level )
{	
    byte c = 0;

    while ( !isalnum( (int)c ) )
	gpg_randomize(&c, 1, level);	
    return c % 127; 
}


int
_gpg_random_is_faked()
{
    if( !is_initialized )
		initialize();
    return faked_rng || quick_test;
}

/****************
 * Return a pointer to a randomized buffer of level 0 and LENGTH bits
 * caller must free the buffer.
 * Note: The returned value is rounded up to bytes.
 */
static byte *
get_random_bytes( size_t nbytes, int level, int secure )
{
    byte *buf, *p;

    if( quick_test && level > 1 )
		level = 1;
    MASK_LEVEL(level);
    if( level == 1 ) {
	rndstats.getbytes1 += nbytes;
	rndstats.ngetbytes1++;
    }
    else if( level >= 2 ) {
	rndstats.getbytes2 += nbytes;
	rndstats.ngetbytes2++;
    }

    buf = secure && secure_alloc ? malloc( nbytes ) : malloc( nbytes );
    for( p = buf; nbytes > 0; ) {
	size_t n = nbytes > POOLSIZE? POOLSIZE : nbytes;
	read_pool( p, n, level );
	nbytes -= n;
	p += n;
    }
    return buf;
}

void *
gpg_random_bytes( size_t nbytes, int level )
{
    return get_random_bytes( nbytes, level, 0 );
}

void *
gpg_random_bytes_secure( size_t nbytes, int level )
{
    return get_random_bytes( nbytes, level, 1 );
}


/****************
 * Mix the pool
 */
static void
mix_pool(byte *pool)
{
    char *hashbuf = pool + POOLSIZE;
    char *p, *pend;
    int i, n;
    RMD160_CONTEXT md;

    rmd160_init( &md );
    /* loop over the pool */
    pend = pool + POOLSIZE;
    memcpy(hashbuf, pend - DIGESTLEN, DIGESTLEN );
    memcpy(hashbuf+DIGESTLEN, pool, BLOCKLEN-DIGESTLEN);
    rmd160_mixblock( &md, hashbuf);
    memcpy(pool, hashbuf, 20 );

    p = pool;
    for( n=1; n < POOLBLOCKS; n++ ) {
	memcpy(hashbuf, p, DIGESTLEN );
	p += DIGESTLEN;
	if( p+DIGESTLEN+BLOCKLEN < pend )
	    memcpy(hashbuf+DIGESTLEN, p+DIGESTLEN, BLOCKLEN-DIGESTLEN);
	else {
	    char *pp = p+DIGESTLEN;
	    for(i=DIGESTLEN; i < BLOCKLEN; i++ ) {
		if( pp >= pend )
		    pp = pool;
		hashbuf[i] = *pp++;
	    }
	}
	rmd160_mixblock( &md, hashbuf);
	memcpy(p, hashbuf, 20 );
    }
    burn_stack (200); /* for the rmd160_mixblock() */
}

void
_gpg_set_random_seed_file( const char *name )
{
    if( seed_file_name )
	random_err("Ohhh jeee, this is a BUG; File %s, Line %d", __FILE__, __LINE__);
    seed_file_name = strdup( name );
}

/****************
 * Read in a seed form the random_seed file
 * and return true if this was successful
 */
static int
read_seed_file()
{
    int fd;
    FILE *fp;
    struct stat sb;
    unsigned char buffer[POOLSIZE];
    int n;

    if( !seed_file_name )
	return 0;
  
	fp = fopen( seed_file_name, "rb" );    
    if( fp == NULL && errno == ENOENT) {
	allow_seed_file_update = 1;
	return 0;
    }
    if( fp == NULL ) {
	random_err("can't open `%s': %s", seed_file_name, strerror(errno));
	return 0;
    }
	fd = fileno(fp);
    if( fstat( fd, &sb ) ) {
	random_err("can't stat `%s': %s", seed_file_name, strerror(errno));
	fclose(fp);
	return 0;
    }
    if( !(sb.st_mode & _S_IFREG) ) {
	fclose(fp);
	return 0;
    }
    if( !sb.st_size ) {
	fclose(fp);
	allow_seed_file_update = 1;
	return 0;
    }
    if( sb.st_size != POOLSIZE ) {
	fclose(fp);
	return 0;
    }
    do {
	n = fread( buffer, 1, POOLSIZE, fp );
    } while( n == -1 && errno == EINTR );
    if( n != POOLSIZE ) {
	random_err("can't read `%s': %s", seed_file_name,strerror(errno));
	fclose(fp);
	return 0;
    }
    fclose(fp);

    add_randomness( buffer, POOLSIZE, 0 );
    /* add some minor entropy to the pool now (this will also force a mixing) */
    {	int x = getpid();
	add_randomness( &x, sizeof(x), 0 );
    }
    {	time_t x = time(NULL);
	add_randomness( &x, sizeof(x), 0 );
    }
    {	clock_t x = clock();
	add_randomness( &x, sizeof(x), 0 );
    }
    /* And read a few bytes from our entropy source.  By using
     * a level of 0 this will not block and might not return anything
     * with some entropy drivers, however the rndlinux driver will use
     * /dev/urandom and return some stuff - Do not read to much as we
     * want to be friendly to the scare system entropy resource. */
    read_random_source( 0, 16, 0 );

    allow_seed_file_update = 1;
    return 1;
}

void
_gpg_update_random_seed_file()
{
    u32 *sp, *dp;
    int i;
	FILE *fp;

    if( !seed_file_name || !is_initialized || !pool_filled )
		return;
    if( !allow_seed_file_update )
		return;

    /* copy the entropy pool to a scratch pool and mix both of them */
    for(i=0,dp=(u32*)keypool, sp=(u32*)rndpool; i < POOLWORDS; i++, dp++, sp++ ) {
	*dp = *sp + ADD_VALUE;
    }
    mix_pool(rndpool); rndstats.mixrnd++;
    mix_pool(keypool); rndstats.mixkey++;
  
    fp = fopen( seed_file_name, "w+b");
    if( fp == NULL ) {
	random_err("can't create `%s': %s\n", seed_file_name, strerror(errno));
	return;
    }
    do {
	i = fwrite( keypool, 1, POOLSIZE, fp );
    } while( i == -1 && errno == EINTR );
    if( i != POOLSIZE ) {
	random_err("can't write `%s': %s\n", seed_file_name, strerror(errno));
    }
    if( fclose(fp) ) {
	random_err("can't close `%s': %s\n", seed_file_name, strerror(errno));	
    }
}


static void
read_pool( byte *buffer, size_t length, int level )
{
    int i;
    u32 *sp, *dp;

    if( length >= POOLSIZE ) {
	random_err("too many random bits requested; the limit is %d\n", POOLSIZE*8-1);
    }
    if( !pool_filled ) {
	if( read_seed_file() )
	    pool_filled = 1;
    }

    /* For level 2 quality (key generation) we alwas make
     * sure that the pool has been seeded enough initially */
    if( level == 2 && !did_initial_extra_seeding ) {
	size_t needed;

	pool_balance = 0;
	needed = length - pool_balance;
	if( needed < POOLSIZE/2 )
	    needed = POOLSIZE/2;
	else if( needed > POOLSIZE )
	    random_err("Ohhh jeee this is a BUG; File %s, Line %d", __FILE__, __LINE__);
	read_random_source( 3, needed, 2 );
	pool_balance += needed;
	did_initial_extra_seeding=1;
    }

    /* for level 2 make sure that there is enough random in the pool */
    if( level == 2 && pool_balance < length ) {
	size_t needed;

	if( pool_balance < 0 )
	    pool_balance = 0;
	needed = length - pool_balance;
	if( needed > POOLSIZE )
	    random_err("Ohhh jeee this is a BUG; File %s, Line %d", __FILE__, __LINE__);
	read_random_source( 3, needed, 2 );
	pool_balance += needed;
    }

    /* make sure the pool is filled */
    while( !pool_filled )	
	random_poll();

    /* do always a fast random poll */
    fast_random_poll();

    if( !level ) { /* no need for cryptographic strong random */
	/* create a new pool */
	for(i=0,dp=(u32*)keypool, sp=(u32*)rndpool; i < POOLWORDS; i++, dp++, sp++ )
	    *dp = *sp + ADD_VALUE;
	/* must mix both pools */
	mix_pool(rndpool); rndstats.mixrnd++;
	mix_pool(keypool); rndstats.mixkey++;
	memcpy( buffer, keypool, length );
    }
    else {
	/* mix the pool (if add_randomness() didn't it) */
	if( !just_mixed ) {
	    mix_pool(rndpool);
	    rndstats.mixrnd++;
	}
	/* create a new pool */
	for(i=0,dp=(u32*)keypool, sp=(u32*)rndpool;
				    i < POOLWORDS; i++, dp++, sp++ )
	    *dp = *sp + ADD_VALUE;
	/* and mix both pools */
	mix_pool(rndpool); rndstats.mixrnd++;
	mix_pool(keypool); rndstats.mixkey++;
	/* read the required data
	 * we use a readpoiter to read from a different postion each
	 * time */
	while( length-- ) {
	    *buffer++ = keypool[pool_readpos++];
	    if( pool_readpos >= POOLSIZE )
		pool_readpos = 0;
	    pool_balance--;
	}
	if( pool_balance < 0 )
	    pool_balance = 0;
	/* and clear the keypool */
	memset( keypool, 0, POOLSIZE );
    }
}


/****************
 * Add LENGTH bytes of randomness from buffer to the pool.
 * source may be used to specify the randomness source.
 * Source is:
 *	0 - used ony for initialization
 *	1 - fast random poll function
 *	2 - normal poll function
 *	3 - used when level 2 random quality has been requested
 *	    to do an extra pool seed.
 */
static void
add_randomness( const void *buffer, size_t length, int source )
{
    const byte *p = buffer;

    if( !is_initialized )
	initialize();
    rndstats.addbytes += length;
    rndstats.naddbytes++;
    while( length-- ) {
	rndpool[pool_writepos++] ^= *p++;
	if( pool_writepos >= POOLSIZE ) {
	    if( source > 1 )
		pool_filled = 1;
	    pool_writepos = 0;
	    mix_pool(rndpool); rndstats.mixrnd++;
	    just_mixed = !length;	
	}
    }
}

static void
random_poll( void )
{
    rndstats.slowpolls++;
    read_random_source( 2, POOLSIZE/5, 1 );
}

void
_gpg_fast_random_poll( void )
{
    static void (*fnc)( void (*)(const void*, size_t, int), int) = NULL;
    static int initialized = 0;

    rndstats.fastpolls++;
    if( !initialized ) {
	if( !is_initialized )
	    initialize();
	initialized = 1;
	fnc = gather_random_fast;
    }
    if( fnc ) {
	(*fnc)( add_randomness, 1 );
	return;
    }

    /* fall back to the generic function */
      
    /* time and clock are availabe on all systems - so
     * we better do it just in case one of the above functions
     * didn't work */
    {	time_t x = time(NULL);
	add_randomness( &x, sizeof(x), 1 );
    }
    {	clock_t x = clock();
	add_randomness( &x, sizeof(x), 1 );
    }
}

static void
read_random_source( int requester, size_t length, int level )
{
    static int (*fnc)(void (*)(const void*, size_t, int), int,
						    size_t, int) = NULL;
    if( !fnc ) {
	if( !is_initialized )
	    initialize();
	fnc = gather_random;
	if( !fnc ) {
	    faked_rng = 1;
	    fnc = gather_faked;
	}
	if( !requester && !length && !level )
	    return; /* init only */
    }
    if( (*fnc)( add_randomness, requester, length, level ) < 0 )
	random_err("No way to gather entropy for the RNG\n");
}


static int
gather_faked( void (*add)(const void*, size_t, int), int requester,
				size_t length, int level )
{
    static int initialized=0;
    size_t n;
    char *buffer, *p;

    if( !initialized ) {
	random_err("WARNING: using insecure random number generator!!\n");
	initialized=1;      
	srand( time(NULL)*getpid());
    }

    p = buffer = malloc( length );
    n = length;  
    while( n-- )
		*p++ = ((unsigned)(1 + (int) (256.0*rand()/(RAND_MAX+1.0)))-1);
    add_randomness( buffer, length, requester );
    free(buffer);
    return 0; /* okay */
}
