
/*   tubo.c  */

/*  A program independent forking object module for gtk based programs.
 *  
 *  Copyright 2000-2012(C)  Edscott Wilson Garcia under GNU GPL
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "tubo_static.i"

// 1. We cannot get rid of grandchild fork because we need two processes 
// at the terminating end of the pipes. This way we will not lose any
// output on broken pipe conditions.
// 2. We cannot replace remote semaphores, since these control the process 
// that will exec. Broken remote semaphores also implies broken remote
// pthread_cond.
// 3. Replacement of local semaphores with pthread_cond is stupid: we would
// have to add a condition variable to the equation.

unsigned 
Tubo_id(void){ return tubo_id;}

pid_t
Tubo_threads (void (*fork_function) (void *),
              void *fork_function_data,
              int *stdin_fd,
              void (*stdout_f) (void *stdout_data,
                                void *stream,
                                int childFD),
              void (*stderr_f) (void *stderr_data,
                                void *stream,
                                int childFD), 
	      void (*tubo_done_f) (void *), 
	      void *user_function_data, 
	      int reap_child,
	      int check_valid_ansi_sequence
) {

    fflush (NULL);
    fork_struct tmpfork, *newfork = NULL;
    static int instance = 0;
    instance++;
    memset (&tmpfork, 0, sizeof (fork_struct));
    tmpfork.reap_child = reap_child;
    tmpfork.check_valid_ansi_sequence = check_valid_ansi_sequence;

    int len = strlen("/tubo-%u-%d") + 6 + 6;
    char *shm_name = (char *) malloc (len+1);
    if (!shm_name){
	fprintf(stderr, "malloc: %s\n", strerror(errno));
	exit(1);
    }
    snprintf(shm_name, len, "/tubo-%u-%d", getpid (), instance);
    int fd = shm_open (shm_name, O_RDWR | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
    if(fd < 0){
        fprintf (stderr, "shm_open(%s): %s\n", shm_name, strerror (errno));
	_exit(1);
    }
#ifdef BROKEN_SHARED_SEMAPHORES
    tmpfork.shm_size = 2 * sizeof (int);
#else
    tmpfork.shm_size = 2 * sizeof (sem_t);
#endif
    if(ftruncate (fd, tmpfork.shm_size) < 0) {
        fprintf (stderr, "ftruncate(%s): %s\n", shm_name, strerror (errno));
	_exit(1);
    }
    tmpfork.remote_semaphore = mmap (NULL, tmpfork.shm_size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    close (fd);
    if(tmpfork.remote_semaphore == MAP_FAILED) {
        shm_unlink (shm_name);
        fprintf (stderr, "rfm_shm_open() mmap: %s\n", strerror (errno));
	_exit(1);
    }
    int i;
    for(i = 0; i < 2; i++) {
#ifdef BROKEN_SHARED_SEMAPHORES
        tmpfork.remote_semaphore[i] = 0;
#else
        if(sem_init (tmpfork.remote_semaphore + i, 1, 0) < 0) {
            fprintf (stderr, "unable to initialize shared semaphore: %s\n", strerror (errno));
	    _exit(1);
        }
#endif
    }
    if(msync (tmpfork.remote_semaphore, tmpfork.shm_size, MS_SYNC) < 0){
        fprintf (stderr, "msync(%s): %s\n", shm_name, strerror (errno));
    }

    for(i = 0; i < 3; i++) {
        tmpfork.tubo[i][0] = tmpfork.tubo[i][1] = -1;

        if(pipe (tmpfork.tubo[i]) == -1) {
            TuboClosePipes (&tmpfork);
            return 0;
        }
    }
    TRACE ("Tubo:  before fork: %d<->%d   %d<->%d   %d<->%d\n",
           tmpfork.tubo[0][0], tmpfork.tubo[0][1], tmpfork.tubo[1][0], tmpfork.tubo[1][1], tmpfork.tubo[2][0], tmpfork.tubo[2][1]);

    tmpfork.fork_function = fork_function;
    tmpfork.fork_function_data = fork_function_data;
    tmpfork.stdout_f = stdout_f;
    tmpfork.stderr_f = stderr_f;
    tmpfork.tubo_done_f = tubo_done_f;
    tmpfork.user_function_data = user_function_data;

    tmpfork.tubo_id = tubo_id++;
    strcpy (tmpfork.shm_name, shm_name);
    free (shm_name);


    // The main fork
    tmpfork.PID = fork ();
    if(tmpfork.PID != 0) {      /* parent */
        newfork = (fork_struct *) malloc (sizeof (fork_struct));
	if (!newfork){
	    fprintf(stderr, "malloc: %s\n", strerror(errno));
	    exit(1);
	}
        memcpy ((void *)newfork, (void *)(&tmpfork), sizeof (fork_struct));
        // create a semaphore so that finished function will
        // not start until all threads are done.
        pthread_mutex_init (&(newfork->mutex_p), NULL);

        newfork->local_semaphore = (sem_t *) malloc (2 * sizeof (sem_t));
        if(stdout_f)
            sem_init (newfork->local_semaphore, 1, 0);
        else
            sem_init (newfork->local_semaphore, 0, 0);
        if(stderr_f)
            sem_init (newfork->local_semaphore + 1, 1, 0);
        else
            sem_init (newfork->local_semaphore + 1, 0, 0);

        TRACE ("Tubo: parent=0x%x child=0x%x\n", (unsigned)getpid (), (unsigned)tmpfork.PID);

        /* INPUT PIPES *************** */

        /* stdin for read, not used: */
        close (newfork->tubo[0][0]);
        TRACE ("Tubo: parent closing unused fd %d\n", newfork->tubo[0][0]);
        newfork->tubo[0][0] = -1;
        /* stdout for write, not used: */
        close (newfork->tubo[1][1]);
        TRACE ("Tubo: parent closing unused fd %d\n", newfork->tubo[1][1]);
        newfork->tubo[1][1] = -1;
        /* stderr for write, not used: */
        close (newfork->tubo[2][1]);
        TRACE ("Tubo: parent closing unused fd %d\n", newfork->tubo[2][1]);
        newfork->tubo[2][1] = -1;

        /* stdin for write: */
        if(stdin_fd != NULL) {
            *stdin_fd = newfork->tubo[0][1];
        }
        // else, stdin is closed by threadwait function...

        /* stdout for read: */
        if(newfork->stdout_f == NULL) {
            TRACE ("Tubo: parent closing fd %d\n", newfork->tubo[1][0]);
            // greenlight to exec:
	    // mutex here would not be necessary...
	    // (see tubo.dia diagram for reason why)
#ifdef BROKEN_SHARED_SEMAPHORES
            newfork->remote_semaphore[0] = 1;
#else
            sem_post (newfork->remote_semaphore);
#endif
            if(msync (newfork->remote_semaphore, newfork->shm_size, MS_SYNC) < 0){
                fprintf (stderr, "msync: %s\n", strerror (errno));
	    }
            // greenlight to fork_finished function:
            TRACE ("Tubo: parent sem_post to semaphore 0\n");
            sem_post (newfork->local_semaphore);
            close (newfork->tubo[1][0]);
            newfork->tubo[1][0] = -1;
        } else {
            TRACE ("Tubo: parent creating stdout thread\n");

	    pthread_t thread;
	    pthread_create(&thread, NULL, stdout_thread_f, (void *)newfork);
	    pthread_detach(thread);
        }
        /* stderr for read: */
        if(newfork->stderr_f == NULL) {
            TRACE ("Tubo: parent closing fd %d\n", newfork->tubo[2][0]);
            // greenlight to exec:
	    // mutex here is necessary to protect shared memory
	    // block where semaphores reside... 
	    // (race with stdout_thread_f)
	    pthread_mutex_lock (&(newfork->mutex_p));
#ifdef BROKEN_SHARED_SEMAPHORES
            newfork->remote_semaphore[1] = 1;
#else
            sem_post (newfork->remote_semaphore + 1);
#endif
            if(msync (newfork->remote_semaphore, newfork->shm_size, MS_SYNC) < 0){
                fprintf (stderr, "msync: %s\n", strerror (errno));
	    }
	    pthread_mutex_unlock (&(newfork->mutex_p));
            // greenlight to fork_finished function:
            TRACE ("Tubo: parent sem_post to semaphore 1\n");
            sem_post (newfork->local_semaphore + 1);
            close (newfork->tubo[2][0]);
            newfork->tubo[2][0] = -1;
        } else {
            TRACE ("Tubo: parent creating stderr thread\n");
	    pthread_t thread;
	    pthread_create(&thread, NULL, stderr_thread_f, (void *)newfork);
	    pthread_detach(thread);
        }
        /* fire off a threaded wait for the child */
	pthread_t thread;
	pthread_create(&thread, NULL, threaded_wait_f, (void *)newfork);
	pthread_detach(thread);

        /* threads are now in place and ready to read from pipes,
         * child process will get green light to exec
         * as soon as stdin/stdout threads are ready... */

        TRACE ("Tubo: The parent process is returning\n");

        TRACE ("Tubo:  parent %d<->%d   %d<->%d   %d<->%d\n",
               newfork->tubo[0][0],
               newfork->tubo[0][1], newfork->tubo[1][0], newfork->tubo[1][1], newfork->tubo[2][0], newfork->tubo[2][1]);
        // this is no longer needed by main thread, but is used by child threads.
        // it will be unmapped and shm_uninked by the wait thread:
        // munmap(newfork->remote_semaphore,2*sizeof(sem_t));

        return newfork->PID;
    } else {                    /* the child */
        if(stdout_f != NULL) {
            TRACE ("Tubo: child wait for semaphore 1 at %s\n", tmpfork.shm_name);
#ifdef BROKEN_SHARED_SEMAPHORES
            while(tmpfork.remote_semaphore[0] == 0) {
                tubo_threadwait ();
            }
#else
            sem_wait (tmpfork.remote_semaphore);
#endif
        }
        if(stderr_f != NULL) {
            TRACE ("Tubo: child wait for semaphore 2 at %s\n", tmpfork.shm_name);
#ifdef BROKEN_SHARED_SEMAPHORES
            while(tmpfork.remote_semaphore[1] == 0) {
                tubo_threadwait ();
            }
#else
            sem_wait (tmpfork.remote_semaphore + 1);
#endif
        }
        newfork = &tmpfork;
        TRACE ("Tubo: child continues...\n");
        // this is no longer needed by main child thread:
        munmap (tmpfork.remote_semaphore, tmpfork.shm_size);

        /* stdin for write, not used: */
        close (newfork->tubo[0][1]);
        newfork->tubo[0][1] = -1;
        /* stdout for read, not used: */
        close (newfork->tubo[1][0]);
        newfork->tubo[1][0] = -1;
        /* stderr for read, not used: */
        close (newfork->tubo[2][0]);
        newfork->tubo[2][0] = -1;

        /* stdin for read: */

        /* if we leave this pipe end open it helps detect errors on
	 * some programs which start an endless scan loop on stdin.
	 * (i.e., nano, dh) Yet this is buggy on commands that accept
	 * pipe redirection from stdin (i.e., gedit)*/
        if(stdin_fd == NULL) {
            close (newfork->tubo[0][0]);
            newfork->tubo[0][0] = -1;
        } else 

        {
            dup2 (newfork->tubo[0][0], 0);      /* stdin */
        }
        /* stdout for write: */
        if(newfork->stdout_f == NULL) {
            //close(newfork->tubo[1][1]); newfork->tubo[1][1] = -1;
        } else {
            dup2 (newfork->tubo[1][1], 1);
        }
        /* stderr for write: */
        if(newfork->stderr_f == NULL) {
            //close(newfork->tubo[2][1]); newfork->tubo[2][1] = -1;
        } else {
            dup2 (newfork->tubo[2][1], 2);
        }

        /* OUTPUT PIPES */
        /*
           struct timespec thread_wait = {
           0, 100000000
           };
           nanosleep(&thread_wait, NULL);
         */
        //fprintf(stderr,"pipe 0x%x.1 open\n",newfork->tubo_id);
        //fprintf(stdout,"pipe 0x%x.2 open\n",newfork->tubo_id);fflush(stdout);
	//
	// this fork is to keep pipes open until all data successfully read.
        grandchildPID = fork ();  

        if(grandchildPID == 0) {
            //fprintf(stderr,"Tubo-id start: %ld\n",(long)(getpid()));
            setpgid (0, 0);     /* or setpgrp(); */
            if(newfork->fork_function)
                (*(newfork->fork_function)) (fork_function_data);
            fprintf (stderr, "Tubo_thread incorrect usage: fork_function must _exit()\n");
            _exit (1);
        }

        signal_connections ();  // pass signals to the grandchild.

        /*TRACE ("Tubo:  child %d<->%d   %d<->%d   %d<->%d\n",
           newfork->tubo[0][0],
           newfork->tubo[0][1],
           newfork->tubo[1][0],
           newfork->tubo[1][1],
           newfork->tubo[2][0],
           newfork->tubo[2][1]); */

        int status;
	// this global variable is for SIGTERM conditions
        term_shm_name = newfork->shm_name;   
	
        waitpid (grandchildPID, &status, 0);
        shm_unlink (newfork->shm_name);
        TRACE("pipe 0x%x.1 closed\n",newfork->tubo_id);
        TRACE("pipe 0x%x.2 closed\n",newfork->tubo_id);fflush(stdout);
        tubo_id++;
        //sleep (1);

        TRACE ("Tubo: shm_unlink(%s)\n", newfork->shm_name);
	// This here is to let other side of pipe know that all pipe output
	// is hencefore done with. Fork finished function may well execute
	// before all pipe output is done with.
	fprintf (stdout, "Tubo-id exit:%d> (%d)\n", 
		newfork->tubo_id, (int)(getpid ()));
	fflush(NULL);
        close (newfork->tubo[0][0]);
        close (newfork->tubo[1][1]);
        close (newfork->tubo[2][1]);
        _exit (1);

    }

}
