/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"

#include "xfdir.i"


void
xfdir_set_monospace(PangoLayout *layout){
    static PangoFontDescription *monospace_desc = NULL;
    static gint fontsize = 0;
    gint newsize = 8;
    const gchar *p = getenv ("RFM_DIAGNOSTICS_FONT_SIZE");
    if(p && strlen (p)) {
	errno=0;
	long value = strtol(p, NULL, 0);
	if (errno == 0){
	    newsize = value;
	}
    }
    if (fontsize != newsize) {
	fontsize = newsize;
	PangoFontDescription *font_desc = pango_font_description_new ();
	pango_font_description_set_family (font_desc, "monospace");
	pango_font_description_set_size (font_desc, fontsize * PANGO_SCALE);
	if (monospace_desc) pango_font_description_free (monospace_desc); 
	monospace_desc = font_desc;
    }

    pango_layout_set_font_description(layout, monospace_desc); 
}




/**
 * SECTION:xfdir
 * @short_description: xfdir structure is a location to place the contents of 
 * the rodent view. The primary xfdir functions allow for the loading and
 * monitoring of local files on a POSIX compliant system.
 * 
 * @section_id:	xfdir
 * @stability: Current
 * @include: primary.h
 *
 **/

/**
 * xfdir_get_local:
 * @view_p: A Rodent view_t pointer
 * @en: An icon entry structure.
 *
 * Generalized xfdir load routine. en->path  should be a valid
 * path or else en->module defined so that module particulars 
 * are used.
 *
 * This is main thread executed.
 **/

void
xfdir_make_layout2(view_t * view_p, population_t *population_p){
    if (!view_p || !population_p || !population_p->en) return;
    record_entry_t *en=population_p->en;
// taxing:    if (!en->path || !en->st || !rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
    if (!en->path || !en->st ){
	return;
    }
    // Use layout2 for file details (if applicable)
    // This function is used by monitor for updates on stat changes.
    gchar *mode_string=rfm_mode_string (en->st->st_mode);
    gchar *grupo=rfm_group_string(en->st);
    gchar *owner=rfm_user_string(en->st);
    gchar *date_string=rfm_date_string (en->st->st_mtime);

    gchar *layout2_text=g_strdup_printf("   <span><i>%s %8s:%-8s %12llu %s</i></span>", 
	    mode_string,
	    owner, grupo,
	    (long long unsigned)en->st->st_size,
	    date_string);
    g_free(mode_string);
    g_free(grupo);
    g_free(owner);
    g_free(date_string);
    population_p->layout2 = 
	gtk_widget_create_pango_layout (view_p->widgets.paper, NULL);
    pango_layout_set_markup(population_p->layout2, layout2_text, -1);
    if (view_p->view_layout.icon_size < TINY_ICON_SIZE){
	xfdir_set_monospace(population_p->layout2);
    }
	
}

xfdir_t *
xfdir_get (view_t * view_p, view_preferences_t *view_preferences_p,
	const record_entry_t * in_en, 
	gint *heartbeat,
	gboolean fullstat) {
    // allocate memory for new xfdir structure and initialize values
    xfdir_t *xfdir_p = (xfdir_t *) malloc (sizeof (xfdir_t));
    if (!xfdir_p) g_error("malloc: %s", strerror(errno));
    memset ((void *)xfdir_p, 0, sizeof (xfdir_t));
    
    xfdir_p->en = rfm_copy_entry(in_en);
    xfdir_p->view_p = view_p;
    // set user defined sort preferences
    // This is already done in rodent_full_reload...
    // rfm_get_view_preferences (view_p, xfdir_p->en);
    
    //if (en && en->st) NOOP("xfdir monitor: xfdir_get: en->st->st_uid  %d\n", en->st->st_uid );

    // This limitation is to make things faster since the desktop will refuse 
    // to map whatever does not fit on the desktop.
    //if (view_p->flags.type ==DESKVIEW_TYPE){
	//xfdir_p->limited_read = TRUE;
    //}
#ifdef DEBUG
	if(view_preferences_p->preferences & __SHOW_HIDDEN){
	    NOOP( "xfdir_get: shows hidden...\n");
	} else {
	    NOOP( "xfdir_get: does *not* show hidden...\n");
	}
#endif
    // do the sort
    if (view_preferences_p) {
	xfdir_p->ascending = view_preferences_p->preferences & SORT_ASCENDING;
	xfdir_p->sort_column = view_preferences_p->sortcolumn;
	if (SHOWS_HIDDEN(view_preferences_p->preferences)) {
	    SET_SHOWS_HIDDEN(xfdir_p->en->type);
	}
    } else {
	xfdir_p->ascending = (DEFAULT_VIEW_PREFERENCES) & SORT_ASCENDING;
	xfdir_p->sort_column = DEFAULT_SORT_COLUMN;
    }
    if (xfdir_p->sort_column != TYPE_SORT && xfdir_p->sort_column != NAME_SORT){
	// we will need to stat records here, so the easiest
	// and quickest way is to enable fullstat (which does not
	// add too much response time, even for large directory listings)
	fullstat=TRUE;
    }
    
    // read local directory information and update xfdir structure
    NOOP("xfdir monitor: starting time on update_xfdir for %s...\n", xfdir_p->en->path);

    NOOP(stderr, "xfdir_get: SHOWS_HIDDEN (xfdir_p->en->type)=%d\n", SHOWS_HIDDEN (xfdir_p->en->type));

    update_xfdir (xfdir_p, fullstat, heartbeat);
    if (heartbeat) *heartbeat = -1;

    if (xfdir_p->pathc == READ_ERROR) {
	NOOP(stderr, "xfdir_p->pathc == READ_ERROR\n");
	g_free(xfdir_p);
	return NULL;
    }


    // now we do the sorting. Stat record is not necessary for
    // name or type sort methods, so barebones should be fine.
    NOOP("xfdir monitor: ---------- sortcolumn=%d\n", xfdir_p->sort_column);

    xfdir_sort (xfdir_p);

    // set default status tag for directory or module (sizetag)
    xfdir_set_entry_tag (&(view_p->widgets), xfdir_p->en, xfdir_p->tama);

    return xfdir_p;
}

/**
 * xfdir_exit_monitor:
 * @xfdir_p:  pointer to an xfdir_t structure
 *
 * Schedule monitor to exit as soon as possible. This function should 
 * be accessed from threads other than the monitor thread. For the monitor thread,
 * use the static exit_monitor()  or rfm_increment_monitor() from outside.
 **/
void
xfdir_exit_monitor (view_t * view_p) {
    if(view_p->mutexes.monitor_id == NULL) {
        view_p->mutexes.monitor_id = g_mutex_new ();
    }
    g_mutex_lock (view_p->mutexes.monitor_id);
    view_p->flags.monitor_id++;
    DBG("xfdir monitor: xfdir_exit_monitor->%d\n", view_p->flags.monitor_id);
    g_mutex_unlock (view_p->mutexes.monitor_id);
}


/**
 * xfdir_count_files:
 * @path: directory path
 * Returns: file count within directory, including hidden files
 *
 * This function is non-recursive.
 **/
gint
xfdir_count_files (const gchar * path) {
    if(!rfm_g_file_test (path, G_FILE_TEST_IS_DIR))
        return 0;
    DIR *directory;
    directory = opendir (path);
    if(!directory)
        return 0;
    gint count = 0;
    struct dirent *d;
    while((d = readdir (directory)) != NULL) {
        if(strcmp (d->d_name, ".") == 0)
            continue;
        if(strcmp (d->d_name, "..") == 0)
            continue;
        count++;
    }
    closedir (directory);
    return count;
}

/**
 * xfdir_count_hidden_files:
 * @path: directory path
 * Returns: hidden file count within directory
 *
 * This function is non-recursive.
 **/
gint
xfdir_count_hidden_files (const gchar * path) {
    if(!rfm_g_file_test (path, G_FILE_TEST_IS_DIR))
        return 0;
    DIR *directory;
    directory = opendir (path);
    if(!directory)
        return 0;
    gint count = 0;
    struct dirent *d;
    while((d = readdir (directory)) != NULL) {
        if(strcmp (d->d_name, ".") == 0)
            continue;
        if(strcmp (d->d_name, "..") == 0)
            continue;
        if(d->d_name[0] != '.')
            continue;
        count++;
    }
    closedir (directory);
    return count;
}

void
xfdir_register_popup(view_t *view_p, GtkWidget *popup){
    if (!view_p || !GTK_IS_WIDGET(popup)){
	g_warning("xfdir_register_popup(): !view_p || !GTK_IS_WIDGET(popup)");
	return;
    }
    g_object_set_data(G_OBJECT(view_p->widgets.paper), 
		"private_popup_widget", popup);
}

/**
 * xfdir_start_monitor:
 * @view_p: A Rodent view_t pointer
 * @en: An icon entry structure.
 *
 * Starts a monitor thread for the loaded @xfdir_p
 * structure. This is done automatically for local files
 * but modules should define a FIXME function to 
 * test for updates and a FIXME function to reload.
 * 
 **/
    void
xfdir_start_monitor (view_t * view_p, xfdir_t * xfdir_p) {

    // (mutex lock might be overkill here)
    g_mutex_lock (view_p->mutexes.monitor_id);
    xfdir_p->monitor_id = view_p->flags.monitor_id;
//    xfdir_p->monitor_id = ++view_p->flags.monitor_id;
    view_p->flags.monitor_enabled=TRUE;
    g_mutex_unlock (view_p->mutexes.monitor_id);

    // start the background monitor thread
    xfdir_p->thread = g_thread_create (thread_monitor_f, (gpointer) xfdir_p, FALSE, NULL);
      
    TRACE("thread(0x%x): xfdir_start_monitor id=%d\n", GPOINTER_TO_INT(xfdir_p->thread), xfdir_p->monitor_id);
    if (xfdir_p->thread == NULL) {
        g_warning("cannot create monitor thread at primary_xfdir.i");
        view_p->flags.monitor_enabled=FALSE;
    }

}

gboolean
xfdir_monitor_control_greenlight(widgets_t *widgets_p){
    if (!widgets_p){
	g_warning("xfdir_monitor_control_greenlight(): widgets_p is NULL\n");
	return FALSE;
    }
    view_t *view_p = widgets_p->view_p;
    if (!view_p) {
	DBG("xfdir_monitor_control_greenlight(): view_p is NULL\n");
	return FALSE;
    }
    NOOP(stderr, "signalling with xfdir_monitor_control_greenlight\n");
    // Just signal here. Monitor thread may even be dead for all that matters.

    g_mutex_lock(view_p->mutexes.monitor_control);
    gboolean active_monitor = view_p->flags.active_monitor;
    view_p->flags.monitor_go = TRUE;
    g_cond_broadcast(view_p->mutexes.monitor_signal);
    g_mutex_unlock(view_p->mutexes.monitor_control);
    return active_monitor;
}

