

static xfdir_t *
get_root_xfdir (view_t * view_p) {
    GSList *plugin_list = rfm_find_plugins ();
    GSList *list;
    gint max_elements = 0;
    for(list = plugin_list; list; list = list->next) {
	if (rfm_is_root_plugin ((gchar *) list->data)) max_elements++;
	else {
	    NOOP(stderr, "%s is not a root plugin\n", (gchar *) list->data);
	}
    }
    // extra elements: home directory and filesystem icon.
    max_elements += 2;  
    xfdir_t *xfdir_p = (xfdir_t *)malloc(sizeof(xfdir_t));
    if (!xfdir_p) g_error("malloc: %s:", strerror(errno));
    memset (xfdir_p, 0, sizeof(xfdir_t));

    xfdir_p->pathc = max_elements;	
    xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
    if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
    memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
    
    

    gint item=0;
    /* create filesystem root icon */
    xfdir_p->gl[item].en=rfm_stat_entry("/", 0);
    xfdir_p->gl[item].en->path = g_strdup("/");
    xfdir_p->gl[item].en->tag = g_strdup(_("This is the root of the filesystem"));
    xfdir_p->gl[item].pathv = g_strdup(_("Root Directory"));
    SET_ROOT_TYPE(xfdir_p->gl[item].en->type);
    /* create homedir icon */
    item++;
    xfdir_p->gl[item].en=rfm_stat_entry(g_get_home_dir(), 0);
    xfdir_p->gl[item].en->path = g_strdup(g_get_home_dir());
    xfdir_p->gl[item].pathv = g_strdup(_("Home Directory"));
    SET_ROOT_TYPE(xfdir_p->gl[item].en->type);

    item++;

    NOOP ("ROOTS: now plugins...\n");
    for(list = plugin_list; list; list = list->next) {
	if (!rfm_is_root_plugin ((gchar *) list->data))continue;
	const gchar *module_name = list->data;
        DBG ("ROOTS: creating population item for module %s\n", module_name);
	xfdir_p->gl[item].en=rfm_mk_entry(0);
	xfdir_p->gl[item].en->st = NULL;
	xfdir_p->gl[item].en->module = module_name; 
	gchar *module_label = 
	    rfm_void (PLUGIN_DIR, (void *)module_name, "module_label");
   
	xfdir_p->gl[item].en->path= module_label;    
	xfdir_p->gl[item].pathv = g_strdup(module_label);
	SET_ROOT_TYPE(xfdir_p->gl[item].en->type);
        item++;
    }
    rfm_cursor_reset(view_p->widgets.window);
    return xfdir_p;
}

 

typedef struct reload_t {
    const view_t *view_p;
    record_entry_t *target_en;
    xfdir_t *xfdir_p;
}reload_t;

typedef struct heartbeat_t{
    const view_t *view_p;
    const record_entry_t *target_en;
    xfdir_t *xfdir_p;
    GMutex *xfdir_mutex;
    GCond *xfdir_signal;
    gint heartbeat;
    view_preferences_t *view_preferences_p;
    GThread *thread;
} heartbeat_t;

static void *
direct_loadxfdir(gpointer data){
    NOOP(stderr, "direct_loadxfdir\n");
    reload_t *reload_p = data;
    view_preferences_t *view_preferences_p = 
	rfm_get_view_preferences (reload_p->view_p->flags.type,
		(record_entry_t *)reload_p->target_en);
    xfdir_t *xfdir_p = xfdir_get ((view_t *)reload_p->view_p, 
	    view_preferences_p,
	    (record_entry_t *)reload_p->target_en,
	    NULL,
	    FALSE 
	    // fullstat (if FALSE, only first FULL_LOAD_COUNT records will be stated) 
	    );
	    
    if(xfdir_p==NULL) {
	DBG ("load_xfdir_structure: xfdir_p is NULL!\n");
    }
    return xfdir_p;
}

static void *
heartbeat_loadxfdir(gpointer data){
    heartbeat_t *heartbeat_p = data;
    xfdir_t *xfdir_p = NULL;

    NOOP( "at heartbeat_loadxfdir\n");
    xfdir_p = xfdir_get ((view_t *)heartbeat_p->view_p, 
	    heartbeat_p->view_preferences_p,
	    (record_entry_t *)heartbeat_p->target_en,
	    &(heartbeat_p->heartbeat),
	    FALSE 
	    // fullstat (if FALSE, only first FULL_LOAD_COUNT records will be stated) 
	    );
    NOOP( "at heartbeat_loadxfdir got xfdir_p\n");
	    
    g_mutex_lock(heartbeat_p->xfdir_mutex);
    heartbeat_p->xfdir_p = xfdir_p;
    g_cond_signal(heartbeat_p->xfdir_signal);
    g_mutex_unlock(heartbeat_p->xfdir_mutex);
    if(xfdir_p==NULL) {
	DBG ("load_xfdir_structure: xfdir_p is NULL!\n");
    }
    return xfdir_p;

}

static 
void *wait_on_thread(gpointer data){
    heartbeat_t *heartbeat_p = data;
    xfdir_t *xfdir_p = g_thread_join(heartbeat_p->thread);

    g_mutex_free(heartbeat_p->xfdir_mutex);
    g_cond_free(heartbeat_p->xfdir_signal);
    g_free(heartbeat_p->view_preferences_p);
    g_free (heartbeat_p);
    return xfdir_p;
}


static
xfdir_t *
load_xfdir_with_timeout (reload_t *reload_p, gboolean with_heartbeat) {
    if (!with_heartbeat){
	return direct_loadxfdir(reload_p);

    }
    heartbeat_t *heartbeat_p = (heartbeat_t *)malloc(sizeof(heartbeat_t));
    if (!heartbeat_p) g_error("malloc heartbeat_p: %s\n",strerror(errno));
    memset(heartbeat_p, 0, sizeof(heartbeat_t));

    // Get user defined preferences for target entry
    heartbeat_p->view_preferences_p = 
	rfm_get_view_preferences (reload_p->view_p->flags.type,
		(record_entry_t *)reload_p->target_en);

#ifdef DEBUG
	if(reload_p->view_p->flags.preferences & __SHOW_HIDDEN){
	    NOOP( "load_xfdir_with_timeout: shows hidden...\n");
	} else {
	    NOOP( "load_xfdir_with_timeout: does *not* show hidden...\n");
	} 
#endif

    heartbeat_p->xfdir_mutex = g_mutex_new();
    heartbeat_p->xfdir_signal = g_cond_new();
    heartbeat_p->view_p = reload_p->view_p;
    heartbeat_p->target_en = reload_p->target_en;

    g_mutex_lock(heartbeat_p->xfdir_mutex);
    NOOP( "Creating wait thread for heartbeat_loadxfdir\n");
    heartbeat_p->thread =
	g_thread_create(heartbeat_loadxfdir, heartbeat_p, TRUE, NULL);
#define RFM_LOAD_TIMEOUT 5
    if (!heartbeat_p->xfdir_p) {
	gint load_timeout = RFM_LOAD_TIMEOUT;
	const gchar *p = getenv ("RFM_LOAD_TIMEOUT");
	if(p && strlen (p)) {
	    errno=0;
	    long value = strtol(p, NULL, 0);
	    if (errno == 0){
		load_timeout = value;
	    }
	}


	// loop will be broken when heartbeat stops.
	// since this is read only, mutex can be skipped.
	while (heartbeat_p->heartbeat >= 0){
	    NOOP("*heartbeat records = %d\n", heartbeat_p->heartbeat);
	    rfm_threadwait();
	}
	// Now that heartbeat has stopped. Wait for the timeout condition.

#if GLIB_MAJOR_VERSION==2 && GTK_MINOR_VERSION<32
	GTimeVal tv;
	g_get_current_time (&tv);
	tv.tv_sec += load_timeout;
	if (!g_cond_timed_wait(heartbeat_p->xfdir_signal, heartbeat_p->xfdir_mutex, &tv))
#else
	gint64 end_time;
	end_time = g_get_monotonic_time () + load_timeout * G_TIME_SPAN_SECOND;
	if (!g_cond_wait_until (heartbeat_p->signal, heartbeat_p->xfdir_mutex, end_time))
#endif
	{
	    g_mutex_unlock(heartbeat_p->xfdir_mutex);
	    DBG("dead heartbeat\n");
	    // Dead heartbeat:
	    // Fire off a wait and cleanup thread.
	    THREAD_CREATE(wait_on_thread, heartbeat_p, "wait_on_thread");
	    return NULL;
	}
    }
    g_mutex_unlock(heartbeat_p->xfdir_mutex);
    xfdir_t *xfdir_p = wait_on_thread(heartbeat_p);
    return xfdir_p;
}


static void
exit_helper_threads(view_t *view_p){
    // Signal monitor to exit.
    xfdir_exit_monitor (view_p);
    // If any thumbnailer is active, signal it to exit:
    if (view_p->flags.thumbnailer_active) {
	view_p->flags.thumbnailer_exit=TRUE;    
    } 
    // FIXME: signal exit to tips threads here (normal or preview)
    // to avoid race collision. This is not yet coded.
}
static void resaturate(view_t *view_p){
    const population_t *population_p = 
	rodent_find_in_population (view_p, view_p->mouse_event.current_mouseX, view_p->mouse_event.current_mouseY);
    if (population_p) {
	    rodent_saturate_item (view_p, population_p);	
    }
}
static void
timeout_message(widgets_t *widgets_p, const gchar *path){
    // ETIMEDOUT       Connection timed out (POSIX.1)
    // EDEADLK         Resource deadlock avoided (POSIX.1)
    gchar *msg = g_strdup_printf(_("Unable to load: %s..."), path);
    GDK_THREADS_ENTER();
    rfm_show_text(widgets_p);
    rfm_diagnostics(widgets_p, "xffm/stock_dialog-error",
	    strerror(ETIMEDOUT), ": ", msg, 
	    " (", strerror(EDEADLK),")\n", NULL);
    GDK_THREADS_LEAVE();
    g_free(msg);
}

static void
get_the_write_lock(view_t *view_p){
    // If we claim write lock here, other gtk events which
    // require read lock will have to wait. That's cool.
    //
    // First we tell other threads that may be doing useless work to quit and
    // release the lock:
    view_p->flags.user_wants_lock=TRUE;
    // Now we can solicit the write lock and wait until obtained:
    gboolean locked = FALSE;
    while (!locked) locked = rfm_population_write_lock (view_p);
    // Since we now have the lock, we allow other thread to proceed with whatever
    // they may be doing. The will automatically wait if they request a read or write lock.
    view_p->flags.user_wants_lock=FALSE; 
}

static void
error_trap(xfdir_t *new_xfdir){
    if (new_xfdir->pathc > 0) return;
    g_warning("No items were loaded into xfdir structure!");
    new_xfdir->pathc = 1;
    new_xfdir->gl = (dir_t *) malloc (sizeof (dir_t));
    if(!new_xfdir->gl) g_error ("malloc: %s", strerror(errno));
    memset (new_xfdir->gl, 0, sizeof (dir_t));
    // root item:
    new_xfdir->gl[0].en = NULL;
    new_xfdir->gl[0].pathv = g_strdup (g_get_host_name ());
    return;
}

static void
transfer_open(view_t *view_p, xfdir_t *new_xfdir){
      // XXX: this should be created in a view_new() function!
      //      we also need a view_destroy() function...( if we
      //      don't have it already)
    // Error trap: if nothing is loaded, then at least put in
    // the localhost icon.
    error_trap(new_xfdir);
    // Set the entry preferences in the view.
    record_entry_t *en = view_p->en;
    view_preferences_t *view_preferences_p = 
	rfm_get_view_preferences (view_p->flags.type, en);
    rfm_set_view_preferences (view_p, view_preferences_p);
    g_free(view_preferences_p);
#ifdef DEBUG
	if(view_p->flags.preferences & __SHOW_HIDDEN){
	    NOOP( "transfer_open: shows hidden...\n");
	} else {
	    NOOP( "transfer_open: does *not* show hidden...\n");
	}
#endif


    NOOP("view_p->view_layout.icon_size = %d\n",
	    view_p->view_layout.icon_size);
    // Set the icon size scale:
    set_icon_size_scale(view_p);
    // Non null entry scenario
    if (en) view_p->module = en->module;
    else view_p->module = NULL;

    widgets_t *widgets_p = &(view_p->widgets);
    g_free (widgets_p->workdir);
    if (en && IS_LOCAL_TYPE(en->type)) widgets_p->workdir = g_strdup (en->path);
    else widgets_p->workdir = g_strdup(g_get_home_dir());
    // set the dtfile quick dir buttons (or menu items in our case)
    if(en && !en->module) {
	if(!en->path || 
		!IS_SDIR(en->type)) {
	    g_warning ("Invalid entry sent to xfdir_get_local");
	} else {
	    set_deepest_dir (view_p);
	}
    }
    return;
}

static gboolean
want_monitor(widgets_t *widgets_p, record_entry_t *en){
    if (!en) return FALSE; 
    // Local files monitor:
    if (en->module){
	// Plugin requesting a file monitor thread:
	if (rfm_void(PLUGIN_DIR, en->module, "module_monitor")) return TRUE;
    } else {
	// Local files monitor:
	if (IS_LOCAL_TYPE(en->type)) return TRUE;
	// Non local types with monitor:
	else if (IS_MONITOR_TYPE(en->type))return TRUE;
    } 
    if (g_path_is_absolute(en->path)) {
	// warning 
	rfm_diagnostics (widgets_p, "xffm/stock_disconnect", NULL);
	rfm_diagnostics (widgets_p, "xffm-tag/red", 
		_("Could not initialize monitoring"), 
		    " (", en->path,
		    ": ", _("Remote Connection..."),")\n", NULL);
    }
    return FALSE;
}

static void
transfer_close(view_t *view_p, xfdir_t *new_xfdir_p){
    record_entry_t *en = view_p->en;
    widgets_t *widgets_p = &(view_p->widgets);
    if (en && en->tag){
	rfm_status (widgets_p, "xffm/stock_dialog-info", en->tag, NULL);
    } else if (en) {
	rfm_status (widgets_p, "xffm/stock_dialog-info", _("Refresh Complete"), NULL);
    } else {
	rfm_status (widgets_p, "xffm/device_computer", _("Installed Plugins"), NULL);
    }



    // start background directory monitor thread, or else free
    // xdfir memory block.
    
    if (want_monitor(widgets_p, en)) {
	TRACE("xfdir_start_monitor...\n");
	g_mutex_lock(view_p->mutexes.monitor_control);
	    view_p->flags.active_monitor = TRUE;
	g_mutex_unlock(view_p->mutexes.monitor_control);
	xfdir_start_monitor (view_p, new_xfdir_p);
    } else {
	// Free xfdir memory block.	
	TRACE("Not starting xfdir monitor...\n");
	g_mutex_lock(view_p->mutexes.monitor_control);
	view_p->flags.active_monitor = FALSE;
	g_mutex_unlock(view_p->mutexes.monitor_control);
	xfdir_free_data (new_xfdir_p);
	g_free (new_xfdir_p);

    }
    if (view_p->flags.type == DESKVIEW_TYPE) {
	// Save desktop directory for navigational desktop
	save_environment_desktop_dir(view_p, en);
    } else {
	// Set the title for the window (does not apply to deskview)
        rodent_set_view_title (view_p);
 	rfm_update_status_line (view_p);
	rodent_set_scroll_position (view_p);
    }
    // resaturate any icon which was previously saturated...
    resaturate(view_p);
}

static record_entry_t *
init_view (view_t * view_p, record_entry_t * en) {
    NOOP ("rodent_population: init_view\n");

    // destroy obsolete entry:
    if(view_p->en) rfm_destroy_entry (view_p->en);
    rodent_destroy_population (view_p);
    view_p->mouse_event.boxX = -1, view_p->mouse_event.boxY = -1;
    view_p->mouse_event.old_X = -1, view_p->mouse_event.old_Y = -1;
    // destroy selection
    if(view_p->selection_list) {
	GSList *tmp=view_p->selection_list;
	for (;tmp && tmp->data; tmp=tmp->next){
	    record_entry_t *en=tmp->data;
	    rfm_destroy_entry(en);
	}
        g_slist_free (view_p->selection_list);
        view_p->selection_list = NULL;
    }
    view_p->mouse_event.mouseX = -1;
    view_p->mouse_event.mouseY = -1;
    view_p->tip_event.tip_timer = 0;
    view_p->widgets.rename = NULL;
    view_p->mouse_event.label_p = NULL;
    view_p->mouse_event.doing_drag_p = NULL;
    view_p->mouse_event.saturated_p = NULL;
    view_p->mouse_event.selected_p = NULL;
    view_p->en = en;

  
    return en;
}

static void 
update_f(view_t *view_p, record_entry_t *target_en, xfdir_t *new_xfdir) {
    if (!new_xfdir){
	g_error("new_xfdir cannot be null here.");
    }

    // Set lock for icontheme tests:
    g_static_rw_lock_reader_lock(&(rfm_global_p->icon_theme_lock));
    // Lock is now in place, icontheme thread will wait.
    //
    // Now set the GDK mutex to hold back all gtk callbacks
    GDK_THREADS_ENTER();
    // We now have GDK exclusive access.
    //
    // Set the write lock for the population
    get_the_write_lock(view_p);
    // We now have the write lock. This will hold off any
    // threads which access the population (and which should
    // have requested a read lock).
    //
    view_p->flags.no_expose = TRUE; 
    // Init view will free population, entry and assign new entry.
    init_view (view_p, target_en);
    transfer_open(view_p, new_xfdir);
    transfer_operations(view_p, new_xfdir);
    transfer_close(view_p, new_xfdir);
    view_p->flags.no_expose = FALSE; 
   
    rfm_population_write_unlock (view_p);

    rodent_expose_all (view_p);

    NOOP(stderr, "reload_f(): 0x%x is unblocking GTK\n", 
	    GPOINTER_TO_INT(g_thread_self()));
	    
    GDK_THREADS_LEAVE();
    // Lift lock. Icon theme checks may now proceed as usual.	
    g_static_rw_lock_reader_unlock(&(rfm_global_p->icon_theme_lock));

    DBG("hard reload thread done...\n");
}



// XXX: cleanup this mutex    g_mutex_lock(view_p->mutexes.reload_mutex);
// XXX: cleanup this rfm_global_p->icon_theme_barrier
// 
// This is a user driven function. Mutual exclusion is to avoid overlapping calls.
//GStaticMutex simultaneous_exclusion = G_STATIC_MUTEX_INIT;
// This is a thread function, must have GDK mutex set for gtk commands...
void *
reload_f (gpointer data) {
    reload_t *reload_p = data;
    view_t *view_p = (view_t *)reload_p->view_p;
    // This mutex will protect against multiple function calls
    // simultaneous on the same view.
    g_mutex_lock(view_p->mutexes.reload_mutex);
    widgets_t *widgets_p = &(view_p->widgets);
    record_entry_t *record_entry_p  = (record_entry_t *)reload_p->target_en;
    xfdir_t *new_xfdir = NULL;


    // Step 2.
    // 2. Get new xfdir. This requires no extra locks nor mutexes...
    // XXX: add roots should work with a xfdir get to simplify this...
    gboolean aborted_load = FALSE;
    if (record_entry_p == NULL){
	// We could wait for monitor exited condition to continue...
	new_xfdir = get_root_xfdir(view_p);
    } else {
	if (IS_LOCAL_TYPE(record_entry_p->type)){
	    // This is a non-heartbeat controlled load. Will block.
	    new_xfdir = load_xfdir_with_timeout(reload_p, FALSE);
	    //new_xfdir = load_xfdir_with_timeout(reload_p, TRUE);
	} else {
	    // This is a heartbeat controlled load.
	    new_xfdir = load_xfdir_with_timeout(reload_p, TRUE);
	}

	// On timeout, null is returned.
	if (!new_xfdir){
	    timeout_message(widgets_p, record_entry_p->path);
	    // If this is not the initial load, then abort reload.
	    rfm_destroy_entry(record_entry_p);
	    if(view_p->go_list && g_list_last (view_p->go_list)){
		GDK_THREADS_ENTER();
		rodent_expose_all (view_p);
		GDK_THREADS_LEAVE();
		aborted_load = TRUE;
	    } else {
		reload_p->target_en = record_entry_p = NULL;
		new_xfdir = get_root_xfdir(view_p);
	    }
	} 
    }
    if (!aborted_load){
	// By now we have a new_xfdir, or valid NULL, so we can do a transfer.
	// target_en will substitute view_p->en downstream.
	exit_helper_threads(view_p);
	NOOP("rodent_full_reload_view(), entering update_f()\n");
	update_f(view_p, reload_p->target_en, new_xfdir);
	sem_post(rfm_global_p->setup_semaphore);
    } else {
	NOOP("rodent_full_reload_view(), reload_f aborted call.\n");
    }
GDK_THREADS_ENTER();
	rfm_cursor_reset (widgets_p->window);
GDK_THREADS_LEAVE();
    g_free(reload_p);
    g_mutex_unlock(view_p->mutexes.reload_mutex);
	NOOP(stderr, "reload_f() is done...\n");
    return GINT_TO_POINTER(!aborted_load);
}

static void
print_loading_msg(widgets_t *widgets_p, record_entry_t *target_en){
    if (!target_en || !widgets_p) return;
    if (target_en->module == NULL){
	rfm_status (widgets_p, "xffm/stock_dialog-info", _("Loading folder..."), NULL);
	return;
    }
    const gchar *name = rfm_void(PLUGIN_DIR, target_en->module,  
		    "module_label"); 
    gchar *text = g_strdup_printf(_("Loading %s..."), 
		(name)? name:target_en->module);
    rfm_status (widgets_p, "xffm/stock_dialog-info", text, NULL);
    g_free(text);
    return;
}

gint
rodent_full_reload_view (view_t *view_p, record_entry_t * target_en) {

    if (rfm_global_p->self != g_thread_self()) {
	g_warning("rodent_full_reload_view is a main thread callback! Call aborted\n");
	return 0;
    }
    widgets_t *widgets_p = &(view_p->widgets);

    NOOP(stderr, "calling print_loading_msg()\n");
    print_loading_msg(widgets_p, target_en);
    
    reload_t *reload_p = (reload_t *)malloc(sizeof(reload_t));
    if (!reload_p) g_error("malloc: %s", strerror(errno));
    memset(reload_p, 0, sizeof(reload_t));
    reload_p->view_p = view_p;
    reload_p->target_en = target_en;
    
    NOOP("rodent_full_reload_view: calling THREAD_CREATE(reload_f), target entry is local: %d\n",
	    (target_en && IS_LOCAL_TYPE(target_en->type)));
    THREAD_CREATE(reload_f, reload_p, "reload_f");
    return 1;
}

