//
//
/* this file is included by rodent_popup.c */
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
typedef struct keybind_t {
    guint key;
    guint mask;
    gpointer callback;
} keybind_t;

typedef struct filechooser_t {
    GtkWidget *parent;
    void *combo_info;
    int filechooser_action;
    const gchar *folder;
    GtkEntry *entry;
    void (*activate_func) (GtkEntry * entry, gpointer activate_user_data);
    gpointer activate_user_data;
    const gchar *title;

} filechooser_t;


/**************************************************************************/
/**************************  pasteboard stuff  ****************************/
/**************************************************************************/

//extern char *src_host;

/*  */
static
int gui_pasteboard_list (GList ** list_p);

static
int gui_pasteboard_transfer (widgets_t * widgets_p,
                             record_entry_t * t_en, 
			     GList * list, 
			     gboolean cut, 
			     gboolean symlink);

	    
static gchar *
strip_path(gchar *command_fmt, const gchar *path){
    if (!path) return command_fmt;
    NOOP("stipping %s\n", command_fmt);
    gchar *end = strstr(command_fmt,path) + strlen(path);
    *strstr(command_fmt, path) = 0;
    gchar *g = g_strconcat(command_fmt, "%","s",end, NULL);
    g_free(command_fmt);
    command_fmt=g;
    NOOP("stipped %s\n", command_fmt);
    return command_fmt;
}

static gchar *
get_text_editor(record_entry_t *en){
    if(!g_path_is_absolute(en->path)) {
	return NULL;
    }
    gchar **text_editors = NULL;
    // Default editor command.
    // We magic the file because the freedesktop standard
    // is borked and changes text/ for application/ on a 
    // number of files. Precisely what should not be done.
    //
    if(!en->mimetype) {
	if (IS_LOCAL_TYPE(en->type)) en->mimemagic = MIME_magic (en->path);
	else en->mimemagic = g_strdup(_("unknown"));
    }
    if (!en->mimetype) en->mimetype = MIME_type (en->path, en->st);
    if(en->mimemagic && strstr (en->mimemagic, "text/")){
	    text_editors = MIME_apps ("text/plain");
    } else if(en->mimetype && strstr (en->mimetype, "text/")){
	    text_editors = MIME_apps ("text/plain");
    }

    // Even magic mimetype may miss the mark. Take care of that here... 
    if(!en->filetype) {
	if (IS_LOCAL_TYPE(en->type)) en->filetype = MIME_file (en->path);
	else en->filetype = g_strdup(_("unknown"));
    }
    if(!text_editors) {
	if(en->filetype && strstr (en->filetype, "text"))
	    text_editors = MIME_apps ("text/plain");
    }
    
    if(!text_editors) {
	return NULL;
    }
    g_strfreev(text_editors);
    /* OK to apply an editor. We will use the default editor */
    gchar *basename=NULL;
    if(getenv ("EDITOR") && strlen (getenv ("EDITOR"))) {
	basename=g_path_get_basename(getenv ("EDITOR"));
	if (strchr(basename, ' ')) *strchr(basename, ' ') = 0;
	gchar *fullpath=g_find_program_in_path(basename);
	// Don't show the EDITOR_CMD if it does not exist
	if (fullpath) {
	    // if nano or vi, then use terminal emulator
	    g_free(basename);
	    basename=g_strdup(getenv ("EDITOR"));
	    if (strcmp(basename, "vi")==0 || strcmp(basename, "nano")==0){
		const gchar *t=getenv("TERMINAL_CMD");
		if (!t || !strlen(t)) t="xterm";
		gchar *b=g_strdup_printf("%s %s %s",
			t, rfm_term_exec_option(t), basename);
		g_free(basename);
		basename = b;
	    }
	    g_free(fullpath);
	} else {
	    g_free(basename);
	    basename=NULL;
	}
    }
    return (basename);
}

static
subthread_t *
create_subthread_p(widgets_t *widgets_p){
    subthread_t *subthread_p=(subthread_t *)malloc(sizeof(subthread_t));
    memset(subthread_p, 0, sizeof(subthread_t));
    subthread_p->widgets_p=widgets_p;

    // default is to use values in widgets_p
    subthread_p->widgets.diagnostics=widgets_p->diagnostics;
    subthread_p->widgets.diagnostics_window=widgets_p->diagnostics_window;

    return subthread_p;
}

static
    void
gui_pasteboard_copy_cut (widgets_t * widgets_p, gboolean cut, GSList ** paste_list) {
    gint len;
    gchar *buffer;
    gchar *files;

    if(*paste_list == NULL){
	g_warning("*paste_list == NULL");
        return;
    }

    if(cut)
        rfm_status (widgets_p, "xffm/stock_dialog-info", _("Cut"), NULL);
    else
        rfm_status (widgets_p, "xffm/stock_dialog-info", _("Copy"), NULL);

    //gdk_flush ();
    rfm_clear_paste_buffer();

    len = 1 + strlen ("#xfvalid_buffer:copy:%%:\n");
    len += strlen (g_get_host_name ());
    GSList *tmp = *paste_list;
    for(; tmp; tmp = tmp->next) {
        gint addlen;
        record_entry_t *en = (record_entry_t *) tmp->data;
        addlen = 0;
        len += (1 + strlen (en->path) + addlen);
    }
    buffer = (gchar *)malloc (len * sizeof (char) + 1);
    if(!buffer) {
        g_warning ("rfm: unable to allocate paste buffer\n");
        return;
    }
    sprintf (buffer, "#xfvalid_buffer:%s:%s:\n", (cut) ? "cut" : "copy", g_get_host_name ());
    files = buffer + strlen (buffer);

    for(tmp = *paste_list; tmp; tmp = tmp->next) {
        record_entry_t *en = (record_entry_t *) tmp->data;
        {
            strcat (buffer, en->path);
            strcat (buffer, "\n");
        }
    }
    NOOP("dbg:len=%d,strlen=%d,data=%s\n",len,strlen(buffer),buffer); 
    rfm_store_paste_buffer(buffer, len);
    g_free (buffer);
    buffer = NULL;

    if(cut)
        rfm_status (widgets_p, "xffm/stock_dialog-info", _("Cut"), NULL);
    else
        rfm_status (widgets_p, "xffm/stock_dialog-info", _("Copy"), NULL);
    view_t *view_p = widgets_p->view_p;
    //view_p->flags.pasteboard_serial++;
    NOOP ("PASTE: view_p->flags.pasteboard_serial=%d\n", view_p->flags.pasteboard_serial);
    gchar *value = g_strdup_printf ("%d", view_p->flags.pasteboard_serial + 1);
    if (rfm_rational (MODULE_DIR, "settings", (void *)"RFM_PASTEBOARD_SERIAL", (void *)value, "mcs_set_var") == NULL){
        g_warning("cannot set RFM_PASTEBOARD_SERIAL");
    }
    g_free (value);
    // This is a callback function, so the GDK_MUTEX is already enabled
    // and it is safe to call rodent_expose_all().
    rodent_expose_all (view_p);
}


static
    int
gui_pasteboard_list (GList ** list_p) {

    // client side pasteboard, via MIT-shm
    gchar *b=rfm_get_paste_buffer();
    if((!b) || (!strlen (b))) {
      no_pasteboard:
	g_free(b);
	return 0;
    }
    gboolean cut;
    gchar *word;
    if((word = strtok (b, ":")) == NULL)
        goto no_pasteboard;
    if(!strstr (word, "#xfvalid_buffer"))
        goto no_pasteboard;
    if((word = strtok (NULL, ":")) == NULL)
        goto no_pasteboard;
    if(strstr (word, "cut"))
        cut = TRUE;
    else
        cut = FALSE;
    if((word = strtok (NULL, ":")) == NULL)
        goto no_pasteboard;
    if(!word) {
        goto no_pasteboard;
    }
    //src_host = g_strdup(word);

    word = word + strlen (word) + 1;
    if(word[0] == '\n') {
        word++;
        if(word[0] == 0)
            goto no_pasteboard;
    } else {
        if((word = strtok (NULL, "\n")) == NULL)
            goto no_pasteboard;
        word = word + strlen (word) + 1;
    }

    /* create list to send to CreateTmpList */
    gint retval = rfm_uri_parse_list (word, list_p);
 
    g_free (b);
    if (retval) {
	if(cut) retval=1;
	else retval= 2;
    }
    return retval;
}

static
    int
gui_pasteboard_transfer (widgets_t * widgets_p,
                         record_entry_t * t_en, 
			 GList * list, 
			 gboolean cut,
			 gboolean symlink) {
    gchar *url = (gchar *) list->data;
    if(!url){
	g_warning("gui_pasteboard_transfer: !url");
        return 0;
    }
 
    if(t_en->module) {
        NOOP ("PASTE: en->module=%s\n", t_en->module);
        if(rfm_natural (PLUGIN_DIR, t_en->module, t_en, "valid_drop_site")) {
            NOOP ("module: valid_drop_site for %s\n", t_en->module);
            rfm_natural (PLUGIN_DIR, t_en->module, t_en, "set_drop_entry");
            if(rfm_rational (PLUGIN_DIR, t_en->module, list, widgets_p, "process_drop")) {
                NOOP ("module: process_drop ok\n");
                /*result=TRUE; */
            }
            rfm_void (PLUGIN_DIR, t_en->module, "clear_drop_entry");
            return 1;
        }
    }
    NOOP ("PASTE: must be local then\n");
    int mode;
    if(cut) {
        mode = TR_MOVE;
    } else {
        mode = TR_COPY;
    }
    if(symlink) {
        mode = TR_LINK;
    }
       
    // Only the uploading notice works here...
    gchar *text=NULL;
    const gchar *icon=NULL;
    //gint type=0;
    gboolean local_target = TRUE;
    //gboolean local_source = TRUE;
    //if (!IS_LOCAL_TYPE(type))local_source = FALSE;
    if (!IS_LOCAL_TYPE(t_en->type))local_target = FALSE;
    if (!local_target){
      switch (mode){
        case TR_COPY:
	case TR_MOVE:
        case TR_COPY_REMOTE:
	case TR_MOVE_REMOTE:
	    icon = "xffm/status_network-transmit";
	    text = g_strdup_printf(_("Uploading file %s"), "...");
	    break;
	default:
	    break;
      }
    } else if (list) {
	// Local target. Test first item of list for remote host.
	const gchar *path = list->data;
	record_entry_t *en = rfm_stat_entry(path, 0);
	if (!IS_LOCAL_TYPE(en->type)) {
	    icon = "xffm/status_network-receive";
	    text = g_strdup_printf(_("Downloading file %s..."), "");
	}
	rfm_destroy_entry(en);
   }


    
    if (text) {
	rfm_diagnostics(widgets_p, "xffm/actions_document-send", NULL);
	rfm_diagnostics(widgets_p, icon, NULL);
	rfm_diagnostics(widgets_p, "xffm_tag/red", text, "\n", NULL);
	g_free(text);
    }
    rodent_cp (mode, widgets_p, list, t_en->path);
    return 1;
}

/**************************************************************************/
/*************************   end of pasteboard stuff **********************/
/**************************************************************************/
static
void rodent_save_workdir_history (char *p);

static
int rodent_diff (widgets_t * widgets_p, char *file1, char *file2);

static
GtkWidget *
thread_add_menu_item (widgets_t * widgets_p,
                   GtkWidget * parent,
                   const gchar * label,
                   const gchar * icon_id,
                   const gchar * name,
                   gpointer callback, 
		   gpointer callback_data, 
		   GtkAccelGroup * accel_group,
		   guint accel_key, 
		   guint accel_mask,
		   gboolean insert);

static
    void
  gui_add_to_keylist (guint in_key, guint mask, gpointer callback);

static
GtkWidget *gui_mk_radio_button (widgets_t * widgets_p,
                                const gchar * label, 
				const gchar * name, 
				GtkWidget * parent, 
				GSList ** radiogroup);

static void
thread_add_autotype_C_widgets (widgets_t * widgets_p,
                            GtkWidget * popup, 
			    gpointer callback, 
			    gpointer on_mount, 
			    gpointer on_unmount);

static
GtkWidget *thread_add_menu_separator (GtkWidget * parent);

/**************************************************************************/

///// popup callbacks....

/* this file is included by rodent_popup.c */

static gchar *
get_command_fmt(record_entry_t *en) {
    gchar *command_fmt = MIME_command (en->mimetype);
    if(!command_fmt) {
        command_fmt = MIME_command (en->mimemagic);
    }
    gboolean is_text = ((en->mimetype && strstr (en->mimetype, "text/")) ||
                        (en->mimemagic && strstr (en->mimemagic, "text/")) || 
                        (en->filetype && strstr (en->filetype, "text")));
    if(!command_fmt && is_text && getenv ("EDITOR_CMD") && strlen (getenv ("EDITOR_CMD"))) {
        command_fmt = g_strdup (getenv ("EDITOR_CMD"));
    }
    return command_fmt;
}

typedef struct execute_t{
    widgets_t *widgets_p;
    GSList *list;
} execute_t;

static void *
execute (void *data) {
    if (rfm_global_p->self != g_thread_self()) {
	g_error("execute() is not a thread function (yet...)\n");
    }
    execute_t *execute_p = data;
    widgets_t *widgets_p = execute_p->widgets_p;
    GSList *selection_list = execute_p->list;
    gpointer retval=GINT_TO_POINTER(1);
    
    view_t *view_p = widgets_p->view_p;
    gchar *command_fmt=NULL;
    

    /* set the working directory */

    g_free (widgets_p->workdir);
    if(view_p->en && IS_LOCAL_TYPE(view_p->en->type) && view_p->en->path)
        widgets_p->workdir = g_strdup (view_p->en->path);
    else
        widgets_p->workdir = g_strdup (g_get_home_dir ());

    NOOP ("on_open_with_activate...\n");
    gchar *files = g_strdup (" ");
    gchar *files_txt;

    gchar *first_path=NULL;
    if(selection_list) {
	files_txt = g_strdup_printf (_("Open with %s"),":   \n\n");
        if(g_slist_length (selection_list)) {
            gchar *tt,
             *ttt;
            files = g_strdup (" ");
            GSList *tmp = selection_list;
            for(; tmp; tmp = tmp->next) {
                record_entry_t *en = (record_entry_t *) tmp->data;
                char *b = g_path_get_basename (en->path);
                gchar *q = rfm_utf_string (rfm_chop_excess (b));
                tt = g_strconcat (files_txt, q, "\n", NULL);
                gchar *esc_path = rfm_esc_string (en->path);
		if (!first_path){
		    first_path=g_strdup(esc_path);
		    command_fmt=get_command_fmt(en);
		}
                ttt = g_strconcat (files, esc_path, " ", NULL);
                g_free (esc_path);
                g_free (files_txt);
                g_free (q);
                g_free (b);
                g_free (files);
                files_txt = tt;
                files = ttt;
            }
        }
    } else {
	// no selection
	files_txt = g_strdup_printf ("%s \n\n", _("Command:"));
    }
    NOOP ("OPEN: files=%s\n", files);
    NOOP ("OPEN: first_path=%s\n", first_path);
    gboolean interm = FALSE;
    const gchar *g;
    {
        gchar *f = g_build_filename (RUN_DBH_FILE, NULL);
        gchar *ff = g_build_filename (RUN_FLAG_FILE, NULL);
        NOOP ("RUN_DBH_FILE=%s RUN_FLAG_FILE=%s\n", f, ff);
	
	gchar *title;
	if (selection_list) {
	    title=g_strdup_printf(_("Open with %s"),"");
	} else {
	    title=g_strdup(_("Execute Shell Command"));
	}
	// Hmmm.... this here will unlock the GDK mutex because a gtk loop is entered...
        g = rodent_get_response_history (NULL,
		title, 
		files_txt,
		_("Console: quickly run single commands -- write a command here and press enter."),
		f, 
		first_path,
		command_fmt, //NULL, // entry text
		ff,
		_("Run in Terminal"),
		GTK_FILE_CHOOSER_ACTION_OPEN,
		"/usr/bin",
		MATCH_COMMAND); 
       g_free (f);
        g_free (ff);
    }
    g_free (first_path);
    g_free (command_fmt);

    g_free (files_txt);
    if(!g) {
        NOOP ("on_open_with_activate... !g\n");
	retval=NULL;
	goto cleanup;
    }
    if(g[strlen (g) + 1])
        interm = TRUE;

    if(selection_list) {
        /* if only one file selected, associate to mimetype... 
	 * but not default unless no other command available 
	 * (i.e., append, not prepend)*/
        if(g_slist_length (selection_list) == 1) {
	    record_entry_t *en = selection_list->data;
	    if(!en->mimetype) {
		if (IS_LOCAL_TYPE(en->type)) {
		    //en->mimetype = en->mimemagic = MIME_magic (en->path);
		    en->mimemagic = MIME_magic (en->path);
		} 
	    }   
	    const gchar *type = en->mimetype;
	    if (!type) type = en->mimemagic;
            if(type) {
                gchar *command_fmt = g_strdup (g);
                if(interm) {
                    gchar *term_command = MIME_mk_terminal_line (command_fmt);
                    g_free (command_fmt);
                    command_fmt = term_command;
                }
		if (strstr(command_fmt, en->path)){
		    command_fmt = 
			strip_path(command_fmt,en->path);
		}
                NOOP(stderr, "OPEN: adding %s --> %s (from %s)\n", type, command_fmt, g);
		rfm_rational(MODULE_DIR,"mime", 
			(void *)(en->mimetype),
			(void *)command_fmt, "mime_append");
		// MIME_add would prepend, which is now deprecated:
		// MIME_add (view_p->mouse_event.selected_p->en->mimetype, command_fmt);
		g_free(command_fmt);
            }
        }
    }
    gchar *command;
    if(strstr (g, "%s")) {
        command = g_strdup_printf (g, files);
    } else {
        command = g_strdup_printf ("%s %s", g, files);
    }
    NOOP ("OPEN: command = %s\n", command);
    g_free (files);

    if(widgets_p->diagnostics_window && *(widgets_p->diagnostics_window)){
	if (!gtk_widget_get_visible(*(widgets_p->diagnostics_window))){
	    rfm_show_text(widgets_p);
	}
    } else {
	rfm_show_text(widgets_p);
    }

    // do the call with argv so that command not saved in lpterm history
    // (but this is broken when we have pipes or redirection)
    gboolean shell_it = FALSE;
    gchar *tokens="|><;&";
    gchar *c=tokens;
    for (c=tokens; *c; c++){
	if (strchr(command, *c)){
	    shell_it=TRUE;
	    break;
	} 
    }
    if (shell_it){
	RFM_THREAD_RUN (widgets_p, command, interm);
    } else {
	RFM_THREAD_RUN2ARGV (widgets_p, command, interm);
    }

    g_free (command);
    // Cleanup
cleanup:
    for (selection_list = execute_p->list; selection_list && selection_list->data;
	    selection_list = selection_list->next){
	record_entry_t *en = selection_list->data;
	rfm_destroy_entry(en);
    }
    if (selection_list) g_slist_free(execute_p->list);
    g_free(data);
    return retval;
}

/*   callback  */
static void
on_open_with_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    execute_t *execute_p = (execute_t *)malloc(sizeof(execute_t));
    if (!execute_p) g_error("cannot allocate execute_p\n");
    memset(execute_p, 0, sizeof(execute_t));
    execute_p->widgets_p = widgets_p;

    GSList *selection_list = view_p->selection_list;
    for (; selection_list && selection_list->data; selection_list = selection_list->next){
	record_entry_t *in_en = selection_list->data;
	record_entry_t *out_en = rfm_copy_entry(in_en);
	execute_p->list = g_slist_append(execute_p->list, out_en); 
    }
    rfm_population_read_unlock (view_p);

    execute(execute_p);
}

static void
gui_on_help_activate (GtkMenuItem * menuitem, gpointer data) {
    widgets_t *widgets_p=data;
    gchar *help_file=g_strdup_printf("%s/doc/%s/RTFM.pdf", PACKAGE_DATA_DIR, RC_DIR);
    // This is a local and absolute path.
    if (g_file_test(help_file, G_FILE_TEST_EXISTS)){
	g_warning("%s: %s", help_file, strerror(ENOENT));
    } else {
	record_entry_t *en=rfm_stat_entry(help_file, 0);
	en->mimetype=MIME_type(en->path, NULL);
	rodent_open_with (widgets_p, en);
	rfm_destroy_entry(en);
    }
    g_free(help_file);
}




static void
destroy_about(GtkWidget *about, gpointer data){
    void *p;
    p=g_object_get_data(G_OBJECT(about), "comments");
    g_free(p);
    p=g_object_get_data(G_OBJECT(about), "program_name");
    g_free(p);
    p=g_object_get_data(G_OBJECT(about), "version");
    g_free(p);
    p=g_object_get_data(G_OBJECT(about), "copyright");
    g_free(p);
    gtk_widget_hide(about);
    gtk_widget_destroy(about);
}

#include "gplv3.h"
static void
 gui_on_about_activate(GtkMenuItem * menuitem, gpointer data){
    gchar *argv[3];
    argv[0] = "rodent";
    argv[1] = "--version";
    argv[2] = NULL;
    widgets_t *widgets_p = (widgets_t *) data;

    rfm_show_text (widgets_p);
    rfm_thread_run_argv (widgets_p, argv, FALSE);
    gchar *tests=NULL;
    if (!tests) {
	tests=g_strdup_printf("%s (%s)", _("Tests"), _("development version"));
    }
     gchar *authors[] = { "Rodent",
	 "   Edscott Wilson García <edscott@xfce.org>", 
	 "",
	 _("Initial idea, basic architecture, much initial source code"),
	 "   Rasca, Berlin",
	 "",
	 tests,
	 "   Gregorio Inda",
	 "   Harold Aling",
	 "   Juri Hamburg",
	 "   Populus Tremula",
	 "",
	 _("Contributors"),
	 "   GNU cp",
	 "      Torbjorn Granlund",
	 "      David MacKenzie",
	 "      Jim Meyering",
	 "   GNU mv",
	 "      Mike Parker",
	 "      David MacKenzie",
	 "      Jim Meyering",
	 "   GNU mv",
	 "      Mike Parker",
	 "      David MacKenzie",
	 "   GNU touch",
	 "      Paul Rubin",
	 "      Arnold Robbins",
	 "      Jim Kingdon",
	 "      David MacKenzie",
	 "      Randy Smith",
	 "   GNU rm",
	 "      Paul Rubin",
	 "      David MacKenzie",
	 "      Richard M. Stallman",
	 "      Jim Meyering",
	 "   GNU shred",
	 "      Colin Plumb",
	 "   libmagic",
	 "      Måns Rullgård",
	 "      Christos Zoulas",
	 "      Guy Harris",
	 "      Rob McMahon",
	 "      Geoff Collyer",
	 "      John Gilmore",
	 "      Ian Darwin",
	 "   GNU ps",
	 "      Branko Lankester",
	 "      Michael K. Johnson",
	 "      Michael Shields",
	 "      Charles Blake",
	 "      Albert Cahalan",
	 "      David Mossberger-Tang",
	 "",
	 _("Open Source"),
	 "   Free Software Foundation, Inc.",
	 "   Nils Rennebarth",
	 "   Bill Wilson",
	 "   Dov Grobgeld",
	 "   Tadej Borovsak",
	 "",
	 _("Contributors to older versions:"),
	 "   Olivier Fourdan",
	 "   Jasper Huijsmans",
	 "   Benedikt Meurer",
	 "   Eduard Roccatello",
	 "   Brian Tarricone",
	 NULL };
     gchar *artists[] = {
	 "Francois Le Clainche <fleclainche at wanadoo.fr>",
	 "Pablo Morales Romero <pg.morales.romero@gmail.com>",
	 NULL};
  //   g_object_set (obj, "authors", authors, NULL);
/*     gchar *writers[];
     g_object_get (obj, "authors", &writers, NULL);
     // do something with writers
     g_strfreev (writers);*/

     gchar *program_name=g_strdup_printf("%s %s", PACKAGE_NAME, TAG);
     gchar *rodent_string=g_strdup_printf(_("This is %s, version %s."), "Rodent Filemanager", TAG); 
     gchar *version=g_strdup_printf("\n(build %s aka xffm-%s)", BUILD, PACKAGE_VERSION);
     gchar *copyright=g_strdup_printf("%s\n%s", COPYRIGHT, _("This is free software with ABSOLUTELY NO WARRANTY."));
     gchar *comments=g_strdup_printf("\
%s\n\
%s\n\n\
%s: %s\n\n\
%s\n\
",
_("Hello World"),
rodent_string,
_("Suggestions, bug reports"), PACKAGE_BUGREPORT,
_("Rodent is fast, small and powerful file manager for the\n\
GNU operating system (but it also works in BSD).\n\
That's one way to look at it. Another way is to call\n\
it a graphic shell (that's probably more accurate).\n\
\n\
Rodent is *not* a filemanager written for dummies.\n\
Emphasis is on ease of use for the advanced user,\n\
not the computer illiterate.\n\
\n\
Rodent is a cross between a command line terminal and an\n\
iconview: a centaur filemanager.\n")
); 
     g_free(rodent_string);
 
     GdkPixbuf *pixbuf=rfm_get_pixbuf("xffm/stock_directory", BIG_ICON_SIZE);

     //     How do we tell the about dialog
     //     to stay above and be sticky for deskview?
     //     We cannot get the GtkWidget pointer if we do it the 
     //     easy way.


     GtkWidget *about= gtk_about_dialog_new();
     // set icon
     gtk_window_set_icon(GTK_WINDOW(about),
	     rfm_get_pixbuf("xffm/stock_about", SIZE_ICON));
     gchar *title=g_strdup_printf("%s Rodent %s", _("About"), TAG);
     gtk_window_set_title(GTK_WINDOW(about), title);
     g_free(title);
    
     gtk_window_stick(GTK_WINDOW(about));
     gtk_window_set_keep_above(GTK_WINDOW(about), TRUE);
     if (widgets_p->view_p->flags.type != DESKVIEW_TYPE){
	 gtk_window_set_transient_for(GTK_WINDOW(about), GTK_WINDOW(widgets_p->window));
	 gtk_window_set_modal(GTK_WINDOW(about), TRUE);
     }
     
     
     const gchar *rfm_team="rodent-translation-team";
     gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG(about), _(rfm_team));
	     // const gchar *translator_credits);
     g_object_set(G_OBJECT(about),
	"artists",		artists,
        "authors",		authors,
	"comments",           comments,
	"copyright",          copyright,
	"license",            GPLV3,
	"logo",               pixbuf,
	"version",            version,
	"website",            HOMEPAGE,
	"program-name",       program_name,
	NULL);

    g_object_set_data(G_OBJECT(about), "version", version);
    g_object_set_data(G_OBJECT(about), "comments", comments);
    g_object_set_data(G_OBJECT(about), "program_name", program_name);

    g_signal_connect (G_OBJECT (about), "destroy_event", G_CALLBACK (destroy_about), widgets_p);
    g_signal_connect (G_OBJECT (about), "delete_event", G_CALLBACK (destroy_about), widgets_p);

    gtk_widget_show(about);
    gint response=gtk_dialog_run(GTK_DIALOG(about));
    if (response==GTK_RESPONSE_CANCEL){
	destroy_about(about, widgets_p);
    }
    /* free these with destroy_about()...
     g_free(comments);
     g_free(program_name);*/


}



static void
on_autotype_S (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en;
    widgets_t *widgets_p = (widgets_t *) user_data;
    en = g_object_get_data(G_OBJECT(menuitem), "record_entry");	    
    if (en && en->path){
	NOOP("on_autotype_S: %s\n", en->path);
	g_free(widgets_p->workdir);
	widgets_p->workdir=g_path_get_dirname(en->path);
	if (!rfm_g_file_test_with_wait(widgets_p->workdir, G_FILE_TEST_IS_DIR)){
	    rodent_time_out(widgets_p, widgets_p->workdir);
	    g_free(widgets_p->workdir);
	    widgets_p->workdir = g_strdup(g_get_home_dir());
	} else {
	    gchar *argv[2]={en->path, NULL};
	    rfm_thread_run_argv(widgets_p, argv, FALSE);
	}
    } 
    return;
}

static void
on_autotype_R (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en;
    widgets_t *widgets_p = (widgets_t *) user_data;
    en = g_object_get_data(G_OBJECT(menuitem), "record_entry");	    
    if (en) {
	if (en->mimetype && 
	    strcmp(en->mimetype, "application/x-desktop")==0 && 
	    rfm_rational(PLUGIN_DIR, "dotdesktop", widgets_p, en,  "double_click")) 
	{
	    return;
	}

	if (en->path && IS_EXE_TYPE(en->type))
	{
	    NOOP("executing %s\n", en->path);
	    g_free(widgets_p->workdir);
	    widgets_p->workdir=g_path_get_dirname(en->path);
	    if (!rfm_g_file_test_with_wait(widgets_p->workdir, G_FILE_TEST_EXISTS)){
		rodent_time_out(widgets_p, widgets_p->workdir);
		g_free(widgets_p->workdir);
		widgets_p->workdir = g_strdup(g_get_home_dir());
	    } else {
		gchar *argv[2]={en->path, NULL};
		rfm_thread_run_argv(widgets_p, argv, TRUE);
	    }
	}
    } 
    return;
}
static void
on_bookmark_add_activate (GtkMenuItem * w, gpointer data) {
    widgets_t *widgets_p = (widgets_t *) data;
     const gchar *path = g_object_get_data(G_OBJECT(w), "path");
     if (!g_path_is_absolute(path)){
	 return; 
     }
     
     if (rodent_bookmarks_add(path)) {
	 GdkRectangle *rect=g_object_get_data(G_OBJECT(w), "rect");
	 if (rect) 
	     rfm_thread_expose_rect (widgets_p->view_p, rect);
	view_t * view_p = widgets_p->view_p;
	rodent_set_view_title (view_p);
     
     }
     return;
}

static void
on_bookmark_remove_activate (GtkMenuItem * w, gpointer data) {
    widgets_t *widgets_p = (widgets_t *) data;
     const gchar *path = g_object_get_data(G_OBJECT(w), "path");
     NOOP("Removing path %s\n", path);
     if (rodent_bookmarks_remove(path)) {
 	 GdkRectangle *rect=g_object_get_data(G_OBJECT(w), "rect");
	 if (rect) 
	    rfm_thread_expose_rect (widgets_p->view_p, rect);
	view_t * view_p = widgets_p->view_p;
	rodent_set_view_title (view_p);
     }
     return;
}

static void
setup_bookmark_menuitem(widgets_t *widgets_p, 
    const population_t *population_p, 
    const gchar *menuitem,
    gboolean on){
    view_t *view_p=widgets_p->view_p;
    record_entry_t *en=(population_p)?population_p->en:view_p->en;
    GtkWidget *w=
	g_object_get_data(G_OBJECT(widgets_p->paper), menuitem);
    if (!w) {
	g_warning("no %s widget!", menuitem);
    } else {
	gchar *basename=g_path_get_basename(en->path);
	gchar *baseutf=rfm_utf_string (basename);
	g_free(basename);
	gchar *text=g_strdup_printf("%s: <b><i>%s</i></b>",
	       (on)?_("Add bookmark"):_("Remove bookmark"), 
	       baseutf);
	g_free(baseutf);
	GtkWidget *label = gtk_bin_get_child (GTK_BIN (w));
	gchar *q = rfm_utf_string (text);
	g_free (text);
	gtk_label_set_markup ((GtkLabel *) label, q);
	g_free (q);

	SHOW_IT (view_p->widgets.paper, menuitem);
	gchar *path = g_object_get_data(G_OBJECT(w), "path");
	g_free(path);
	g_object_set_data(G_OBJECT(w), "path",   g_strdup(en->path));
	NOOP("setting path to %s\n", en->path);
	if (population_p) {
	    GdkRectangle *rect=g_object_get_data(G_OBJECT(w), "rect");
	    if (!rect) rect=(GdkRectangle *)malloc(sizeof(GdkRectangle));
	    if (rfm_get_population_icon_rect(view_p, population_p, rect)) {
		g_object_set_data(G_OBJECT(w), "rect",   rect);
	    } else {
		g_object_set_data(G_OBJECT(w), "rect",   NULL);
	    }
	} else {
	    g_object_set_data(G_OBJECT(w), "rect",   NULL);
	}
    }
}

static void
on_new_window_activate (GtkMenuItem * w, gpointer user_data) {
    NOOP("on_new_window_activate\n");
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    gchar *new_path=NULL;
    if (view_p->module) {
	new_path=g_strconcat("exec:rodent-", view_p->module, NULL);
    } else if (view_p->en) {
	 if (!rfm_g_file_test_with_wait(view_p->en->path, G_FILE_TEST_IS_DIR)){
	     rodent_time_out(widgets_p, view_p->en->path);
	     return; 
	 }
	new_path=g_strdup(view_p->en->path);
    }	

    rodent_new_gridview(widgets_p, new_path);
    g_free(new_path);
    return;
}



static void
rodent_open_in_terminal_activate (GtkMenuItem * menuitem, gpointer user_data) {
    gchar *shell = rfm_xterm_shell();
    if (!shell) return;
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    g_free (widgets_p->workdir);
    if(!view_p->en 
	    || !view_p->en->path 
	    || !g_path_is_absolute(view_p->en->path)){
        widgets_p->workdir = g_strdup (g_get_home_dir ());
    } else { 
	if (!rfm_g_file_test_with_wait(view_p->en->path, G_FILE_TEST_IS_DIR)){
            rfm_confirm (NULL, GTK_MESSAGE_ERROR, strerror (EEXIST), NULL, NULL);
	    g_free(shell);
	    return;

	} else  widgets_p->workdir = g_strdup (view_p->en->path);
    }

     gchar *argv[] = { shell, NULL };
    // use argv version so that it won't go into history file...
    rfm_thread_run_argv (widgets_p, argv, TRUE);
    g_free(shell);

    return;
}

static void
rodent_differences_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    gchar *file1 = NULL;
    gchar *file2 = NULL;

    if(view_p->selection_list) {
        file1 = ((record_entry_t *) (view_p->selection_list->data))->path;
        if(g_slist_length (view_p->selection_list) > 1){
            file2 = ((record_entry_t *) (view_p->selection_list->next->data))->path;
	}
    }
    NOOP ("rodent-diff %s %s\n", (file1 ? file1 : "null"), (file2 ? file2 : "null"));

    if(!view_p->en || !view_p->en->path || 
	    !rfm_g_file_test_with_wait(view_p->en->path, G_FILE_TEST_EXISTS)){
        widgets_p->workdir = g_strdup (g_get_home_dir ());
    } else {
        widgets_p->workdir = g_strdup (view_p->en->path);
    }
    if (file1 && !rfm_g_file_test_with_wait(file1, G_FILE_TEST_EXISTS)) {
	 rodent_time_out(widgets_p, file1);
    }
    if (file2 && !rfm_g_file_test_with_wait(file2, G_FILE_TEST_EXISTS)) {
	 rodent_time_out(widgets_p, file2);
    }
    rodent_diff (widgets_p, file1, file2);
}

static void
on_settings_show (GtkMenuItem * w, gpointer data) {
    // widgets_p was used to set dialog as transient, but alas,
    // no more. Transient windows (gtk style) are for dummies...
    // Okay, okay, everybody is a dummy now and then, 
    // but not always!
    //widgets_t *widgets_p = (widgets_t *) data;
    if(rfm_void (MODULE_DIR, "settings", "module_active")) {
        rfm_void (MODULE_DIR, "settings", "run_rfm_settings_dialog");
    } else {
        g_warning ("on_settings_show(): cannot load libsettings");
    }
    return;

}

static void
on_newdir_activate (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = NULL;
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(view_p) en = view_p->en;
    if(!en || !en->path) return;

    if (!rodent_entry_available(widgets_p, view_p->en)) return; 

    gchar *label = g_strdup_printf ("%s/", en->path);
    gchar *q = rfm_utf_string (rfm_chop_excess (label));
    const gchar *g = rodent_get_response (NULL, _("Create New Folder"), q,
                                          _("Name"));
    g_free (label);
    g_free (q);

    if(g && strlen (g)) {
      gchar *p = g_build_filename (en->path, g, NULL);
      // 
      if(!rfm_locate_path(view_p, p))
      {
        gchar *argv[4];
        int j=0;
        argv[j++]="mkdir";
        argv[j++]=p;
        argv[j++]=NULL;

	if (en->st->st_mode == 0) {
	    g_warning("This should not happen: en->st->st_mode == 0\n");
	    return;
	}	

	if (rfm_write_ok(en->st)){
	    rfm_thread_run_argv (widgets_p, argv, FALSE);
	} else {
	    gchar *base=g_path_get_basename(p);
	    gchar *operation=g_strconcat("mkdir ", base, NULL);
	    g_free(base);
	    if (rfm_confirm_sudo(widgets_p, en->path, 
		    _("write failed"), operation)){
		RFM_TRY_SUDO (widgets_p, argv, FALSE);
		rfm_show_text(widgets_p);
		rfm_diagnostics(widgets_p, "xffm/stock_properties",NULL);
		rfm_diagnostics(widgets_p, "xffm_tag/green",
			_("Don't forget"), " ", NULL);
		rfm_diagnostics(widgets_p, "xffm_tag/magenta",
		    "chown pid:gid", " ", p, "\n", NULL);
	    }
	    g_free(operation);
	}
      }
      else {
            rfm_confirm (NULL, GTK_MESSAGE_ERROR, strerror (EEXIST), NULL, NULL);
      }
      g_free(p);
    }
}

static void
on_newfile_activate (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = NULL;
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(view_p){
        en = view_p->en;
    }
    if(!en || !en->path) {
	return;
    }
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 

    gchar *label = g_strdup_printf ("%s/", en->path);
    gchar *q = rfm_utf_string (rfm_chop_excess (label));
    const gchar *g = rodent_get_response (NULL, _("New file"), q, _("Name"));
    g_free (q);
    g_free (label);
    if(g && strlen (g)) {
        gchar *p = g_build_filename (en->path, g, NULL);
        if(!rfm_locate_path(view_p, p)) {
	    gchar *argv[3];
	    int j=0;
	    argv[j++]="touch";
	    argv[j++]=p;
	    argv[j++]=NULL;
	    

	    if (rfm_write_ok(en->st)){
		rfm_thread_run_argv (widgets_p, argv, FALSE);
	    } else {
		gchar *base=g_path_get_basename(p);
		gchar *operation=g_strconcat("touch ", base, NULL);
		g_free(base);
		if (rfm_confirm_sudo(widgets_p, en->path, 
			_("write failed"), operation)){
		    RFM_TRY_SUDO (widgets_p, argv, FALSE);
		    rfm_show_text(widgets_p);
		    rfm_diagnostics(widgets_p, "xffm/stock_properties",NULL);
		    rfm_diagnostics(widgets_p, "xffm_tag/green",
			    _("Don't forget"), " ", NULL);
		    rfm_diagnostics(widgets_p, "xffm_tag/magenta",
			"chown pid:gid", " ", p, "\n", NULL);
		}
		g_free(operation);
	    }
         } else {
            rfm_confirm (NULL, GTK_MESSAGE_ERROR, strerror (EEXIST), NULL, NULL);
        }
        g_free (p);
    }
}


// Determines the work directory of the command to be executed
// to do things like extract a tar file to a selected diretory
static gchar * 
autofunction_workdir (view_t * view_p, record_entry_t * en, const gchar * querypath) {
    widgets_t *widgets_p;
    widgets_p = &(view_p->widgets);
    static gchar *last_dir = NULL;
    const gchar *g;

    if(querypath) {
        gchar *f = g_build_filename (WORKDIR_DBH_FILE, NULL);
        const gchar *folder;
        if(view_p->en && view_p->en->path && IS_SDIR(view_p->en->type)) {
            folder = view_p->en->path;
        } else {
            folder = g_get_home_dir ();
        }
        NOOP ("COMBO: last_dir is %s\n", last_dir);
	//const gchar *default_dir=last_dir;
	const gchar *default_dir=NULL;
	if (!default_dir) default_dir=en->path;
	if (!rodent_entry_available(widgets_p, en)) {
	    default_dir=g_get_home_dir();
	} else if (!rfm_g_file_test_with_wait(default_dir, G_FILE_TEST_IS_DIR) || 
		access(en->path, R_OK|W_OK|X_OK) < 0) {
	    default_dir=g_get_home_dir();
	}
        g = rodent_get_response_history (NULL, _(querypath), 
			    _("Path"), _("Select directory"),
			    f, NULL,
                            default_dir,
                            NULL, NULL,
                            GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                            folder,
			    MATCH_FILE);
        g_free (f);

        if(!g)
            return NULL;
	gboolean exists = rfm_g_file_test_with_wait (g, G_FILE_TEST_EXISTS);
	gboolean isdir = rfm_g_file_test_with_wait (g, G_FILE_TEST_IS_DIR);

	
	if(exists && !isdir) {
            rfm_status (widgets_p, "xffm/stock_dialog-warning", strerror (ENOTDIR), NULL);
            return NULL;
        }
        if(!exists) {
            gchar *b = g_strdup_printf ("%s: %s\n%s...", g,
                                        strerror (ENOENT),_("Create New Folder"));
            if(!rfm_confirm (widgets_p, GTK_MESSAGE_QUESTION, b, _("Cancel"), NULL)) {
                g_free (b);
                return NULL;
            }
            g_free (b);
            if(g_mkdir_with_parents (g, 0750) < 0) {
                rfm_confirm (widgets_p, GTK_MESSAGE_ERROR, strerror (errno), NULL, NULL);
                return NULL;
            }
        }

    
        g_free (last_dir);
        last_dir = g_strdup (g);
        NOOP ("COMBO: setting last_dir to %s\n", last_dir);
        if(isdir)
            rodent_save_workdir_history (widgets_p->workdir);
        else {
            rfm_show_text(widgets_p);
            rfm_diagnostics (widgets_p, "xffm/stock_dialog-error", NULL);
            rfm_status (widgets_p, "xffm_tag/stderr", g, ": ", strerror (errno), "\n", NULL);
            return NULL;
        }
        return g_strdup(g);
    }

    return NULL;
}

static void
module_goto (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    const gchar *module_name = g_object_get_data (G_OBJECT (menuitem), "module_name");
    TRACE("module_goto(): %s\n", module_name);
    if(module_name) {
      if (view_p->child_constructor) {
	TRACE("module_goto(): using child_constructor function.\n");
	// Module name is constructed with "rodent-[module_name]".
	// This cuts down on the code.
	gchar *m = g_strconcat("exec:rodent-",module_name, NULL);
        (*view_p->child_constructor) (widgets_p, (char *)m);
	g_free(m);
      } else {
        rodent_push_view_go_history (view_p);
        record_entry_t *new_en = rfm_mk_entry (0);
        SET_ROOT_TYPE (new_en->type);
        new_en->module = module_name;
        new_en->path = rfm_void (PLUGIN_DIR, module_name, "module_label");
	if (!rodent_refresh (widgets_p, new_en)){ rfm_destroy_entry(new_en); }
      }
    } else {
        g_warning ("cannot get path from module-goto menu");
    }
    return;
}

static void
level_goto (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    const gchar *path;
    path = g_object_get_data (G_OBJECT (menuitem), "path");
    NOOP("path is %s\n", path);
    if(path) {
        if(view_p->child_constructor) {
            (*view_p->child_constructor) (widgets_p, (char *)path);
        } else {
            rodent_push_view_go_history (view_p);
	    record_entry_t *new_en;

	    if (rfm_g_file_test_with_wait(path, G_FILE_TEST_EXISTS)){
		     new_en = rfm_stat_entry (path, 0);
	    } else {
		rfm_confirm (NULL, GTK_MESSAGE_ERROR, strerror (EEXIST), NULL, NULL);
		return;
	    }   
	    if (new_en && !new_en->module) {
		rfm_save_to_go_history ((gchar *) path);
	    }
	    if (!rodent_refresh (widgets_p, new_en)){ 
		rfm_destroy_entry(new_en); 
	    }
	    
        }
    } else {
        g_warning ("cannot get path from level-goto menu");
    }
    return;
}

// autotype C is "open with" operator which is determined by
// means of the command associated data item of the widget
static void
execute_autotype (GtkMenuItem * menuitem, gpointer data) {

    view_t *view_p = g_object_get_data (G_OBJECT (menuitem), "view_p");
    const gchar *command;
    const gchar *dirname;
    gchar *new_command=NULL;
    const gchar *querypath,
     *output_arg,
     *output_ext;
    record_entry_t *en;
    widgets_t *widgets_p = &(view_p->widgets);

    en = view_p->en;

    output_arg = g_object_get_data (G_OBJECT (menuitem), "output_arg");
    command = g_object_get_data (G_OBJECT (menuitem), "command");
    dirname = g_object_get_data (G_OBJECT (menuitem), "workdir");
    querypath = g_object_get_data (G_OBJECT (menuitem), "querypath");
    output_ext = g_object_get_data (G_OBJECT (menuitem), "output_ext");

    // Assign command as default command here, if applicable.
    if (g_object_get_data(G_OBJECT(menuitem), "CTL_SET")) {
	NOOP("CTL is set\n");
        if(g_slist_length (view_p->selection_list) == 1 && view_p->mouse_event.selected_p) {
	    // must substitute path in command line for %s
	    gchar *command_fmt=g_strdup(command);
	    if (view_p->mouse_event.selected_p->en->path &&
		strstr(command_fmt, view_p->mouse_event.selected_p->en->path))
	    {
		command_fmt=strip_path(command_fmt,view_p->mouse_event.selected_p->en->path);
	    }

	    if(!view_p->mouse_event.selected_p->en->mimetype) {
		if (IS_LOCAL_TYPE(view_p->mouse_event.selected_p->en->type)) 
		    view_p->mouse_event.selected_p->en->mimemagic = 
			MIME_magic (view_p->mouse_event.selected_p->en->path);
		else view_p->mouse_event.selected_p->en->mimemagic = 
		    g_strdup(_("unknown"));
	    }    
            if(view_p->mouse_event.selected_p->en->mimetype) {
                NOOP (stderr, "OPEN: adding %s --> %s \n", view_p->mouse_event.selected_p->en->mimetype, command_fmt);
		// This sets it as default command for the specified mimetype
		MIME_add (view_p->mouse_event.selected_p->en->mimetype, command_fmt);
            }
	    g_free(command_fmt);
        }
    }




    NOOP("execute_autotype: %s\n", command);
    gint argc;
    gchar **argv;
    GError *error=NULL;

    if(!g_shell_parse_argv (command, &argc, &argv, &error)) {
        g_warning("execute_autotype(%s): %s", command, error->message);
        g_error_free (error);
        g_strfreev (argv);
        NOOP ("!g_shell_parse_argv at gui_autostuff\n");
        return;
    }

// internal commands:
    if (strcmp(argv[0], "rodent-newtab")==0){
        if(view_p->tab_constructor) {
            (*(view_p->tab_constructor))(widgets_p, argv[1]);
        }
        return;
    }
    if (strcmp(argv[0], "rodent-newwin")==0){
	rodent_new_gridview(widgets_p, argv[1]);
        return;
    }
    // We don't use this any more:
    g_strfreev (argv);

    gchar *workdir=NULL;
    if(querypath) {
        workdir=autofunction_workdir (view_p, en, querypath);
        if (!workdir) return;
    }

    if (dirname) {// XXX I wonder what this is for?
        g_free (widgets_p->workdir);
        widgets_p->workdir = g_strdup (dirname);
    }
    if(output_ext) {
        gchar *basename = g_path_get_basename (output_arg);
        gchar *output_path=g_strconcat("\"",workdir, G_DIR_SEPARATOR_S, basename, output_ext,"\"",NULL);
        gchar *esc_path = g_strconcat("\"",basename,"\"",NULL);
        g_free(basename);
        basename=esc_path;

        gchar *s=strstr(command,"%s");
        if (s) {
        // XXX this should have at least two %s tokens...
            new_command = g_strdup_printf(command, output_path, basename);
        } else {
                NOOP("no %%s in command");
            new_command = g_strconcat (command, " ", output_path, " ", basename, NULL);
        }
        command = (const gchar *)new_command;
        g_free (output_path);
        g_free (basename);
    } else if (workdir) {
        g_free (widgets_p->workdir);
        widgets_p->workdir = g_strdup (workdir); 
    }
    NOOP ("output_arg=%s, command=%s, querypath(string)=%s, output_ext=%s, dirname=%s workdir=%s\n", output_arg, command, querypath,
           output_ext, dirname, widgets_p->workdir);


    rfm_show_text(widgets_p);
    RFM_THREAD_RUN2ARGV (widgets_p, (void *)command, FALSE);
        
    g_free (new_command);
    g_free (workdir);


}

static void
rodent_up_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(!view_p) {
        g_warning ("!view_p");
    } else {
        if(view_p->population_pp && view_p->population_pp[0]) {
            if(view_p->population_pp[0]->en && !IS_DUMMY_TYPE (view_p->population_pp[0]->en->type))
                return;
            rodent_push_view_go_history (view_p);
            record_entry_t *en = rfm_copy_entry (view_p->population_pp[0]->en);
	    if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
        }
    }
}


/*********************** private callbacks ************************************/
#define RENAME_CASO 	0
#define DUPLICATE_CASO 	1
#define SYMLINK_CASO 	2

static gboolean
easy_data_test (view_t * view_p) {
    if(!view_p->en || !view_p->mouse_event.selected_p || !view_p->mouse_event.selected_p->en) {
        rfm_show_text(&(view_p->widgets));
        rfm_diagnostics (&(view_p->widgets), "xffm/stock_dialog-error", NULL);
        rfm_diagnostics (&(view_p->widgets), "xffm_tag/stderr", strerror (EINVAL), "\n", NULL);
        return FALSE;
    }
    return TRUE;
}

static void
rodent_rename_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!easy_data_test (view_p)) {
        g_warning ("!easy_data_test");
        return;
    }
    if(!rfm_population_try_read_lock (view_p)) {
        g_warning ("cannot get readlock!");
        return;
    }

    rodent_mk_text_entry (view_p, view_p->mouse_event.selected_p, RENAME_CASO);
    rfm_population_read_unlock (view_p);
}

static void
rodent_duplicate_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!easy_data_test (view_p))
        return;
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }

    rodent_mk_text_entry (view_p, view_p->mouse_event.selected_p, DUPLICATE_CASO);
    rfm_population_read_unlock (view_p);
}

static void
rodent_symlink_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!easy_data_test (view_p))
        return;
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    rodent_mk_text_entry (view_p, view_p->mouse_event.selected_p, SYMLINK_CASO);
    rfm_population_read_unlock (view_p);
}

static void
rodent_unselect_all_view (view_t * view_p) {
    rfm_cursor_wait (view_p->widgets.window);
    rodent_unselect_all_pixbuf (view_p);
 //   is not necessary: rodent_expose_all (view_p);
    rfm_cursor_reset (view_p->widgets.window);
}

static int
rodent_select_all_view (view_t * view_p) {
    population_t **population_pp;
    int items = 0;
    if(!view_p || !view_p->en)
        return items;
    rfm_cursor_wait (view_p->widgets.window);
    population_pp = view_p->population_pp;
    for(; population_pp && *population_pp; population_pp++) {
        if((*population_pp)->en == NULL)
            continue;
        if(IS_DUMMY_TYPE ((*population_pp)->en->type))
            continue;
        items++;
        NOOP ("selected item %d", items);
        //
        rodent_select_pixbuf (view_p, *population_pp);
	rfm_expose_item (view_p, *population_pp);
    }
 //   is not necessary: rodent_expose_all (view_p);
    rfm_cursor_reset (view_p->widgets.window);
    return items;
}

enum {
    END_MATCH,
    START_MATCH,
    TOMATO_IN_SANDWICH_MATCH,
    BREAD_ON_SANDWICH_MATCH,
    EXACT_MATCH
};

static gint
rodent_select_byfilter_view (struct view_t *view_p, const gchar * select_filter) {
    population_t **population_pp;
    gint items = 0;
    if(!view_p || !view_p->en)
        return items;
    if(!select_filter || !strlen (select_filter))
        return items;
    if(strcmp (select_filter, "*") == 0)
        return rodent_select_all_view (view_p);
    rfm_cursor_wait (view_p->widgets.window);

    gchar *filter = g_strdup (select_filter);
    memset (filter, 0, strlen (select_filter));
    int j = 0,
        i = 0,
        caso;
    for(; i < strlen (select_filter); i++) {
        if(select_filter[i] != '*')
            filter[j++] = select_filter[i];
    }

    if(select_filter[0] == '*' && select_filter[strlen (select_filter) - 1] == '*') {
        caso = TOMATO_IN_SANDWICH_MATCH;
    } else if(select_filter[0] == '*') {
        caso = END_MATCH;
    } else if(select_filter[strlen (select_filter) - 1] == '*') {
        caso = START_MATCH;
    } else if(strchr (select_filter, (int)'*')) {
        caso = BREAD_ON_SANDWICH_MATCH;
    } else {
        caso = EXACT_MATCH;
    }

    rodent_unselect_all_pixbuf (view_p);
    population_pp = view_p->population_pp;
    for(; population_pp && *population_pp; population_pp++) {
        gchar *f;
        gboolean match = FALSE;
        if((*population_pp)->en == NULL)
            continue;
        if((*population_pp)->en->path == NULL)
            continue;
        if(IS_DUMMY_TYPE ((*population_pp)->en->type))
            continue;
        f = g_path_get_basename ((*population_pp)->en->path);
        gchar *p = strcasestr (f, filter);

        if(p && caso == END_MATCH) {
            gchar *pp;
            do {
                pp = p;
                p = strcasestr (p + 1, filter);
            }
            while(p);
            p = pp;
        }

        if(p) {
            NOOP ("p=%s, filter=%s\n", p, filter);
            switch (caso) {
            case TOMATO_IN_SANDWICH_MATCH:
                match = TRUE;
                break;
            case END_MATCH:
                if(p[strlen (filter)] == 0)
                    match = TRUE;
                break;
            case START_MATCH:
                if(p == f)
                    match = TRUE;
                break;
            case EXACT_MATCH:
                if(strlen (p) == strlen (filter))
                    match = TRUE;
                break;
            case BREAD_ON_SANDWICH_MATCH:
                //not implemented. Need to do 2 matches.
                //break;
            default:           //START_END_MATCH
                match = TRUE;
                break;

            }
        }
        if(match) {
            items++;
            rodent_select_pixbuf (view_p, *population_pp);
	    rfm_expose_item (view_p, *population_pp);
        }
        g_free (f);
    }
    rfm_cursor_reset (view_p->widgets.window);
    gchar *plural_text=g_strdup_printf (ngettext ("%'u item", "%'u items",
                                               items),items);
    gchar *g = g_strdup_printf ("%s: %s", _("Selection"), plural_text);
    g_free(plural_text);

    rfm_status (&(view_p->widgets), "xffm/stock_dialog-info", g, NULL);
    g_free (g);
    g_free (filter);

    return items;
}

static void
rodent_unselect_all_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(!view_p)
        return;
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    rodent_unselect_all_view (view_p);
    rfm_population_read_unlock (view_p);
}

static void
rodent_select_all_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    int items = 0;
    if(!view_p)
        return;
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }

    rfm_status (&(view_p->widgets), "xffm/stock_dialog-info", _("Select All"), "...", NULL);
    rfm_cursor_wait (view_p->widgets.window);
    items = rodent_select_all_view (view_p);
    gchar *plural_text=g_strdup_printf (
	ngettext ("%'u item", "%'u items", g_slist_length(view_p->selection_list)),
	g_slist_length(view_p->selection_list));
    gchar *g = g_strdup_printf ("%s: %s", _("Selection"), plural_text);
    g_free(plural_text);

    rfm_status (&(view_p->widgets), "xffm/stock_dialog-info", g, NULL);
    g_free (g);
   
    rfm_cursor_reset (view_p->widgets.window);
    rfm_population_read_unlock (view_p);
}

static void
rodent_touch_activate (GtkMenuItem * menuitem, gpointer user_data) {
    //record_entry_t *en = NULL;
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    if (view_p && g_slist_length(view_p->selection_list) >= 1){
	subthread_t *subthread_p=create_subthread_p(widgets_p);
	GSList *tmp=view_p->selection_list;
	for (;tmp && tmp->data; tmp=tmp->next) {
	    record_entry_t *en=tmp->data;
	    subthread_p->selection_list=
		g_slist_append(subthread_p->selection_list, g_strdup (en->path));
	}
	THREAD_CREATE(touch_dialog, subthread_p, "touch_dialog");
    }

    rfm_population_read_unlock (view_p);
    return;
}

/***   properties_callbacks  ***/

static void
rodent_on_prop_activate (GtkMenuItem * menuitem, gpointer data) {
    widgets_t *widgets_p = (widgets_t *) data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!view_p){
        return ;
    }
    if (!view_p->selection_list || g_slist_length (view_p->selection_list) < 1){
        return ;
    }
    if (!rfm_population_read_lock (view_p)) return;
    if(view_p->module && 
	    rfm_rational (PLUGIN_DIR, view_p->module,
		widgets_p,  view_p->selection_list, "do_properties")
	    ) {
        // module has done the properties thing here;
    } else {
	rfm_natural(MODULE_DIR, "properties", widgets_p, "do_prop");
    }
    rfm_population_read_unlock (view_p);
}


static void
rodent_select_by_filter (GtkMenuItem * menuitem, gpointer user_data) {
    const gchar *filter_s;
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(!view_p || !view_p->en || !view_p->en->path)
        return;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    gchar *b = g_strdup_printf ("%s/", view_p->en->path);
    gchar *label = rfm_utf_string (rfm_chop_excess (b));
    filter_s = rodent_get_response (&(view_p->widgets), _("Select Items Matching..."), label, "*");
    g_free (label);
    g_free (b);

    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    if(filter_s && strlen (filter_s)) {
        rodent_select_byfilter_view (view_p, filter_s);
    }
    rfm_population_read_unlock (view_p);
    return;
}

static void
rodent_hidden_toggled (GtkToggleButton * togglebutton, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p;
    view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!user_data)
        return;
    if(view_p->flags.disable_toggle_refresh)
        return;
    view_p->flags.preferences ^= __SHOW_HIDDEN;
    rfm_save_view_preferences (view_p, view_p->en);
    NOOP ("rodent_refresh by rodent_hidden_toggled... %s\n", (view_p->flags.preferences & __SHOW_HIDDEN) ? "ON" : "Off");
    record_entry_t *en = rfm_copy_entry (view_p->en);
    if (SHOWS_HIDDEN(view_p->flags.preferences)) {
	SET_SHOWS_HIDDEN(en->type);
    } else {
	UNSET_SHOWS_HIDDEN(en->type);
    }

    if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
}

static void
rodent_preview_toggled (GtkToggleButton * togglebutton, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!user_data)
        return;
    if(view_p->flags.disable_toggle_refresh)
        return;
    view_p->flags.preferences ^= __SHOW_IMAGES;
    rfm_save_view_preferences (view_p, view_p->en);
    NOOP ("rodent_refresh by rodent_preview_toggled... %s\n", (view_p->flags.preferences & __SHOW_IMAGES) ? "ON" : "Off");
    record_entry_t *en = rfm_copy_entry (view_p->en);
    if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
}

/* sort type radio: */
static void
rodent_on_radio (GtkCheckMenuItem * checkmenuitem, gpointer user_data) {
    view_t *view_p = (view_t *) g_object_get_data (G_OBJECT (checkmenuitem), "view_p");
    if(!view_p)
        return;
    widgets_t *widgets_p = &(view_p->widgets);
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    view_p->flags.sortcolumn = GPOINTER_TO_INT (user_data);
    switch (GPOINTER_TO_INT (user_data)){
	case DATE_SORT:
	case SIZE_SORT:
	case MODE_SORT:
	view_p->flags.preferences &= (SORT_ASCENDING ^ 0xffffffff);
		break;
	default:
	view_p->flags.preferences |= SORT_ASCENDING;
    }
    rfm_save_view_preferences (view_p, view_p->en);
    if(!view_p->flags.disable_toggle_refresh) {
        record_entry_t *en = rfm_copy_entry (view_p->en);
	if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
    }
}

static void
on_ascending (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!view_p){
	return;
    }
//    if (view_p->flags.type == DESKVIEW_TYPE) { rodent_erase_deskview_geometries (); }
	
    view_p->flags.preferences |= SORT_ASCENDING;
    rfm_save_view_preferences (view_p, view_p->en);
    if(!view_p->flags.disable_toggle_refresh) {
        record_entry_t *en = rfm_copy_entry (view_p->en);
	if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
    }
    return;
}

static void
on_descending (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!view_p)
        return;
//    if (view_p->flags.type == DESKVIEW_TYPE) { rodent_erase_deskview_geometries (); }
    view_p->flags.preferences &= (SORT_ASCENDING ^ 0xffffffff);
    rfm_save_view_preferences (view_p, view_p->en);
    if(!view_p->flags.disable_toggle_refresh) {
        record_entry_t *en = rfm_copy_entry (view_p->en);
	if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
    }
    return;
}

static void
private_mount (widgets_t *widgets_p, int mounted) {
    record_entry_t *en = NULL;
    gchar *mount_point;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(view_p->mouse_event.selected_p && view_p->mouse_event.selected_p->en) {
        en = view_p->mouse_event.selected_p->en;
    }
    NOOP ("MOUNT: private_mount %d (0x%x)\n", mounted, GPOINTER_TO_INT(view_p));
    if(!en || !en->path) {
        g_warning ("no entry to mount");
        return;
    }
    if(mounted){
        SET_MOUNTED_TYPE (en->type);
    } else{
        UNSET_MOUNTED_TYPE (en->type);
    }
    mount_point = (gchar *) FSTAB_fstab_mount (widgets_p, en);
    rodent_unselect_all_activate (NULL, (gpointer) widgets_p);
    rfm_cursor_reset (widgets_p->window);

    NOOP ("MOUNT: private_mount done\n");
    return;
}

static void
rodent_mount (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    private_mount (widgets_p, FALSE);
    rfm_population_read_unlock (view_p);
    return;
}

static void
rodent_unmount (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    private_mount (widgets_p, TRUE);
    rfm_population_read_unlock (view_p);
    return;
}

/*   pasteboard callbacks   */
// XXX what the heck with t_en? does not seemed used for anything...
static void
private_paste (widgets_t * widgets_p, view_t * view_p, gboolean symlink) {
    gboolean cut;
    GList *list = NULL;
    record_entry_t *t_en = NULL;
    int i;


    
    if((t_en = view_p->en) == NULL){
	g_warning("(t_en = view_p->en) == NULL");
        return;
    }


    if((i = gui_pasteboard_list (&list)) == 0){
	g_warning("(i = gui_pasteboard_list (&list)) == 0");
        return;
    }
    if(i == 1)
        cut = TRUE;
    else
        cut = FALSE;
    i = gui_pasteboard_transfer (widgets_p, t_en, list, cut, symlink);

    list = rfm_uri_free_list (list);

    if(i) {
        if(cut){
	    rfm_clear_paste_buffer();
	}	    
    }
    return;
}

static void
copy_cut_callback (gboolean cut, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!view_p->selection_list)
        return;

    gui_pasteboard_copy_cut (widgets_p, cut, &(view_p->selection_list));
    // unselect stuff if it is cut or copied.
    GSList *tmp=view_p->selection_list;
    for (;tmp && tmp->data; tmp=tmp->next){
	record_entry_t *en=tmp->data;
	rfm_destroy_entry(en);
    }
    g_slist_free (view_p->selection_list);
    view_p->selection_list = NULL;

    if(rfm_population_try_read_lock (view_p)) {
        population_t **pp;
        for(pp = view_p->population_pp; pp && *pp; pp++) {
            population_t *population_p = *pp;
            population_p->flags  &= (POPULATION_SELECTED ^ 0xffffffff);
        }
        rfm_population_read_unlock (view_p);
    }

}

static void
rodent_cut_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    copy_cut_callback (TRUE, user_data);
    rfm_population_read_unlock (view_p);
    return;
}

static void
rodent_copy_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    copy_cut_callback (FALSE, user_data);
    rfm_population_read_unlock (view_p);
    return;
}

static void
paste_callback (gboolean symlink, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(!view_p->en)
        return;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    //rfm_details->pastepath = view_p->en->path;
    private_paste (widgets_p, view_p, symlink);
}

static void
rodent_paste_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
	rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning", "Unable to obtain readlock... aborting paste command. Please retry.",NULL);
        return;
    }
    paste_callback (FALSE, user_data);
    rfm_population_read_unlock (view_p);
    return;
}


static void
rodent_run_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    // force desktop diagnostics so window gets shown regardless
    // of RFM_DISABLE_DIAGNOSTICS setting.
    gboolean visible=rfm_diagnostics_is_visible(widgets_p);
    if (!visible) rfm_show_text(widgets_p);
        
    execute_t *execute_p = (execute_t *)malloc(sizeof(execute_t));
    if (!execute_p) g_error("cannot allocate execute_p\n");
    memset(execute_p, 0, sizeof(execute_t));
    execute_p->widgets_p = widgets_p;

    rfm_population_read_unlock (view_p);

    if (!execute (execute_p)){
	if (!visible && widgets_p->diagnostics_window==NULL){
	    rfm_hide_text(widgets_p);
	}
    }

}

static void
rodent_pop_view_go_history (view_t * view_p) {
    GList *last;
    if(!view_p->go_list)
        return;
    last = g_list_last (view_p->go_list);
    if(!last) {
        g_list_free (view_p->go_list);
        view_p->go_list = NULL;
        return;
    }
    view_p->go_list = g_list_remove (view_p->go_list, last->data);
    if(!g_list_length (view_p->go_list)) {
        g_list_free (view_p->go_list);
        view_p->go_list = NULL;
        return;
    }
    return;
}

static void
goup_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if(!view_p) {
        g_warning ("!view_p");
        return;
    }

    if(!view_p->en || !view_p->en->path)
        return;
        
    record_entry_t *en = NULL;
    if (view_p->population_pp && view_p->population_pp[0]){
	if (view_p->population_pp[0]->en && 
		view_p->population_pp[0]->en->module) {
	    en =rfm_copy_entry(view_p->population_pp[0]->en);
	} else if (view_p->population_pp[0]->en) {
	    gchar *path=g_strdup(view_p->population_pp[0]->en->path);
	    if (g_path_is_absolute(path)){
		while (!rfm_g_file_test_with_wait(path, G_FILE_TEST_IS_DIR)){
		    gchar *d = g_path_get_dirname(path);
		    g_free(path);
		    path = d;
		}
	    } else {
		g_error("goup_activate(): this should not happen\n");
	    }
	    en =rfm_stat_entry(path, 0);
	    g_free(path);
	}
    } 



    if (!en) {
	rodent_host_activate (menuitem, user_data);
	return;
    }
    if (IS_SDIR(en->type)){
        rodent_push_view_go_history (view_p);
    }
    if(view_p->child_constructor) {
	if (en->module) {
	  gchar *m=g_strdup_printf("exec:rodent-%s",
		  (gchar *)rfm_natural (PLUGIN_DIR, en->module, NULL, "module_name")
		  );
            (*view_p->child_constructor) (widgets_p, (char *)m);
	  g_free(m);
	} else {
          (*view_p->child_constructor) (widgets_p, en->path);
	}
    } else {
	if (IS_SDIR(en->type)){
	    rodent_push_view_go_history (view_p);
	}
	if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
    }

}

////////////////////  rodent_popup.i

static void
  rodent_mk_module_popup_menu (const gchar * module_name, view_t * view_p, record_entry_t * en);
static void execute_autotype (GtkMenuItem * menuitem, gpointer user_data);
static void rodent_mount (GtkMenuItem * menuitem, gpointer user_data);
static void rodent_unmount (GtkMenuItem * menuitem, gpointer user_data);

static void
rodent_save_workdir_history (char *p) {
    gchar *f = g_build_filename (WORKDIR_DBH_FILE, NULL);
    COMBOBOX_save_to_history (f, p);
    g_free (f);
}
/***********************************************************************************/

static void
remove_menu_item (GtkWidget * widget, gpointer data) {
    NOOP ("removing 0x%lx\n", (unsigned long)widget);
    gtk_widget_destroy (widget);
}

static void
close_menu_item_activate (GtkMenuItem * menuitem, gpointer data) {
    widgets_t *widgets_p = data;
    view_t *view_p = widgets_p->view_p;
    rodent_destroy_view_item (view_p);
    if (!xfdir_monitor_control_greenlight(widgets_p)){
	rodent_trigger_reload(view_p);
    }
    if (strstr(rfm_global_p->argv[0], "rodent-desk")){
	if (!fork()){
	    sleep(1);
	    execlp("killall", "killall", "rodent-desk", NULL);
	    _exit(123);
	}
    }
    return;
}


static void
thread_mk_module_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu)
{

    if (rfm_global_p->self == g_thread_self()){
	g_error("thread_mk_module_menu_items() is a thread function, only to be called from LWP\n");
    }
   GSList *list;
    GSList *plugin_list = rfm_find_plugins ();
    gint module_number = 0;
    for(list = plugin_list; list; list = list->next) {
        if(!rfm_is_root_plugin ((gchar *) (list->data))) {
	    NOOP(stderr, "%s is not a root module\n", 
		    (gchar *) (list->data));
	    continue;
	}
        
        if(++module_number > DEEPEST_MODULE_MENU_LEVELS)
            break;
        gchar *name = g_strdup ((gchar *) (list->data));
        const gchar *icon = rfm_get_plugin_icon (name);
        if(!icon){
	    icon = "xffm/stock_go-forward";
	}
        const gchar *label = rfm_get_plugin_label(name);
        if(!label){
            label = name;
	}
        GtkWidget *w = thread_add_menu_item (widgets_p, item22_menu, label, icon, 
		(accel_group==NULL)?NULL:name,
		module_goto, widgets_p, NULL, 0, 0, FALSE);
        g_object_set_data (G_OBJECT (w), "module_name", (gpointer) name);
    }
}

static void
thread_mk_bookmarks_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{


    gint level;


    const gchar *id= "xffm/emblem_link";
    for(level = 0; level < DEEPEST_BOOK_MENU_LEVELS; level++) {
        gchar *name = g_strdup_printf ("%s-%d",
		(accel_group)?"bookmark":"static-bookmark", level);
        thread_add_menu_item (widgets_p, item22_menu, name, id, name, level_goto, widgets_p, NULL, 0, 0, FALSE);
	g_free(name);
    }
    
    if (!accel_group) {
	// if accel group is NULL, then the construction is a static menu which is
	// regenerated when needed. In this case we need to set the bookmarks.
	rodent_bookmark_set_menuitems(widgets_p, "static-bookmark");
	
	return;
    }

    thread_add_menu_item (widgets_p, item22_menu, "subbookmark_item_yes", "xffm/places_user-bookmarks/composite/stock_yes", "subbookmark_item_yes", on_bookmark_add_activate, widgets_p, NULL, 0, 0, FALSE);
    thread_add_menu_item (widgets_p, item22_menu, "subbookmark_item_no", "xffm/places_user-bookmarks/composite/stock_no", "subbookmark_item_no", on_bookmark_remove_activate, widgets_p, NULL, 0, 0, FALSE);

//    thread_add_menu_separator (item22_menu);
}


static void
thread_mk_directory_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{

    if (!accel_group) return;

    gint level;

    const gchar *id = "xffm/stock_goto-last";
    for(level = 0; level < DEEPEST_DIR_MENU_LEVELS; level++) {
        gchar *name = g_strdup_printf ("level-%d", level);
        thread_add_menu_item (widgets_p, item22_menu, name, id, name, level_goto, widgets_p, NULL, 0, 0, FALSE);
	g_free(name);
    }
}

static void
thread_mk_goback_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{

    if (!accel_group) return;

    gint level;

    const gchar *id = "xffm/stock_goto-last";
    for(level = 0; level < DEEPEST_HISTORY_MENU_LEVELS; level++) {
        gchar *name = g_strdup_printf ("goback_item-%d", level);
        thread_add_menu_item (widgets_p, item22_menu, name, id, name, level_goto, widgets_p, NULL, 0, 0, FALSE);
	g_free(name);
    }
}

static void
thread_navigation_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{
     GtkWidget *w;
    //////// Places (Bookmarks) menu:
    GtkWidget *places_menu;
    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    gchar *g=g_strdup_printf("%s (%s)", _("Places"), _("Bookmarks"));
    places_menu = rodent_mk_menu (widgets_p, g,     /* label */
                   (accel_group==NULL)?NULL:"places_menu",    /* name */
                    item22_menu,    /* parent */
                                NULL,   /* callback (or NULL) */
                                "xffm/places_user-bookmarks");  /* iconfile (or NULL) */
    g_free(g);

    // Bookmarks:
    thread_mk_bookmarks_menu_items (widgets_p, accel_group, places_menu);
    w = thread_add_menu_separator (places_menu);

    // Root items:
    // thread_add_menu_item always protects with GDK mutex
    thread_add_menu_item (widgets_p, places_menu,
                       g_get_host_name (),
                       "xffm/device_computer", 
		       (accel_group==NULL)?NULL:"host2",
                       (gpointer) rodent_host_activate, 
		       widgets_p, accel_group, 
		       GDK_KEY_End, GDK_CONTROL_MASK, FALSE);
    // thread_add_menu_item always protects with GDK mutex
    thread_add_menu_item (widgets_p, places_menu,
                       _("Home"),
                       "xffm/stock_home", 
		       (accel_group==NULL)?NULL:"home2",
                       (gpointer) rodent_home_activate, 
		       widgets_p, accel_group, 
		       GDK_KEY_Home, GDK_CONTROL_MASK, FALSE);

    // module elements:
    // thread_mk_module_menu_items always protects with GDK mutex
    thread_mk_module_menu_items (widgets_p, accel_group, places_menu);

     widgets_p->navigation_popup = places_menu;


    //////  Go to menu /////

    

      GtkWidget *goto_menu;
      goto_menu = rodent_mk_menu (widgets_p, _("Go to"),     /* label */
                   (accel_group==NULL)?NULL:"goto_menu",    /* name */
                    item22_menu,    /* parent */
                    NULL,   /* callback (or NULL) */
                    "xffm/stock_go-forward");  /* iconfile (or NULL) */
    // History driven Go to dialog:
    // thread_add_menu_item always protects with GDK mutex
    thread_add_menu_item (widgets_p, goto_menu,
                       _("Choose Location"),
                       "xffm/stock_jump-to", 
		       (accel_group==NULL)?NULL:"goto2",
                       (gpointer) rodent_jump_to_activate, 
		       widgets_p, accel_group, 
		       GDK_KEY_Right, GDK_CONTROL_MASK, FALSE);
    // Go up menu item
    // thread_add_menu_item always protects with GDK mutex
    thread_add_menu_item (widgets_p,
                       item22_menu,
                       _("Go up"),
                       "xffm/stock_go-up",
                       (accel_group==NULL)?NULL:"goup_menuitem", 
		       (gpointer) goup_activate, 
		       widgets_p, accel_group, 
		       GDK_KEY_Up, GDK_CONTROL_MASK, FALSE);

    // Go back menu item 
    // (only shown if there is a back item in current session and view_p)
    // thread_add_menu_item always protects with GDK mutex
    thread_add_menu_item (widgets_p, item22_menu,
                       _("Go back"),
                       "xffm/stock_go-back", 
		       (accel_group==NULL)?NULL:"go_back_menuitem",
                       (gpointer) rodent_back_activate, 
		       widgets_p, accel_group, 
		       GDK_KEY_Left, GDK_CONTROL_MASK, FALSE);
      GtkWidget *goback_menu;
      goback_menu = rodent_mk_menu (widgets_p, _("History list"), 
	           /* label "History...""Go backward in history"*/
                   (accel_group==NULL)?NULL:"goback_menu",    /* name */
                    goto_menu,    /* parent */
                    NULL,   /* callback (or NULL) */
                    "xffm/stock_goto-first");  /* iconfile (or NULL) */
	
    thread_mk_goback_menu_items (widgets_p, accel_group, goback_menu);
      gtk_widget_show(goback_menu);

    w = thread_add_menu_separator (goto_menu);

    if (accel_group) {
	// Bookmarks menu
	// navigation bar equivalent. 
	thread_mk_directory_menu_items (widgets_p, accel_group, goto_menu);
    }

 		
    // thread_add_menu_item always protects with GDK mutex
    thread_add_menu_item (widgets_p,
		   //places_menu,
		   item22_menu,
		   _("Reload"),
		   "xffm/stock_refresh",
		   (accel_group==NULL)?NULL:"refresh3", 
		   (gpointer) rodent_refresh_activate, 
		   widgets_p, accel_group, 
		   GDK_KEY_Down, GDK_CONTROL_MASK, FALSE);

    if (accel_group==NULL && 
	    g_object_get_data(G_OBJECT(widgets_p->paper), "nonmain_applications_menu"))
    {
	    g_object_set_data(G_OBJECT(widgets_p->paper), "nonmain_applications_menu", NULL);
    }
    thread_add_menu_item (widgets_p,
                   item22_menu,
                   _("Applications"),
                   "xffm/category_accessories",
		   (accel_group==NULL)?"nonmain_applications_menu":"applications_menu", 
		   NULL, 
		   widgets_p, NULL, 
		   0, 0, FALSE);
    


    if (accel_group) {
	/* shift mask: */
	/*add_to_keylist(accel_key, accel_mask,  callback,  callback_data); */
	gui_add_to_keylist (GDK_KEY_Up, GDK_CONTROL_MASK, (gpointer) rodent_up_activate);
    }

    // End of Go category ////////////
    // End category separator:
    w = thread_add_menu_separator (item22_menu);

    if (accel_group) {
	RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "navigation_separator");
    }


}

static void
thread_view_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{
    if (rfm_global_p->self == g_thread_self()){
	g_error("thread_view_menu_items() is a thread function, only to be called from LWP\n");
    }
    GtkWidget *w;
    //view_t *view_p=widgets_p->view_p;
    //

    GtkWidget *sort1_menu;
    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread

    sort1_menu = rodent_mk_menu (widgets_p, _("Change Sort Order"),  /* label */
                                 "sort1",       /* name */
                                 item22_menu,   /* parent */
                                 NULL,  /* callback (or NULL) */
                                 NULL);       /* iconfile (or NULL) */

    thread_add_menu_item (widgets_p, sort1_menu,
                       _("Sort data in descending order"),
                       "xffm/stock_sort-descending", "descending1", 
		       (gpointer) on_descending, widgets_p, NULL, 0, 0, FALSE);
    thread_add_menu_item (widgets_p, sort1_menu,
                       _("Sort data in ascending order"),
                       "xffm/stock_sort-ascending", "ascending1", 
		       (gpointer) on_ascending, widgets_p, NULL, 0, 0, FALSE);

    {
        GSList *radiogroup = NULL;
        GtkWidget *unsorted6,
         *name6,
         *size6,
         *date6,
         *owner6,
         *group6,
         *mode6;

GDK_THREADS_ENTER();
        unsorted6 = gui_mk_radio_button (widgets_p, _("Type"), "unsorted6", sort1_menu, &radiogroup);
        name6 = gui_mk_radio_button (widgets_p, _("By Name"), "name6", sort1_menu, &radiogroup);
        size6 = gui_mk_radio_button (widgets_p, _("By Size"), "size6", sort1_menu, &radiogroup);
        date6 = gui_mk_radio_button (widgets_p, _("By Date"), "date6", sort1_menu, &radiogroup);
        owner6 = gui_mk_radio_button (widgets_p, _("By Owner"), "owner6", sort1_menu, &radiogroup);
        group6 = gui_mk_radio_button (widgets_p, _("By Group"), "group6", sort1_menu, &radiogroup);
        mode6 = gui_mk_radio_button (widgets_p, _("By Permissions"), "mode6", sort1_menu, &radiogroup);
        {
            view_t *view_p = widgets_p->view_p;
            if(view_p->flags.sortcolumn == TYPE_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) unsorted6, TRUE);
            else if(view_p->flags.sortcolumn == NAME_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) name6, TRUE);
            else if(view_p->flags.sortcolumn == SIZE_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) size6, TRUE);
            else if(view_p->flags.sortcolumn == DATE_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) date6, TRUE);
            else if(view_p->flags.sortcolumn == OWNER_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) owner6, TRUE);
            else if(view_p->flags.sortcolumn == GROUP_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) group6, TRUE);
            else if(view_p->flags.sortcolumn == MODE_SORT)
                gtk_check_menu_item_set_active ((GtkCheckMenuItem *) mode6, TRUE);

            g_signal_connect ((gpointer) unsorted6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (TYPE_SORT));
            g_object_set_data (G_OBJECT (unsorted6), "view_p", view_p);

            g_signal_connect ((gpointer) name6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (NAME_SORT));
            g_object_set_data (G_OBJECT (name6), "view_p", view_p);

            g_signal_connect ((gpointer) size6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (SIZE_SORT));
            g_object_set_data (G_OBJECT (size6), "view_p", view_p);

            g_signal_connect ((gpointer) date6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (DATE_SORT));
            g_object_set_data (G_OBJECT (date6), "view_p", view_p);

            g_signal_connect ((gpointer) owner6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (OWNER_SORT));
            g_object_set_data (G_OBJECT (owner6), "view_p", view_p);

            g_signal_connect ((gpointer) group6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (GROUP_SORT));
            g_object_set_data (G_OBJECT (group6), "view_p", view_p);

            g_signal_connect ((gpointer) mode6, "toggled", G_CALLBACK (rodent_on_radio), GINT_TO_POINTER (MODE_SORT));
            g_object_set_data (G_OBJECT (mode6), "view_p", view_p);
        }

    }

    /* end of sort menu items */

    w = gtk_check_menu_item_new_with_mnemonic (_("Show hidden files"));
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (item22_menu), w);
    RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "show_hidden_menuitem");
    g_signal_connect ((gpointer) w, "toggled", G_CALLBACK (rodent_hidden_toggled), widgets_p);



    w = gtk_check_menu_item_new_with_mnemonic (_("Show preview"));
    gtk_container_add (GTK_CONTAINER (item22_menu), w);
    RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "preview_images_menuitem");
    gtk_widget_show (w);
    g_signal_connect ((gpointer) w, "toggled", G_CALLBACK (rodent_preview_toggled), widgets_p);
GDK_THREADS_LEAVE();

}

static void
thread_size_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group,
	GtkWidget * item22_menu) 
{
    GtkWidget *zoom_menu;
    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    zoom_menu = rodent_mk_menu (widgets_p, _("Icon size"),     /* label */
                   (accel_group==NULL)?NULL:"zoom_menu",    /* name */
                    item22_menu,    /* parent */
                                NULL,   /* callback (or NULL) */
                                "xffm/stock_zoom-fit");  /* iconfile (or NULL) */

    GtkWidget *w;
    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Details"),
                       NULL, 
		       (accel_group==NULL)?NULL:"compact_iconsize",
                       (gpointer) rodent_details_iconsize, 
		       widgets_p, 
		       NULL, 0, 0, FALSE);
    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Compact"),
                       NULL, 
		       (accel_group==NULL)?NULL:"tiny_iconsize",
                       (gpointer) rodent_compact_iconsize, 
		       widgets_p, 
		       NULL, 0, 0, FALSE);
    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Normal"),
                       NULL, 
		       (accel_group==NULL)?NULL:"normal_iconsize",
                       (gpointer) rodent_normal_iconsize, 
		       widgets_p, 
		       NULL, 0, 0, FALSE);
    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Big"),
                       NULL, 
		       (accel_group==NULL)?NULL:"big_iconsize",
                       (gpointer) rodent_big_iconsize, 
		       widgets_p, 
		       NULL, 0, 0, FALSE);
    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Huge"),
                       NULL, 
		       (accel_group==NULL)?NULL:"huge_iconsize",
                       (gpointer) rodent_huge_iconsize, 
		       widgets_p, 
		       NULL, 0, 0, FALSE);
    /////////////////////////////

    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Increase the view size"),
                       "xffm/stock_zoom-in", 
		       (accel_group==NULL)?NULL:"plus_iconsize",
                       (gpointer) rodent_increase_iconsize, 
		       widgets_p, 

		       accel_group, GDK_KEY_KP_Add, GDK_CONTROL_MASK, FALSE);
//  extra key binding for Boxes without keypads 
    if (accel_group) gui_add_to_keylist(GDK_KEY_plus, GDK_CONTROL_MASK, rodent_increase_iconsize);

    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Decrease the view size"),
                       "xffm/stock_zoom-out", 
		       (accel_group==NULL)?NULL:"minus_iconsize",
                       (gpointer) rodent_decrease_iconsize, 
		       widgets_p, 
		       accel_group, GDK_KEY_KP_Subtract, GDK_CONTROL_MASK, FALSE);

    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Default Size"),
                       "xffm/stock_zoom-100", 
		       (accel_group==NULL)?NULL:"default_iconsize",
                       (gpointer) rodent_default_iconsize, 
		       widgets_p, 
		       accel_group, GDK_KEY_0, GDK_CONTROL_MASK, FALSE);

    thread_add_menu_item (widgets_p, zoom_menu,
                       _("Apply to all instances"),
                       "xffm/stock_zoom-100", 
		       (accel_group==NULL)?NULL:"default_iconsize_all",
                       (gpointer) rodent_default_iconsize_all, 
		       widgets_p, 
		       NULL, 0, 0, FALSE);

//  extra key binding for Boxes without keypads 
    if (accel_group) gui_add_to_keylist(GDK_KEY_minus, GDK_CONTROL_MASK, rodent_decrease_iconsize);

    w = thread_add_menu_separator (item22_menu);
    if (accel_group) RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "view_separator");

}

static void
thread_file_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{
    NOOP ("POPUPx: thread_file_menu_items...\n");
    //view_t *view_p=widgets_p->view_p;
    {
        gchar *openwith=g_strdup_printf(_("Open with %s"),"...");
        gchar *string[4] = { "", "", openwith, NULL };
        gchar *iconfile[4] = { "xffm/stock_execute",  "xffm/stock_execute","xffm/stock_open",   NULL };
        gchar *id[4] = { "autotype_Prun", "autotype_Srun","open_with_menuitem",   NULL };
        gpointer callback[4] = { (gpointer) on_autotype_R, (gpointer) on_autotype_S, (gpointer) on_open_with_activate, NULL };
        guint akey[4] = {0, 0, GDK_KEY_o, 0};
        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, item22_menu,
                               _(string[i]),
                               iconfile[i],
                               (accel_group==NULL)?NULL:id[i], 
			       callback[i], widgets_p, ((akey[i]) ? 
				   accel_group : NULL), akey[i], 
			       GDK_CONTROL_MASK, (i==0));
        }
    }
    GtkWidget *w = thread_add_menu_separator (item22_menu);
    if (accel_group) RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "file_separator2");
    GtkWidget *new_menu;
    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    new_menu = rodent_mk_menu (widgets_p, _("New"),     /* label */
                               "new_menu",      /* name */
                               item22_menu,     /* parent */
                               NULL,    /* callback (or NULL) */
                               "xffm/stock_new"); /* iconfile (or NULL) */

    {
        gchar *string[4] = { N_("Create new file in the given directory"), 
	    N_("Create a new empty folder inside this folder"),
            NULL
        };
        gchar *iconfile[4] = { "xffm/stock_new", "xffm/actions_folder-new", NULL };
        gchar *id[4] = { "newfile_menuitem", "newdirectory_menuitem",
	    NULL
        };
        gpointer callback[4] = { (gpointer) on_newfile_activate, (gpointer) on_newdir_activate };
        guint akey[4] = { GDK_KEY_Insert, GDK_KEY_Insert, 0 };
        guint amask[5] = { GDK_CONTROL_MASK, GDK_MOD1_MASK, 0 };
        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, new_menu,
                               _(string[i]),
                               iconfile[i], 
			       (accel_group==NULL)?NULL:id[i],
			       callback[i], 
			       widgets_p, 
			       ((akey[i]) ? accel_group : NULL),
			       akey[i], amask[i], FALSE);
        }
    }
}

static void
thread_exec_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu)
{
    if (accel_group != NULL){
        gchar *string[] = {
	    N_("Add bookmark"),
	    N_("Remove bookmark"),
            NULL
        };
        gchar *iconfile[] = { 
	    "xffm/places_user-bookmarks/composite/stock_yes",
	    "xffm/places_user-bookmarks/composite/stock_no",
            NULL
        };
        gchar *id[] = { 
	    "addbookmark_menuitem",
	    "removebookmark_menuitem",
            NULL
        };
        gpointer callback[] = {
	    (gpointer) on_bookmark_add_activate,
	    (gpointer) on_bookmark_remove_activate,
	    NULL 
        };
        guint akey[] = { 
	    0,0,0
        };
        guint amask[] = { 
	    0,0,0
        };
        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, item22_menu,
                               _(string[i]),
                               iconfile[i], 
			       id[i], 
			       callback[i], widgets_p, 
			       NULL,
			       akey[i], amask[i], FALSE);
        }
    }
    {
        gchar *string[] = {
	    N_("Open in New Window"),
	    N_("Open terminal here"),
            N_("Execute Shell Command"), 
	    N_("Search"), //"Find..." "Search"
            N_("Compare Files or Folders"), 
            NULL
        };
        gchar *iconfile[] = { 
	    "xffm/actions_window-new",
	    "xffm/apps_terminal",
            "xffm/stock_execute", 
	    "xffm/stock_find",
            "xffm/apps_diff-tool", 
            NULL
        };
        gchar *id[] = { 
	    "newwindow",
	    "terminal2",
            "run2", 
	    "find2",
            "differences2",
            NULL
        };
        gpointer callback[] = {
	    (gpointer) on_new_window_activate,
            (gpointer) rodent_open_in_terminal_activate,
            (gpointer) rodent_run_activate,
            (gpointer) rodent_glob_activate,
            (gpointer) rodent_differences_activate, 
	    NULL 
        };
        guint akey[] = { 
	    GDK_KEY_Page_Up,
	    GDK_KEY_t,
            GDK_KEY_e,
	    GDK_KEY_f,
            0,
	    0 
        };
        guint amask[] = { 
	    GDK_CONTROL_MASK,
	    GDK_CONTROL_MASK,
            GDK_CONTROL_MASK, GDK_CONTROL_MASK,
            GDK_CONTROL_MASK, 0 
        };
        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, item22_menu,
                               _(string[i]),
                               iconfile[i], 
			       (accel_group==NULL)?NULL:id[i], 
			       callback[i], widgets_p, 
			       ((akey[i]) ? accel_group : NULL), 
			       akey[i], amask[i], FALSE);
        }
    }

    GtkWidget *w = thread_add_menu_separator (item22_menu);
    if (accel_group) RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "file_separator");

}

static void
thread_fileexec_menu_items (widgets_t * widgets_p, GtkAccelGroup * accel_group, GtkWidget * item22_menu) {

    {
        gchar *string[] = {
	    N_("Properties"), 
            N_("Mount the volume associated with this folder"), 
            N_("Unmount the volume associated with this folder"),
	    N_("Rename"),
            N_("Duplicate each selected item"), //N_("D_uplicate"), 
	    N_("Create a symbolic link for each selected item"),   //N_("_Link Here"),
            N_("Touch")" (touch)", 
            NULL
        };
        gchar *iconfile[] = { 
	    "xffm/stock_properties", 
	    "xffm/stock_yes", 
	    "xffm/stock_no",
	    "xffm/stock_save-as",
            "xffm/stock_dnd-multiple", 
	    "xffm/stock_convert",
            "xffm/stock_index", 
            NULL
        };
        gchar *id[] = { 
	    "properties2", 
	    "mountP",
	    "unmountP",
	    "rename_menuitem",
            "duplicate_menuitem",
	    "symlink_menuitem",
            "touch_menuitem",
            NULL
        };
        gpointer callback[] = {  
	    (gpointer) rodent_on_prop_activate,
	    NULL, 
	    NULL,
            (gpointer) rodent_rename_activate,
            (gpointer) rodent_duplicate_activate,
            (gpointer) rodent_symlink_activate,
            (gpointer) rodent_touch_activate,
            NULL
        };
        guint akey[] = { 
	    GDK_KEY_p,
	    0, 
	    0,
	    GDK_KEY_r,
            GDK_KEY_d,
	    GDK_KEY_s,
            GDK_KEY_t, 
            0
        };
        guint amask[] = { 
	    GDK_CONTROL_MASK, 
	    0, 0,
            GDK_CONTROL_MASK, GDK_CONTROL_MASK,
            GDK_MOD1_MASK, GDK_CONTROL_MASK,
            0
        };
        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, item22_menu,
                               _(string[i]),
                               iconfile[i],
			       (accel_group==NULL)?NULL:id[i],
			       callback[i], widgets_p, ((akey[i]) ? 
				   accel_group : NULL), 
			       akey[i], amask[i], FALSE);
        }
    }

    GtkWidget *w = thread_add_menu_separator (item22_menu);
    if (accel_group) RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "file_separator");

}

static void
thread_edit_menu_items (widgets_t * widgets_p, 
	GtkAccelGroup * accel_group, 
	GtkWidget * item22_menu) 
{
    GtkWidget *w;
    GtkWidget *select_menu;
    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    select_menu = rodent_mk_menu (widgets_p, _("Select"),       /* label */
                                  "select_menu",        /* name */
                                  item22_menu,  /* parent */
                                  NULL, /* callback (or NULL) */
                                  "xffm/actions_select-items");        /* iconfile (or NULL) */

    {
        gchar *string[4] = { N_("Select All"), 
	    N_("Select Items Matching..."), 
	    N_("Unselect"),
            NULL
        };
        gchar *iconfile[4] = { "xffm/stock_select-all", "xffm/actions_select-items", "xffm/actions_select-none", NULL };
        gchar *id[4] = { "select_all_menuitem", "select_byfilter_menuitem",
            "unselect_all_menuitem", NULL
        };
        gpointer callback[4] = { (gpointer) rodent_select_all_activate,
            (gpointer) rodent_select_by_filter,
            (gpointer) rodent_unselect_all_activate, NULL
        };
        guint akey[4] = { GDK_KEY_a, GDK_KEY_s, GDK_KEY_u, 0 };
        guint amask[4] = { GDK_MOD1_MASK, GDK_MOD1_MASK, GDK_MOD1_MASK, 0 };

        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, select_menu,
                               _(string[i]),
                               iconfile[i], id[i], callback[i], widgets_p, 
			       ((akey[i]) ? 
				accel_group : NULL),
			       akey[i], amask[i], FALSE);
        }
    }
    //view_t *view_p=widgets_p->view_p;
    {
        gchar *string[6] = { N_("Cut"), 
	    N_("Copy"), 
	    N_("Paste"),
            N_("Delete"),
            NULL
        };
        gchar *iconfile[6] = { "xffm/stock_cut", "xffm/stock_copy", "xffm/stock_paste",
            "xffm/stock_delete", NULL
        };
        gchar *id[6] = { "cut_menuitem", "copy_menuitem", "paste_menuitem",
            "remove_menuitem", NULL
        };
        gpointer callback[6] = { (gpointer) rodent_cut_activate, (gpointer) rodent_copy_activate,
            (gpointer) rodent_paste_activate,
            (gpointer) rodent_remove_activate, NULL
        };
        guint akey[6] = { GDK_KEY_x, GDK_KEY_c, GDK_KEY_v, 0, GDK_KEY_Delete, 0 };
        guint amask[6] = { GDK_CONTROL_MASK, GDK_CONTROL_MASK, GDK_CONTROL_MASK,
            0, 0
        };
        int i;

        for(i = 0; string[i]; i++) {
            thread_add_menu_item (widgets_p, item22_menu,
                               _(string[i]),
                               iconfile[i], id[i], callback[i], widgets_p, ((akey[i]) ? accel_group : NULL), akey[i], amask[i], FALSE);
        }
    }

    w = thread_add_menu_separator (item22_menu);
    RODENT_HOOKUP_OBJECT (widgets_p->paper, w, "edit_separator");
}

static
void
thread_general_menu_items (widgets_t * widgets_p, GtkAccelGroup * accel_group, GtkWidget * item22_menu){

    thread_add_menu_item (widgets_p, item22_menu,
                       _("Personal settings"),
                       "xffm/stock_preferences",
                       (accel_group==NULL)?NULL:"settings2",
		       (gpointer) on_settings_show, widgets_p, 
		       accel_group, GDK_KEY_p, GDK_MOD1_MASK, FALSE);
    

    thread_add_menu_item (widgets_p, item22_menu,
                       _("About"),
                       "xffm/stock_about",
                       (accel_group==NULL)?NULL:"about2", 
		       (gpointer) gui_on_about_activate, widgets_p, 
		       accel_group, 
		       GDK_KEY_i, GDK_MOD1_MASK, FALSE);
    
    thread_add_menu_item (widgets_p, item22_menu,
                       _("Help"),
                       "xffm/stock_help",
                       (accel_group==NULL)?NULL:"help_pdf", 
		       (gpointer) gui_on_help_activate, widgets_p, 
		       accel_group,
		       GDK_KEY_h, GDK_MOD1_MASK, FALSE);

    thread_add_menu_item (widgets_p, item22_menu,
                       _("Close"),
                       "xffm/stock_close",
                       (accel_group==NULL)?NULL:"exit2", 
		       (gpointer) close_menu_item_activate, widgets_p, 
		       accel_group, 
		       GDK_KEY_q, GDK_CONTROL_MASK, FALSE);

    if (accel_group) {
	RODENT_HOOKUP_OBJECT (widgets_p->paper, item22_menu, "item22_menu");
    }
}

static GtkWidget *
thread_mk_popup_menu (widgets_t * widgets_p, 
	GtkWidget * parent,
	GtkAccelGroup * accel_group) 
{
    if (rfm_global_p->self == g_thread_self()){
	g_error("thread_mk_popup_menu() is a thread function, only to be called from LWP\n");
    }
    GtkWidget *w;
    GtkWidget *item22_menu;
NOOP("thread_mk_popup_menu now...\n");
    if(!accel_group) {
GDK_THREADS_ENTER();
        accel_group = gtk_accel_group_new ();
        gtk_window_add_accel_group (GTK_WINDOW (widgets_p->window), accel_group);
GDK_THREADS_LEAVE();
    }

    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    item22_menu = rodent_mk_menu (widgets_p, "item22",  /* label */
                                  "item22",     /* name */
                                  NULL, //parent, /* parent */
                                  NULL, NULL);  /* callback (or NULL) */

    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    w = rodent_mk_menu (widgets_p, "",  /* label */
                        "module1",      /* name */
                        item22_menu,    /* parent */
                        NULL,   /* callback (or NULL) */
                        NULL);  /* iconfile (or NULL) */

    /* navigation menu-items */

    // thread_navigation_menu_items autoprotects with GDK mutex, always.
    thread_navigation_menu_items (widgets_p, accel_group, item22_menu);

    /* file menu-items */
    //  file menu-items autoprotect with GDK mutex, always.
    thread_file_menu_items (widgets_p, accel_group, item22_menu);
    thread_exec_menu_items (widgets_p, accel_group, item22_menu);
    thread_fileexec_menu_items (widgets_p, accel_group, item22_menu);

    /* view menu-items */
    thread_view_menu_items (widgets_p, accel_group, item22_menu);
    thread_size_menu_items (widgets_p, accel_group, item22_menu);

    /* edit menu items */
    thread_edit_menu_items (widgets_p, accel_group, item22_menu);

    /* general group of menuitems: */
    thread_general_menu_items (widgets_p, accel_group, item22_menu);
NOOP("thread_mk_popup_menu done!\n");
    return item22_menu;

}

/////////////////////////////////////////////77
//////  rodent_menu

/*************************** specific commands to menu ***************/

static int
rodent_diff (widgets_t * widgets_p, char *file1, char *file2) {
    gchar *argv[4];
    int i = 0;
    argv[i++] = "rodent-diff";
    if(file1)
        argv[i++] = file1;
    if(file2)
        argv[i++] = file2;
    argv[i] = NULL;

    rfm_thread_run_argv (widgets_p, argv, FALSE);
    return TRUE;
}

/****************** gtk functions for callbacks *********************/
// mount_item: shows the menu mount item if fstab module available
static void
mount_item (view_t * view_p, const population_t * population_p) {
    if(g_slist_length (view_p->selection_list) != 1)
        return;
    if(rfm_void (PLUGIN_DIR, "fstab", "is_root_module") == NULL)
        return;

    gboolean mounted = FALSE;
    gboolean in_fstab = FALSE;
    gboolean partition = FSTAB_is_partition_type (population_p->en);
    if(partition || IS_SDIR(population_p->en->type)) {
        mounted = FSTAB_entry_is_mounted (population_p->en);
        in_fstab = FSTAB_is_in_fstab (population_p->en->path);
    }
    gboolean isofs = 
	(population_p->en->mimetype && 
	 strstr(population_p->en->mimetype, "application/x-cd-image")) ||
	(population_p->en->mimemagic && 
	 strstr(population_p->en->mimemagic, "application/x-cd-image"));
    // This does not good. Mount volume will be /dev/loopX
    if (isofs) {
        mounted = FSTAB_entry_is_mounted (population_p->en);
    }
 

    NOOP ("POPUPx  %s is mounted=%d\n", population_p->en->path, mounted);
    gboolean do_mount_item = mounted || in_fstab || partition || isofs;
    if(do_mount_item) {
        if(mounted)
            SHOW_IT (view_p->widgets.paper, "unmountP");
        else
            SHOW_IT (view_p->widgets.paper, "mountP");
    }
}

static void
remove_item (view_t * view_p, const population_t * population_p) {
    if(!view_p || !population_p || !population_p->en || !population_p->en->path)
        return;
    if(IS_UP_TYPE(population_p->en->type))return;
    /* for local remove: */
    if(IS_BROKEN_LNK (population_p->en->type)
       || 
       g_path_is_absolute(population_p->en->path)
       ) {
        SHOW_IT (view_p->widgets.paper, "remove_menuitem");
    }
}

static void
properties_item (view_t * view_p, const population_t * population_p) {
    if(!view_p || !population_p || !population_p->en || !population_p->en->path)
        return;

    if (rfm_void(MODULE_DIR, "properties", "module_active")) {
      if(IS_BROKEN_LNK (population_p->en->type) ||
       g_path_is_absolute (population_p->en->path) ) {
        SHOW_IT (view_p->widgets.paper, "properties2");
    }}

    if (rfm_void(MODULE_DIR, "fgr", "module_active")) {
      if(IS_SDIR(population_p->en->type)) {
        SHOW_IT (view_p->widgets.paper, "find2");
    }}
}

// Just valid paths in this baby for now (no modules, no rodent root)
static void
setup_goback_menu (widgets_t * widgets_p){
    view_t *view_p = widgets_p->view_p;
    gint level;
    GList *history=g_list_last (view_p->go_list);
    GSList *used_list=NULL;
    for(level = 0; 
	    history && level < DEEPEST_HISTORY_MENU_LEVELS;
	    history = history->prev) {
	gchar *name = g_strdup_printf ("goback_item-%d", level);
        GtkWidget *a = rodent_get_widget_by_name (widgets_p->paper, name);
	GtkWidget *label = gtk_bin_get_child (GTK_BIN (a));
	record_entry_t *en=history->data;
	if (!en) continue; // no rodent root for now
	else {
	    if (en->module) continue;// no modules for now
	}
	if (!IS_SDIR(en->type)) continue;
	GSList *tmp=used_list;
	gboolean repeat=FALSE;
	for (;tmp && tmp->data; tmp=tmp->next){
	    if (strcmp((gchar *)tmp->data, en->path)==0) {
		repeat=TRUE;
		break;
	    }
	} 
	if (repeat) continue;
	
	gchar *utf = rfm_utf_string (en->path);
	gtk_label_set_text ((GtkLabel *) label, utf);
	g_free (utf);
	gchar *old_path;
	if((old_path = g_object_get_data (G_OBJECT (a), "path")) != NULL) {
	    g_object_set_data (G_OBJECT (a), "path", NULL);
	    g_free (old_path);
	}
	g_object_set_data (G_OBJECT (a), "path", g_strdup(en->path));
	SHOW_IT (view_p->widgets.paper, name);

	g_free (name);
	level++;
    }
    for (; level < DEEPEST_HISTORY_MENU_LEVELS; level++){
	gchar *name = g_strdup_printf ("goback_item-%d", level);
        GtkWidget *a = rodent_get_widget_by_name (widgets_p->paper, name);
	gchar *old_path;
	if((old_path = g_object_get_data (G_OBJECT (a), "path")) != NULL) {
	    g_object_set_data (G_OBJECT (a), "path", NULL);
	    g_free (old_path);
	}
	HIDE_IT (view_p->widgets.paper, name);
	g_free (name);
    }
    return;
}

static void
recursive_dirname (widgets_t * widgets_p, char *path, int level) {
    view_t *view_p = widgets_p->view_p;
    if(!path || strcmp (path, "/") == 0)
        return;
    if(level >= DEEPEST_DIR_MENU_LEVELS)
        return;
    gchar *b = g_path_get_dirname (path);
    gchar *name = g_strdup_printf ("level-%d", level);

    GtkWidget *a = rodent_get_widget_by_name (widgets_p->paper, name);
    GtkWidget *label = gtk_bin_get_child (GTK_BIN (a));
    gchar *q = rfm_utf_string (b);
    gtk_label_set_text ((GtkLabel *) label, q);
    g_free (q);
    gchar *old_b;
    if((old_b = g_object_get_data (G_OBJECT (a), "path")) != NULL) {
        g_object_set_data (G_OBJECT (a), "path", NULL);
        g_free (old_b);
    }
    g_object_set_data (G_OBJECT (a), "path", (gpointer) b);
    SHOW_IT (view_p->widgets.paper, name);

    recursive_dirname (widgets_p, b, level + 1);
    g_free (name);
    return;
}

static void
refresh_item (view_t * view_p, const population_t * population_p) {
    // dotdesktop_menu will not exist if dotdesktop module is not present
    gboolean have_dotdesktop =
	GPOINTER_TO_INT(rfm_void(PLUGIN_DIR, "dotdesktop", "is_root_module"));
    if (have_dotdesktop) {
	GtkWidget *applications_menu=
	    g_object_get_data(G_OBJECT(view_p->widgets.paper), 
		    "applications_menu");
	if (applications_menu) {
	    GtkWidget *popup_widget=
		rfm_rational(PLUGIN_DIR, "dotdesktop",&(view_p->widgets),
			applications_menu, 
			"dotdesktop_nondetached_menu");
	    if (popup_widget) {
		gtk_widget_show_all(applications_menu);
	    }
	}
    }	
    SHOW_IT (view_p->widgets.paper, "places_menu");
    HIDE_IT (view_p->widgets.paper, "subbookmark_item_yes");
    HIDE_IT (view_p->widgets.paper, "subbookmark_item_no");
   
    SHOW_IT (view_p->widgets.paper, "goto_menu");
    if(view_p->go_list) {
        SHOW_IT (view_p->widgets.paper, "go_back_menuitem");
	//SHOW_IT (view_p->widgets.paper, "goback_menu");
    } else {
        HIDE_IT (view_p->widgets.paper, "go_back_menuitem");
	HIDE_IT (view_p->widgets.paper, "goback_menu");
    }
    SHOW_IT (view_p->widgets.paper, "goto2");
    SHOW_IT (view_p->widgets.paper, "newwindow");
    SHOW_IT (view_p->widgets.paper, "host2");
    SHOW_IT (view_p->widgets.paper, "home2");
    SHOW_IT (view_p->widgets.paper, "help_pdf");

    SHOW_IT (view_p->widgets.paper, "zoom_menu");

    if (view_p->flags.type == DESKVIEW_TYPE) {
	if (view_p->view_layout.icon_size < BIG_ICON_SIZE) {
	    SHOW_IT (view_p->widgets.paper, "plus_iconsize");
	}
	if (view_p->view_layout.icon_size >= TINY_ICON_SIZE) {
	    SHOW_IT (view_p->widgets.paper, "minus_iconsize");
	}
    }
    if (rfm_get_default_size() != view_p->view_layout.icon_size){
	SHOW_IT (view_p->widgets.paper, "default_iconsize");
    }
    SHOW_IT (view_p->widgets.paper, "default_iconsize_all");

    TRACE("refresh_item: rodent_bookmark_set_menuitems\n");
    rodent_bookmark_set_menuitems(&(view_p->widgets), "bookmark");
    gint i;
    for(i = 0; i < DEEPEST_BOOK_MENU_LEVELS; i++) {
        gchar *name = g_strdup_printf ("bookmark-%d", i);
	GtkWidget *menuitem = g_object_get_data(G_OBJECT(view_p->widgets.paper), name);
	 if (!menuitem){
	     g_warning("refresh_item(): widget %s not found", name);
	     g_free(name);
	     continue;
	 }
	 gchar *path=g_object_get_data(G_OBJECT(menuitem), "path");
	 if (!path) {
	     HIDE_IT (view_p->widgets.paper, name);
	 } else {
	     SHOW_IT (view_p->widgets.paper, name);
	 }
	 g_free(name);
    }
   
    for(i = 0; i < DEEPEST_DIR_MENU_LEVELS; i++) {
        char *name = g_strdup_printf ("level-%d", i);
        HIDE_IT (view_p->widgets.paper, name);
        g_free (name);
    }
    /* this shows valid directories to goto */
    recursive_dirname (&(view_p->widgets), view_p->deepest_dir, 0);
    setup_goback_menu (&(view_p->widgets));


    SHOW_IT (view_p->widgets.paper, "navigation_separator");

    gchar **w,
     *widget_name[] = { "show_hidden_menuitem", "preview_images_menuitem", "new_menu", NULL };
    SHOW_IT (view_p->widgets.paper, "refresh3");
    if(view_p->en && view_p->en->path) {
        gchar *d = g_path_get_dirname (view_p->en->path);
            
	SHOW_IT (view_p->widgets.paper, "goup_menuitem");
        g_free (d);
    }

    for(w = widget_name; *w; w++) {
        if(view_p->en && view_p->en->path) {
            if(view_p->module) {
                if(rfm_void (PLUGIN_DIR, view_p->module, *w))
                    SHOW_IT (view_p->widgets.paper, *w);
            } else if(IS_SDIR(view_p->en->type)) {
                SHOW_IT (view_p->widgets.paper, *w);
            }
        }
    }

    SHOW_IT (view_p->widgets.paper, "view_separator");

    SHOW_IT (view_p->widgets.paper, "sort1");

    if(view_p->flags.preferences & SORT_ASCENDING){
        HIDE_IT (view_p->widgets.paper, "ascending1");
        SHOW_IT (view_p->widgets.paper, "descending1");
    } else {
	HIDE_IT (view_p->widgets.paper, "descending1");
        SHOW_IT (view_p->widgets.paper, "ascending1");
    }

    SHOW_IT (view_p->widgets.paper, "run2");
    if(rfm_valid_pasteboard (view_p) && view_p->en && view_p->en->path
	    && 
	      IS_SDIR(view_p->en->type)
	    )
    {
        SHOW_IT (view_p->widgets.paper, "paste_menuitem");
    }

    if(view_p->constructor) {
        SHOW_IT (view_p->widgets.paper, "newwindow");

    }

    if(getenv ("TERMINAL_CMD") && strlen (getenv ("TERMINAL_CMD"))) {
        gchar *term = g_find_program_in_path (getenv ("TERMINAL_CMD"));
	if (!term){
	    const gchar **p=terminals_v;
	    for (;p && *p; p++){
		term = g_find_program_in_path (*p);
		if (term) break;
	    }
	}

	if (term) {
	    SHOW_IT (view_p->widgets.paper, "terminal2");
	    g_free(term);
	} else {

	    const gchar *t=getenv ("TERMINAL_CMD");
	    g_warning("Terminal command \"%s\" not found", t);
	}
    }
    if (rfm_void(MODULE_DIR, "fgr", "module_active")) {
	SHOW_IT (view_p->widgets.paper, "find2");
    }
    SHOW_IT (view_p->widgets.paper, "differences2");
    SHOW_IT (view_p->widgets.paper, "file_separator");

    if(view_p->en) {
        SHOW_IT (view_p->widgets.paper, "select_menu");
        SHOW_IT (view_p->widgets.paper, "select_all_menuitem");
        SHOW_IT (view_p->widgets.paper, "select_byfilter_menuitem");
        SHOW_IT (view_p->widgets.paper, "unselect_all_menuitem");
        SHOW_IT (view_p->widgets.paper, "edit_separator");
    }
    if (rfm_void(MODULE_DIR, "settings", "module_active")) {
	SHOW_IT (view_p->widgets.paper, "settings2");
    }
    SHOW_IT (view_p->widgets.paper, "about2");
    //if (view_p->flags.type != DESKVIEW_TYPE) 
    SHOW_IT (view_p->widgets.paper, "exit2");

}

#define MENU_IN_MODULE(x) (view_p->module && rfm_void(PLUGIN_DIR,view_p->module,(x)))

static void
open_item (view_t * view_p, const population_t * population_p) {
    NOOP ("POPUPx: open_item...\n");
    gchar *symbols[] = {
        "open_with_menuitem", "copy_menuitem", "cut_menuitem",
        "rename_menuitem", "duplicate_menuitem",
        "touch_menuitem", "symlink_menuitem",
        NULL
    };
    if (population_p->en && population_p->en->mimetype &&
	    rfm_void(PLUGIN_DIR, "dotdesktop", "module_active") &&
	    strcmp(population_p->en->mimetype, "application/x-desktop")==0 ) {
	NOOP ("showing autotype_Prun\n");
        SHOW_IT (view_p->widgets.paper, "autotype_Prun");
    }
    gboolean multiple_select[] = { TRUE, TRUE, TRUE, FALSE, FALSE, TRUE, FALSE, 0 };
    int i;

    if(!view_p || !population_p || !population_p->en || !population_p->en->path)
        return;
    if(IS_UP_TYPE(population_p->en->type))return;

    for(i = 0; symbols[i]; i++) {
        HIDE_IT (view_p->widgets.paper, symbols[i]);
        if(IS_BROKEN_LNK(population_p->en->type) ||
	    g_path_is_absolute(population_p->en->path) ||
	   MENU_IN_MODULE (symbols[i]) ) {
            if(g_slist_length (view_p->selection_list) > 1 && !multiple_select[i])
                continue;
            SHOW_IT (view_p->widgets.paper, "file_separator2");
            SHOW_IT (view_p->widgets.paper, "file_separator");
            SHOW_IT (view_p->widgets.paper, symbols[i]);
        }
    }

    if(IS_SDIR(view_p->en->type)
       || MENU_IN_MODULE ("paste_menuitem")) {
        if(rfm_valid_pasteboard (view_p) && view_p->en && view_p->en->path) {
            SHOW_IT (view_p->widgets.paper, "paste_menuitem");
        }
    }

    if(g_slist_length (view_p->selection_list) > 0 &&
       g_slist_length (view_p->selection_list) < 3 &&
       view_p->en && view_p->en->path && IS_SDIR(view_p->en->type)) {
        SHOW_IT (view_p->widgets.paper, "differences2");
    }



}

static void
set_menu_toggles (view_t * view_p) {
    {
        NOOP ("setting sort toggles now...\n");
        view_p->flags.disable_toggle_refresh = TRUE;
        if(view_p->flags.sortcolumn == TYPE_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "unsorted6"), TRUE);
        else if(view_p->flags.sortcolumn == NAME_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "name6"), TRUE);
        else if(view_p->flags.sortcolumn == SIZE_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "size6"), TRUE);
        else if(view_p->flags.sortcolumn == DATE_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "date6"), TRUE);
        else if(view_p->flags.sortcolumn == OWNER_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "owner6"), TRUE);
        else if(view_p->flags.sortcolumn == GROUP_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "group6"), TRUE);
        else if(view_p->flags.sortcolumn == MODE_SORT)
            gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                            rodent_get_widget_by_name (view_p->widgets.paper, "mode6"), TRUE);

        NOOP ("before setting __SHOW_IMAGES toggle now...%s\n", (view_p->flags.preferences & __SHOW_IMAGES) ? "ON" : "Off");
        gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                        rodent_get_widget_by_name (view_p->widgets.paper,
                                                                   "preview_images_menuitem"), view_p->flags.preferences & __SHOW_IMAGES);

        NOOP ("after setting __SHOW_IMAGES toggle now...%s\n", (view_p->flags.preferences & __SHOW_IMAGES) ? "ON" : "Off");
        gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
                                        rodent_get_widget_by_name (view_p->widgets.paper,
                                                                   "show_hidden_menuitem"), view_p->flags.preferences & __SHOW_HIDDEN);
        NOOP ("setting __SHOW_HIDDEN toggle now...%s\n", (view_p->flags.preferences & __SHOW_HIDDEN) ? "ON" : "Off");

        view_p->flags.disable_toggle_refresh = FALSE;

    }
}

////////////////////////////////////////////////
////   menu.i

static GList *key_list = NULL;

/* Use as many different names for as many different
 * autotype menu entries for one file extension type. */
static char *auto_C_name[] = {
    "ejecutar1",
    "ejecutar2",
    "ejecutar3",
    "ejecutar4",
    "ejecutar5",
    "ejecutar6",
    "ejecutar7",
    "ejecutar8",
    "ejecutar9",
    "ejecutar10",
    NULL
};

static
gboolean set_default_app (GtkWidget      *widget,
	                   GdkEventButton *event,
                           gpointer        data){
    if (event->state & GDK_CONTROL_MASK){
	g_object_set_data(G_OBJECT(widget), "CTL_SET", GINT_TO_POINTER(1));
    } else {
	g_object_set_data(G_OBJECT(widget), "CTL_SET", NULL);
    }
    return FALSE;
}

static GtkWidget *
thread_create_menu_entry (widgets_t * widgets_p, 
	gchar *name, 
	gchar *label, 
	GtkWidget * parent, 
	void *callback,
	void *data, 
	gint offset,
	const gchar *tooltip_text) {

    if (rfm_global_p->self == g_thread_self()){
	g_error("thread_create_menu_entry() is a thread function, only to be called from LWP\n");
    }
GDK_THREADS_ENTER();
    GtkWidget *w = gtk_image_menu_item_new_with_label (label);
    if(offset < 0)
        gtk_container_add (GTK_CONTAINER (parent), w);
    else {
        gtk_menu_shell_insert (GTK_MENU_SHELL (parent), w, offset);
    }
GDK_THREADS_LEAVE();

    const gchar *icon_id=NULL;
    if (strlen(label)){
	icon_id = (const gchar *)
	    rfm_natural(PLUGIN_DIR, "dotdesktop", label, "get_exec_icon");
	if (!icon_id) {
	    NOOP("1. command icon \"%s\" not found!\n",label);
	}
    }
    if (!icon_id) {
	icon_id = "xffm/stock_execute";
    }
    GdkPixbuf *p = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
//    GdkPixbuf *p = rfm_get_pixbuf ("xffm/stock_execute", SIZE_BUTTON);
    GtkWidget *image = NULL;
GDK_THREADS_ENTER();
    if(p) {
	image = gtk_image_new_from_pixbuf (p);
	gtk_widget_show (image);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (w), image);
	g_object_set (GTK_IMAGE_MENU_ITEM (w), "always-show-image", TRUE, NULL);
    } 
    NOOP ("created %s at 0x%lx window\n", name, (unsigned long)widgets_p->paper);
    g_object_set_data (G_OBJECT (widgets_p->paper), name, w);
    g_object_set_data (G_OBJECT (w), "view_p", widgets_p->view_p);
    if (tooltip_text) {
	gtk_widget_set_tooltip_text(w, tooltip_text);
	g_signal_connect((gpointer) w, "button-press-event", 
		G_CALLBACK (set_default_app), data);
    }
    g_signal_connect ((gpointer) w, "activate", G_CALLBACK (callback), data);
GDK_THREADS_LEAVE();
    
    return w;
}

static void
clean_object_data(GtkWidget *a, const gchar *data_id)
{
    gchar *old_text = g_object_get_data (G_OBJECT (a), data_id);
    if((old_text = g_object_get_data (G_OBJECT (a), data_id)) != NULL) {
        g_object_set_data (G_OBJECT (a), data_id, NULL);
        g_free (old_text);
    }
}

static int
set_auto_command (widgets_t * widgets_p,
                  int j,
                  GtkWidget * popup,
                  const gchar * name,
                  const gchar * alt_label,
                  const gchar * app,
                  const gchar * path
                  ) {
    int jj;

    gchar *dirname = g_path_get_dirname (path);
    gchar *basename = g_path_get_basename (path);

    GtkWidget *a = rodent_get_widget_by_name (widgets_p->paper, name);

    const gchar * output_ext=MIME_command_output_ext(app);
    view_t *view_p=widgets_p->view_p;
    NOOP ("AUTO: set_auto_command() now... %s\n", app);
    if(!MIME_is_valid_command (app)) {
        if (strcmp(app, "rodent-newtab")==0 && view_p->tab_constructor) {
            NOOP (">> set_auto_command: %s name=%s\n", app, name);
        }
        else if (strcmp(app, "rodent-newwin")==0) {
            NOOP (">> set_auto_command: %s name=%s\n", app, name);
        }
        else {
            NOOP ("%s is not valid command", app);
            return j;
        }
    }
    GtkWidget *label;
    gchar *command;
    if(strncmp (app, "sudo -A ", strlen ("sudo -A ")) == 0) {
        if(getuid () == 0)
            return j;           /* don't sudoize root */
        if(!strlen (app + strlen ("sudo -A "))
           || !MIME_is_valid_command (app + strlen ("sudo -A ")))
            return j;
    }

    NOOP("looking for command icon \"%s\"\n",app);
    gchar *app_no_args = g_strdup(app);
    if (strchr(app_no_args, ' ')) *strchr(app_no_args, ' ')=0;
    const gchar *icon_id=(const gchar *)
	rfm_natural(PLUGIN_DIR, "dotdesktop", (void *)app_no_args, "get_exec_icon");
	
    // try custom mime icon
    if (!icon_id){
	icon_id=MIME_command_icon (app_no_args);
	NOOP("icon id= %s, %s\n", icon_id, app_no_args);
    }

    // try named icon
    gchar *icon=NULL;
    if (!icon_id) {
	NOOP("try named icon for %s\n", app_no_args);
	icon=ICON_get_filename_from_basename(app_no_args);
	// This should always be local and absolute.
	if (g_file_test(icon, G_FILE_TEST_EXISTS)){
	    icon_id = icon;
	} 
    }
    g_free(app_no_args);

    if (!icon_id) {
        NOOP("2. command icon \"%s\" not found!\n",app);
        icon_id="xffm/stock_execute";
    }

    if (icon_id) {
      GdkPixbuf *p = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
      if(p) {
        GtkWidget *image = gtk_image_menu_item_get_image (GTK_IMAGE_MENU_ITEM (a));
	if (GTK_IS_WIDGET(image)) {
	    gtk_widget_destroy(image);
	}
        image = gtk_image_new_from_pixbuf (p);
        gtk_widget_show (image);
            NOOP("setting icon for menu item \"%s\"\n",app);
        gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (a), image);
      } else {
            NOOP("unable to get pixbuf for menu item \"%s\"\n",app);
      }
    }
    g_free(icon);


    label = gtk_bin_get_child (GTK_BIN (a));
    if(alt_label) { 
        gtk_label_set_text ((GtkLabel *) label, rfm_utf_string (alt_label));
    } else {
        /* allow mixed utf8-eucjp  paths */
        gchar *new_label;
        {
            const gchar *text = MIME_command_text (app);
	    const gchar *text2 = MIME_command_text2 (app);
            NOOP("MIME: looking for %s in hash: %s %s\n",app, text, text2);
            gchar *q;
            if (text) {
		gchar *texto;
		if (text2){
		    texto=g_strconcat(_(text), _(text2), NULL);
		} else {
		    texto=g_strdup(_(text));
		}
                NOOP("MIME: found\n");
                if (strstr(texto,"%s")!=NULL) {
                    q = MIME_mk_command_line (texto, basename);
                } else {
                    if (output_ext) 
                        q = g_strdup_printf("%s (%s)", texto, output_ext);
                    else
                        q = g_strdup(texto);
                }
		g_free(texto);
            }
            else {
                NOOP("MIME: not found\n");
                q = MIME_mk_command_line (app, basename);
            }
            new_label = rfm_utf_string (q);
            g_free (q);
        }
        gtk_label_set_text ((GtkLabel *) label, new_label);
        g_free (new_label);
    }

    if(output_ext) {
        NOOP ("command is:  %s output_ext is; %s\n", app, output_ext);
        command = g_strdup (app);
    } else {
        /* this is to use relative paths for arguments for all
         * commands except those where the output directory (or
         * widgets_p->workdir) is requested from the user.
         * */
        NOOP ("making command from (%s, %s)\n", app, path);
        command = MIME_mk_command_line (app, path);
    }

    /* check for duplicates */
    {
        gint argcp1;
        gchar **argvp1;
        GError *error = NULL;
        if(!g_shell_parse_argv ((const gchar *)command, &argcp1, &argvp1, &error)) {
            g_free (command);
            g_error_free (error);
            g_strfreev (argvp1);
            NOOP ("!g_shell_parse_argv at gui_autostuff\n");
            return j;
        }
        for(jj = 0; jj < j; jj++) {
            gboolean is_duplicate = TRUE;
            gint ac,
              argcp2;
            gchar **argvp2;
            gchar *set_command = g_object_get_data (G_OBJECT (rodent_get_widget_by_name (widgets_p->paper, auto_C_name[jj])),
                                                    "command");
            NOOP ("AUTO: j=%d: %s == %s\n", jj, set_command, command);
            if(!g_shell_parse_argv ((const gchar *)set_command, &argcp2, &argvp2, &error)) {
                g_free (basename);
                g_free (command);
                g_error_free (error);
                g_strfreev (argvp2);
                NOOP ("!g_shell_parse_argv at gui_autostuff\n");
                return j;
            }
            if(argcp2 != argcp1) {
                is_duplicate = FALSE;
            } else
                for(ac = 0; ac < argcp2; ac++) {
                    if(strcmp (argvp2[ac], argvp1[ac])) {
                        is_duplicate = FALSE;
                        break;
                    }
                }
            NOOP ("0x%lx, 0x%lx\n", (unsigned long)argvp1, (unsigned long)argvp2);

            g_strfreev (argvp2);
            argvp2 = NULL;
            /*if (strcmp(set_command,command)==0) */
            if(is_duplicate) {
                g_free (basename);
                g_free (command);
                NOOP ("AUTO: is_duplicate at gui_autostuff\n");
                return j;
            }
        }                       /* end for */
        g_strfreev (argvp1);
        argvp1 = NULL;
    }
    /* OK, by now we know the command is not duplicated... */
    // But is it a valid command??
    

    /* now let's free up any left over data from previous
     * generated menus... */
    clean_object_data(a, "command");
    clean_object_data(a, "workdir");
    g_object_set_data (G_OBJECT (a), "command", (gpointer) command);
    g_object_set_data (G_OBJECT (a), "workdir", (gpointer) dirname);
    if(MIME_command_output (app)) {
        g_object_set_data (G_OBJECT (a), "querypath", (gpointer)_("Specify Output Directory..."));
    } else {
        g_object_set_data (G_OBJECT (a), "querypath", NULL);
    }
    if(MIME_force_diagnostics (app)) {
        g_object_set_data (G_OBJECT (a), "force_diagnostics", GINT_TO_POINTER(1));
    } else {
        g_object_set_data (G_OBJECT (a), "force_diagnostics", NULL);
    }

    if(output_ext) {
        g_object_set_data (G_OBJECT (a), "output_arg", g_strdup (path));
    } else {
        clean_object_data(a, "output_arg");
        g_object_set_data (G_OBJECT (a), "output_arg", NULL);
    }
    NOOP ("AUTO:command=%s, output_arg=%s\n", command, path);
    g_object_set_data (G_OBJECT (a), "output_ext", (gpointer) output_ext);
    NOOP ("AUTO:now showing %s, active widget=0x%lx\n", name, (unsigned long)widgets_p->paper);
    SHOW_IT (widgets_p->paper, name);
    j++;
    g_free (basename);
    return j;
}

static void
gui_autostuff (widgets_t * widgets_p,
               GtkWidget * popup, 
	       record_entry_t * en, 
	       gpointer callback, 
	       gpointer on_mount, 
	       gpointer on_unmount) {
    gint j = 0;
        TRACE("gui_autostuff()...\n");

    // On first call to function, create the menu item containers
    if(!rodent_get_widget_by_name (widgets_p->paper, auto_C_name[0])) {
        NOOP(stderr, "creating auto menu elements from gui_autostuff\n");
        thread_add_autotype_C_widgets (widgets_p, popup, callback, on_mount, on_unmount);
    }
    // Special "open in new window" item for partition types
    // XXX: This should probably be done by the fstab module, not here.
    if(FSTAB_is_partition_type (en)) {
        NOOP(stderr, "test FSTAB_is_partition_type (en)...\n");
        // only first item (open in new window...)
        gchar *mount_point = FSTAB_get_mnt_dir (en->path);
        if(mount_point) {
	    gchar *cc=g_strdup("rodent-forked");
            j = set_auto_command (widgets_p, j, popup, auto_C_name[j], 
            _("Open in New Window"), cc, mount_point);
	    g_free(cc);
            g_free (mount_point);
        }
    }
    // Special run command for dot desktop files.
    // XXX: as above, this should be done by the module...
    else if(IS_LOCAL_TYPE(en->type) && 
	    en->mimetype && strcmp(en->mimetype, "application/x-desktop")==0  ) {
	if (rfm_void(PLUGIN_DIR, "dotdesktop", "module_active")){
	    // get the Icon
	    const gchar *Icon=rfm_natural(PLUGIN_DIR, "dotdesktop", en, "item_icon_id");
	    // get the Name
	    gchar *Name=rfm_natural(PLUGIN_DIR, "dotdesktop", en->path, "item_name");
	    GtkWidget *a = 
		rodent_get_widget_by_name (widgets_p->paper, "autotype_Prun");
	    if (Name) {
		GtkWidget *label;
		GtkWidget *a;
		a =	rodent_get_widget_by_name (widgets_p->paper, "autotype_Prun");
		label = gtk_bin_get_child (GTK_BIN (a));
		gchar *q = rfm_utf_string (Name);
		gtk_label_set_text ((GtkLabel *) label, q);
		g_free (q);
		g_object_set_data(G_OBJECT(a), "record_entry", en);
	    }
	    if (Icon) {
		GdkPixbuf *p = rfm_get_pixbuf (Icon, SIZE_BUTTON);
		if(p) {
		    GtkWidget *image =
			gtk_image_menu_item_get_image (GTK_IMAGE_MENU_ITEM (a));
		    if (GTK_IS_WIDGET(image)) {
			gtk_widget_destroy(image);
		    }
		    image = gtk_image_new_from_pixbuf (p);
		    gtk_widget_show (image);
		    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (a), image);
		} 
	    }
	    if (Icon && Name){
		SHOW_IT (widgets_p->paper, "autotype_Prun");
	    }
	    g_free (Name);
	}
    }
    // Executable files get the special "run in terminal" menu item:
    else if(IS_EXE_TYPE(en->type)) {
	if (IS_SDIR(en->type)) {
	    // ignore
	} else {
	    gchar *cmd;
	    GtkWidget *label;
	    GtkWidget  *a = rodent_get_widget_by_name (widgets_p->paper, "autotype_Prun");
	    label = gtk_bin_get_child (GTK_BIN (a));
	    g_object_set_data(G_OBJECT(a), "record_entry", en);
	    cmd = g_strdup_printf ("%s (%s)", _("Run in terminal window"), _("Is executable"));
	    gchar *q = rfm_utf_string (cmd);
	    gtk_label_set_text ((GtkLabel *) label, q);
	    g_free (cmd);
	    g_free (q);
	    GdkPixbuf *p = rfm_get_pixbuf ("xffm/apps_terminal", SIZE_BUTTON);
	    if(p) {
		GtkWidget *image =
		    gtk_image_menu_item_get_image (GTK_IMAGE_MENU_ITEM (a));
		if (GTK_IS_WIDGET(image)) {
		    gtk_widget_destroy(image);
		}
		image = gtk_image_new_from_pixbuf (p);
		gtk_widget_show (image);
		gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (a), image);
	    } 
	    SHOW_IT (widgets_p->paper, "autotype_Prun");
	    // Run out of terminal..
	    a = rodent_get_widget_by_name (widgets_p->paper, "autotype_Srun");
	    label = gtk_bin_get_child (GTK_BIN (a));
	    g_object_set_data(G_OBJECT(a), "record_entry", en);
	    cmd = g_strdup_printf ("%s (%s)", _("Run"), _("Is executable"));
	    q = rfm_utf_string (cmd);
	    gtk_label_set_text ((GtkLabel *) label, q);
	    g_free (cmd);
	    g_free (q);
	    SHOW_IT (widgets_p->paper, "autotype_Srun");

	}
    }
   

    // Mimetype associations. 
    // These mimetype associations are defined in the file
    // "applications-module.xml" and harvested from
    // installed dot desktop files by the dotdesktop module.
    //
    // Associations should be done to the basic mimetype 
    // (not any alias) for sanity sake.
    //
    // This is the nitty gritty. 
    // Suggest an application based on the file's mimetype.
    //
    NOOP(stderr, "AUTO: menu nitty-gritty now\n");
    gchar *magic_type = NULL;
    gchar *mime_type = NULL;
    if (IS_LOCAL_TYPE(en->type)) magic_type = MIME_magic (en->path);
    else magic_type = g_strdup(_("unknown"));
    // G_FILE_TEST_EXISTS is a downer on remote connections
    if (g_path_is_absolute(en->path)) {
	mime_type = MIME_type (en->path, en->st);
    }

    // These are the menu items set from dotdesktop files 
    // and "applications-module.xml"

    if(g_path_is_absolute(en->path)) {
        int k;
        NOOP(stderr, "AUTO: menu nitty-gritty mime-type=%s magic-type=%s\n", mime_type, magic_type);

        // here we use the freedesktop and magic
        // mimetype to get the run program.
        for(k = 0; k < 2; k++) {
            gchar **apps = NULL;
            int i = 0;
            if(k == 0) {
                apps = MIME_apps (mime_type);
                NOOP(stderr, "AUTO: menu nitty-gritty (%s) apps=0x%lx\n", mime_type, (long)apps);
            } else {
                apps = MIME_apps (magic_type);
                NOOP(stderr, "AUTO: menu nitty-gritty (%s) apps=0x%lx\n", magic_type, (long)apps);
            }

            if(apps && apps[i])
                for(; apps && apps[i]; i++) {
                    NOOP ("AUTO: j=%d, i=%d %s -> %s\n", j, i, auto_C_name[j], apps[i]);fflush(NULL);

                    if(!auto_C_name[j])
                        continue;
                    j = set_auto_command (widgets_p, j, popup, auto_C_name[j], NULL,
                                          // the command:
                                          apps[i], en->path);
                }
            g_strfreev (apps);
        }

    }
    
    NOOP(stderr, "AUTO: editor test...\n");
    gboolean has_editor=FALSE;
    gint k;
    gchar **apps = MIME_apps (mime_type);
    for (k=0; k<j && apps && apps[k]; k++){
	gint i;
	for (i=0; editors_v[i]; i++){
	    if (strstr(apps[k], editors_v[i])) {
		has_editor=TRUE;
		break;
	    }
	    if (has_editor) break;
	}
    }
    g_strfreev (apps);
    if (!has_editor){
	// Default editor command.
	gchar *text_editor = get_text_editor(en);

	if (text_editor){
	    j = set_auto_command (widgets_p, j, popup, 
		    auto_C_name[j], NULL,
		    // the command:
		     text_editor, en->path);
		g_free(text_editor);
	}
    }


    if(g_path_is_absolute(en->path)) {
        SHOW_IT (widgets_p->paper, "open_with_menuitem");
    }
    NOOP(stderr, "AUTO: done!\n");

    g_free (magic_type);
    g_free (mime_type);
    return;
}

typedef struct extra_key_t {
    GtkWidget *check1;
    GtkEntry *entry;
    gboolean flag1;
    gboolean flag2;
    const gchar *response;
    const gchar *flagfile;
    const gchar *history_file;
} extra_key_t;

static void
activate_entry (GtkEntry * entry, gpointer data) {
    gtk_dialog_response ((GtkDialog *) data, GTK_RESPONSE_YES);
}

static void
cancel_entry (GtkEntry * entry, gpointer data) {
    gtk_dialog_response ((GtkDialog *) data, GTK_RESPONSE_CANCEL);
}

static GMutex *flags_mutex=NULL;
static void
recover_flags (extra_key_t * extra_key_p) {
    DBHashTable *runflags;
    GString *gs;
    int *flags;

    if(flags_mutex==NULL) flags_mutex=g_mutex_new();
    g_mutex_lock(flags_mutex);
    runflags = dbh_new (extra_key_p->flagfile, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE);

    if(runflags  == NULL) {
        DBG ("Cannot open %s\n", extra_key_p->flagfile);
        extra_key_p->flag1 = 0;
        extra_key_p->flag2 = 0;
	g_mutex_unlock(flags_mutex);
        return;
    }
    gs = g_string_new (extra_key_p->response);
    sprintf ((char *)DBH_KEY (runflags), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    flags = (int *)runflags->data;
    dbh_load (runflags);
    extra_key_p->flag1 = flags[0];
    extra_key_p->flag2 = flags[1];
    dbh_close (runflags);
    g_mutex_unlock(flags_mutex);

    DBG ("recover_flags(): flags recovered from dbh file for %s, flag1=%d flag2=%d\n",
         extra_key_p->response, extra_key_p->flag1, extra_key_p->flag2);
}

static void
save_flags (extra_key_t * extra_key_p) {
    DBHashTable *runflags;
    GString *gs;
    int *flags;

    if(flags_mutex==NULL) flags_mutex=g_mutex_new();
    g_mutex_lock(flags_mutex);
    runflags = dbh_new (extra_key_p->flagfile, NULL, DBH_PARALLEL_SAFE);

    if(runflags == NULL) {
        DBG ("Creating cache file: %s", extra_key_p->flagfile);
	unsigned char keylength=11;
	runflags = dbh_new (extra_key_p->flagfile, &keylength, DBH_CREATE|DBH_PARALLEL_SAFE);
        if(runflags == NULL) {
            g_warning ("Cannot create %s\n", extra_key_p->flagfile);
	    g_mutex_unlock(flags_mutex);
	    return;
        }
    }
    gs = g_string_new (extra_key_p->response);
    sprintf ((char *)DBH_KEY (runflags), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    flags = (int *)runflags->data;
    flags[0] = extra_key_p->flag1;
    flags[1] = extra_key_p->flag2;
    dbh_set_recordsize (runflags, 2 * sizeof (int));

    dbh_update (runflags);
    dbh_close (runflags);
    g_mutex_unlock(flags_mutex);
    NOOP ("flags saved in dbh file  %s flag1=%d flag2=%d\n", extra_key_p->flagfile, extra_key_p->flag1, extra_key_p->flag2);

}

static int
extra_key_completionR (gpointer data) {
    extra_key_t *extra_key_p = (extra_key_t *) data;

    NOOP("extra_key_completionR\n");
    if(!extra_key_p) {
        g_warning ("!extra_key_p");
        return FALSE;
    }
    if(!GTK_IS_ENTRY (extra_key_p->entry))
        return FALSE;
    extra_key_p->response = gtk_entry_get_text (extra_key_p->entry);

    recover_flags (extra_key_p);
    gtk_toggle_button_set_active ((GtkToggleButton *) extra_key_p->check1, extra_key_p->flag1);
    return FALSE;
}

static void
toggle_flag (GtkToggleButton * togglebutton, gpointer user_data) {
    extra_key_t *extra_key_p = (extra_key_t *) user_data;
    if(gtk_toggle_button_get_active (togglebutton))
        extra_key_p->flag1 = TRUE;
    else
        extra_key_p->flag1 = FALSE;
    save_flags (extra_key_p);
}

static gpointer
thread_preload_f (gpointer data) {
    const gchar *folder = (const gchar *)data;
    if(folder && rfm_g_file_test_with_wait (folder, G_FILE_TEST_IS_DIR)) {
        GError *error = NULL;
        const gchar *path;
        GDir *dir = g_dir_open (folder, 0, &error);
        while((path = g_dir_read_name (dir)) != NULL) {
            /* just to avoid compiler optimization: */
            if(strcmp (path, ".") == 0)
                g_warning ("This should never happen: strcmp (path, \".\") == 0");
        }
        g_dir_close (dir);
    }

    return NULL;
}

static
    void
filechooser (GtkButton * button, gpointer user_data) {
    filechooser_t *filechooser_p = (filechooser_t *) user_data;
    const gchar *text=filechooser_p->title;
    if (text==NULL) {
      if(filechooser_p->filechooser_action == GTK_FILE_CHOOSER_ACTION_OPEN) {
        text = _("Select File");
      } else if(filechooser_p->filechooser_action == GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER) {
        text = _("Select Folder");
      } 
    }
      
      GtkWidget *dialog = gtk_file_chooser_dialog_new (text,
                                                     GTK_WINDOW (filechooser_p->parent),
                                                     filechooser_p->filechooser_action,
                                                     GTK_STOCK_CANCEL,
                                                     GTK_RESPONSE_CANCEL,
                                                     GTK_STOCK_OPEN,
                                                     GTK_RESPONSE_ACCEPT,
                                                     NULL);
    
      // not necessary when using ~/.gtkbookmarks file:
      // rodent_set_filechooser_bookmarks(GTK_FILE_CHOOSER (dialog));

     gtk_file_chooser_set_use_preview_label(GTK_FILE_CHOOSER (dialog), FALSE);
    gtk_file_chooser_set_preview_widget_active (GTK_FILE_CHOOSER (dialog), FALSE);
    gtk_file_chooser_set_action (GTK_FILE_CHOOSER (dialog), filechooser_p->filechooser_action);
    //gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), filechooser_p->folder);

    // XXX gtk_file_chooser_set_show_hidden isn't working. Gtk bug?
    //     Maybe. It will not work if you set the current folder with
    //	   gtk_file_chooser_set_current_folder()
    gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER (dialog), TRUE);

    if(gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
        char *filename;
        void *combo_info = filechooser_p->combo_info;
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	if (rfm_void(MODULE_DIR, "combobox", "module_active")) {
	    COMBOBOX_set_entry (combo_info, filename);
	} else {
	    gtk_entry_set_text (filechooser_p->entry, filename);
	}
        NOOP ("Got %s\n", filename);
        g_free (filename);
	gtk_widget_destroy (dialog);
	if(filechooser_p->activate_func) {
	     NOOP("filechooser_p->activate_func \n");
            (*(filechooser_p->activate_func)) (filechooser_p->entry, filechooser_p->activate_user_data);
	}
	else NOOP("filechooser_p->activate_func is NULL\n");

    } else {
	gtk_widget_destroy (dialog);
    }

}

static void
preload (const gchar * folder) {
    THREAD_CREATE (thread_preload_f, (gpointer) (folder), "thread_preload_f");
}

static void
thread_add_autotype_C_widgets (widgets_t * widgets_p,
                            GtkWidget * popup, 
			    gpointer callback, 
			    gpointer on_mount, 
			    gpointer on_unmount) {
    int i;
    GtkWidget *w;
    if (rfm_global_p->self == g_thread_self()){
	g_error("thread_add_autotype_C_widgets() is a thread function, only to be called from LWP\n");
    }

    /* check for optional fstab plugin... */
    if(rfm_void (PLUGIN_DIR, "fstab", "module_active")) {
        if(on_mount) {
GDK_THREADS_ENTER();
	    w=g_object_get_data(G_OBJECT(widgets_p->paper), "mountP");
            if(w){
                g_signal_connect ((gpointer) w, 
			"activate", G_CALLBACK (on_mount), widgets_p);
	    }
GDK_THREADS_LEAVE();
        }
        if(on_unmount) {
            //w = rodent_get_widget_by_name (widgets_p->paper, "unmountP");
GDK_THREADS_ENTER();
	    w=g_object_get_data(G_OBJECT(widgets_p->paper), "unmountP");
            if(w){
                g_signal_connect ((gpointer) w,
			"activate", G_CALLBACK (on_unmount), widgets_p);
	    }
GDK_THREADS_LEAVE();
        }
    }

    if(callback){
        for(i = 0; auto_C_name[i]; i++) {
//	    const gchar *tooltip_text=NULL;
	    const gchar *tooltip_text=_("CONTROL click to set command as default for mimetype");
            thread_create_menu_entry (widgets_p, auto_C_name[i], "", 
		    popup, callback, NULL, i + 6, tooltip_text);
        }
    }

}

/* gui_autofunction for directories works different: 
 * directories: the path argument is relative, so we must do a chdir
 * 		to the path directory.
 * 		The output specified will carry along the fullpath for 
 * 		output. The workdir will be that where the directory 
 * 		resides. We must do a chdir here.
 * files:	the path argument is absolute. So we just do a chdir to
 * 		the specified workdir before execution.
 *
 * 		*/

/* keybindings... accel group will not work if menus is not around. This is a
 * gtk feature, so we have to do our own keybinding. 
 **/

/* function to add a keybind */

static
    void
gui_add_to_keylist (guint in_key, guint mask, gpointer callback) {
    GList *tmp;
    keybind_t *keybind_p;
    guint key;

    if(in_key < 128) {
        NOOP ("key=toupper(in_key)\n");
        key = toupper (in_key);
    } else
        key = in_key;

    for(tmp = key_list; tmp; tmp = tmp->next) {
        keybind_p = (keybind_t *) tmp->data;
        if(!keybind_p)
            continue;
        if(key == keybind_p->key && mask == keybind_p->mask)
            return;
    }
    NOOP ("adding to keylist=%u callback=0x%lx\n", key, (unsigned long)callback);
    keybind_p = (keybind_t *) malloc (sizeof (keybind_t));
    memset (keybind_p, 0, sizeof (keybind_t));
    keybind_p->key = key;
    keybind_p->mask = mask;
    keybind_p->callback = callback;
    key_list = g_list_append (key_list, keybind_p);
    return;
}

/* function to run callback with keybind values */
typedef void (*keybind_f) (GtkMenuItem *, gpointer);

static
    int
gui_keybind_exec (guint in_key, guint mask, void *data) {
    GList *tmp;
    void (*callback) (GtkMenuItem * menuitem, gpointer data);
    guint key;

    if(in_key < 128)
        key = toupper (in_key);
    else
        key = in_key;

    for(tmp = key_list; tmp; tmp = tmp->next) {
        keybind_t *keybind_p = (keybind_t *) tmp->data;
        if(!keybind_p)
            continue;
        NOOP ("key: %u=%u?\n", (char)keybind_p->key, key);
        if(key == keybind_p->key && mask == keybind_p->mask) {
            callback = (keybind_f) keybind_p->callback;
            NOOP ("callback=0x%lx, data=0x%lx\n", (unsigned long)callback, (unsigned long)data);
            (*callback) (NULL, data);
            return TRUE;
        }
    }
    NOOP ("no keybinding...\n");
    return FALSE;
}

static
GtkWidget *
thread_add_menu_separator (GtkWidget * parent) {
    GtkWidget *w = gtk_menu_item_new ();
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (parent), w);
    gtk_widget_set_sensitive (w, FALSE);
    return w;
}


/* function valid for only one radio group: */
static
GtkWidget *
gui_mk_radio_button (widgets_t * widgets_p, const gchar * label, const gchar * name, GtkWidget * parent, GSList ** radiogroup) {
    GtkWidget *w = gtk_radio_menu_item_new_with_mnemonic (*radiogroup, label);

    /*w = gtk_radio_menu_item_new_with_mnemonic (*radiogroup,label); */
    *radiogroup = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (w));
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (parent), w);
    //gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (w), TRUE);
    RODENT_HOOKUP_OBJECT (widgets_p->paper, w, name);
    return w;
}

static
GtkWidget *
thread_add_menu_item (widgets_t * widgets_p,
                   GtkWidget * parent,
                   const gchar * label,
                   const gchar * icon_id,
                   const gchar * name,
                   gpointer callback, 
		   gpointer callback_data,
		   GtkAccelGroup * accel_group,
		   guint accel_key,
		   guint accel_mask,
		   gboolean insert) {
    if (rfm_global_p->self == g_thread_self()){
	g_error("thread_add_menu_item() is a thread function, only to be called from LWP\n");
    }
    GDK_THREADS_ENTER();
    GtkWidget *w = gtk_image_menu_item_new_with_mnemonic (label);
    gtk_image_menu_item_set_always_show_image ((GtkImageMenuItem *) w, TRUE);
    GDK_THREADS_LEAVE();
    NOOP ("thread_add_menu_item\n");
    if(icon_id){
        rodent_mk_pixmap_menu (icon_id, w, MENU_PIXMAP);
    }
    gtk_widget_show (w);
    if (insert) {
        gtk_menu_shell_insert (GTK_MENU_SHELL (parent), w, 0);
    } else {
	gtk_container_add (GTK_CONTAINER (parent), w);
    }
    if (name) RODENT_HOOKUP_OBJECT (widgets_p->paper, w, name);
    if(callback)
        g_signal_connect ((gpointer) w, "activate", G_CALLBACK (callback), callback_data);
    if(accel_key) {
        gui_add_to_keylist (accel_key, accel_mask, callback);
    }
    if(accel_group){
        gtk_widget_add_accelerator (w, "activate", accel_group, accel_key, accel_mask, GTK_ACCEL_VISIBLE);
    }
    return w;
}

static
    void
rodent_mk_module_popup_menu (const gchar * module_name, view_t * view_p, record_entry_t * en) {
    gchar *txt;
    if(!module_name)
        return;
    txt = rfm_void (PLUGIN_DIR, module_name, "module_label");

    /* set the submenu label */
    if(txt) {
        GtkWidget *label,
         *a = rodent_get_widget_by_name (view_p->widgets.paper, "module1");
        label = gtk_bin_get_child (GTK_BIN (a));
        gchar *q = rfm_utf_string (txt);
        gtk_label_set_text ((GtkLabel *) label, q);
        g_free (q);
        g_free (txt);
    }
    /* remove anything left over */
    gtk_container_foreach ((GtkContainer *)
                           rodent_get_widget_by_name (view_p->widgets.paper, "module1_menu"), remove_menu_item, NULL);

    /* add new stuff to submenu */
    rfm_rational (PLUGIN_DIR,
                       module_name,
                       (void
                        *)(rodent_get_widget_by_name (view_p->widgets.paper, "module1_menu")), &(view_p->widgets), "extend_popup");

}


