//
//
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


   
static gboolean
environment_condition(const gchar *environment_string, const gchar *current_value){
    const gchar *environment_value=getenv (environment_string);
    if (!environment_value && !current_value) return FALSE;
    else if (!environment_value && current_value) return TRUE;
    else if (environment_value && !current_value) return TRUE;
    else if (strcmp(environment_value, current_value))  return TRUE;
    return FALSE;
}
   
static gchar *
environment_dup(const gchar *environment_string){
    if (getenv(environment_string)) return g_strdup(getenv(environment_string));
    return NULL;
}

// XXX background image works for the iconview too.
//     Just create a different environment variable
//     to do configuration separate from deskview...
static gboolean
background_test(view_t *view_p) {
    gboolean background_condition=(view_p->flags.type == ICONVIEW_TYPE)? FALSE:
	environment_condition("RFM_DESKTOP_IMAGE",view_p->eyecandy.desktop_bg_file);
    gboolean deskcolor_condition=(view_p->flags.type == ICONVIEW_TYPE)? FALSE:
	environment_condition("RFM_DESKTOP_COLOR",view_p->eyecandy.desktop_color);
    gboolean gridcolor_condition=(view_p->flags.type == DESKVIEW_TYPE)? FALSE:
	environment_condition("RFM_ICONVIEW_COLOR",view_p->eyecandy.iconview_color);
	
    if (deskcolor_condition &&
	    getenv("RFM_DESKTOP_IMAGE") &&
	    strlen(getenv("RFM_DESKTOP_IMAGE"))) background_condition = TRUE;

    if (background_condition){
	// eliminate the old pixbuf
	/*if (view_p->background_pixbuf) {
	    g_object_unref(view_p->background_pixbuf);
	}
	view_p->background_pixbuf=NULL;*/
	if (view_p->eyecandy.pixmap) {
	    Display *display=
		gdk_x11_display_get_xdisplay(gdk_display_get_default());
	    XFreePixmap(display, view_p->eyecandy.pixmap);
	    view_p->eyecandy.pixmap=None;
	}
	// create new background
	g_free(view_p->eyecandy.desktop_bg_file);
	view_p->eyecandy.desktop_bg_file = environment_dup("RFM_DESKTOP_IMAGE");
	if(view_p->eyecandy.desktop_bg_file && 
		!g_file_test (view_p->eyecandy.desktop_bg_file, G_FILE_TEST_EXISTS)) {
	    if (strlen(view_p->eyecandy.desktop_bg_file)){
		g_warning ("background image \"%s\" does not exist", view_p->eyecandy.desktop_bg_file);
	    }
	} else {
	    /*view_p->background_pixbuf = 
		rfm_create_background_pixbuf (view_p->eyecandy.desktop_bg_file, -1, -1);*/
	    view_p->eyecandy.pixmap=
		rfm_create_background_pixmap (view_p->eyecandy.desktop_bg_file);

	}
    }
    if (deskcolor_condition){
	g_free (view_p->eyecandy.desktop_color);
	view_p->eyecandy.desktop_color = environment_dup("RFM_DESKTOP_COLOR");
    }
    if (gridcolor_condition){
	g_free (view_p->eyecandy.iconview_color);
	view_p->eyecandy.iconview_color = environment_dup("RFM_ICONVIEW_COLOR");
    }
    return (gridcolor_condition | deskcolor_condition | background_condition);
}

static void
clean_rectangle (view_t * view_p, int x, int y, int w, int h, cairo_t *gdk_context, gint xoffset, gint yoffset) {
    if (w == 0 || h == 0) return;
    NOOP("clean_rectangle()\n");

    if (view_p->eyecandy.pixmap) { // background pixmap is available
        DBG("clean_rectangle(): background pixbuf is available\n");
        
	
	// lets use Pixmap instead. 

	if (w > view_p->view_layout.root_w) w=view_p->view_layout.root_w;
	if (h > view_p->view_layout.root_h) w=view_p->view_layout.root_h;
	Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
	Pixmap xpixmap = XCreatePixmap (display, GDK_ROOT_WINDOW (), w, h, 			view_p->view_layout.root_d);
	Visual *visual = gdk_x11_visual_get_xvisual(gdk_visual_get_system());
	GC graphic_context = XCreateGC(display, xpixmap, 0, 0);
	XCopyArea(display, view_p->eyecandy.pixmap, xpixmap, graphic_context,
		xoffset, yoffset, w, h, 0, 0);
	//XCopyArea(display, src, dest, gc, src_x, src_y, width, height,  dest_x, dest_y)
	NOOP ("PIXMAP x,y=%d,%d w,h=%d,%d\n", x, y, w, h);
	cairo_surface_t *Xsurface = cairo_xlib_surface_create(
		display, xpixmap, visual, w, h);

	cairo_surface_t *surface = cairo_surface_create_similar(
		cairo_get_target(gdk_context),
		CAIRO_CONTENT_COLOR_ALPHA, 
		w,h);
	cairo_t *context=cairo_create(surface);
	cairo_set_source_surface (context, Xsurface, 0, 0);
	cairo_paint(context);
	cairo_destroy(context);
	XFreePixmap(display, xpixmap);

	XFreeGC(display, graphic_context);

	cairo_set_source_surface(gdk_context, surface, 0, 0);
	cairo_rectangle(gdk_context, x, y, w, h);
	cairo_fill(gdk_context);
	
	cairo_surface_destroy(surface);
    } else {
	// select background color
	GdkColor *color=rfm_get_gdk_color (view_p, BACKGROUND_COLOR);


	cairo_set_source_rgb(gdk_context, 
		(double)(color->red)/0xffff,
		(double)(color->green)/0xffff, 
		(double)(color->blue)/0xffff);
	g_free(color);
	cairo_rectangle(gdk_context, x, y, w, h);
	cairo_fill(gdk_context);
    }

    
}


#define RED 0
#define GREEN 1
#define CYAN 2
#define MAGENTA 3
#define GREY 4
#define BGCOLOR 5
#define RUBBERBAND_COLOR 6
#define RUBBERBAND_COLOR_DEFINITION 0.35, 0.65, 0.65

static void 
saturated_rectangle(view_t * view_p, gint x, gint y, gint width, gint height, cairo_t *gdk_context, gint color) {
	// saturated rectangle (thin margin)
	gboolean is_dark_bg=rfm_is_dark_background(view_p);
	
	switch (color){
	    case RUBBERBAND_COLOR:
		// No outline here... (use same color as fill color)
		cairo_set_source_rgb(gdk_context, RUBBERBAND_COLOR_DEFINITION);
		break;
	    case BGCOLOR:
		// No outline here... (use same color as fill color)
		if (is_dark_bg){
		    cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
		} else {
		    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.95);
		}
		break;
	    case RED:
		// No outline here... (use same color as fill color)
		if (is_dark_bg){
		    cairo_set_source_rgb(gdk_context, 0.950, 0.30, 0.30);
		} else {
		    cairo_set_source_rgb(gdk_context, 0.850, 0.0, 0.0);
		}
		break;
	    case GREEN:
		cairo_set_source_rgb(gdk_context, 0.0, 1.0, 0.0);
		break;
	    case CYAN:
		cairo_set_source_rgb(gdk_context, 0.0, 1.0, 1.0);
		break;
	    case MAGENTA:
		cairo_set_source_rgb(gdk_context, 1.0, 0.0, 1.0);
	    case GREY:
		cairo_set_source_rgb(gdk_context, 0.30, 0.30, 0.30);
		break;
	}
	cairo_rectangle(gdk_context, x, y, width, height);
	cairo_fill(gdk_context);
	
	clean_rectangle (view_p, x + 1, y + 1, width-2, height-2, gdk_context, 0, 0);
	if (color == RED) {
	    if (is_dark_bg){
		cairo_set_source_rgb(gdk_context, 0.75, 0.75, 0.95);
	    } else {
		cairo_set_source_rgb(gdk_context, 0.3, 0.3, 0.7);
	    }
	} else if (color == BGCOLOR) {
	    if (is_dark_bg){
		cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.95);
	    } else {
		cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
	    }
	}  else if (color == RUBBERBAND_COLOR) {
	    // lighter blue background 
	    cairo_set_source_rgb(gdk_context, RUBBERBAND_COLOR_DEFINITION);
	} else {
	    // blue background 
	    cairo_set_source_rgb(gdk_context, 0.45, 0.65, 0.95);
	}
//	cairo_set_source_rgba(gdk_context, 0.0, 0.25, 0.85, 0.45);
	cairo_rectangle(gdk_context,  x + 1, y + 1, width-2, height-2);
	cairo_fill(gdk_context);
}

#define DETAILS_EXPOSE 0x01
#define COMPACT_EXPOSE 0x02
#define NORMAL_EXPOSE 0x04
#define SATURATED_LABEL_EXPOSE 0x08
#define SATURATED_ICON_EXPOSE 0x010
#define SATURATED_EXPOSE (SATURATED_LABEL_EXPOSE | SATURATED_ICON_EXPOSE)
#define SELECTED_EXPOSE 0x020
#define RUBBERBAND_EXPOSE 0x040
#define DARK_BACKGROUND_EXPOSE 0x080

static void
erase_text (view_t * view_p, population_t * population_p, cairo_t *gdk_context,
	gint xoffset, gint yoffset, gint expose_type) {
    GdkRectangle label_rect;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) return;

    GdkRectangle item_rect;
    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;

    // margin considerations
    item_rect.x += view_p->view_layout.margin_left;
    item_rect.y += view_p->view_layout.margin_top;
    label_rect.x += view_p->view_layout.margin_left;
    label_rect.y += view_p->view_layout.margin_top;

    gint width = CELLWIDTH(view_p);
    //gint height = label_rect.height;


    // Clean rectangle sets up the clip area for expose event...
    clean_rectangle (view_p, 
		item_rect.x-xoffset,//label_rect.x-xoffset, 
		label_rect.y - 1 -yoffset,
		width,
		CELLHEIGHT(view_p) - ICON_SIZE(view_p) - TEXTSPACING, // full_height
		gdk_context,
		xoffset, yoffset);

    if ((expose_type & SELECTED_EXPOSE) && (expose_type & RUBBERBAND_EXPOSE)) {
	saturated_rectangle(view_p, 
		item_rect.x -1 - xoffset, 
		label_rect.y - 1 - yoffset,
		CELLWIDTH(view_p) + 1, 
		CELLHEIGHT(view_p) - ICON_SIZE(view_p) - TEXTSPACING, // full_height
		gdk_context, RUBBERBAND_COLOR);
    } else if (expose_type & SATURATED_LABEL_EXPOSE) {
	// draw saturation box
	if (expose_type & (DETAILS_EXPOSE | COMPACT_EXPOSE)) {
		saturated_rectangle(view_p, 
			label_rect.x - 1 -xoffset, 
			label_rect.y - 1 -yoffset,
			label_rect.width + 2, 
			label_rect.height + 2, 
			gdk_context, RED);
	} else {
	    PangoRectangle rect;
	    pango_layout_get_pixel_extents (population_p->layout_full, NULL, &rect);
		saturated_rectangle(view_p, 
			label_rect.x - 1 -xoffset, 
			label_rect.y - 1 -yoffset,
			rect.width + 2, 
			rect.height + 2, 
			gdk_context, RED);
	}
    } /*else if (expose_type & SATURATED_ICON_EXPOSE) {
		saturated_rectangle(view_p, 
			label_rect.x - 1 -xoffset, 
			label_rect.y - 1 -yoffset,
			width + 2, 
			height + 2, 
			gdk_context, BGCOLOR);
    } */
}


static void
draw_mask(cairo_t *gdk_context, PangoLayout 	*layout, gint X, gint Y, gint expose_type) {
    if (expose_type & DARK_BACKGROUND_EXPOSE) {
	cairo_set_source_rgb(gdk_context, 0.0, 0.0, 0.0);
    } else {
	cairo_set_source_rgb(gdk_context, 0.99, 0.99, 0.99);
    }
    cairo_move_to (gdk_context,  X+1, Y+1);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X-1, Y-1);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X+1, Y-1);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X-1, Y+1);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X-1, Y);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X+1, Y);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X, Y+1);
    pango_cairo_show_layout (gdk_context, layout);
    cairo_move_to (gdk_context,  X, Y-1);
    pango_cairo_show_layout (gdk_context, layout);
}

static void
redraw_text (view_t * view_p, population_t * population_p, cairo_t *gdk_context,
	gint xoffset, gint yoffset) {

    // Figure out what we are dealing with from the beginning...
    gint expose_type = 0;
    // Three basic types of view: details, compact and normal
    if (view_p->view_layout.icon_size < TINY_ICON_SIZE){
	expose_type |= DETAILS_EXPOSE;
    } else if (view_p->view_layout.icon_size == TINY_ICON_SIZE){
	expose_type |= COMPACT_EXPOSE;
    } else {
	expose_type |= NORMAL_EXPOSE;
    }
    if (population_p->flags & POPULATION_SATURATED) {
	expose_type |= SATURATED_ICON_EXPOSE;
    } else if (population_p->flags & LABEL_SATURATED) {
	expose_type |= SATURATED_LABEL_EXPOSE;
    } else if (population_p->flags & POPULATION_SELECTED) {
	expose_type |= SELECTED_EXPOSE;
    }
    if (view_p->mouse_event.rubberbanding) {
	expose_type |= RUBBERBAND_EXPOSE;
    }
    if (rfm_is_dark_background(view_p)) {
	expose_type |= DARK_BACKGROUND_EXPOSE;
    }

    erase_text (view_p, population_p, gdk_context, xoffset, yoffset, expose_type); 
    GdkRectangle label_rect;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) return;

    // margin considerations
    label_rect.x += view_p->view_layout.margin_left;
    label_rect.y += view_p->view_layout.margin_top;
 
    population_p->labelX=label_rect.x;
    population_p->labelY=label_rect.y;
    
    // redraw_text_outline
    // (not for saturated labels...)
    // This only applies when we have a background image, really.
    gboolean do_outline = (view_p->flags.type == DESKVIEW_TYPE &&
	    getenv ("RFM_DESKTOP_IMAGE") && 
	    strlen (getenv ("RFM_DESKTOP_IMAGE")) );
    if (do_outline && !(expose_type & SATURATED_EXPOSE)) {
	GdkColor *color=rfm_get_gdk_color (view_p, BACKGROUND_COLOR);
	cairo_set_source_rgb(gdk_context, 
		(double)(color->red)/0xffff, 
		(double)(color->green)/0xffff, 
		(double)(color->blue)/0xffff);
	g_free(color);    
	gint i, j;
	
	gint x = population_p->labelX-xoffset;
	gint y = population_p->labelY-yoffset;
	NOOP("outline for %s\n", population_p->en->path);
	for(i = x; i < x + 3; i++) {
	    for(j = y; j < y + 3; j++) {
		//if(i == x && j == y) continue;
		cairo_move_to (gdk_context, i,j);
		pango_cairo_show_layout (gdk_context, population_p->layout);
		if(population_p->layout2) {
		gint X=(view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
			i:
			i+population_p->logical_rect.width +2;
		gint Y=(view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
			j + population_p->logical_rect.height:
			j;
		cairo_move_to (gdk_context, X, Y);
		pango_cairo_show_layout (gdk_context, population_p->layout2);
		}
	    }
	}
    }

    // redraw layouts;
    gint X = population_p->labelX-xoffset + 1;
    gint Y = population_p->labelY-yoffset + 1;
    if (!population_p->layout) {
	g_warning("Layout should not be NULL\n");
	return;
    }

    // Redefine text color.
    if (expose_type & SATURATED_EXPOSE) {
	if (expose_type & DARK_BACKGROUND_EXPOSE) {
	    cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
	} else {
	    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.95);
	}
    } else {
	if ((expose_type & SELECTED_EXPOSE) && (expose_type & RUBBERBAND_EXPOSE)){
	    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.05);
	} else { 
	    GdkColor *color=
		    rfm_get_gdk_color (view_p, rodent_text_color(population_p));
	    cairo_set_source_rgb(gdk_context, 
		    (double)(color->red)/0xffff, 
		    (double)(color->green)/0xffff,
		    (double)(color->blue)/0xffff);
	    g_free(color);
	}
    }


#define DARK_SAT_COLOR 0.95, 0.95, 0.95
#define LIGHT_SAT_COLOR  0.05, 0.05, 0.05
    if(expose_type & SATURATED_ICON_EXPOSE){
	if (expose_type & SATURATED_LABEL_EXPOSE && !(expose_type & (DETAILS_EXPOSE | COMPACT_EXPOSE))) {
		if(population_p->layout_full)
		    draw_mask(gdk_context,  population_p->layout_full, X, Y, expose_type);
	} else {
		if(population_p->layout)
		    draw_mask(gdk_context,  population_p->layout, X, Y, expose_type);
	}
	if (expose_type & DARK_BACKGROUND_EXPOSE) {
	    cairo_set_source_rgb(gdk_context, DARK_SAT_COLOR);
	} else {
	    cairo_set_source_rgb(gdk_context, LIGHT_SAT_COLOR);
	}
    }
    cairo_move_to (gdk_context,  X, Y);
    // XXX: We have a race crash here. Let's see if this mutex will 
    // 	    avoid it. crash in
    // 	    pango_layout_get_iter () from /usr/lib/libpango-1.0.so.0
    // 	    called from
    // 	    pango_renderer_draw_layout () from /usr/lib/libpango-1.0.so.0
    GStaticMutex pango_draw_mutex = G_STATIC_MUTEX_INIT;
    if (expose_type & SATURATED_LABEL_EXPOSE && !(expose_type & (DETAILS_EXPOSE | COMPACT_EXPOSE))) {
	if(population_p->layout_full){
	    g_static_mutex_lock(&pango_draw_mutex);
	    pango_cairo_show_layout (gdk_context, population_p->layout_full);
	    g_static_mutex_unlock(&pango_draw_mutex);
	}
    } else {
	if(population_p->layout){
	    g_static_mutex_lock(&pango_draw_mutex);
	    pango_cairo_show_layout (gdk_context, population_p->layout);
	    g_static_mutex_unlock(&pango_draw_mutex);
	}
    }

    // Figure out layout2 position
    if (population_p->layout2 && !(expose_type & COMPACT_EXPOSE)){
	if(expose_type & DETAILS_EXPOSE){
	    gint max_tiny_width=view_p->view_layout.name_width;
	    X = population_p->labelX + max_tiny_width -xoffset +1;
	    Y = population_p->labelY-yoffset + 1;
	    if(expose_type & SATURATED_EXPOSE){
		// mask
		draw_mask(gdk_context,  population_p->layout2, X, Y, expose_type);
		if (expose_type & DARK_BACKGROUND_EXPOSE) {
		    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.0);
		} else {
		    cairo_set_source_rgb(gdk_context, 0.0, 0.0, 0.95);
		}
	    }
	} else if (!(expose_type & SATURATED_LABEL_EXPOSE)){
	    X = population_p->labelX-xoffset + 1;
	    Y = population_p->labelY + population_p->logical_rect.height-yoffset + 1;
	    if (expose_type & SATURATED_ICON_EXPOSE){
		draw_mask(gdk_context,  population_p->layout2, X, Y, expose_type);
		if (expose_type & DARK_BACKGROUND_EXPOSE) {
		    cairo_set_source_rgb(gdk_context, DARK_SAT_COLOR);
		} else {
		    cairo_set_source_rgb(gdk_context, LIGHT_SAT_COLOR);
		}
	    }
	}
	if (!(expose_type & DETAILS_EXPOSE) && (expose_type & SELECTED_EXPOSE) 
		&& (expose_type & RUBBERBAND_EXPOSE)){
	    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.05);
	} 	
	cairo_move_to (gdk_context,  X, Y);
	if (expose_type & DETAILS_EXPOSE || !(expose_type & SATURATED_LABEL_EXPOSE)){
	    pango_cairo_show_layout (gdk_context, population_p->layout2);
	}
    }
}

static void
insert_layout (view_t * view_p, population_t * population_p, cairo_t *context,
	gint xoffset, gint yoffset) {
    if(!view_p || !population_p) {
        g_warning ("function parameter is null");
        return;
    }
    redraw_text (view_p, population_p, context, xoffset, yoffset);

}

// readlock should be enabled before entering this function...
static void
insert_pixbuf (view_t * view_p, 
	population_t * population_p,
	cairo_t *gdk_context,
	gint xoffset, gint yoffset) {
    if (!population_p->pixbuf) return;
    //if (population_p->x < 0) return;
    // insert pixbuf
	
    GdkRectangle item_rect;
    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;

    // margin considerations
    item_rect.x += view_p->view_layout.margin_left;
    item_rect.y += view_p->view_layout.margin_top;
 
    GdkRectangle icon_rect;
    if (!rfm_get_population_icon_rect(view_p, population_p, &icon_rect))return;

    // margin considerations
    icon_rect.x += view_p->view_layout.margin_left;
    icon_rect.y += view_p->view_layout.margin_top;
	    
    clean_rectangle (view_p,
		item_rect.x-xoffset, 
		item_rect.y-yoffset,
		item_rect.width,
		ICON_SIZE(view_p), 
		gdk_context,
		xoffset, yoffset);

    gint x = (view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
	    item_rect.x + ((CELLWIDTH(view_p) - ICON_SIZE(view_p)) / 2):
	    item_rect.x;
 
    // saturated item background 
    // (the color is for the thin margin around the rectangle
    //  and will also determine the rectangle fill color)
    if(population_p->flags & POPULATION_SATURATED) {
		saturated_rectangle(view_p, 
			x - xoffset,
			item_rect.y - yoffset,
			ICON_SIZE(view_p) + 2, 
			ICON_SIZE(view_p) + 2, gdk_context, 
			RED);
     } else if (population_p->flags & POPULATION_SELECTED) {
	if(view_p->mouse_event.rubberbanding) {
	        // rubberband box
		saturated_rectangle(view_p, 
			item_rect.x - xoffset,
			item_rect.y - yoffset,
			CELLWIDTH(view_p), 
			CELLHEIGHT(view_p), 
			gdk_context, 
			RUBBERBAND_COLOR);
	} else {
		// simply select
		saturated_rectangle(view_p, 
			x - xoffset,
			item_rect.y - yoffset,
			ICON_SIZE(view_p) + 2, 
			ICON_SIZE(view_p) + 2, gdk_context, 
			MAGENTA);
	}
     } else {
	    clean_rectangle (view_p,
		icon_rect.x-xoffset, 
		icon_rect.y-yoffset,
		icon_rect.width + 5,
		icon_rect.height + 5,
		gdk_context,
		xoffset, yoffset);
    }
    // Special stuff for thumbnails (but not for folders)
    if(population_p->en && !IS_SDIR(population_p->en->type) && 
	    (population_p->flags & POPULATION_THUMBNAILED) ) {
	NOOP ("MIME: %s is image type\n", population_p->en->mimetype);
	// cute shadow
	if (!(population_p->flags & POPULATION_SATURATED) &&
	    !(population_p->flags & POPULATION_SELECTED)
	    ) {
	    cairo_set_source_rgba(gdk_context, 0.0, 0.0, 0.0, 0.35);
	    cairo_rectangle(gdk_context, 
		icon_rect.x+2-xoffset,
		icon_rect.y+2-yoffset,
		icon_rect.width + 3,
		icon_rect.height + 3);
	    cairo_fill(gdk_context);
	}
	// black frame
	cairo_set_source_rgb(gdk_context, 0.0, 0.0, 0.0);
	
	cairo_rectangle(gdk_context, 
		icon_rect.x-xoffset,
		icon_rect.y-yoffset,
		icon_rect.width + 2,
		icon_rect.height + 2);
	cairo_fill(gdk_context);

	// fill in with white color, for the benefit of transparent previews.
	if (population_p->flags & POPULATION_SELECTED) {
	    cairo_set_source_rgba(gdk_context, 1.0, 1.0, 1.0, 0.450);
	} else {
	    cairo_set_source_rgb(gdk_context, 1.0, 1.0, 1.0);
	}
	cairo_rectangle(gdk_context, 
		icon_rect.x+1-xoffset,
		icon_rect.y+1-yoffset,
		icon_rect.width,
		icon_rect.height);
	cairo_fill(gdk_context);
	
    }
    // icon pixbuf
    NOOP("---> inserting %s at %d,%d\n",
	    population_p->en->path, icon_rect.x+1, icon_rect.y+1);


    if (GDK_IS_PIXBUF(population_p->pixbuf)){
		gdk_cairo_set_source_pixbuf(gdk_context, 
				population_p->pixbuf,
				icon_rect.x+1-xoffset,
				icon_rect.y+1-yoffset);
		if(population_p->flags & POPULATION_SELECTED) {
			cairo_paint_with_alpha(gdk_context, 0.450);
		} else {
			cairo_paint (gdk_context); 
		}
    }
    
    // Entry dependent pixbuf modifiers
    if (!population_p->en) goto done;

#if 10
    GdkPixbuf *emblem_pixbuf=NULL;
    // Emblems...
    // Module emblems
    if (POPULATION_MODULE(population_p)){
	// ask POPULATION_MODULE for emblem
	const gchar *emblem=(const gchar *)
	    rfm_natural(PLUGIN_DIR, POPULATION_MODULE(population_p), 
		    population_p->en, "get_emblem");
	if (emblem) {
	  emblem_pixbuf =
	    rfm_get_pixbuf (emblem, REDUCED_ICONSIZE(view_p)/1.8);
	  if (GDK_IS_PIXBUF(emblem_pixbuf)){
	    gdk_cairo_set_source_pixbuf(gdk_context, 
		emblem_pixbuf,
		icon_rect.x+1-xoffset+icon_rect.width - icon_rect.width/1.8, 
		icon_rect.y+1-yoffset);
//		icon_rect.y+1-yoffset+icon_rect.height/1.8);
	    cairo_paint_with_alpha(gdk_context, 0.85);
	  }
	}
    }


    // Up type icon emblems...
    if (IS_UP_TYPE (population_p->en->type)){
	gboolean have_icons_plugin =
	    GPOINTER_TO_INT(rfm_void(MODULE_DIR, "icons", "module_active"));
	if (have_icons_plugin) {
	    if(view_p->flags.preferences & __SHOW_HIDDEN){
	      emblem_pixbuf = 
		  rfm_get_pixbuf ("xffm/emblem_show-hidden", REDUCED_ICONSIZE(view_p)/3);
	      if (GDK_IS_PIXBUF(emblem_pixbuf)){
		gdk_cairo_set_source_pixbuf(gdk_context, 
		  emblem_pixbuf,
		  icon_rect.x+1-xoffset, 
		  icon_rect.y+1-yoffset);
		cairo_paint_with_alpha(gdk_context, 0.85);
	      }
	    }
	    if(view_p->flags.preferences & __SHOW_IMAGES){
	      emblem_pixbuf = 
		  rfm_get_pixbuf ("xffm/emblem_show-previews", REDUCED_ICONSIZE(view_p)/6);
	      if (GDK_IS_PIXBUF(emblem_pixbuf)){
		  gdk_cairo_set_source_pixbuf(gdk_context, 
		    emblem_pixbuf,
		    icon_rect.x+1-xoffset, 
		    icon_rect.y+1-yoffset + icon_rect.height - icon_rect.width/6);
		  cairo_paint_with_alpha(gdk_context, 0.85);
	      }
	    }
	}

    }

    // Readonly emblem
    gboolean no_write_emblem=FALSE;
    if (IS_LOCAL_TYPE(population_p->en->type)) {
	no_write_emblem = 
	    !IS_UP_TYPE(population_p->en->type) &&
	    !IS_SDIR(population_p->en->type) &&
	    IS_NOWRITE_TYPE(population_p->en->type);
	   // !rfm_write_ok_path(population_p->en->path);
    }
    if (population_p->en && no_write_emblem){
	gint size = REDUCED_ICONSIZE(view_p) * 0.30;
	emblem_pixbuf =
	    rfm_get_pixbuf("xffm/emblem_readonly", size);	
	if (GDK_IS_PIXBUF(emblem_pixbuf)){
	    gdk_cairo_set_source_pixbuf(gdk_context,
		emblem_pixbuf,	
		icon_rect.x+1-xoffset + icon_rect.width - icon_rect.width/3, 
		icon_rect.y+1-yoffset);
	    cairo_paint_with_alpha(gdk_context, 0.85);
	}
    }

    
    // Bookmark emblem
    if (!IS_UP_TYPE(population_p->en->type) &&
	    rodent_path_has_bookmark(population_p->en->path)){
	emblem_pixbuf =
	    rodent_get_bookmark_emblem(REDUCED_ICONSIZE(view_p));
	if (GDK_IS_PIXBUF(emblem_pixbuf)){
	    gdk_cairo_set_source_pixbuf(gdk_context, 
		emblem_pixbuf,
		icon_rect.x+1-xoffset + icon_rect.width - icon_rect.width/3, 
		icon_rect.y+1-yoffset);
	    cairo_paint_with_alpha(gdk_context, 0.85);
	}
    }

    // Pasteboard emblems...
    GdkPixbuf *pasteboard_pixbuf=NULL;
    // CUT emblem
    if (population_p->flags & POPULATION_CUT) {
	pasteboard_pixbuf = rfm_get_pixbuf ("xffm/stock_cut", REDUCED_ICONSIZE(view_p)/3);
    }
    // COPIED emblem
    else if (population_p->flags & POPULATION_COPIED) {
	pasteboard_pixbuf = rfm_get_pixbuf ("xffm/stock_copy", REDUCED_ICONSIZE(view_p)/3);
    }
    if (GDK_IS_PIXBUF(pasteboard_pixbuf)) {
	gdk_cairo_set_source_pixbuf(gdk_context, 
	    pasteboard_pixbuf,
	    icon_rect.x+1-xoffset, 
	    icon_rect.y+1-yoffset);
	cairo_paint_with_alpha(gdk_context, 0.85);
    }

    // Symlink emblem...
    if (IS_SLNK(population_p->en->type) && 
	    !IS_UP_TYPE(population_p->en->type)) {
	emblem_pixbuf =
	    rfm_get_pixbuf ("xffm/emblem_link", REDUCED_ICONSIZE(view_p)/2);
      if (GDK_IS_PIXBUF(emblem_pixbuf)) {
	int height = gdk_pixbuf_get_height (population_p->pixbuf) -
	    gdk_pixbuf_get_height (emblem_pixbuf);
	// "SW"

	gdk_cairo_set_source_pixbuf(gdk_context, 
	    emblem_pixbuf,
	    icon_rect.x+1-xoffset, 
	    icon_rect.y+1-yoffset+height);
	cairo_paint_with_alpha(gdk_context, 0.85);
      }

    }
#endif
done:
    return;
}

static gint
get_minimum(view_t *view_p, GSList *render_list, gboolean getX, gint min){
    GSList *tmp;
    for (tmp=render_list; tmp && tmp->data; tmp=tmp->next){
	population_t *population_p=tmp->data;
#if 0
	// hack test
	gboolean OK = FALSE;
	population_t **population_pp = view_p->population_pp;
	for (;population_pp && *population_pp; population_pp++){
	    if (population_p == *population_pp){
		OK = TRUE;
		break;
	    }
	}
	if (!OK) g_error("get_minimum(): invalid population_p!\n");
#endif
	gint c;
	if (getX) {
	    c=population_p->column * CELLWIDTH(view_p) + view_p->view_layout.margin_left;
	} else {
	    c=population_p->row * CELLHEIGHT(view_p) + view_p->view_layout.margin_top;
	}
	if (c < min) min=c;

    }
    return min;
}

static gint
get_minimum_x(view_t *view_p, GSList *render_list, gint min){
    return get_minimum(view_p, render_list, TRUE, min);
}

static gint
get_minimum_y(view_t *view_p, GSList *render_list, gint min){
    return get_minimum(view_p, render_list, FALSE, min);
}

static
void
render_slist(view_t *view_p, GSList *render_list, cairo_t *context,
	gint xoffset, gint yoffset){
    GSList *slist;
    for (slist=render_list; slist && slist->data; slist=slist->next){
	population_t *population_p=slist->data;
	insert_pixbuf (view_p, population_p, context, xoffset, yoffset);
	insert_layout (view_p, population_p, context, xoffset, yoffset);
    }
}

typedef struct expose_slist_t {
    view_t *view_p;
    GSList *render_list;
    GdkRectangle area;
} expose_slist_t;

expose_slist_t *mk_expose_slist_p(view_t *view_p, GSList *render_list, GdkRectangle *area){
    expose_slist_t *expose_slist_p =
	(expose_slist_t *)malloc(sizeof(expose_slist_t));
    if (!expose_slist_p) g_error("malloc: %s", strerror(errno));
    memcpy(&(expose_slist_p->area), area, sizeof(GdkRectangle));
    expose_slist_p->view_p = view_p;
    expose_slist_p->render_list = render_list;
    return expose_slist_p;
}

void free_expose_slist_p(expose_slist_t *expose_slist_p) {
    if (!expose_slist_p) return;
    if (expose_slist_p->render_list) {
	g_slist_free(expose_slist_p->render_list);
    }
    g_free(expose_slist_p);

}

// The offset values for x and y are for using the sandbox
// cairo surface with origin at 0,0 to avoid flashing.
// This function is not thread safe because gdk_cairo_create()
// uses XSetClipRectangles(), which is apparently non threadable.
static
void *
expose_slist_f(gpointer data){
//expose_slist(view_t *view_p, GSList *render_list, GdkRectangle *area){
    
    expose_slist_t *expose_slist_p = data;
    view_t *view_p = expose_slist_p->view_p;
    GSList *render_list = expose_slist_p->render_list;
    GdkRectangle *area = &(expose_slist_p->area);

    if (!area){
	g_warning("area != NULL not met (expose_slist)");
	return NULL;
    }
    NOOP ("expose_slist_f(): rendering %d,%d  %d, %d\n",area->x, area->y, area->width, area->height);
    // First, clean rectangle. This is done either by copying the relevant
    // portion of the background image, or simply filling the rectangle
    // with the background color.

    gint xoffset=0;
    gint yoffset=0;
    // disable USE_OFFSCREEN_SURFACE only for debugging purposes
    // USE_OFFSCREEN_SURFACE eliminates expose flashing
#define USE_OFFSCREEN_SURFACE 1
#ifdef USE_OFFSCREEN_SURFACE
    cairo_t *display_context =  
	    gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
    cairo_surface_t *surface;
    surface =	cairo_surface_create_similar(
		cairo_get_target(display_context),
		CAIRO_CONTENT_COLOR_ALPHA, 
		area->width, 
		area->height);
    xoffset=area->x;
    yoffset=area->y;
    cairo_t *gdk_context=cairo_create(surface);
#else
    cairo_t *gdk_context =  
	    gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
#endif   

    // clean entire expose area
    clean_rectangle (view_p,
		 area->x-xoffset, 
		 area->y-yoffset, 
		 area->width,
		 area->height,
		 gdk_context,
		 xoffset, yoffset);
    


    // Render the affected items.
    // Only if something is listed, of course.
    if (render_list) {
	render_slist(view_p, render_list, gdk_context, 
	    xoffset, yoffset);
    }

    cairo_destroy(gdk_context);

#ifdef USE_OFFSCREEN_SURFACE
    // Flash copy the surface to the display
    cairo_set_source_surface (display_context, surface, 
	    xoffset, yoffset);
    cairo_paint(display_context);
    cairo_surface_destroy(surface);
    cairo_destroy(display_context);
#endif
    return NULL;

}


