/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm_libs.h"
#include "rfm_modules.h"
#include "primary-run.i"

/////////////////////// Module wraparounds ////////////////////////////
//
#if 0
// Use rfm_thread_run_argv or rfm_thread_run?
// rfm_thread_run_argv will execute directly, not via a shell, and will
// not be saved in the lpterm history

// rfm_thread_run_argv:
// This modality will execute the command without shell


// rfm_thread_run:
// This modality will execute the command via "sh -c" and allow pipes and 
// redirection. and will be saved in the lpterm history file

pid_t
rfm_thread_run (widgets_t * in_widgets_p, const gchar * command, gboolean interm) {
    void **arg = rfm_argument_container_new(3);
    arg[0] = in_widgets_p;
    arg[1] = (void *)command;
    arg[2] = GINT_TO_POINTER(interm);

    pid_t child = GPOINTER_TO_INT(
	    rfm_natural(MODULE_DIR, "run", arg, "m_thread_run"));
    return child;
}



pid_t
rfm_try_sudo (widgets_t * in_widgets_p, gchar ** argv, gboolean interm) {
    void **arg = rfm_argument_container_new(3);
    arg[0] = in_widgets_p;
    arg[1] = argv;
    arg[2] = GINT_TO_POINTER(interm);

    pid_t child = GPOINTER_TO_INT(
	    rfm_natural(MODULE_DIR, "run", arg, "m_try_sudo"));
    return child;
}


pid_t
rfm_thread_run2argv (widgets_t * widgets_p, const gchar * command, gboolean interm) {
    void **arg = rfm_argument_container_new(3);
    arg[0] = widgets_p;
    arg[1] = (void *)command;
    arg[2] = GINT_TO_POINTER(interm);
    pid_t child = GPOINTER_TO_INT(
	    rfm_natural(MODULE_DIR, "run", arg, "m_thread_run2argv"));
    return child;
}
#endif
    
/*static pid_t
vector_run(const void **vector){
    void **arg = rfm_argument_container_new(7, vector);
    pid_t child = GPOINTER_TO_INT(
	    rfm_natural(MODULE_DIR, "run", arg, "m_thread_run_argv"));
    return child;
}*/

pid_t
rfm_thread_run_argv (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), NULL,  NULL, NULL, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));
}

pid_t
rfm_thread_run_argv_full (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm,
	gint *stdin_fd,
	void (*stdout_f) (void *stdout_data,
                      void *stream,
                      int childFD),
	void (*stderr_f) (void *stdout_data,
                      void *stream,
                      int childFD),
	void (*tubo_done_f) (void *data)
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), stdin_fd,  stdout_f, stderr_f, tubo_done_f};
    return GPOINTER_TO_INT(rfm_vector_run(MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

 }

pid_t
rfm_thread_run_argv_with_stdin (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm, 
	gint *stdin_fd
    	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), stdin_fd,  NULL, NULL, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

 
}

pid_t
rfm_thread_run_argv_with_stdout (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm, 
	void (*stdout_f) (void *stdout_data,
                      void *stream,
                      int childFD)
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), NULL,  stdout_f, NULL, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

    
}

pid_t
rfm_thread_run_argv_with_stderr (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm, 
	void (*stderr_f) (void *stderr_data,
                      void *stream,
                      int childFD)
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), NULL,  NULL, stderr_f, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

}
////////////////////////////////////////////////////////////////////////////

void
rfm_recover_flags (gchar * in_cmd, gboolean * interm, gboolean * hold) {
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *g = g_build_filename ( RUN_FLAG_FILE, NULL);
    if((runflags = dbh_new (g, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE)) == NULL) {
        DBG ("Cannot open %s\n", g);
        *interm = 0;
        *hold = 0;
        return;
    }
    gs = g_string_new (in_cmd);
    sprintf ((char *)DBH_KEY (runflags), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    flags = (int *)runflags->data;
    dbh_load (runflags);
    *interm = flags[0];
    *hold = flags[1];
    dbh_close (runflags);

    NOOP ("flags recovered from dbh file for %s, interm=%d hold=%d\n", in_cmd, *interm, *hold);
}

// this function is run in a gthread at startup. 
// mutex is to protect from main thread running the
// save history function.
static GMutex *command_history_mutex=NULL;
gpointer
rfm_load_sh_command_history (gpointer data) {
    // This thread is not barrier subject
    if (!command_history_mutex) {
	command_history_mutex=g_mutex_new();
    }
    view_t *view_p = (view_t *) data;
    gchar *history = g_build_filename (LP_TERMINAL_HISTORY, NULL);
    TRACE ("rfm_load_sh_command_history(): %s\n", history);
    g_mutex_lock(command_history_mutex);
    g_list_free (view_p->sh_command);
    GList *p;
    for(p = view_p->sh_command; p; p = p->next) {
        g_free (p->data);
    }
    view_p->sh_command = NULL;
    view_p->sh_command = g_list_append (view_p->sh_command, g_strdup (""));
    view_p->sh_command_counter = 0;

    FILE *sh_history = fopen (history, "r");
    if(sh_history) {
        // we should put a file lock here...
        DBG ("rfm_load_sh_command_history(): readlock for %s\n", history);
        rfm_lock ();
        gchar line[2048];
        memset (line, 0, 2048);
        while(fgets (line, 2047, sh_history) && !feof (sh_history)) {
            NOOP ("HISTORY: line %s\n", line);
            if(strchr (line, '\n'))
                *strchr (line, '\n') = 0;
            if(strlen (line) == 0)
                continue;
            // skip invalid commands (except cd):
            if(!MIME_is_valid_command (line)) {
                if(strcmp (line, "cd") != 0 && strncmp (line, "cd ", strlen ("cd ")) != 0) {
                    NOOP ("HISTORY: invalid history command in %s: %s\n", history, line);
                    continue;
                }
            }
	    gchar *newline=compact_line(line);
	    GList *element=find_in_history_list(view_p->sh_command, newline);

	    if (element) { 
		// remove old element
		gchar *data=element->data;
		view_p->sh_command = g_list_remove(view_p->sh_command, data);
		g_free(data);
	    }
	    // put at top of the pile
	    view_p->sh_command =
		    g_list_insert_before (view_p->sh_command, g_list_last (view_p->sh_command), newline);
        }
        rfm_unlock ();
        DBG ("rfm_load_sh_command_history(): readunlock for %s\n", history);
        fclose (sh_history);
        view_p->sh_command_counter = g_list_length (view_p->sh_command) - 1;
    }
    g_free (history);
    g_mutex_unlock(command_history_mutex);
    return NULL;
}

void
rfm_save_sh_command_history (view_t * view_p, const gchar * command) {
    if (!view_p){
	return;
    }
    if (!command_history_mutex) {
	command_history_mutex=g_mutex_new();
    }
    GList *p;
    g_mutex_lock(command_history_mutex);
    p = g_list_previous (g_list_last (view_p->sh_command));
    if(!command || !strlen (command)){
	g_mutex_unlock(command_history_mutex);
        return;
    }
    gchar *command_p = g_strdup (command);
    g_strstrip (command_p);
    // if repeat of last command, skip it.
    if(!p || strcmp (command, (char *)p->data)) {
        view_p->sh_command = g_list_insert_before (view_p->sh_command, g_list_last (view_p->sh_command), command_p);
        //view_p->sh_command = g_list_append(view_p->sh_command, command_p);

        /* don't save to file if invalid command */
        if(!MIME_is_valid_command (command_p)) {
            if(strcmp (command_p, "cd") != 0 && strncmp (command_p, "cd ", strlen ("cd ")) != 0) {
                g_warning ("not saving %s", command_p);
                view_p->sh_command_counter = g_list_length (view_p->sh_command) - 1;
		g_mutex_unlock(command_history_mutex);
                return;
            }
        }
        // here we will rewrite the history, removing any item
        // which duplicate the command about to be saved.
        // This effectively moves the command to the bottom
        // of the list.
        // 

        gchar *history = g_build_filename (LP_TERMINAL_HISTORY, NULL);

        // read it first to synchronize with other rodent instances
        GList *disk_history = NULL;
        FILE *sh_history = fopen (history, "r");

        if(sh_history) {
            // we should put a file lock here...
            rfm_lock ();
            char line[2048];
            memset (line, 0, 2048);
            while(fgets (line, 2047, sh_history) && !feof (sh_history)) {
                if(strchr (line, '\n')) {
                    *strchr (line, '\n') = 0;
                }
                if(strcmp (line, command_p) != 0) {
                    disk_history = g_list_append (disk_history, g_strdup (line));
                }
            }
            rfm_unlock ();
            fclose (sh_history);
        }
        disk_history = g_list_append (disk_history, g_strdup (command_p));

        sh_history = fopen (history, "w");
        if(sh_history) {
            // we should put a file lock here...
            rfm_lock ();
            GList *p;
            for(p = g_list_first (disk_history); p && p->data; p = p->next) {
                fprintf (sh_history, "%s\n", (char *)p->data);
                g_free (p->data);
            }
            rfm_unlock ();
            fclose (sh_history);
        }
        g_list_free (disk_history);
        g_free (history);
    }
    view_p->sh_command_counter = g_list_length (view_p->sh_command) - 1;
    TRACE ("rfm_save_sh_command_history(); command_counter=%d\n",
	    view_p->sh_command_counter);
    g_mutex_unlock(command_history_mutex);
    return;
}

 
const gchar * 
rfm_term_exec_option(const gchar *terminal) {
    const gchar *exec_option = "-e";
    gchar *t = g_path_get_basename (terminal);
    if(strcmp (t, "gnome-terminal") == 0 || strcmp (t, "Terminal") == 0)
            exec_option = "-x";
    g_free(t);
    return exec_option;
}

const gchar *
rfm_what_term (void) {
    const gchar *term=getenv ("TERMINAL_CMD");
    gchar *t=NULL;
    if(term && strlen (term)) {
        t = g_find_program_in_path (term);
    }
    if(!t) {
	    const gchar **p=terminals_v;
	    for (;p && *p; p++){
		t = g_find_program_in_path (*p);
		if (t) {
		    term=*p;
		    break;  
		}  
	    }
    }
    if (t) {
	g_free(t);
	return term;
    }
    g_warning ("TERMINAL_CMD=%s: %s", getenv ("TERMINAL_CMD"), strerror (ENOENT));

    return NULL;
}

