/*
 * Copyright (C) 2003-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __MODULES_C__
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm_libs.h"
#include "rfm_modules.h"

static GHashTable *module_hash = NULL;
static GMutex *module_hash_mutex = NULL;


const gchar *
rfm_get_module_popup_id(void){
    return "module_popup_widget";
}

gboolean 
rfm_is_root_plugin(const gchar *module_name){
    if (!module_name) return FALSE;
    // new items:
    if (strncmp(module_name, "root-", strlen("root-"))==0) return TRUE;
    // old items:
    if (strcmp(module_name, "fstab")==0) return TRUE;
    if (strcmp(module_name, "dotdesktop")==0) return TRUE;
    if (strcmp(module_name, "ps")==0) return TRUE;
    if (strcmp(module_name, "fuse")==0) return TRUE;
    if (strcmp(module_name, "smb")==0) return TRUE;
    return FALSE;
}

const gchar * 
rfm_get_plugin_icon(const gchar *module_name){
    if (!module_name) return NULL;
    // old items:
    if (strcmp(module_name, "fstab")==0) return "xffm/stock_harddisk";
    if (strcmp(module_name, "dotdesktop")==0) return "xffm/places_folder-system";
    if (strcmp(module_name, "ps")==0) return "xffm/stock_execute";
    if (strcmp(module_name, "fuse")==0) return "xffm/places_folder-remote";
    if (strcmp(module_name, "smb")==0) return "xffm/places_network-workgroup/composite3/places_network-server";
    // new items:
    if (strncmp(module_name, "root-", strlen("root-"))==0){
	return rfm_void (PLUGIN_DIR, module_name, "module_icon_id");
    }
    return NULL;
}

const gchar * 
rfm_get_plugin_label(const gchar *module_name){
    if (!module_name) return NULL;
    // old items:
    if (strcmp(module_name, "fstab")==0) return _("Mount Point");
    if (strcmp(module_name, "dotdesktop")==0) return _("Application Launcher...");
    if (strcmp(module_name, "ps")==0) return _("System Processes");
    if (strcmp(module_name, "fuse")==0) return _("Mount user-space filesystems (FUSE)");
    if (strcmp(module_name, "smb")==0) return _("SMB Browser");
    // new items:
    if (strncmp(module_name, "root-", strlen("root-"))==0){
	return rfm_void (PLUGIN_DIR, module_name, "module_label");
    }
    return NULL;
}

void
rfm_sanity_check (int argc, char **argv, int librfm_serial) {
    if(librfm_serial != LIBRFM_SERIAL) {
        GtkWidget *dialog;
	/* This warning text is only for developers modifying and rebuilding */
        gchar *p = g_strdup_printf ("%s needs to be recompiled \n(has obsolete library headers)", argv[0]);
        gtk_init (&argc, &argv);
        dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK, "%s", p);
        gtk_dialog_run (GTK_DIALOG (dialog));
        exit (0);
    }
    return;
}

static gint
compare_strings (gconstpointer a, gconstpointer b) {
    return (strcmp ((char *)a, (char *)b));
}

#if 10
GSList *
rfm_find_plugins (void) {
    static GSList *plugin_list = NULL;
    static gsize initialized = 0;
    // Only once...
    if (g_once_init_enter(&initialized)){
	GError *error = NULL;
	gchar *full_libdir = g_build_filename (LIBDIR, PLUGIN_DIR, NULL);

	GDir *dir;
	if((dir = g_dir_open (full_libdir, 0, &error)) != NULL) {
	    const gchar *name;
	    while((name = g_dir_read_name (dir)) != NULL) {
		gchar *plugin;
		if(strncmp (name, "lib", strlen ("lib")))
		    plugin = g_strdup (name);
		else
		    plugin = g_strdup (name + strlen ("lib"));
		if(strchr (plugin, '.'))
		    plugin = strtok (plugin, ".");

		if(g_slist_find_custom (plugin_list, plugin, compare_strings)) {
		    g_free (plugin);
		} else {
		    NOOP ("modules: g_slist_append %s\n", plugin);
		    plugin_list = g_slist_append (plugin_list, plugin);
		}
	    }
	    g_dir_close (dir);
	}
	g_free (full_libdir);
	g_once_init_leave(&initialized, 1);
    }
    return plugin_list;
}
#else
// just for debug tests. Will leak list with each call...
GSList *
rfm_find_plugins (void) {
    GSList *plugin_list = NULL;
	GError *error = NULL;
	gchar *full_libdir = g_build_filename (LIBDIR, PLUGIN_DIR, NULL);

	GDir *dir;
	if((dir = g_dir_open (full_libdir, 0, &error)) != NULL) {
	    const gchar *name;
	    while((name = g_dir_read_name (dir)) != NULL) {
		gchar *plugin;
		if(strncmp (name, "lib", strlen ("lib")))
		    plugin = g_strdup (name);
		else
		    plugin = g_strdup (name + strlen ("lib"));
		if(strchr (plugin, '.')) *(strchr (plugin, '.'))=0;

		if(g_slist_find_custom (plugin_list, plugin, compare_strings)) {
		    g_free (plugin);
		} else {
		    NOOP ("modules: g_slist_append %s\n", plugin);
		    plugin_list = g_slist_prepend (plugin_list, plugin);
		}
	    }
	    g_dir_close (dir);
	}
	g_free (full_libdir);
    return plugin_list;
}
#endif


/*************************************************************/

//void unload_module (const gchar * module_name);

static void *
module_error(const gchar * module_name, gchar *module, gchar *warning){
g_mutex_lock(module_hash_mutex);
    g_hash_table_insert (module_hash, (gpointer) module_name, GINT_TO_POINTER(-1));
g_mutex_unlock(module_hash_mutex);
    g_warning ("Module %s (%s): %s\n", module, module_name, warning);
    g_free (warning);
    g_free (module);
    return NULL;
}

GModule *
get_module_info (const gchar * librarydir, const gchar * module_name) {
    GModule *module_cm = NULL;
    void *(*module_sanity) (void);
    int init_version;

    gchar *full_libdir;
    if(librarydir){
	full_libdir = g_build_filename (LIBDIR, librarydir, NULL);
    } else {
	full_libdir = g_strdup (LIBDIR);
    }
    gchar *module = g_module_build_path (full_libdir, module_name);
    g_free (full_libdir);

    if(!rfm_g_file_test (module, G_FILE_TEST_EXISTS)) {
	DBG ("Module %s does not exist!\n", module);
	g_free (module);
	return NULL;
    }

    NOOP ("getting module info for %s (LIBDIR=%s)\n", module_name, LIBDIR);

    if(!module_hash) {
        NOOP ("get_module_info: creating module hash\n");
	if (!module_hash_mutex){
	    module_hash_mutex=g_mutex_new();
	}
        module_hash = g_hash_table_new (g_str_hash, g_str_equal);
        if(!module_hash)
            g_assert_not_reached ();
    }
    NOOP ("looking for module in hash: %s\n", module_name);
    g_mutex_lock(module_hash_mutex);
    module_cm = g_hash_table_lookup (module_hash, module_name);
    g_mutex_unlock(module_hash_mutex);
    if (GPOINTER_TO_INT(module_cm) == -1){
	NOOP("Module %s cannot be loaded.\n", module);
	g_free(module);
	return NULL;
    }
    if(module_cm != NULL) {
        NOOP ("module info for %s found (already loaded) 0x%x\n",
		module,
		GPOINTER_TO_INT(module_cm));
	g_free(module);
        return module_cm;
    } 

    TRACE ("get_module_info(): RSS changes with load of %s\n", module);
    // Here's the rub
#ifdef DEBUG_NOOP
    module_cm = g_module_open (module,
	    G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL);
#else
    module_cm = g_module_open (module,
	    G_MODULE_BIND_LOCAL);
#endif

    if(!module_cm) {
	return module_error(module_name, module, 
		g_strdup("g_module_open() == NULL\nmodule cannot be opened: Check if correctly installed or unresolved symbols within...\n****\n"));
    }

    // Check for module sanity 
    if(!g_module_symbol (module_cm, "module_sanity", (gpointer) & (module_sanity))) {
	return module_error(module_name, module, 
		g_strdup ("Module is not sane!\nDoes module specify: \"LIBRFM_MODULE\"?"));
    } 
    
    // Check for construction with correct structure serial (types.h)
    init_version = GPOINTER_TO_INT ((*module_sanity) ());
    if(init_version != LIBRFM_SERIAL) {
	return module_error(module_name, module, 
		g_strdup ("Module is compiled with obsolete headers (not loaded)"));
    }



    NOOP ("adding module to module hash: %s (0x%lx)\n", module_name, (unsigned long)module_cm);
g_mutex_lock(module_hash_mutex);
    g_hash_table_insert (module_hash, (gpointer) module_name, module_cm);
g_mutex_unlock(module_hash_mutex);
    NOOP ("get_module_info: module %s successfully loaded\n", module);
    g_free (module);
    return module_cm;
    
}


void *
rfm_void (const gchar * librarydir, const gchar * module_name, const gchar * function_id) {
    gchar *(*function) (void);
    if(!librarydir || !module_name || !function_id)
        return NULL;
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm) return NULL;

    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        /*if (strcmp(function_id,"submodule_name")!=0 && strcmp(function_id,"submodule_dir")!=0) */
        {
            NOOP(stderr, "g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) ();
}

void *
rfm_natural (const gchar * librarydir, const gchar * module_name, void *n, const gchar * function_id) {
    gchar *(*function) (void *n);
    if(!librarydir || !module_name || !function_id){
	DBG("!librarydir (%s)|| !module_name (%s)|| !function_id(%s)\n",
		librarydir, module_name, function_id);
        return NULL;
    }
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm){
	NOOP("!module_cm\n");
        return NULL;
    }
    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        if(strcmp (function_id, "submodule_name") != 0 && strcmp (function_id, "submodule_dir") != 0) {
            NOOP (stderr, "g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) (n);
}

void *
rfm_rational (const gchar * librarydir, const gchar * module_name, void *p, void *q, const gchar * function_id) {
    gchar *(*function) (void *p, void *q);
    if(!librarydir || !module_name || !function_id)
        return NULL;
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm)
        return NULL;
    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        if(strcmp (function_id, "submodule_name") != 0 && strcmp (function_id, "submodule_dir") != 0) {
            NOOP (stderr, "g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) (p, q);
}

void *
rfm_complex (const gchar * librarydir, const gchar * module_name, void *p, void *q, void *r, const gchar * function_id) {
    gchar *(*function) (void *p, void *q, void *r);
    if(!librarydir || !module_name || !function_id)
        return NULL;
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm)
        return NULL;
    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        if(strcmp (function_id, "submodule_name") != 0 && strcmp (function_id, "submodule_dir") != 0) {
            NOOP (stderr, "g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) (p, q, r);
}

static void **
argument_container_new(gint size, const void **in_vector){
    void **argument_container = (void **)malloc(size * sizeof(void *));
    if (!argument_container) g_error("malloc %s\n", strerror(errno));
    memset(argument_container, 0, size * sizeof(void *));
    if (in_vector){
	gint i;
	for (i=0; i < size; i++) argument_container[i] = (void *)in_vector[i];
    }
    return argument_container;
}

   
G_MODULE_EXPORT
void *
rfm_vector_run(const gchar * librarydir, const gchar * module_name, 
	void *vector_size_p, const void **vector, const gchar *function_id){

    void **arg = argument_container_new(GPOINTER_TO_INT(vector_size_p), vector);
    NOOP(stderr, "$$$  vector run %d: %s\n", GPOINTER_TO_INT(vector_size_p), function_id);
    return rfm_natural(MODULE_DIR, module_name, arg, function_id);
}


#if 0
void *
load_module (const gchar * librarydir, const gchar * module_name) {
    GModule *module_cm = get_module_info (librarydir, module_name);
    NOOP ("finished load attempt of module %s (%s)\n", module_name, librarydir);
    if(!module_cm) {
        g_warning ("Show stopper: cannot get module info for %s", module_name);
        exit (0);
    }
    return GINT_TO_POINTER(1);
}

void
unload_module (const gchar * module_name) {

    if(!module_hash)
        return;
    g_mutex_lock(module_hash_mutex);
    GModule *module_cm = g_hash_table_lookup (module_hash, module_name);
    g_mutex_unlock(module_hash_mutex);
    if(!module_cm) {
        NOOP ("module %s is not loaded\n", module_name);
        return;
    }

    if(!g_module_close (module_cm)) {
        g_warning ("g_module_close (%s) failed\n", module_name);
    } else {
    g_mutex_lock(module_hash_mutex);
        if(!g_hash_table_remove (module_hash, module_name)) {
            g_warning ("could not remove %s from module hash", module_name);
        }
    g_mutex_unlock(module_hash_mutex);
        DBG ("module %s unloaded\n", module_name);
    }
}
#endif

/*********************************************************************************************/


