/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**  miscelaneous */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#define EXPAND      TRUE
#define FILL        TRUE
#define NOEXPAND    FALSE
#define NOFILL      FALSE


#include "rfm_libs.h"
#include "rfm_modules.h"
#include "primary-misc.i"

typedef struct sequence_t {
    const gchar *id;
    const gchar *sequence;
} sequence_t;

static sequence_t sequence_p[] = {
        {"xffm_tag/black", "[30m"},
        {"xffm_tag/red", "[31m"},
        {"xffm_tag/red", "[31;01m"},
        {"xffm_tag/green", "[32m"},
        {"xffm_tag/green", "[32;01m"},
        {"xffm_tag/yellow", "[33m"},
        {"xffm_tag/yellow", "[33;01m"},
        {"xffm_tag/blue", "[34m"},
        {"xffm_tag/blue", "[34;01m"},
        {"xffm_tag/magenta", "[35m"},
        {"xffm_tag/magenta", "[35;01m"},
        {"xffm_tag/cyan", "[36m"},
        {"xffm_tag/cyan", "[36;01m"},
        {"xffm_tag/white", "[37m"},
        {"xffm_tag/white", "[37;01m"},
        {"xffm_tag/bold", "[1m"},
        {"xffm_tag/bold", "[1m;01"},
        {"xffm_tag/italic", "[4m"},
        {"xffm_tag/italic", "[4m;01"},
        {NULL, "[m"},
        {NULL, "[0m"},
        {NULL, "[22m"},
        {NULL, "[24m"},
        {NULL, NULL}            // this marks the end of sequences.
};


void
rfm_set_store_data_from_list(GtkListStore *list_store, GSList **list){
    GtkTreeIter iter;
    gtk_list_store_clear (list_store);
    GSList *p = *list;
    for (; p && p->data; p=p->next) {
	gtk_list_store_append (list_store, &iter);
	gtk_list_store_set (list_store, &iter,
                          0, (gchar *)p->data,
                          -1);
      /* Note: The store will keep a copy of the string internally, 
       * so the list may be freed */
    }
}


char *
rfm_display_host_name (GtkWidget * widget){
    Window xid=GDK_WINDOW_XID(gtk_widget_get_window(widget));
    char *name = NULL;
    unsigned char *property_data;
    unsigned long items, remaining;
    int actual_format;
    Atom atomo, actual_atom;

    atomo = XInternAtom(gdk_x11_display_get_xdisplay(gdk_display_get_default()), 
	    "WM_CLIENT_MACHINE", FALSE);
    if(XGetWindowProperty(gdk_x11_display_get_xdisplay(gdk_display_get_default()), 
		xid,
		atomo, 0, 255, FALSE, XA_STRING, 
		&actual_atom, &actual_format, 
		&items, &remaining, &property_data) == Success)
    {
	NOOP("property_data=%s\n",((property_data)?property_data:(unsigned char *)"null")); 
	if (!property_data) name = g_strdup("localhost");
	else {
	    name = g_strdup((const gchar *)property_data);
	    XFree(property_data);
	}
    }
    else name = g_strdup(g_get_host_name());
    return name;
}

GtkWidget *
rfm_mk_little_button (const gchar * icon_id, void *callback, void *callback_data, const gchar * tooltip_text) {
    GtkWidget *button;
    button = gtk_button_new ();
    gtk_widget_set_can_focus (button, FALSE);

    gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
    if(icon_id) {
        GdkPixbuf *pb;
        GtkWidget *image;
        pb = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
        image = gtk_image_new_from_pixbuf (pb);
        gtk_widget_show (image);
        gtk_container_add (GTK_CONTAINER (button), image);
    }
    if(tooltip_text && strlen (tooltip_text)) {
	gtk_widget_set_tooltip_text (button, tooltip_text);
    }
    if(callback) {
        NOOP ("DIAGNOSTICS: run button connected to callback\n");
        g_signal_connect ((gpointer) button, "clicked", G_CALLBACK (callback), callback_data);
    }
    return button;
}

void
rfm_cursor_wait (GtkWidget * widget) {
    static GdkCursor *cursor = NULL;
    if(!widget) return;
    if(!cursor) cursor = gdk_cursor_new (GDK_WATCH);
    gdk_window_set_cursor (gtk_widget_get_window(widget), cursor);
}

void
rfm_cursor_reset (GtkWidget * widget) {
    if(!widget) return;
    gdk_window_set_cursor (gtk_widget_get_window(widget), NULL);
}

    

static void
clear_diagnostics_window (GtkButton * button, gpointer data) {
    widgets_t *widgets_p = data;
    NOOP ("clear_diagnostics_window() now \n");
    if (widgets_p->diagnostics_window==NULL) {
	g_error("widgets_p->diagnostics_window==NULL");
    }
    if (*(widgets_p->diagnostics_window)==NULL) {
	g_error("*(widgets_p->diagnostics_window==NULL)");
    }
    //gtk_widget_hide (*(widgets_p->diagnostics_window));
    rfm_clear_text (widgets_p);
    return;
}

#define ICONOFY_BUTTON 1
#ifdef ICONOFY_BUTTON
// This function is necessary because fvwm does not put in a minimize
// button for child windows of the desktop window. Actually, no decorations
// are used for any child windows of the desktop window.
static void
iconofy_diagnostics_window (GtkButton * button, gpointer data) {
    widgets_t *widgets_p = data;
    NOOP ("iconofy_diagnostics_window() now \n");
    NOOP("*(widgets_p->diagnostics_window) = 0x%x, widgets=0x%x diagnostics_window=0x%x\n",
	    0,//GPOINTER_TO_INT(*(widgets_p->diagnostics_window)),
	    GPOINTER_TO_INT(widgets_p),
	    GPOINTER_TO_INT(widgets_p->diagnostics_window));
    if (widgets_p->diagnostics_window==NULL) g_error("widgets_p->diagnostics_window==NULL");
    if (*(widgets_p->diagnostics_window)==NULL) g_error("*(widgets_p->diagnostics_window==NULL)");
    gtk_window_iconify(GTK_WINDOW(*(widgets_p->diagnostics_window)));
    return;
}
#endif


static gboolean
destroy_diagnostics_window (GtkWidget * widget, GdkEvent * event, gpointer data) {
    //return TRUE;
    // Just hide it.
    widgets_t *widgets_p = data;
    NOOP ("destroy_diagnostics_window() now \n");
    if (widgets_p->diagnostics_window==NULL) {
	g_error("widgets_p->diagnostics_window==NULL");
    }
    if (*(widgets_p->diagnostics_window)==NULL) return TRUE;
    gtk_window_iconify(GTK_WINDOW(*(widgets_p->diagnostics_window)));
    // Hiding does not work right since if input is flowing to
    // the widget, GTK gets confused and will not show it again
    // (at least with fvwm).
    //gtk_widget_hide(*(widgets_p->diagnostics_window));
    // Let's try simply setting the configuration option to hide it...
    rfm_rational (MODULE_DIR, "settings", (void *)"RFM_ENABLE_DESKTOP_DIAGNOSTICS", (void *)"", "mcs_set_var");

    return TRUE;
}

static void
close_diagnostics_window (GtkWidget * widget, gpointer data) {
    destroy_diagnostics_window (NULL, NULL, data);
}

GtkWidget *
rfm_create_diagnostics_window (widgets_t * widgets_p) {
    if (widgets_p->diagnostics_window==NULL) {
	g_error("This should never happen: widgets_p->diagnostics_window==NULL");
    }

    if (*(widgets_p->diagnostics_window) != NULL) {
	g_warning("rfm_create_diagnostics_window(): diagnostics_window already exists.");
	return *(widgets_p->diagnostics_window);
    }
    GtkWidget *dialog,
     *button,
     *scrolledwindow,
     *hbox;
    //dialog = gtk_dialog_new ();
    //// THis instead:
    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);


    gtk_window_stick (GTK_WINDOW(dialog));
 

    *(widgets_p->diagnostics_window) = dialog;

    
    NOOP("*(widgets_p->diagnostics_window) = 0x%x, widgets=0x%x diagnostics_window=0x%x\n",
	    GPOINTER_TO_INT(dialog),
	    GPOINTER_TO_INT(widgets_p),
	    GPOINTER_TO_INT(widgets_p->diagnostics_window));
    g_object_set_data(G_OBJECT(dialog), "widgets_p", widgets_p);
    gchar *title=g_strdup(_("Console Message Viewer"));
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    gtk_window_set_icon (GTK_WINDOW (dialog), 
	    rfm_get_pixbuf("xffm/apps_terminal", SIZE_ICON));
    g_free(title);
    gtk_window_set_resizable (GTK_WINDOW (dialog), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

    /* create diagnostics textview */
    if (widgets_p->diagnostics==NULL){
	widgets_p->diagnostics =
	    (GtkWidget **)malloc(sizeof(GtkWidget *));
	if (!widgets_p->diagnostics) g_error("malloc: %s", strerror(errno));
    }
    *(widgets_p->diagnostics) = gtk_text_view_new ();
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_set_size_request (scrolledwindow, 600, 400);

    gtk_container_add (GTK_CONTAINER (scrolledwindow), *(widgets_p->diagnostics));
    gtk_container_set_border_width (GTK_CONTAINER (*(widgets_p->diagnostics)), 2);
    
    gtk_widget_set_can_focus (*(widgets_p->diagnostics), FALSE);
    gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (*(widgets_p->diagnostics)), GTK_WRAP_WORD);
    gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (*(widgets_p->diagnostics)), FALSE);

    	gint size = 10; //8 This only works for desktop....
        PangoFontDescription *font_desc = pango_font_description_new ();
        pango_font_description_set_family (font_desc, "monospace");
        pango_font_description_set_size (font_desc, size * PANGO_SCALE);
#ifdef USE_GTK2
	gtk_widget_modify_font (*(widgets_p->diagnostics), font_desc);
#else
	gtk_widget_override_font (*(widgets_p->diagnostics), font_desc);
#endif
	g_object_set_data(G_OBJECT(*(widgets_p->diagnostics)), 
		"font_desc", font_desc);

    GtkWidget *vbox = rfm_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (dialog), vbox);

    gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 0);

    hbox = rfm_hbox_new (FALSE, 0);

    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    button = rfm_dialog_button ("xffm/stock_close", _("Close"));
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (close_diagnostics_window), widgets_p);

    button = rfm_dialog_button ("xffm/stock_clear", _("Clear"));
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (clear_diagnostics_window), widgets_p);

#ifdef ICONOFY_BUTTON
    button = rfm_dialog_button ("xffm/stock_goto-bottom", _("Iconify"));
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (iconofy_diagnostics_window), widgets_p);
#endif


    g_signal_connect (G_OBJECT (dialog), "destroy_event", G_CALLBACK (destroy_diagnostics_window), widgets_p);
    g_signal_connect (G_OBJECT (dialog), "delete_event", G_CALLBACK (destroy_diagnostics_window), widgets_p);

    widgets_p->button_space = rfm_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), widgets_p->button_space, FALSE, FALSE, 0);
    gtk_widget_realize (dialog);
    if(!getenv ("RFM_ENABLE_DESKTOP_DIAGNOSTICS") ||
	  strlen (getenv ("RFM_ENABLE_DESKTOP_DIAGNOSTICS"))==0)
    {
	gtk_widget_hide (dialog);
        //gtk_window_iconify(GTK_WINDOW(*(widgets_p->diagnostics_window)));
    } else {
        gtk_window_iconify(GTK_WINDOW(*(widgets_p->diagnostics_window)));
	gtk_widget_show_all (dialog);
    }
    
    return dialog;

}

void
rfm_update_status_line (view_t * view_p) {
    if(!view_p->widgets.status) return;
    // Item selected:
    if(view_p->selection_list && g_slist_length (view_p->selection_list) == 1) {
        record_entry_t *en = view_p->selection_list->data;
        if(en && en->path) {
            GError *error = NULL;
            const gchar *id = "xffm/stock_file";
	    if (en && IS_SDIR(en->type)){
		if (IS_LOCAL_TYPE(en->type)) id = "xffm/stock_directory";
		else id ="xffm/places_folder-remote";
	    } else if (en && en->mimetype) id = en->mimetype; 
            gchar *g = g_path_get_basename (en->path);
            gchar *gg = g_locale_to_utf8 (g, -1, NULL, NULL, &error);
            if(gg)
                rfm_status (&(view_p->widgets), id, gg, NULL);
            else {
                g_warning ("g_locale_to_utf8(%s): %s", g, error->message);
                g_error_free (error);
            }
            g_free (g);
            g_free (gg);
        }
    } 
    // Items selected:
    else if(view_p->selection_list && g_slist_length (view_p->selection_list) >= 1) {
        gchar *g=
            g_strdup_printf (ngettext ("%'d item selected", "%'d items selected",
                g_slist_length (view_p->selection_list)),
                    g_slist_length (view_p->selection_list));
        rfm_status (&(view_p->widgets), "xffm/stock_info", g, NULL);
    } 
    // Use en->tag for status line.
    else {
        const gchar *icon = "xffm/emblem_link";
        if(view_p->en){
	    if (view_p->en->module){
		const gchar *m_icon = rfm_void(PLUGIN_DIR, view_p->en->module,
			"module_icon_id");
		if (m_icon) icon = m_icon;

	    }
	    if (view_p->en->tag == NULL) {
		gint items=0;
		for (;view_p->population_pp && view_p->population_pp[items]; items++);
		if (IS_SDIR(view_p->en->type)){
		    if (IS_LOCAL_TYPE(view_p->en->type)) {
			icon = "xffm/stock_directory";
		    } else icon ="xffm/places_folder-remote";
		}
		gchar *g = g_strdup_printf (ngettext (" (containing %'d item)", " (containing %'d items)", items), items);
		gchar *b = g_path_get_basename(view_p->en->path);
		view_p->en->tag = g_strdup_printf ("%s %s", b, g);
		g_free(b);
		g_free(g);
	    }
	    rfm_status (&(view_p->widgets), icon,  view_p->en->tag, NULL);
	}
    }
}

typedef struct lpterm_colors_t {
    const gchar *id;
    GdkColor fore;
    GdkColor back;
} lpterm_colors_t;

static
GtkTextTag *
resolve_tag (GtkTextBuffer * buffer, const gchar * id) {

    GtkTextTag *tag = NULL;
    lpterm_colors_t lpterm_colors_p[] = {
        {"xffm_tag/command",
         {101, 0x5858, 0x3434, 0xcfcf}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/stderr",
         {102, 0xcccc, 0, 0}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/command_id",
         {103, 0x0000, 0x0000, 0xffff}, {201, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/green",
         {104, 0x0000, 0x8888, 0x0000}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/red",
         {105, 0xaaaa, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/blue",
         {106, 0x0000, 0x0000, 0xffff}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/yellow",
         {107, 0xffff, 0xffff, 0x0000}, {202, 0x0000, 0x0000, 0x0000}},
        {"xffm_tag/magenta",
         {108, 0x8888, 0x0000, 0x8888}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/cyan",
         {109, 0x0000, 0x8888, 0x8888}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/grey",
         {110, 0x8888, 0x8888, 0x8888}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/black",
         {1, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff}},
        {"xffm_tag/white",
         {0, 0xffff, 0xffff, 0xffff}, {0, 0x0000, 0x0000, 0x0000}},
        {NULL,
         {0, 0, 0, 0}, {0, 0, 0, 0}}

    };

    tag = gtk_text_tag_table_lookup (gtk_text_buffer_get_tag_table (buffer), id);
    // bold an italic don't work...
    if(!tag) {                  // bold and italic tags
        if(strcmp (id, "xffm_tag/bold") == 0) {
            GdkColor fore = { 300, 0x0a0a, 0x0505, 0xc3c3 };
            tag = gtk_text_buffer_create_tag (buffer, id, "weight", PANGO_WEIGHT_BOLD, "foreground_gdk", &fore, NULL);
        } else if(strcmp (id, "xffm_tag/italic") == 0) {
            GdkColor fore = { 300, 0x2b2b, 0x6e6e, 0x3333 };
            tag = gtk_text_buffer_create_tag (buffer, id, "style", PANGO_STYLE_ITALIC, "foreground_gdk", &fore, NULL);
        }
    }
    if(!tag) {                  // color tags
        /*NOOP("dbg:creating tag %s\n",tag_id); */
        lpterm_colors_t *p;
        for(p = lpterm_colors_p; p && p->id; p++) {
            if(strcmp (id, p->id) == 0) {
                tag = gtk_text_buffer_create_tag (buffer, p->id, "background_gdk", &(p->back), "foreground_gdk", &(p->fore), NULL);
                break;
            }
        }
    }

    return tag;
}

void
rfm_clear_sh_command_history (view_t * view_p, gboolean disk_too) {
    if(disk_too) {
        gchar *history = g_build_filename ( LP_TERMINAL_HISTORY, NULL);
        unlink (history);
    }
    GList *p;
    for(p = g_list_first (view_p->sh_command); p && p->data; p = p->next) {
        g_free (p->data);
    }
    g_list_free (view_p->sh_command);
    view_p->sh_command = NULL;
    view_p->sh_command_counter = 0;
}


// thread function (null)
void
rfm_dump_output (void *user_data, void *stream, int childFD) {
    return;
}

// This is a thread function, must have GDK mutex set for gtk commands...
void
rfm_operate_stderr (void *user_data, void *stream, int childFD) {
    widgets_t *widgets_p = user_data;
    view_t *view_p = widgets_p->view_p;
    if (view_p && view_p->flags.status ==STATUS_EXIT) return;
    char *line;
    line = (char *)stream;

    if(line[0] != '\n') {
GDK_THREADS_ENTER ();
        rfm_diagnostics (widgets_p, "xffm_tag/stderr", line, NULL);
GDK_THREADS_LEAVE ();
    }
    return;
}


// This is a thread function, must have GDK mutex set for gtk commands...
void
rfm_operate_stdout (void *user_data, void *stream, int childFD) {
    widgets_t *widgets_p = user_data;
    view_t *view_p = widgets_p->view_p;
    if (view_p && view_p->flags.status ==STATUS_EXIT) return;
    char *line;
    line = (char *)stream;
    NOOP ("FORK stdout: %s\n", line);

    if(line[0] == '\n')
        return;

/*    this is directly in tubo library
 *    if (!valid_ansi_sequence(line)) {
        NOOP("invalid ansi sequence!\n");
        return;
    }*/


    // eliminate all ^G found (as in nano output)


    //int bell=0x07; // bell
    int bs=0x08;   // backspace
    //int ht=0x09;   // horizontal tab
    //int lf=0x0a;   // linefeed
    //int vt=0x0b;   // vertical tab
    //int ff=0x0c;   // formfeed
    //int cr=0x0d;   // carriage return

    // apply all ^H (bs) found (as in rar output)
    int i,
      j;
    gchar *outline = g_strdup (line);
    for(i = 0, j = 0; line[i]; i++) {
	/*if (line[i] == bell) {
	    gdk_window_beep (widgets_p->window->window);
	} else*/ 
	if(line[i] == bs && j > 0){
            j--;
	} else {
            outline[j] = line[i];
            j++;
        }
    }
    outline[j] = 0;

    //NOOP ("%s\n",outline);

    if(strncmp (line, "Tubo-id exit:", strlen ("Tubo-id exit:")) == 0) {
            if(strchr (line, '\n')) *strchr (line, '\n') = 0;
GDK_THREADS_ENTER ();
            rfm_diagnostics (widgets_p, "xffm/stock_stop", NULL);
            rfm_diagnostics (widgets_p, "xffm_tag/command_id", strchr (line, ':') + 1, ".", NULL);
            rfm_diagnostics (widgets_p, NULL, "\n", NULL);
GDK_THREADS_LEAVE ();
    } else {
GDK_THREADS_ENTER ();
        rfm_diagnostics (widgets_p, NULL, outline, NULL);
GDK_THREADS_LEAVE ();
    }
    g_thread_yield();
    g_free(outline);
    return;
}

gchar **rfm_split(const gchar *text,const gchar token){
    gint count=0;
    gchar *c=(gchar *)text;
    while ((c=strchr(c,token))!=NULL){ 
	count++;
	c++;
    } 
    if (count==0) return NULL;
    gchar **split = (gchar **)malloc((count+2)*sizeof(gchar *));
    if (!split) g_error("malloc: %s", strerror(errno));
    c=g_strdup(text);
    gint i=0;
    if (*c != token) split[i++]=c;
    while ((c=strchr(c,token))!=NULL){ 
	if (i==0) {
	    *c=' ';
	    g_strchug(c);
	} else {
	    *c=0;
	    c++;
	}
	split[i++]=c;
    } 
    split[i]=NULL;
    return split;
}

gchar **split_esc(const gchar *text){
    return (rfm_split(text, 0x1b));
}



static void
insert_string (GtkTextBuffer * buffer, const gchar * s, GtkTextTag * tag) {
    if(!s)
        return;
    GtkTextIter start,
      end,
      real_end;
    gint line = gtk_text_buffer_get_line_count (buffer);
    gchar *a = NULL;
    if(strchr (s, 0x1B)) {      //vt escape sequence
	// do a split
	gchar **split=split_esc(s);
	if (!split) {
	    g_warning("insert_string(): split_esc barfed");
	    return;
	}
	gchar **pp=split;
	for (;pp && *pp; pp++){
	    const gchar *tag=NULL;
	    const gchar *sequence=NULL;
	    sequence_t *p;
	    for(p = sequence_p; p && p->sequence; p++) {
		if(strncmp (*pp, p->sequence, strlen (p->sequence)) == 0) {
		    tag=p->id;
		    sequence=p->sequence;
		    break;
		}
	    }
	    if (!tag){
		tag="xffm_tag/black";
		//g_warning("VT escape secuence not reconized: <esc>%s",*pp);
	    }
	    gint offset=(sequence)?strlen(sequence):0;
	    gchar *string=*pp+offset;
	    NOOP("(%s) token:%s-->\"%s\"\n",
		    tag, (sequence)?sequence:"None", string);
	    insert_string (buffer, string, resolve_tag(buffer, tag));
	}
	g_free(split[0]);
	g_free(split);
	// done
	return;
    }

    GtkTextMark *mark = gtk_text_buffer_get_mark (buffer, "rfm-ow");
    if(strchr (s, 0x0D)) {      //CR
        gchar *aa = g_strdup (s);
        *strchr (aa, 0x0D) = 0;
        insert_string (buffer, aa, tag);
        g_free (aa);
        aa = strchr (s, 0x0D) + 1;
        if(mark) {
            gtk_text_buffer_get_iter_at_line (buffer, &start, line);
            gtk_text_buffer_move_mark (buffer, mark, &start);
        }
        insert_string (buffer, aa, tag);
        g_free (a);
        // we're done
        return;
    }

    gchar *q = rfm_utf_string (s);
    /// this should be mutex protected since this function is being called
    //  from threads all over the place.
    static GMutex *insert_mutex=NULL;
    if (insert_mutex==NULL) {
        insert_mutex=g_mutex_new();
    }
#if 10
// test 1
    g_mutex_lock(insert_mutex);
#if 10
// test 3
    if(mark == NULL) {
        gtk_text_buffer_get_end_iter (buffer, &end);
        gtk_text_buffer_create_mark (buffer, "rfm-ow", &end, FALSE);
    } else {
        gtk_text_buffer_get_iter_at_mark (buffer, &end, mark);
    }

    gtk_text_buffer_get_iter_at_line (buffer, &start, line);
    gtk_text_buffer_get_end_iter (buffer, &real_end);

    // overwrite section
    gchar *pretext = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
    gchar *posttext = gtk_text_buffer_get_text (buffer, &end, &real_end, TRUE);
    long prechars = g_utf8_strlen (pretext, -1);
    long postchars = g_utf8_strlen (posttext, -1);
    long qchars = g_utf8_strlen (q, -1);
    g_free (pretext);
    g_free (posttext);
    if(qchars < postchars) {
        gtk_text_buffer_get_iter_at_line_offset (buffer, &real_end, line, prechars + qchars);
    }
    /////
    gtk_text_buffer_delete (buffer, &end, &real_end);
#else
    gtk_text_buffer_get_end_iter (buffer, &end);

// end test 3
#endif
    if(tag) {
        gtk_text_buffer_insert_with_tags (buffer, &end, q, -1, tag, NULL);
    } else {
        gtk_text_buffer_insert (buffer, &end, q, -1);
    }
    g_mutex_unlock(insert_mutex);
// end test 1
#endif
    g_free (q);
    g_free (a);
    return;
}


static void
set_font_family (GtkWidget * widget, const gchar *in_family) {
    if (!in_family) g_error("in_family cannot be NULL\n");
    if (!GTK_IS_WIDGET(widget)) return;
    gint fontsize = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), 
		"fontsize"));

    gint newsize=8; // default font size.
    const gchar *p = getenv ("RFM_DIAGNOSTICS_FONT_SIZE");
    if(p && strlen (p)) {
	errno=0;
	long value = strtol(p, NULL, 0);
	if (errno == 0){
	    newsize = value;
	}
    }

    if(newsize != fontsize) {
	DBG("XXX setting %s fontsize  %d -> %d \n", in_family, fontsize, newsize);
	fontsize = newsize;
	gchar *font_id = g_strdup_printf("%s_font_desc", in_family);
        PangoFontDescription *old_desc = 
	    g_object_get_data(G_OBJECT(widget), font_id);
        PangoFontDescription *font_desc = pango_font_description_new ();
        pango_font_description_set_family (font_desc, in_family);
        pango_font_description_set_size (font_desc, fontsize * PANGO_SCALE);
#ifdef USE_GTK2
	gtk_widget_modify_font (widget, font_desc);
#else
	gtk_widget_override_font (widget, font_desc);
#endif
	g_object_set_data(G_OBJECT(widget), 
		"fontsize", GINT_TO_POINTER(fontsize));
	g_object_set_data(G_OBJECT(widget), font_id, font_desc);
	g_free(font_id);
	if (old_desc) pango_font_description_free (old_desc); 
    }
}


void
rfm_status (widgets_t * widgets_p, const gchar * id, ...
    ) {
    if (!widgets_p) return;
    va_list var_args;
    char *t;
    GtkTextIter start,
      end;
    GtkTextBuffer *buffer;
    GdkPixbuf *icono;
    int icono_width = 0;

    if (!widgets_p) {
        return;
    }
//    if(!widgets_p) return;

    if(!widgets_p->status || !GTK_IS_TEXT_VIEW (widgets_p->status)) {
#ifdef DEBUG_NOOP
        /*g_warning("rfm_status: status==NULL"); */
        va_start (var_args, id);
        do {
            t = va_arg (var_args, char *);
            if(t && strlen (t)) {
                NOOP ("%s\n", t);
            }
        }
        while(t);
        va_end (var_args);
#endif
        return;
    }
    // call this function when status line is created, 
    // but that will not work for user changes in font size...
    if (id && strcmp(id, "xffm/apps_terminal")==0) {
         set_font_family (widgets_p->status, "monospace");
    } else {
         set_font_family (widgets_p->status, "sans");
    }
     // call this function if you really want it when status line is created:
    // set_sans (widgets_p->status);
    /*gtk_text_view_set_justification ((GtkTextView *) widgets_p->status,GTK_JUSTIFY_LEFT); */
    buffer = gtk_text_view_get_buffer ((GtkTextView *) widgets_p->status);
    gtk_text_buffer_set_text (buffer, " ", -1);
    gtk_text_buffer_get_bounds (buffer, &start, &end);

    if(!id){
        id = "xffm/emote_cool";
    }
    icono = rfm_get_pixbuf (id, SIZE_BUTTON);

    if(GDK_IS_PIXBUF(icono)) {
        gtk_text_buffer_insert_pixbuf (buffer, &start, icono);
        icono_width = gdk_pixbuf_get_width (icono);
    }

    {
        char *string = NULL;    /*,*tag; */
        gchar *f = NULL;
        va_start (var_args, id);
        do {
            t = va_arg (var_args, char *);
            if(t && strlen (t)) {
                if(!f)
                    f = g_strdup ("");
                string = g_strconcat (f, t, NULL);
                g_free (f);
                f = string;
            }
        }
        while(t);
        va_end (var_args);
        if(string) {
            insert_string (buffer, string, NULL);
            g_free (string);
        }
    }
    g_object_set_data (G_OBJECT (widgets_p->status), "clean", (gpointer) ((long)0x01));
}


    // For normal diagnostics, default is to show stdout if diagnostics
    // is visible, and to show diagnostics automatically whenever output
    // to stderr is detected.

void
rfm_diagnostics (widgets_t * widgets_p, const gchar * id, ...
    ) {
    char *t;
    va_list var_args;
    GtkTextIter start, end;
    GtkTextMark *mark;
    GtkTextBuffer *buffer;
    GdkPixbuf *icono = NULL;

    if (!widgets_p) {
        return;
    }
    view_t *view_p = widgets_p->view_p;
    if (view_p && view_p->flags.status == STATUS_EXIT){
	NOOP(stderr, "dead view 0x%x\n", GPOINTER_TO_INT(view_p));
	return;
    }
    gboolean diagnostics_is_visible=rfm_diagnostics_is_visible (widgets_p);

    if(!diagnostics_is_visible) {
        NOOP ("!rfm_diagnostics_is_visible(diagnostics)\n");
        return;
    }
//    if (!widgets_p->diagnostics || !gtk_widget_get_visible(*widgets_p->diagnostics)){
    if (!widgets_p->diagnostics || *(widgets_p->diagnostics)==NULL){
        NOOP ("!widgets_p->diagnostics\n");
	return;
    }

    gboolean is_a_deskview=(widgets_p->diagnostics_window && *(widgets_p->diagnostics_window));
    if (is_a_deskview){
        NOOP ("is_a_deskview: gtk_widget_show_all\n");
	if(getenv ("RFM_ENABLE_DESKTOP_DIAGNOSTICS") &&
	    strlen (getenv ("RFM_ENABLE_DESKTOP_DIAGNOSTICS"))!=0)
	{
	    gtk_widget_show_all(*(widgets_p->diagnostics_window));
	    //gtk_window_deiconify(GTK_WINDOW(*(widgets_p->diagnostics_window)));
	}
    }

#ifdef DEBUG_NOOP
    NOOP("rfm_diagnostics(): diagnostics==NULL\n");
    va_start (var_args, id);
    do {
	t = va_arg (var_args, char *);
	if(t && strlen (t)) {
	    NOOP ("%s\n", t);
	}
    }
    while(t);
    va_end (var_args);
#endif

    // call this function when diagnostics is created, 
    // but that only works for desktop diagnostics,
    // not for iconview diagnostics, I wonder why...
     set_font_family (*(widgets_p->diagnostics), "monospace");

    /****  */
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (*(widgets_p->diagnostics)));

    /*NOOP("NOOP:rfm_diagnostics\n"); */
    gtk_text_buffer_get_bounds (buffer, &start, &end);

    GtkTextTag *tag = NULL;
    //NOOP("DIAGNOSTICS: id= %s\n",id);
    if(id) {
        if(strncmp (id, "xffm_tag/", strlen ("xffm_tag/")) == 0) {
            tag = resolve_tag (buffer, id);
        } else {
            icono = rfm_get_pixbuf (id, SIZE_BUTTON);
        }
    }
    if(icono) {
        gtk_text_buffer_insert_pixbuf (buffer, &end, icono);
    }

    va_start (var_args, id);
    do {
        t = va_arg (var_args, char *);
        if(t && strlen (t)) {
            insert_string (buffer, t, tag);
        }
    }
    while(t);
    va_end (var_args);
    gtk_text_buffer_get_bounds (buffer, &start, &end);
    mark = gtk_text_buffer_create_mark (buffer, "scrollmark", &end, FALSE);
    gtk_text_view_scroll_to_mark (GTK_TEXT_VIEW (*(widgets_p->diagnostics)), mark, 0.2,    /*gdouble within_margin, */
                                  FALSE, 0.0, 0.0);
    gtk_text_buffer_delete_mark (buffer, mark);
    gint line_count = gtk_text_buffer_get_line_count (buffer);
    glong max_lines_in_buffer = 1000;
    if (getenv("RFM_MAXIMUM_DIAGNOSTIC_LINES") && 
	    strlen(getenv("RFM_MAXIMUM_DIAGNOSTIC_LINES"))){
	errno = 0;
	max_lines_in_buffer = 
	    strtol(getenv("RFM_MAXIMUM_DIAGNOSTIC_LINES"), NULL, 10);
	if (errno){
	    max_lines_in_buffer = 1000;
	}

    }
    if (line_count > max_lines_in_buffer) {
	//gtk_text_buffer_set_text(buffer, "", -1); //clear buffer
	
	NOOP(stderr, "line count is %d, deleting %d lines\n", 
		line_count, line_count-max_lines_in_buffer);
	gtk_text_buffer_get_iter_at_line (buffer, &start, 0);
	gtk_text_buffer_get_iter_at_line (buffer, &end, line_count - max_lines_in_buffer);
	gtk_text_buffer_delete (buffer, &start, &end);
    }
}

void
rfm_show_text (widgets_t * widgets_p) {

    if(!widgets_p) return;
    //gboolean diagnostics_is_visible=rfm_diagnostics_is_visible (widgets_p);
    gboolean is_a_deskview = (widgets_p->diagnostics_window != NULL);



    if(is_a_deskview){
	if (getenv ("RFM_ENABLE_DESKTOP_DIAGNOSTICS") &&
	  strlen (getenv ("RFM_ENABLE_DESKTOP_DIAGNOSTICS"))!=0)
	{
	    gtk_widget_show_all(*(widgets_p->diagnostics_window));
	    //gtk_window_deiconify(GTK_WINDOW(*(widgets_p->diagnostics_window)));
	} 
        return;
    }

    // If already visible, then there's nothing more to do.
    // if (diagnostics_is_visible) return;

    if(widgets_p->vpane){
	GtkAllocation allocation;
	gtk_widget_get_allocation (widgets_p->vpane, &allocation);
	if (allocation.height > 50) {
	    gdouble position = 
		gtk_paned_get_position (GTK_PANED(widgets_p->vpane));
	    if(position > allocation.height * 0.90) {
		gtk_paned_set_position (GTK_PANED (widgets_p->vpane), allocation.height * 0.75);
	    }
	}
    }
    return;
}

gboolean
rfm_diagnostics_is_visible (widgets_t * widgets_p) {
    if (!widgets_p) return FALSE;

    // Are we dealing with a deskview?
    gboolean is_a_deskview = (widgets_p->diagnostics_window != NULL);
    if (is_a_deskview){
	// Does the deskview output window exist?
	// If no, then create it.
	if (*(widgets_p->diagnostics_window)==NULL){
	    rfm_create_diagnostics_window(widgets_p); 
	}
	// We return TRUE even if item is iconized
	return TRUE;
    }

    // By this point we should be dealing with views with diagnostics area.
    // If the view does not have a diagnostics area defined, return false.
    if(widgets_p->diagnostics == NULL || *widgets_p->diagnostics == NULL) {
	return FALSE;
    }


    // By now we should have a diagnostics area defined within the vertical pane.
    if(widgets_p->vpane){
	// Whatever...
	return TRUE;
/*	GtkAllocation allocation;
	gtk_widget_get_allocation (widgets_p->vpane, &allocation);
	if(gtk_paned_get_position (
		    GTK_PANED(widgets_p->vpane)) > allocation.height * 0.95)
	{
	    return FALSE;
	}*/
    }
        
    return TRUE;
}

void
rfm_hide_text (widgets_t * widgets_p) {
    if(widgets_p->diagnostics==NULL || *(widgets_p->diagnostics)==NULL){
        return;
    }
    if(!widgets_p->vpane){
        return;
    }
	NOOP(stderr, "widgets_p->vpane = 0x%x\n", GPOINTER_TO_INT(widgets_p->vpane));

    GtkAllocation allocation;
    gtk_widget_get_allocation (widgets_p->vpane, &allocation);
    NOOP(stderr, "HACK set pane position to %d\n", allocation.height);

    gtk_paned_set_position (GTK_PANED (widgets_p->vpane), allocation.height);
    gtk_paned_set_position (GTK_PANED (widgets_p->vpane), 1000);
}

void
rfm_clear_text (widgets_t * widgets_p) {
    GtkTextIter start,
      end;
    GtkTextBuffer *buffer;
    if(widgets_p->diagnostics==NULL || *(widgets_p->diagnostics)==NULL)
        return;
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (*(widgets_p->diagnostics)));
    gtk_text_buffer_get_bounds (buffer, &start, &end);
    gtk_text_buffer_delete (buffer, &start, &end);
    if (widgets_p->diagnostics_window==NULL) {
	// This is not applicable to diagnostics_window:
	rfm_hide_text (widgets_p);
    }
    g_object_ref (G_OBJECT(buffer)); 
    gtk_text_view_set_buffer(GTK_TEXT_VIEW (*(widgets_p->diagnostics)), gtk_text_buffer_new(NULL));
    g_object_ref_sink (G_OBJECT(buffer));
    g_object_unref (G_OBJECT(buffer)); 

}

void
rfm_clear_text_window (GtkButton * button, gpointer data) {

    widgets_t *widgets_p = (widgets_t *) data;
    if(widgets_p->diagnostics==NULL || *(widgets_p->diagnostics)==NULL)
        return;
    rfm_clear_text (widgets_p);
    //under certain condition, like doing a clear after a threaded remove,
    //update status line may race with invalid view_p->selection_list
    //rfm_update_status_line (widgets_p->view_p);
}

GtkWidget *
rfm_dialog_button (const char *icon_id, const char *text) {
    GtkWidget *button = gtk_button_new_with_label (text);

    if(icon_id) {
        GdkPixbuf *pb = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
        GtkWidget *image = gtk_image_new_from_pixbuf (pb);
	gtk_button_set_image(GTK_BUTTON(button), image);
    }
    gtk_widget_show_all (button);

    return button;

}


G_MODULE_EXPORT gboolean
rfm_confirm (widgets_t * widgets_p, 
	// type: GTK_MESSAGE_INFO, GTK_MESSAGE_WARNING, GTK_MESSAGE_QUESTION, GTK_MESSAGE_ERROR
	gint type,
//	GtkMessageType type,
	const gchar * text, 
	const gchar * action_false,  // if NULL, button not shown
	const gchar * action_true   // if NULL, "Ok" button shown
	) {
    gint response = GTK_RESPONSE_NONE;
    NOOP ("DIAGNOSTICS: rfm_confirm\n");
    GtkWidget *dialog = confirm_dialog (widgets_p, type, text, action_false, action_true);
    if(!dialog){
        return FALSE;
    }
    NOOP ("DIAGNOSTICS: rfm_confirm-- gtk_dialog_run\n");
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);
    gtk_widget_destroy (dialog);

    if(response == GTK_RESPONSE_YES)
        return TRUE;
    else
        return FALSE;
}

gchar *
rfm_get_passphrase (widgets_t * widgets_p, const gchar * ptext) {
    GtkWidget *dialog = passwd_dialog (widgets_p, ptext, 2, NULL, NULL);
    gint response = GTK_RESPONSE_NONE;
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);

    gchar *password = NULL;

    if(response == GTK_RESPONSE_YES){
	GtkWidget *entry = g_object_get_data(G_OBJECT(dialog), "passwd");
	const gchar *c = gtk_entry_get_text(GTK_ENTRY(entry));
	if (c) password = g_strdup(c);
	gtk_entry_set_text(GTK_ENTRY(entry), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXX");
    } 
    gtk_widget_destroy (dialog);
    return password;
}

gchar *
rfm_get_password_with_default (widgets_t * widgets_p, const gchar * ptext, const gchar * default_value) {
    GtkWidget *dialog = passwd_dialog (widgets_p, ptext, 1, NULL, default_value);
    gint response = GTK_RESPONSE_NONE;
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);

    gchar *password = NULL;

    if(response == GTK_RESPONSE_YES){
	GtkWidget *entry = g_object_get_data(G_OBJECT(dialog), "passwd");
	const gchar *c = gtk_entry_get_text(GTK_ENTRY(entry));
	if (c) password = g_strdup(c);
	gtk_entry_set_text(GTK_ENTRY(entry), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXX");
    } 
    gtk_widget_destroy (dialog);
    return password;
}

gchar *
rfm_get_password (widgets_t * widgets_p, const gchar * ptext) {
    return rfm_get_password_with_default(widgets_p, ptext, NULL); 
}

gchar *
rfm_get_user_with_default (widgets_t * widgets_p, const gchar * ptext, const gchar * default_value) {
    GtkWidget *dialog = passwd_dialog (widgets_p, ptext, 0, NULL, default_value);
    gint response = GTK_RESPONSE_NONE;
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);

    gchar *password = NULL;

    if(response == GTK_RESPONSE_YES){
	GtkWidget *entry = g_object_get_data(G_OBJECT(dialog), "passwd");
	const gchar *c = gtk_entry_get_text(GTK_ENTRY(entry));
	if (c) password = g_strdup(c);
	gtk_entry_set_text(GTK_ENTRY(entry), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXX");
    } 
    gtk_widget_destroy (dialog);
    return password;
}

gchar *
rfm_get_user (widgets_t * widgets_p, const gchar * ptext) {
    return rfm_get_user_with_default (widgets_p, ptext, NULL);
}

gchar *
rfm_get_string (widgets_t * widgets_p, const gchar * ptext, const gchar * qtext) {
    GtkWidget *dialog = passwd_dialog (widgets_p, ptext, 0, qtext, NULL);
    gint response = GTK_RESPONSE_NONE;
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);

    gchar *password = NULL;

    if(response == GTK_RESPONSE_YES){
	GtkWidget *entry = g_object_get_data(G_OBJECT(dialog), "passwd");
	const gchar *c = gtk_entry_get_text(GTK_ENTRY(entry));
	if (c) password = g_strdup(c);
	gtk_entry_set_text(GTK_ENTRY(entry), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXX");
    } 
    gtk_widget_destroy (dialog);
    return password;
}


gboolean rfm_confirm_sudo(widgets_t *widgets_p,
       const gchar *tgt, 
       const gchar *failed, 
       const gchar *operation){

	gchar *altoperation=
	    g_strconcat(_("sudo"), " ", operation, NULL);
	gchar *text=g_strconcat(
	    _("Command:"), " \"", operation, "\"",
	    "\n\n",
	    failed,"\n",
	    _("Permission denied"), ": ", tgt, "\n\n",
	     _("Try to approach a problem from different angles."), "\n\n",
	    _("Do you want to retry?"), "\n",
	     	    _("Alternate:"), " \"", altoperation, "\"",
	    "\n",
	    NULL
		);
	gboolean retval= rfm_confirm(widgets_p, GTK_MESSAGE_QUESTION,
		text, _("No"), altoperation);
	g_free(altoperation);
	g_free(text);
	return retval;
}

