//
//
// Condition variable:
static gboolean glob_done = FALSE;
// Condition mutex:
static GMutex *glob_mutex = NULL;
// Condition signal:
static GCond *glob_signal = NULL;

// Hash tables:
static GHashTable *icon_hash = NULL;
static GHashTable *icon_exec_hash = NULL;
static GHashTable *string_hash = NULL;
static GHashTable *reverse_string_hash = NULL;
static GHashTable *category_hash = NULL;

// Hash table mutexes:
static GStaticMutex icon_hash_mutex = G_STATIC_MUTEX_INIT;
static GStaticMutex exec_hash_mutex = G_STATIC_MUTEX_INIT;
static GStaticMutex string_hash_mutex = G_STATIC_MUTEX_INIT;
static GStaticMutex category_hash_mutex = G_STATIC_MUTEX_INIT;

// Single linked lists:
static GSList *category_list = NULL;
static GSList *path_list = NULL;

// Popup menu mutex:
static GStaticMutex popup_mutex = G_STATIC_MUTEX_INIT;

static gint desktop_serial=0;

typedef struct dotdesktop_t {
    const gchar *category;
    const gchar *string;
    const gchar *icon;
    GSList *application_list;
    gboolean popup;
} dotdesktop_t;

static dotdesktop_t dotdesktop_v[]={
    // These are categories which will appear in popup menu, 
    // in the specified order. Category translation is refined. 
    // Icon specification is hard coded.
    {"Rodent", NULL, "xffm/category_Rodent", 
	NULL, FALSE},
    {"Utility", N_("Accessories"), "xffm/category_accessories", 
	NULL, TRUE},
    {"Graphics", N_("Graphics applications"), "xffm/category_graphics", 
	NULL, TRUE},
    {"System", N_("System Tools"), "xffm/category_utilities", 
	NULL, TRUE},
    {"Network", N_("Internet and Network"), "xffm/category_internet", 
	NULL, TRUE},
    {"Game", N_("Games and amusements"), "xffm/category_games", 
	NULL, TRUE},
    {"Office", N_("Office Applications"), "xffm/category_office", 
	NULL, TRUE},
    {"Development", N_("Tools for software development"), "xffm/category_development",
       	NULL, TRUE},
    {"AudioVideo", N_("Applications related to audio and video"), "xffm/category_multimedia",
       	NULL, TRUE},
    {"Settings", N_("Personal preferences"),"xffm/preferences_personal",  
	NULL, TRUE},

    // These categories will not appear in popup menu but
    // translation may be refined and icon hard coded.
    {N_("Documentation"), NULL, "xffm/category_help", 
	NULL, FALSE},
    {N_("Accessibility"), NULL, NULL, 
	NULL, FALSE},
    {"Qt", NULL, NULL,
       	NULL, FALSE},
    {"GTK", NULL, "xffm/category_GTK", 
	NULL, FALSE},
    {"X-XFCE", NULL, "xffm/category_XFCE", 
	NULL, FALSE},
    {"GNOME", NULL, "xffm/category_GNOME", 
	NULL, FALSE},
    {"KDE", NULL, "xffm/category_KDE", 
	NULL, FALSE}, 

    // These categories will not appear in popup menu
    // and the translation will be refined.
    {"WordProcessor", N_("Word Processors"), NULL, 
	NULL, FALSE},
    {"DesktopSettings", N_("Desktop settings"), NULL, 
	NULL, FALSE},
    {"Player", N_("Media Player"), NULL, 
	NULL, FALSE}, 
    {"DiscBurning", N_("Video Disc Recorder"), NULL, 
	NULL, FALSE},
    {"PackageManager", N_("Package Manager"), NULL, 
	NULL, FALSE},
    {"HardwareSettings", N_("Hardware"), NULL, 
	NULL, FALSE},
    {"InstantMessaging", N_("Instant Messaging"), NULL, 
	NULL, FALSE},
    {"RasterGraphics", N_("Raster Graphics"), NULL, 
	NULL, FALSE},
    {"VectorGraphics", N_("Scalable Vector Graphics"), NULL, 
	NULL, FALSE},
    {"ContactManagement", N_("Contacts"), NULL, 
	NULL, FALSE},
    {"WebBrowser", N_("Web browser"), NULL, 
	NULL, FALSE},
    {"LogicGame", N_("Logic Games"), NULL, 
	NULL, FALSE},
    {"2DGraphics", N_("2D Graphics"), NULL, 
	NULL, FALSE},
    {"TerminalEmulator", N_("Terminal Emulator"), NULL, 
	NULL, FALSE},
    {"Recorder", N_("Sound Recorder"), NULL, 
	NULL, FALSE},
    {"BoardGame", N_("Board Games"), NULL, 
	NULL, FALSE},
    {"AudioVideoEditing", N_("Audio Editor"), NULL, 
	NULL, FALSE},
    {"BlocksGame", N_("Blocks Games"), NULL, 
	NULL, FALSE},
    {"CardGame", N_("Card Games"), NULL, 
	NULL, FALSE},
    {"Filesystem", N_("File System"), NULL, 
	NULL, FALSE},
    {"FileTools", N_("Batch tools for multiple files"), NULL, 
	NULL, FALSE},
    {"FileManager", N_("File Management"), NULL, 
	NULL, FALSE},
    {"FileTransfer", N_("File Transfer"), NULL, 
	NULL, FALSE},
    {"RemoteAccess", N_("Remote"), NULL, 
	NULL, FALSE},

    {NULL, NULL, "", NULL, FALSE}
};

// These are regular categories with acceptable translations
const gchar *translation_v[]={
    N_("Audio"),
    N_("Music"),
    N_("Application"),
    N_("Viewer"),
    N_("Email"),
    N_("Calendar"),
    N_("Archiving"),
    N_("Compression"),
    N_("Calculator"),
    N_("TextEditor"),
    N_("Clock"),
    N_("Core"),
    N_("Monitor"),
    N_("Spreadsheet"),
    N_("Science"),
    N_("Math"),
    N_("Video"),
    N_("Security"),
    N_("Photography"),
    N_("Scanning"),
    N_("Printing"),
    N_("P2P"),
    N_("Dictionary"),
    N_("Mixer"),
    N_("TrayIcon"),
    NULL
};

static
const gchar *
icon_by_path(gchar *path);

static gchar *
get_hash_key (const gchar * pre_key) {
    GString *gs = g_string_new (pre_key);
    gchar *key;
    key = g_strdup_printf ("%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    return key;
}

static
gchar * 
get_desktop_string(const gchar *key, const gchar *file){
	GKeyFile *key_file;
	GError *error=NULL;
	gchar *value;
	key_file=g_key_file_new ();

	if (!g_key_file_load_from_file (key_file, file, 0, &error)){
		DBG("get_desktop_string(): %s (%s)",error->message,file);
		g_error_free(error);
		return NULL;
	}

	if (!g_key_file_has_group(key_file, "Desktop Entry") ||
		!g_key_file_has_key(key_file, "Desktop Entry", key, NULL)){
	    g_key_file_free (key_file);
	    return NULL;
	}
	value=g_key_file_get_locale_string    (key_file,"Desktop Entry",key,NULL,&error);
	if (error){
		g_warning("%s (%s)",error->message,file);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	if (strcmp(key, "Exec")==0){
	    gchar *test_exec=g_strdup(value);
	    if (strchr(test_exec, ' ')) *strchr(test_exec, ' ') = 0;
	    gchar *command=g_find_program_in_path(test_exec);
	    if (!command) {
		DBG("get_desktop_string(): %s -> %s not found in path: %s\n", value, test_exec, file);
		g_free(value);
		value=NULL;
	    }
	    g_free(test_exec);
	    g_free(command);
	}

	return value;
}

static
int 
get_desktop_bool(const gchar *key, const gchar *file){
	GKeyFile *key_file;
	GError *error=NULL;
	gboolean value=FALSE;
	key_file=g_key_file_new ();
	if (!g_key_file_load_from_file (key_file, file, 0, &error)){
		DBG("get_desktop_bool(): %s (%s)",error->message,file);
		g_error_free(error);
		return FALSE;
	}
	if (!g_key_file_has_group(key_file, "Desktop Entry") ||
		!g_key_file_has_key(key_file, "Desktop Entry", key, NULL)){
	    g_key_file_free (key_file);
	    return FALSE;
	}
	value=g_key_file_get_boolean(key_file,"Desktop Entry",key,&error);
	if (error){
		g_warning("%s (%s)",error->message,file);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	return value;
}

static
gboolean
execute_dot_desktop(widgets_t *widgets_p, gchar *path, GList *target_list)
{
    gchar *exec=rfm_natural(PLUGIN_DIR, "dotdesktop", path, "item_exec");
    gchar *targets=NULL;



    if (target_list) {
	GList *tmp=target_list;
	for (; tmp && tmp->data; tmp=tmp->next){
	    gchar *esc_string=rfm_esc_string((gchar *)tmp->data);
	    gchar *p=g_strconcat((targets)?targets:"", " ", esc_string, NULL);
	    g_free(esc_string);
	    g_free(targets);
	    targets=p;
	}
    }    

    if (exec){
	gchar *command=g_strdup(exec);
	if (strchr(exec,' ')) *strchr(exec,' ')=0;

	// this is to chop off -f %F and %U stuff which is
	// not processed by double click (but should be DnD)
	if (strchr(command, '%')) {
	    if (targets!=NULL) {
		NOOP("targets=%s\n", targets);
		// replace the funky F or U or whatever for non nerdified format
		*(strchr(command, '%')+1)='s';
		// construct the full command line
		gchar *p=g_strdup_printf(command, targets);
		g_free(targets);
		g_free(command);
		command=p;
		NOOP("command=%s\n", command);
	    } else {
		if (strchr(command,' ')) *strchr(command,' ')=0;
	    }
	} else if (targets!=NULL) {
		// format specifier not found.
		// Assume default argv layout.
		// construct the full command line
		gchar *p=g_strdup_printf("%s %s", command, targets);
		g_free(targets);
		g_free(command);
		command=p;
	}


	gchar *fullexec=g_find_program_in_path(exec);
	if (fullexec && rfm_g_file_test(fullexec, G_FILE_TEST_IS_EXECUTABLE)){
	    rfm_show_text(widgets_p);
	    gboolean in_terminal=GPOINTER_TO_INT(
		    rfm_natural(PLUGIN_DIR, "dotdesktop", path, "exec_in_terminal"));
	    RFM_THREAD_RUN2ARGV (widgets_p, command, in_terminal);
	    g_free(command);
	    g_free(exec);
	    g_free(fullexec);
	    return TRUE;
	} 
	gchar *text=g_strdup_printf(_("File \"%s\" does not exist or is not executable."), exec);
	rfm_confirm(widgets_p, GTK_MESSAGE_ERROR, text,
		    NULL, NULL);
	g_free(command);
	g_free(exec);
	g_free(fullexec);
    }
    return FALSE;
}



static const gchar *
put_icon_in_hash(const gchar *path, const gchar *icon){
    if (!icon || !path) return NULL;
    const gchar *iconpath=NULL;
    if (!rfm_g_file_test(icon, G_FILE_TEST_EXISTS)){
	gchar *basename = (g_path_is_absolute(icon))? 
	    g_path_get_basename(icon) : g_strdup(icon);
	if (strchr(basename, '.')) *strrchr(basename, '.')=0;
	iconpath=ICON_get_filename_from_basename(basename);
	g_free(basename);
	if (!iconpath) {
	    DBG("put_icon_in_hash(): icon %s not found in icon path\n", icon);
	    return NULL;
	} 
    } else {
	iconpath = g_strdup(icon);
    }

    gchar *hash_key = NULL;

    hash_key = get_hash_key(path);
    g_static_mutex_lock(&icon_hash_mutex);
    if (g_hash_table_lookup(icon_hash, (gpointer) hash_key)){
	NOOP("c.put_icon_in_hash(): already in hash %s (%s) -> %s\n", 
		path, hash_key, iconpath);
	g_static_mutex_unlock(&icon_hash_mutex);	
	g_free(hash_key);
	return iconpath;
    } else {
	g_hash_table_insert (icon_hash, (gpointer) hash_key, (gpointer) iconpath);
    }
    g_static_mutex_unlock(&icon_hash_mutex);	
    g_free(hash_key);

    gchar *exec=get_desktop_string("Exec", path);
    if (exec) {
	if (strchr(exec, '%')) {
	    // replace the funky F or U or whatever for non nerdified format
	    *(strchr(exec, '%')+1)='s';
	}
	hash_key = get_hash_key(exec);
	g_static_mutex_lock(&exec_hash_mutex);
	if (g_hash_table_lookup(icon_exec_hash, (gpointer) hash_key)){
	    NOOP("a.put_icon_in_hash(): already in hash %s (%s) -> %s\n", 
		path, hash_key, iconpath);
	} else {
	    g_hash_table_insert (icon_exec_hash, (gpointer) hash_key,
		(gpointer) iconpath);
	}
	g_static_mutex_unlock(&exec_hash_mutex);
	g_free(hash_key);
	
	// Hash the plain command as well, if applicable
	if (strchr(exec, ' ')){
	    *(strchr(exec, ' ')) = 0;
	    hash_key = get_hash_key(exec);
	    NOOP("hash insert %s: %s -> %s\n", path, exec, iconpath);
	    g_static_mutex_lock(&exec_hash_mutex);
	    if (g_hash_table_lookup(icon_exec_hash, (gpointer) hash_key)){
		NOOP("b.put_icon_in_hash(): already in hash %s (%s) -> %s\n", 
		path, hash_key, iconpath);
	    } else {
		g_hash_table_insert (icon_exec_hash, (gpointer) hash_key,
		    (gpointer) iconpath);
	    }
	    g_static_mutex_unlock(&exec_hash_mutex);
	    g_free(hash_key);
	}
	g_free(exec);
    }
  
    return iconpath;
}

static gpointer
glob_dir_f( gpointer data){

    glob_t stack_glob_v;
    gchar *prefix[]={NULL, "/usr", "/usr/local", NULL};
    prefix[0] = (gchar *)g_get_user_data_dir();

    gint i;
    // don't hog up the disk on startup.
    g_thread_yield();
    for (i=0; i<5; i++) rfm_threadwait();
    //forcing bug condition now for 10 seconds...
    //sleep(10);

    for (i=0; prefix[i]; i++){
	gint flags;
	gchar *directory=g_strdup_printf("%s/share/applications/*.desktop", prefix[i]);
	if (i==0) flags = 0;
	else flags =  GLOB_APPEND;
	//gint glob_result = 
	glob(directory, flags, NULL, &stack_glob_v);
	g_free(directory);
    }

    gint gi;
    for (gi=0; gi < stack_glob_v.gl_pathc; gi++){
	NOOP("glob=%s\n", stack_glob_v.gl_pathv[gi]);
	GKeyFile *key_file;
	GError *error=NULL;
	key_file=g_key_file_new ();

	const gchar *key="Categories";
	if (!g_key_file_load_from_file (key_file, stack_glob_v.gl_pathv[gi], 0, &error)){
		NOOP("%s (%s)",error->message,stack_glob_v.gl_pathv[gi]);
		g_error_free(error);
		continue;
	}
	if (!g_key_file_has_group(key_file, "Desktop Entry") ||
		!g_key_file_has_key(key_file, "Desktop Entry", key, NULL)){
	    g_key_file_free (key_file);
		continue;
	}
	gchar *value=g_key_file_get_string    (key_file,"Desktop Entry",key,&error);
	if (error){
		g_warning("%s (%s)",error->message,stack_glob_v.gl_pathv[gi]);
		g_error_free(error);
	}
	g_key_file_free (key_file);
	gchar **values=rfm_split(value, ';');


// create array of standard categories, data is gslist
// foreach category add path to desktop file to category glist.
// categories will be the icons of the module,
// on each icon click, we will get the paths of all items in the categories for
//  the xfdir structure. No monitor enabled, or monitor globbed directories and reglob
//  on modifications....

	gchar **p=values;
	NOOP("value=%s\n", value);
	for (; p && *p; p++){
	    if (strlen(*p)==0) continue;
	    g_static_mutex_lock(&category_hash_mutex);
	    dotdesktop_t *dotdesktop_p=g_hash_table_lookup(category_hash, *p);
	    g_static_mutex_unlock(&category_hash_mutex);
	    if (dotdesktop_p == NULL) {
		// Add new category
		NOOP("new category=%s\n", *p);
		NOOP("    {N_(\"%s\"), N_(\"\")},\n", *p);
		gchar *cat=g_strdup(*p);
	    
		dotdesktop_p = (dotdesktop_t *)
		    malloc(sizeof(dotdesktop_t));
		if (!dotdesktop_p) g_error("malloc: %s", strerror(errno));
		g_static_mutex_lock(&category_hash_mutex);
		g_hash_table_insert(category_hash, cat, 
			dotdesktop_p);
		g_static_mutex_unlock(&category_hash_mutex);
		memset(dotdesktop_p, 0, sizeof(dotdesktop_t));
		dotdesktop_p->category=cat;

		dotdesktop_t *qq=dotdesktop_v;
		dotdesktop_p->icon=NULL;
		for (; qq && qq->category; qq++){
		    if (strcmp(qq->category, cat)==0) {
			dotdesktop_p->string=qq->string;
			dotdesktop_p->popup=qq->popup;
			dotdesktop_p->icon=qq->icon;
			break;
		    }
		}
		// If no icon is assigned, then use the icon of the
		// first dot desktop element found in that category...
		if (dotdesktop_p->icon==NULL) {
		    dotdesktop_p->icon = icon_by_path(stack_glob_v.gl_pathv[gi]);
		}
	        g_static_mutex_lock(&string_hash_mutex);
		category_list = 
			g_slist_prepend(category_list, dotdesktop_p);
		g_hash_table_insert(reverse_string_hash, _(cat), cat);
		if (dotdesktop_p->string) {
		    g_hash_table_insert(reverse_string_hash,
			    (void *)_(dotdesktop_p->string), cat);
		}
	        g_static_mutex_unlock(&string_hash_mutex);
	    }
	    // add item to category 
		
	    dotdesktop_p->application_list = 
			g_slist_prepend(dotdesktop_p->application_list,
				g_strdup(stack_glob_v.gl_pathv[gi]));

	}
	g_free(values);


    }
    globfree(&stack_glob_v);

    
    g_mutex_lock(glob_mutex);
    glob_done = TRUE;
    g_cond_broadcast(glob_signal);
    g_mutex_unlock(glob_mutex);
    NOOP("dotdesktop init... glob_done\n");

    return NULL;
}


static
void *
full_init(void){
    g_mutex_lock(glob_mutex);
    if (!glob_done) g_cond_wait(glob_signal, glob_mutex);
    g_mutex_unlock(glob_mutex);
    return GINT_TO_POINTER(1);
}

static 
void * 
private_get_xfdir(xfdir_t *xfdir_p){
    full_init();
    // 1.
    // glob /usr/share/applications and /usr/local/share/applications
    // for dotdesktop files
    // 2.
    // Create/append categories hash
    //   key:
    //      - category name
    //   data:
    //      - category name (translatable)
    //      - category gslist
    // 3.Create/append category gslist
    //   data:
    //      - name (locale dependent)
    //      - exec
    //      - comment (locale dependent)
    //      - create/add mimetype hash (this will feed applications module)
    //
    // 1.
    // glob /usr/share/applications and /usr/local/share/applications
    // for dotdesktop files

    //
    gint parent_category=0;
    
    record_entry_t *p_en=rfm_copy_entry(xfdir_p->en);
    if (p_en && p_en->st) {
	parent_category=p_en->st->st_uid;
	// this is very important, since xfdir_p->st gets
	// put into view_p->en->st, which is used for reload
	memcpy(xfdir_p->en->st, p_en->st, sizeof(struct stat));
    }


 
    if (parent_category==0) {
	// No parent. These are categories.
	xfdir_p->pathc=1; 
	gint i;

	xfdir_p->pathc = g_slist_length(category_list) + 1;
	NOOP("pathc=%d\n", (gint)xfdir_p->pathc);


	xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
	if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
	memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
	// Up is rodent root level.
	xfdir_p->gl[0].en = NULL;
	NOOP("up set to root level\n");
	xfdir_p->gl[0].pathv = g_strdup (g_get_host_name ());

	GSList *tmp=category_list;
	gint id=1;
	for (i=1;tmp && tmp->data; tmp=tmp->next,id++){
	    dotdesktop_t *category_p=tmp->data;
	    // Translation:
	    g_static_mutex_lock(&string_hash_mutex);
	    const gchar *string=g_hash_table_lookup(string_hash, 
		category_p->category);
	    g_static_mutex_unlock(&string_hash_mutex);
	    if (!string) string = category_p->category;

	    xfdir_p->gl[i].pathv = g_strdup(_(string));
	    xfdir_p->gl[i].en=rfm_mk_entry(0);
	    xfdir_p->gl[i].en->type=0; /* remove local-type attributes */
	    xfdir_p->gl[i].en->parent_module = MODULE_NAME;
	    xfdir_p->gl[i].en->module = MODULE_NAME;
	    xfdir_p->gl[i].en->path = g_strdup(_(string));

	    xfdir_p->gl[i].en->st = (struct stat*) malloc(sizeof(struct stat));
	    if (!xfdir_p->gl[i].en->st) g_error("malloc: %s", strerror(errno));
	    memset(xfdir_p->gl[i].en->st, 0, sizeof(struct stat));
	    xfdir_p->gl[i].en->st->st_uid=id;
	    NOOP("path=%s module=%s\n", 
			xfdir_p->gl[i].en->path, xfdir_p->gl[i].en->module);
	    i++;
	}
    } else {
	parent_category--;
	GSList *tmp = g_slist_nth(category_list, parent_category);
	GSList *list=NULL;
	if (tmp) {
	    dotdesktop_t *q=tmp->data;
	    NOOP("loading category %s\n", q->category); 
	    list=q->application_list;

	    xfdir_p->pathc = g_slist_length(list)+1;
	} else {
	    xfdir_p->pathc = 1;
	}
	NOOP("category=%d, pathc=%d\n", parent_category, (gint)xfdir_p->pathc);
	// up item is module root 
	// xfdir_p->pathc++;

	xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
	if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
	memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
	xfdir_p->gl[0].pathv = g_strdup(MODULE_LABEL);
	xfdir_p->gl[0].en=rfm_mk_entry(0);
	xfdir_p->gl[0].en->parent_module = MODULE_NAME;
	xfdir_p->gl[0].en->module = MODULE_NAME;
	xfdir_p->gl[0].en->st = NULL;	
	xfdir_p->gl[0].en->path=g_strdup(MODULE_LABEL);
        SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
	SET_UP_TYPE(xfdir_p->gl[0].en->type);

	gint i;
	for (i=1; list && list->data;list=list->next, i++){
	    gchar *path=list->data;
	    gchar *name=get_desktop_string("Name", path);
	    if (name) {
		xfdir_p->gl[i].pathv = name;
	    } else {
		xfdir_p->gl[i].pathv = g_path_get_basename(path);
	    }
	    xfdir_p->gl[i].en=rfm_stat_entry(path, 0);
	    xfdir_p->gl[i].en->type=0; 
	    xfdir_p->gl[i].en->module = MODULE_NAME;
	    xfdir_p->gl[i].en->path = g_strdup(path);
	    xfdir_p->gl[i].en->mimetype = g_strdup("application/x-desktop");

	    NOOP("category=%d path=%s\n",
		    parent_category, xfdir_p->gl[i].en->path);
	}

    }
    rfm_destroy_entry(p_en);
    return GINT_TO_POINTER(1);
}


static void
menu_exec(GtkMenuItem *m, gpointer data){
    gchar *path=data;
    widgets_t *widgets_p=g_object_get_data(G_OBJECT(m), "widgets_p");
    NOOP("execute %s", path);
    execute_dot_desktop(widgets_p, path, NULL);
}

	// Dotdesktop define icon for items.
static void *
populate_menuicons(gpointer data){
    g_static_mutex_lock(&popup_mutex);
    GSList **menulist=data;
    GSList *tmp = *menulist;
    for (;tmp && tmp->data; tmp=tmp->next){
	GtkWidget *menuitem = tmp->data;
	const gchar *path = g_object_get_data(G_OBJECT(menuitem), "path");
	gchar *hash_key = get_hash_key(path);
	g_static_mutex_lock(&icon_hash_mutex);
	const gchar *iconpath =
	    ((gchar *) g_hash_table_lookup (icon_hash, hash_key));
	g_static_mutex_unlock(&icon_hash_mutex);
	g_free(hash_key);
	if (!iconpath) {
	    gchar *icon=get_desktop_string("Icon", path);
	    if (!icon) icon=g_strdup("xffm/generic_executable");
	    iconpath=put_icon_in_hash(path, icon);
	    g_free(icon);
	}
GDK_THREADS_ENTER();
	// Put pixbuf for icon into menu item.
	rodent_mk_pixmap_menu((iconpath)?iconpath:"xffm/generic_executable", menuitem, MENU_PIXMAP);
GDK_THREADS_LEAVE();
    }
    g_slist_free(*menulist);
    g_free(menulist);
    g_static_mutex_unlock(&popup_mutex);
    return NULL;
}

static void *
populate_icon_hash_f(void *p){
     // wait for module initialization to complete...
    full_init();
    GSList *cat_list=category_list;
    for (; cat_list && cat_list->data; cat_list=cat_list->next){
	dotdesktop_t *dotdesktop_p=cat_list->data;
	GSList *app_list=dotdesktop_p->application_list;
	for (;app_list && app_list->data; app_list = app_list->next){
	    const gchar *path=app_list->data;
	    gchar *icon=get_desktop_string("Icon", path);
	    if (icon) {
		put_icon_in_hash(path, icon);
		g_free(icon);
	    }
	}
    }
	    
    return NULL;
}

static void *
populate_mimetype_hash_f(void *p){
     // wait for module initialization to complete...
    full_init();
    TRACE("populate_mimetype_hash_f()...\n");
    GSList *cat_list=category_list;
    for (; cat_list && cat_list->data; cat_list=cat_list->next){
	dotdesktop_t *dotdesktop_p=cat_list->data;
	GSList *app_list=dotdesktop_p->application_list;
	for (;app_list && app_list->data; app_list = app_list->next){
	    const gchar *path=app_list->data;
	    gchar *mimetypes=get_desktop_string("MimeType", path);
	    if (mimetypes) {
		gchar *command=get_desktop_string("Exec", path);
		if (command) {
		    if (strchr(command, '%')) {
			// replace the funky F or U or whatever for non nerdified format
			*(strchr(command, '%')+1)='s';
		    }
		    gchar **types = g_strsplit (mimetypes, ";", -1);
		    gchar **type_p=types;
		    for (;type_p && *type_p; type_p++){
			if (!strchr(*type_p, '/')) continue;
			NOOP("Appending mimecommand: %s --> %s\n", *type_p, command);
			rfm_rational(MODULE_DIR,"mime", (void *)(*type_p),
				(void *)command, "mime_append");
		    }		    
		    g_free(command);
		    g_strfreev(types);
		}
		
		g_free(mimetypes);
	    }
	}
    }
	    
    return NULL;
}

// This is a thread function, must have GDK mutex set for gtk commands...
static void 
populate_submenu ( widgets_t *widgets_p, const gchar *popup_name){
     // wait for module initialization to complete...
    full_init();
    GtkWidget *popup_widget=
	g_object_get_data(G_OBJECT(widgets_p->paper), popup_name);

    g_static_mutex_lock(&popup_mutex);

    GHashTable *populate_hash=g_hash_table_new(g_str_hash, g_str_equal);

    GtkWidget *w;
    GSList **menu_items_list =
	(GSList **)malloc(sizeof(GSList *));
    if (!menu_items_list) g_error("malloc: %s", strerror(errno));
    *menu_items_list=NULL;

    gint iv;
    for (iv=0; dotdesktop_v[iv].category != NULL; iv++){
	if (!dotdesktop_v[iv].popup) continue;
	// Find category pointer, category_p
	GSList *tmp=category_list;
	dotdesktop_t *category_p=NULL;
	for (;tmp && tmp->data; tmp=tmp->next){
	    dotdesktop_t *qqq = tmp->data;
	    if (strcmp(dotdesktop_v[iv].category, qqq->category)==0){
		category_p=tmp->data;
		break;
	    }
	}
	if (!category_p) {
	    NOOP("Category %s not found!\n", dotdesktop_v[iv].category);
	    continue;
	}
	NOOP("dotdesktop_popup loop: %s\n", category_p->category);
	gchar *hash_key = get_hash_key(category_p->icon);
	g_static_mutex_lock(&icon_hash_mutex);
	const gchar *iconpath = ((gchar *) g_hash_table_lookup (icon_hash, hash_key));
	g_static_mutex_unlock(&icon_hash_mutex);
	g_free(hash_key);
	if (!iconpath) iconpath = put_icon_in_hash(category_p->category, category_p->icon);
	
	g_static_mutex_lock(&string_hash_mutex);
	const gchar *string=g_hash_table_lookup(string_hash, 
		category_p->category);
	g_static_mutex_unlock(&string_hash_mutex);
	if (!string) string = category_p->category;

	// rodent_mk_menu autoprotects with GDK mutex when 
	// called from non main thread.
	w = rodent_mk_menu(
	    widgets_p,  		/* window */
	    _(string), 	/* label */
	    NULL,   		/* name */
	    popup_widget, 	/* parent */
	    NULL,		/* callback (or NULL)*/
	    category_p->icon);//iconpath); 	/* icon id*/
	GtkWidget *v;
	GSList *list=category_p->application_list;
	
	for (; list && list->data; list=list->next){
	    if (get_desktop_bool("NoDisplay", (gchar *)list->data)) continue;
	    
	    // Item will appear only in first menu category listed in dotdesktop file.
	    if (g_hash_table_lookup(populate_hash, list->data)){
		continue;
	    } else {
		g_hash_table_insert(populate_hash, list->data, GINT_TO_POINTER(1));
	    }
	    
	    // Static list of pointers to dotdesktop files
	    gchar *path=NULL;
	    GSList *tmp=path_list;
	    for (;tmp && tmp->data; tmp=tmp->next){
		if (strcmp((gchar *)list->data, (gchar *)tmp->data)==0){
		    path=tmp->data;
		}
	    }
	    if (!path){
		path = g_strdup((gchar *)list->data);
		path_list = g_slist_prepend(path_list, path);
	    }

	    // Dotdesktop exec command for menu item.
	    gchar *exec=get_desktop_string("Exec", path);
	    if (!exec) {
		DBG("populate_submenu(): %s: no exec!\n", path);
		continue;
	    }
	    if (strchr(exec,' ')) *strchr(exec,' ')=0;

	    // Dotdesktop defined name for item.
	    gchar *name=get_desktop_string("Name", path);
	    if (strcmp(name,exec)){
		gchar *text=g_strdup_printf("%s (%s)", name, exec);
		g_free(name);
		name=text;
	    } 
	    g_free(exec);

	    gchar *comment=get_desktop_string("Comment", path);
	    // Create menu item with retrieved name.
GDK_THREADS_ENTER();
	    v = gtk_image_menu_item_new_with_mnemonic (name);
	    if (comment){
		gtk_widget_set_tooltip_text(v, comment);
		g_free(comment);
	    }
	    g_object_set_data(G_OBJECT(v),"widgets_p",widgets_p);
	    g_object_set_data(G_OBJECT(v),"path",path);
GDK_THREADS_LEAVE();
	    *menu_items_list = g_slist_prepend(*menu_items_list, v);

	    // Connect signal and show.
GDK_THREADS_ENTER();
	    gtk_container_add (GTK_CONTAINER (w), v);
	    g_signal_connect ((gpointer) v, "activate", G_CALLBACK (menu_exec), path);
	    gtk_widget_show (v);
GDK_THREADS_LEAVE();
	    g_free(name);	
	}
GDK_THREADS_ENTER();
	gtk_widget_show(w);
GDK_THREADS_LEAVE();
    }
    // Clean up popup hash table 
    g_hash_table_destroy(populate_hash);
    // Update menuitems icons.
    THREAD_CREATE(populate_menuicons, menu_items_list, "populate_menuicons");

    g_static_mutex_unlock(&popup_mutex);
    return ;
	
}

typedef struct populate_submenu_t {
    widgets_t *widgets_p;
    gchar *id;
}populate_submenu_t;
static gpointer 
populate_submenu2 (gpointer data){
    populate_submenu_t *populate_submenu_p=data;
    populate_submenu(populate_submenu_p->widgets_p, populate_submenu_p->id);
    g_free(populate_submenu_p->id);
    g_free(populate_submenu_p);
    return NULL;
}

static void *
dotdesktop_double_click(void *p, void *q){
    record_entry_t *en=q;
    if (!en) {
	return NULL;
    }
    if (!en->path || !rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)) {
	return NULL;
    }
    widgets_t *widgets_p=p;
    execute_dot_desktop(widgets_p, en->path, NULL);
    return GINT_TO_POINTER(1); 
}

static void *
dotdesktop_entry_tip(void *p){
    record_entry_t *en=p;
    if (!en || !en->path ) return NULL;


    gchar *name=get_desktop_string("Name",en->path);
    gchar *generic_name=get_desktop_string("GenericName",en->path);
    gchar *exec=get_desktop_string("Exec",en->path);
    gboolean in_term=get_desktop_bool("Terminal",en->path);

    gchar *text = g_strconcat(
	    name,
	    (generic_name)?"\n(":"",
	    (generic_name)?generic_name:"",
	    (generic_name)?")":"",
	    "\n","\n", _("Command to run when clicked:"), " ", exec,
	    "\n","\n", _("Terminal application"), ": ",
	    (in_term)?_("Yes"):_("No"),
	    NULL);
    gchar *tip = rfm_utf_string(text);
    g_free(name);
    g_free(generic_name);
    g_free(exec);
    g_free(text);

    return (tip);
}

static
const gchar *
icon_by_path(gchar *path){
    gchar *hash_key = get_hash_key(path);
    g_static_mutex_lock(&icon_hash_mutex);
    const gchar *iconpath = ((gchar *) g_hash_table_lookup (icon_hash, hash_key));
    g_static_mutex_unlock(&icon_hash_mutex);
    g_free(hash_key);
    if (iconpath) {
	NOOP("%s: hashed icon=%s\n", path, iconpath);
	return (void *)iconpath;
    }
    gchar *icon=get_desktop_string("Icon", path);
    NOOP("%s: icon=%s\n", path, iconpath);
    iconpath=put_icon_in_hash(path, icon);
    if (!iconpath) {
	g_free(icon);
	return "application/x-desktop";
    }
    return iconpath;
}

static
void *dotdesktop_item_icon_id(void *p){
    if (p==NULL){
	return "xffm/generic_executable";
    } 
    record_entry_t *en=(record_entry_t *)p;
    if (IS_UP_TYPE(en->type)) return "xffm/stock_go-up";
    NOOP("resolving icon for %s (%s)\n", en->path, en->mimetype);
    gboolean icon_module=GPOINTER_TO_INT(rfm_void(MODULE_DIR, "icons", "module_active"));
    if (en->mimetype && strcmp(en->mimetype,"application/x-desktop")==0) {
	if (!icon_module) return "xffm/stock_file/composite/stock_yes";
	return (void *) icon_by_path(en->path);
    }
    // By this point we are dealing with a category.
    if (!icon_module) return "xffm/stock_directory";

    // Input value p is the translated category string.
    g_static_mutex_lock(&string_hash_mutex);
    const gchar *string = g_hash_table_lookup(reverse_string_hash, en->path);
    g_static_mutex_unlock(&string_hash_mutex);
    
    if (!string){
	DBG("dotdesktop_item_icon_id(): string==NULL (%s)\n", en->path);
	string=en->path;
    }
    // Try hard coded categories first.
    dotdesktop_t *category_p=dotdesktop_v;
    for (; category_p && category_p->category; category_p++){
	if (en && strcasecmp(string, category_p->category)==0){
	    if (!category_p->icon){
		break;
	    }

	    NOOP("got icon:%s\n",category_p->icon);
	    gchar *hash_key = get_hash_key(_(category_p->category));
	    g_static_mutex_lock(&icon_hash_mutex);
	    const gchar *iconpath = ((gchar *) g_hash_table_lookup (icon_hash, hash_key));
	    g_static_mutex_unlock(&icon_hash_mutex);
	    g_free(hash_key);
	    if (!iconpath) iconpath=put_icon_in_hash(_(category_p->category), category_p->icon);
	    if (iconpath) {
		NOOP("returning icon: %s\n", iconpath);
		return (void *)iconpath;
	    }
	}
    }
    
    // Soft coded categories: index is by translated string
    // category_hash is nontranslated, basic.

    g_static_mutex_lock(&category_hash_mutex);
    dotdesktop_t *dotdesktop_p=g_hash_table_lookup(category_hash, string);
    g_static_mutex_unlock(&category_hash_mutex);
    if (dotdesktop_p && dotdesktop_p->icon){
	return (void *)dotdesktop_p->icon;
    }
    NOOP("no icon for %s\n", string);
    if (en->st == NULL){
	// this is the module root item
	static gchar *icon=NULL;
	if (icon == NULL){
	    icon = g_strdup_printf("%s/icons/Rodent/scalable/places/folder-system.svg", PACKAGE_DATA_DIR);
	}
	return icon;
    }
    DBG("dotdesktop_item_icon_id(): could not find icon:%s\n",en->path);
    return "xffm/generic_executable";
}

static void *
dotdesktop_hide_local_menu_items (void *p, void *q) {
    //record_entry_t *en = q;
    widgets_t *widgets_p = p;
    if(!p)
        return NULL;


    gchar *symbols[] = {
        "duplicate_menuitem",
        "symlink_menuitem",
        "touch_menuitem",
        "rename_menuitem",
        "paste_menuitem",
	"sort1",
	"select_menu",
	"paste_menuitem",
	"edit_separator",
	"view_separator",
        NULL
    };
    gchar **pp;
    for(pp = symbols; *pp; pp++) {
	NOOP("hiding: %s\n", *pp);
        HIDE_IT (widgets_p->paper, *pp);
    }
    return GINT_TO_POINTER (1);
}

static off_t
basedir_sum (const gchar * dir) {
    off_t sum = 0;


    NOOP ("dotdesktop: recurse_basedir_sum(%s)\n", dir);
    if(rfm_g_file_test (dir, G_FILE_TEST_IS_DIR)) {
	if(!rfm_g_file_test (dir, G_FILE_TEST_IS_SYMLINK)) {
	    struct stat st;
	    if(stat (dir, &st) == 0) {
		sum += st.st_mtime;
		sum += st.st_dev;
	    }
	}
    }
    return sum;
}

static gpointer
monitor_f(gpointer data) {
    off_t status=0;
    gchar *prefix[]={PACKAGE_DATA_DIR, "/usr/share", "/usr/local/share", NULL};
    gchar *package_dir = g_build_filename(PACKAGE_DATA_DIR, "applications", NULL);
    while (1){
	g_mutex_lock(glob_mutex);
	if (!glob_done) g_cond_wait(glob_signal, glob_mutex);
	g_mutex_unlock(glob_mutex);
	// Test for reload condition
	off_t new_status=0;
	gint i;
	for (i=0; prefix[i]; i++){
	    gchar *directory=g_build_filename(prefix[i], "applications", NULL);
	    if (i == 0 || strcmp(directory,package_dir) != 0) {
		new_status += basedir_sum(directory);
	    }
	    g_free(directory);
	}
	if (status == 0) {
	    status = new_status;
	}
	
	NOOP("monitor_f(dotdesktop-module.i): status %ld = %ld\n",
		(glong) status, (glong)new_status);	
	if (status != new_status) {
	    status = new_status;
	    g_static_mutex_lock(&popup_mutex);
	    desktop_serial++;
	    g_mutex_lock(glob_mutex);
	    glob_done=FALSE;
	    g_mutex_unlock(glob_mutex);
	    // Clear old values:
	    GSList *tmp1=category_list;
	    for (;tmp1 && tmp1->data; tmp1=tmp1->next){
		dotdesktop_t *q=tmp1->data;
		GSList *tmp=q->application_list;
		for (; tmp && tmp->data; tmp=tmp->next){
		    g_free(tmp->data);
		}
		g_slist_free(q->application_list);
		q->application_list=NULL;
		g_static_mutex_lock(&category_hash_mutex);
		g_hash_table_steal(category_hash, q->category);
		g_static_mutex_unlock(&category_hash_mutex);
		g_free(q);
	    }
	    g_slist_free(category_list);
	    category_list=NULL;
	    // Set new values:
	    glob_dir_f(NULL);

	    // Inform other threads that everthing is cool.
	    g_mutex_lock(glob_mutex);
	    glob_done=TRUE;
	    g_cond_broadcast(glob_signal);
	    g_mutex_unlock(glob_mutex);
	    g_static_mutex_unlock(&popup_mutex);
	    DBG("monitor_f(dotdesktop-module.i): Reloaded applications...\n");
	}
	sleep(1);
    }
    g_free(package_dir);

    return NULL;
}


